import { JSONValue } from '@lumino/coreutils';
import { IObservableList, ObservableList } from './observablelist';
/**
 * An object which knows how to serialize and
 * deserialize the type T.
 */
export interface ISerializer<T> {
    /**
     * Convert the object to JSON.
     */
    toJSON(value: T): JSONValue;
    /**
     * Deserialize the object from JSON.
     */
    fromJSON(value: JSONValue): T;
}
/**
 * An observable list that supports undo/redo.
 */
export interface IObservableUndoableList<T> extends IObservableList<T> {
    /**
     * Whether the object can redo changes.
     */
    readonly canRedo: boolean;
    /**
     * Whether the object can undo changes.
     */
    readonly canUndo: boolean;
    /**
     * Begin a compound operation.
     *
     * @param isUndoAble - Whether the operation is undoable.
     *   The default is `false`.
     */
    beginCompoundOperation(isUndoAble?: boolean): void;
    /**
     * End a compound operation.
     */
    endCompoundOperation(): void;
    /**
     * Undo an operation.
     */
    undo(): void;
    /**
     * Redo an operation.
     */
    redo(): void;
    /**
     * Clear the change stack.
     */
    clearUndo(): void;
}
/**
 * A concrete implementation of an observable undoable list.
 */
export declare class ObservableUndoableList<T> extends ObservableList<T> implements IObservableUndoableList<T> {
    /**
     * Construct a new undoable observable list.
     */
    constructor(serializer: ISerializer<T>);
    /**
     * Whether the object can redo changes.
     */
    get canRedo(): boolean;
    /**
     * Whether the object can undo changes.
     */
    get canUndo(): boolean;
    /**
     * Begin a compound operation.
     *
     * @param isUndoAble - Whether the operation is undoable.
     *   The default is `true`.
     */
    beginCompoundOperation(isUndoAble?: boolean): void;
    /**
     * End a compound operation.
     */
    endCompoundOperation(): void;
    /**
     * Undo an operation.
     */
    undo(): void;
    /**
     * Redo an operation.
     */
    redo(): void;
    /**
     * Clear the change stack.
     */
    clearUndo(): void;
    /**
     * Handle a change in the list.
     */
    private _onListChanged;
    /**
     * Undo a change event.
     */
    private _undoChange;
    /**
     * Redo a change event.
     */
    private _redoChange;
    /**
     * Copy a change as JSON.
     */
    private _copyChange;
    private _inCompound;
    private _isUndoable;
    private _madeCompoundChange;
    private _index;
    private _stack;
    private _serializer;
}
/**
 * Namespace for ObservableUndoableList utilities.
 */
export declare namespace ObservableUndoableList {
    /**
     * A default, identity serializer.
     */
    class IdentitySerializer<T extends JSONValue> implements ISerializer<T> {
        /**
         * Identity serialize.
         */
        toJSON(value: T): JSONValue;
        /**
         * Identity deserialize.
         */
        fromJSON(value: JSONValue): T;
    }
}
