///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoQuery.cc
// ------------
// Cego query implementation         
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoQuery
// 
// Description: All modifying queries
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////


#include <lfcbase/BigDecimal.h>
#include <lfcbase/Datetime.h>

#include "CegoQueryException.h"
#include "CegoOutput.h"
#include "CegoQuery.h"
#include "CegoDatabaseFormater.h"

#include <string.h>
#include <stdlib.h>

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, CegoQuery::TransactionOp to)
{
    if ( to == CegoQuery::START )
	_mode = START_QUERY;
    else if ( to == CegoQuery::COMMIT )
	_mode = COMMIT_QUERY;
    else if ( to == CegoQuery::ROLLBACK )
	_mode = ROLLBACK_QUERY;
    
    _pGTM = pGTM;
    _tableSet = tableSet;
    _pPred = 0;
    _pSelect = 0;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName, const ListT<CegoField>& schema, const ListT<ListT<CegoExpr*> >& exprListArray)
{
    _mode = INSERT_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableSet = tableSet;
    _schema = schema;
    _pSelect = 0;
    _exprListArray = exprListArray;
    _pPred = 0;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName, const ListT<CegoField>& schema, CegoSelect *pSelect)
{
    _mode = INSERTBYSELECT_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableSet = tableSet;
    _schema = schema;
    _pSelect = pSelect;
    _pPred = 0;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName,  const Chain& tableAlias,  CegoPredDesc* pPred)
{
    _mode = DELETE_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableAlias = tableAlias;
    _tableSet = tableSet;
    _pPred = pPred;
    _pSelect = 0;    
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName, const Chain& tableAlias,  CegoPredDesc* pPred, ListT<CegoField>& updSchema, const ListT<CegoExpr*>& updList, const ListT<CegoReturnVar*> retVarList, bool returnOnFirst)
{
    _mode = UPDATE_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableAlias = tableAlias;
    _tableSet = tableSet;
    _pPred = pPred;
    _schema = updSchema;
    _pSelect = 0;
    _updList = updList;
    _retVarList = retVarList;
    _returnOnFirst = returnOnFirst;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName, bool doLock)
{
    if ( doLock )
	_mode = LOCK_QUERY;
    else
	_mode = UNLOCK_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableSet = tableSet;
    _pPred = 0;
    _pSelect = 0;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& tableName, const ListT<CegoAlterDesc>& alterList)
{
    _mode = ALTER_QUERY;
    _pGTM = pGTM;
    _tableName = tableName;
    _tableSet = tableSet;
    _alterList = alterList;
    _pPred = 0;
    _pSelect = 0;
}

CegoQuery::CegoQuery(CegoDistManager* pGTM, const Chain& tableSet, const Chain& objName, CegoObject::ObjectType objType, const Chain& newObjName )
{
    _mode = RENAME_QUERY;
    _pGTM = pGTM;
    _tableSet = tableSet;
    _objName = objName;
    _objType = objType;
    _newObjName = newObjName;
    _pPred = 0;
    _pSelect = 0;
}

CegoQuery::~CegoQuery()
{
    CegoExpr **pExpr = _updList.First();
    while ( pExpr )
    {
	delete *pExpr;
	pExpr = _updList.Next();
    }

    ListT<CegoExpr*> *pExprList = _exprListArray.First();
    while ( pExprList )
    {
	CegoExpr **pExpr = pExprList->First();
	while ( pExpr )
	{
	    delete *pExpr;
	    pExpr = pExprList->Next();
	}
	pExprList = _exprListArray.Next();
    }

    if ( _pPred )
	delete _pPred;
    if ( _pSelect )
	delete _pSelect;
    
}

void CegoQuery::setBlock(CegoProcBlock *pBlock)
{
    CegoExpr** pExpr = _updList.First();
    while ( pExpr )
    {
	(*pExpr)->setBlock(pBlock);
	pExpr = _updList.Next();
    }

    ListT<CegoExpr*> *pExprList = _exprListArray.First();
    while ( pExprList )
    {
	CegoExpr **pExpr = pExprList->First();
	while ( pExpr )
	{
	    (*pExpr)->setBlock(pBlock);
	    pExpr = pExprList->Next();
	}
	pExprList = _exprListArray.Next();
    }

    if ( _pPred )
    {
	_pPred->setBlock(pBlock);
    }
    return ;
}

Chain CegoQuery::execute(CegoProcBlock *pBlock)
{   
    if ( _pGTM == 0 )
	throw Exception(EXLOC, "No valid table manager set up");

    _affCount = 0;
    
    switch ( _mode )
    {
    case START_QUERY:
    {
	try
	{
	    _pGTM->startDistTransaction(_tableSet);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	return Chain("Transaction started");
    }
    case COMMIT_QUERY:
    {
	long opCount = 0;
	try
	{
	    opCount = _pGTM->endDistTransaction(_tableSet);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	Chain msg = Chain(opCount) + Chain(" operations committed");
	return msg;
    }
    case ROLLBACK_QUERY:
    {
	long opCount = 0;
	try
	{
	    opCount = _pGTM->rollbackDistTransaction(_tableSet);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	Chain msg = Chain(opCount) + Chain(" operations rollbacked");
	return msg;
    }	
    case LOCK_QUERY:
    {
	try
	{
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);    
	    _pGTM->lockTable(tabSetId, _tableName);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	Chain msg = Chain("Table ") + _tableName + Chain(" locked");
	return msg;
    }	
    case UNLOCK_QUERY:
    {
	try
	{
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    _pGTM->unlockTable(tabSetId, _tableName);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	Chain msg = Chain("Table ") + _tableName + Chain(" unlocked");
	return msg;
    }	
    case INSERT_QUERY:
    {
	
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}

	CegoTableObject oe;

	try
	{
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    if ( _pGTM->objectExists(tabSetId, _tableName, CegoObject::ALIAS ))
	    {
		CegoAliasObject ao;
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::ALIAS, ao);

		// get table object for alias
		_pGTM->getDistObject(_tableSet, ao.getTabName(), CegoObject::TABLE, oe);

		if (  _schema.isEmpty() )
		{
		    // set alias schema to original table schema
		    ao.setSchema(oe.getSchema());		    
		    // modify schema with alias definitons
		    ao.mapSchema();

		    // now we can setup target schema
		    _schema = ao.getSchema();
		}
				
		CegoQueryHelper::mapAliasSchema(_schema, ao.getAliasList());		
		
	    }
	    else
	    {
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::TABLE, oe);
	    }
	}
	catch ( Exception e )
	{

	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	
	if (  _schema.isEmpty() )
	{   
	    _schema = oe.getSchema();
	}
	else
	{
	    /* complete missing attributes in schema */
	    ListT<CegoField> checkSchema = oe.getSchema();

	    CegoField *pF = checkSchema.First();
	    while ( pF )
	    {
		CegoField *pCF = _schema.Find(*pF);
		if ( pCF == 0  )
		{
		    if ( pF->getValue().getValue() )
		    {
			_schema.Insert( *pF );
			
			ListT<CegoExpr*> *pExprList = _exprListArray.First();
			while ( pExprList )
			{			    
			    CegoFieldValue fv = pF->getValue();
			    CegoExpr* pExpr = new CegoExpr( new CegoTerm( new CegoFactor( fv )));
			    
			    pExprList->Insert(pExpr);
			    pExprList = _exprListArray.Next();
			}		    
		    }
		    else if ( pF->isNullable() )
		    {
			_schema.Insert( *pF );

			ListT<CegoExpr*> *pExprList = _exprListArray.First();
			while ( pExprList )
			{
			    CegoFieldValue fv;
			    CegoExpr* pExpr = new CegoExpr( new CegoTerm( new CegoFactor( fv )));
			    pExprList->Insert(pExpr);
			    pExprList = _exprListArray.Next();
			}			
		    }
		    else
		    {
			throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Attribute ") + pF->getAttrName() + " not nullable");    
		    }
		}
		else
		{
		    pCF->setType(pF->getType());
		    pCF->setLength(pF->getLength());
		    pCF->setDim(pF->getDim());
		}
		pF = checkSchema.Next();
	    }
	    pF = _schema.First();
	    while ( pF )
	    {
		CegoField *pUF = oe.getSchema().Find(*pF);
		if ( pUF )
		{
		    pF->setValue(pUF->getValue());
		    pF->setNullable(pUF->isNullable());
		}
		else
		{
		    throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Unknown attribute ") + pF->getAttrName() );
		}
		
		pF = _schema.Next();
	    }
	}
	
	try
	{
	    ListT< ListT<CegoField> > fva;
	    
	    ListT<CegoExpr*> *pExprList = _exprListArray.First();
	    while ( pExprList )
	    {	   
		CegoField* pFV = _schema.First();
		CegoExpr** pExpr = pExprList->First();
		
		while ( pFV && pExpr)
		{	    
		    // treat trigger value list
		    (*pExpr)->setFieldListArray(pBlock->getTriggerValueList());
		    (*pExpr)->setBlock(pBlock);

		    CegoFieldValue fv = (*pExpr)->evalFieldValue();

		    try
		    {
			CegoQueryHelper::prepareFieldValue(pFV, fv, _pGTM, oe.getTabSetId());
		    }
		    catch ( Exception e )
		    {
			Chain msg;
			e.pop(msg);			
			throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
		    }

		    pFV->setValue(fv);

		    pFV = _schema.Next();
		    pExpr = pExprList->Next();
		    
		}
		if ( pFV || pExpr )
		{		    
		    throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Mismatched argument count for value list"));
		}	    

		fva.Insert(_schema);

		pExprList = _exprListArray.Next();
	    }
	    
	    try
	    {				
		_pGTM->insertDistDataTable(oe, fva);
		
		_pGTM->getDBMng()->cleanCache( oe.getTabSetId(), CegoObject::TABLE, oe.getTabName());
		
		_affCount+=fva.Size();
	    }
	    catch ( Exception e )
	    {
		Chain msg;
		e.pop(msg);
		throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
	    }	       
	}
	catch ( CegoQueryException e )
	{
	    // clean clobs and blobs
	    
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    
	    CegoField *pF = _schema.First();
	
	    while ( pF)
	    {
		if ( pF->getValue().getType() == BLOB_TYPE && pF->getValue().getValue() != 0 )
		{
		    PageIdType pageId;
		    memcpy(&pageId, pF->getValue().getValue(), sizeof(PageIdType));
		    
		    _pGTM->decreaseBlobRef(tabSetId, pageId);
		}
		
		if ( pF->getValue().getType() == CLOB_TYPE && pF->getValue().getValue() != 0 )
		{
		    PageIdType pageId;
		    memcpy(&pageId, pF->getValue().getValue(), sizeof(PageIdType));
		    _pGTM->decreaseClobRef(tabSetId, pageId);
		}
		
		pF = _schema.Next();
	    }

	    throw e;
	   
	}
	
	return Chain(_affCount) + Chain(" tuples inserted");
    }
    case INSERTBYSELECT_QUERY:
    {
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}
 
	CegoTableObject oe;

	try
	{
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    if ( _pGTM->objectExists(tabSetId, _tableName, CegoObject::ALIAS ))
	    {
		CegoAliasObject ae;
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::ALIAS, ae);

		CegoQueryHelper::mapAliasSchema(_schema, ae.getAliasList());
		
		// get table object for alias
		_pGTM->getDistObject(_tableSet, ae.getTabName(), CegoObject::TABLE, oe);
	    }
	    else
	    {
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::TABLE, oe);
	    }
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}
	
	
	if (  _schema.isEmpty() )
	{   
	    _schema = oe.getSchema();
	}
	else
	{	
	    /* complete missing attributes in schema */
	    ListT<CegoField> checkSchema = oe.getSchema();
	    
	    CegoField *pF = checkSchema.First();
	    while ( pF )
	    {
		CegoField *pCF = _schema.Find(*pF);
		if ( pCF == 0  )
		{
		    if ( pF->getValue().getValue() )
		    {
			_schema.Insert( *pF );			    
		    }
		    else if ( pF->isNullable() )
		    {
			_schema.Insert( *pF );				
		    }
		    else
		    {
			throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Attribute ") + pF->getAttrName() + " not nullable");    
		    }
		}
		else
		{
		    pCF->setType(pF->getType());
		    pCF->setLength(pF->getLength());
		    pCF->setDim(pF->getDim());		   
		}
		pF = checkSchema.Next();	    
	    }
	    
	    pF = _schema.First();
	    while ( pF )
	    {
		CegoField *pUF = oe.getSchema().Find(*pF);
		if ( pUF )
		{
		    pF->setValue(pUF->getValue());
		    pF->setNullable(pUF->isNullable());
		}
		else
		{
		    throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Unknown attribute ") + pF->getAttrName() );
		}
		
		pF = _schema.Next();
	    }
	}
	
	int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);    
	ListT<CegoField> fl;
	bool moreTuple = false;
	
	try
	{
	    _pSelect->setProcBlock(pBlock);
	    _pSelect->setTabSetId(tabSetId);
	    _pSelect->prepare();	    
	    moreTuple = _pSelect->nextTuple(fl);		
	}
	catch ( Exception e )
	{
	    _pSelect->cleanUp();
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
	}

	ListT< ListT<CegoField> > fva;
	
	while ( moreTuple || fva.Size() > 0 )
	{	    
	    try
	    {
		if ( moreTuple )
		{
		    ListT<CegoField> insertSchema = _schema;
		    CegoField* pFV = insertSchema.First();
		    CegoField* pSF = fl.First();
		    
		    while ( pFV && pSF)
		    {		    
			CegoFieldValue fv = pSF->getValue();
			
			try
			{
			    CegoQueryHelper::prepareFieldValue(pFV, fv, _pGTM, oe.getTabSetId());
			}
			catch ( Exception e )
			{
			    Chain msg;
			    e.pop(msg);			
			    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
			}
			
			// we have to create a local copy, since the tuple are buffered in fva
			// and the corresponding data pages could be replaced in bufferpool ...
			pFV->setValue(fv.getLocalCopy()); 
			pFV = insertSchema.Next();
			pSF = fl.Next();	    
		    }
		    	           
		    fva.Insert(insertSchema);
		}
		
		if ( fva.Size() == MAX_CLUSTERED_INSERT || moreTuple == false)
		{
		    try
		    {
			_pGTM->insertDistDataTable(oe, fva);
			_affCount += fva.Size();
			fva.Empty();
		    }
		    catch ( Exception e )
		    {
			Chain msg;
			e.pop(msg);
			throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
		    }
		}
	    }
	    catch ( CegoQueryException e )
	    {		    
		// clean clobs and blobs
		
		int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
		
		ListT<CegoField> *pFLA = fva.First();
		while ( pFLA )
		{
		    CegoField *pF = pFLA->First();
		    
		    while ( pF)
		    {
			if ( pF->getValue().getType() == BLOB_TYPE && pF->getValue().getValue() != 0 )
			{
			    PageIdType pageId;
			    memcpy(&pageId, pF->getValue().getValue(), sizeof(PageIdType));			    
			    _pGTM->decreaseBlobRef(tabSetId, pageId);
			}
			
			if ( pF->getValue().getType() == CLOB_TYPE && pF->getValue().getValue() != 0 )
			{
			    PageIdType pageId;
			    memcpy(&pageId, pF->getValue().getValue(), sizeof(PageIdType));
			    _pGTM->decreaseClobRef(tabSetId, pageId);
			}
			
			pF = pFLA->Next();
		    }
		    pFLA=fva.Next();
		}
			    
		throw e;		
	    }
	    
	    if ( moreTuple )
	    {
		try
		{
		    moreTuple = _pSelect->nextTuple(fl);		
		}
		catch ( Exception e )
		{
		    _pSelect->cleanUp();
		    Chain msg;
		    e.pop(msg);
		    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
		}
	    }	    
	}    
	
	_pSelect->cleanUp();
	
	_pGTM->getDBMng()->cleanCache( oe.getTabSetId(), CegoObject::TABLE, oe.getTabName());	    
	
	return Chain(_affCount) + Chain(" tuples inserted");
    }
    case DELETE_QUERY:
    {
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}
	
	CegoTableObject oe;
	
	try 
	{	    
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    if ( _pGTM->objectExists(tabSetId, _tableName, CegoObject::ALIAS ))
	    {
		CegoAliasObject ae;
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::ALIAS, ae);

		CegoQueryHelper::mapAliasPredicate(_pPred, _tableAlias, ae.getAliasList());
		
		// get table object for alias
		_pGTM->getDistObject(_tableSet, ae.getTabName(), CegoObject::TABLE, oe);
	    }
	    else
	    {
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::TABLE, oe);
	    }
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);
	}

	try 
	{	    
	    oe.setTabAlias(_tableAlias);
	    _affCount = _pGTM->deleteDistDataTable(oe, _pPred, pBlock);
	    _pGTM->getDBMng()->cleanCache(oe.getTabSetId(), CegoObject::TABLE, oe.getTabName() );
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}

	return Chain(_affCount) + Chain(" tuples deleted");
    }
    case UPDATE_QUERY:
    {      
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}

	CegoTableObject oe;

	try 
	{
	    int tabSetId = _pGTM->getDBMng()->getTabSetId(_tableSet);
	    if ( _pGTM->objectExists(tabSetId, _tableName, CegoObject::ALIAS ))
	    {
		CegoAliasObject ae;
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::ALIAS, ae);

		CegoQueryHelper::mapAliasPredicate(_pPred, _tableAlias, ae.getAliasList());
		CegoQueryHelper::mapAliasSchema(_schema, ae.getAliasList());
		
		// get table object for alias
		_pGTM->getDistObject(_tableSet, ae.getTabName(), CegoObject::TABLE, oe);
	    }
	    else
	    {
		_pGTM->getDistObject(_tableSet, _tableName, CegoObject::TABLE, oe);
	    }
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}

	oe.setTabAlias(_tableAlias);
	
	CegoField *pF = _schema.First();
	while ( pF )
	{	    
	    if ( _tableName != _tableAlias )
	    {	    
		pF->setTableAlias(_tableAlias);
	    }

	    CegoField *pUF = oe.getSchema().Find(*pF);
	    if ( pUF )
	    {		
		pF->setLength(pUF->getLength());
		pF->setDim(pUF->getDim());		
		pF->setType(pUF->getType());
		pF->setNullable(pUF->isNullable());		
		pF->setValue(pUF->getValue());
	    }
	    else
	    {
		throw CegoQueryException(COREOP_EXCEP, EXLOC, Chain("Unknown attribute ") + pF->getAttrName() );
	    }
	    
	    pF = _schema.Next();
	}

	ListT<CegoField> returnList;
	
	try 
	{
	    //_affCount = _pGTM->updateDistDataTable(oe, _pPred, _schema, _updList, _retVarList.Size() > 0, returnList,  pBlock);
	    _affCount = _pGTM->updateDistDataTable(oe, _pPred, _schema, _updList, _returnOnFirst, returnList,  pBlock);
	    _pGTM->getDBMng()->cleanCache( oe.getTabSetId(), CegoObject::TABLE, oe.getTabName());
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);	    
	}

	/*
	CegoField *pX = returnList.First();
	while ( pX )
	{
	    cout << "Schema : " << pX->getAttrName() << " = " << pX->getValue().toChain() << endl;
	    pX = returnList.Next();
	}
	*/
	
	CegoReturnVar **pRV = _retVarList.First();
	while ( pRV ) 
	{	    
	    if ( pBlock )
	    {	
		CegoExpr *pExpr = (*pRV)->getExpr();

		// we need to create an additional reference to use the setFieldListArray method
		ListT<CegoField>* fl[2];
		// fl[0] = &_schema;
		fl[0] = &returnList;
		fl[1] = 0;
				
		pExpr->setFieldListArray(fl);
		CegoFieldValue fv = pExpr->evalFieldValue();
		pBlock->setValue((*pRV)->getVarName(), fv); 
	    }
	    pRV = _retVarList.Next();
	}

	return Chain(_affCount) + Chain(" tuples updated");
    }
    case ALTER_QUERY:
    {
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}
 
	CegoTableObject oe;

	try
	{
	    _pGTM->getDistObject(_tableSet, _tableName, CegoObject::TABLE, oe);
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	}

	try 
	{
	    _pGTM->alterDistDataTable(oe, _alterList);
	    _affCount++;
	    _pGTM->getDBMng()->cleanCache(oe.getTabSetId(), CegoObject::TABLE, oe.getTabName());
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);	    
	}
	    
	return Chain("Table altered");
    }
    case RENAME_QUERY:
    {
	if ( _pGTM->getAutoCommit() == false )
	{
	    if ( _pGTM->getDistTid(_tableSet) == 0 )
	    {
		_pGTM->startDistTransaction(_tableSet);
	    }
	}

	if ( _objType == CegoObject::TABLE )
	{
	    CegoTableObject oe;
	    
	    try
	    {		
		_pGTM->getDistObject(_tableSet, _objName, _objType, oe);	
	    }
	    catch ( Exception e )
	    {
		Chain msg;
		e.pop(msg);
		throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);    
	    }
	    _pGTM->getDBMng()->cleanCache(oe.getTabSetId(), CegoObject::TABLE, oe.getTabName() );	    
	}
	
	try 
	{
	    _pGTM->renameDistObject(_tableSet, _objName, _objType, _newObjName);
	    _affCount++;
	}
	catch ( Exception e )
	{
	    Chain msg;
	    e.pop(msg);
	    throw CegoQueryException(COREOP_EXCEP, EXLOC, msg);   
	}
	return Chain("Object renamed");	
    }
    }
}

long CegoQuery::getAffectedCount() const
{
    return _affCount;
}

Chain CegoQuery::toChain(const Chain& indent) const
{
    Chain s;
    switch ( _mode )
    {
    case START_QUERY:
    {
	s = Chain("start transaction");
	break;
    }
    case COMMIT_QUERY:
    {
	s = Chain("commit");
	break;
    }
    case ROLLBACK_QUERY:
    {
	s = Chain("rollback");
	break;
    }
    case LOCK_QUERY:
    {
	s = Chain("lock table ") + _tableName;
	break;
    }
    case UNLOCK_QUERY:
    {
	s = Chain("unlock table ") + _tableName;
	break;
    }
    case INSERT_QUERY:
    {	
	s = Chain("insert into ") + _tableName;

	if ( _schema.Size() > 0 )
	{
	    CegoField* pFV = _schema.First();
	    
	    s += " ( ";	
	    
	    while ( pFV )
	    {
		s += pFV->getAttrName();
		pFV = _schema.Next();
		if ( pFV  )
		    s += Chain(", ");
	    }
	    
	    s += Chain(" )"); 
	}
	s += Chain("\n") + indent + Chain("values ");

	ListT<CegoExpr*> *pExprList = _exprListArray.First();
	while ( pExprList )
	{
	    s += Chain("("); 
	
	    CegoExpr **pExpr = pExprList->First();
	    while ( pExpr )
	    {
		s += (*pExpr)->toChain();
		pExpr = pExprList->Next();
		if ( pExpr )
		    s += Chain(", ");
	    }
	    s += Chain(")");
	    pExprList = _exprListArray.Next();
	    if ( pExprList )
		s += Chain(", ");		
	}
	break;
    }
    case INSERTBYSELECT_QUERY:
    {
	s = Chain("insert into ") + _tableName;

	if ( _schema.Size() > 0 )
	{
	    CegoField* pFV = _schema.First();
	    
	    s += Chain(" ( ");	
	    
	    while ( pFV )
	    {
		s += pFV->getAttrName();
		pFV = _schema.Next();
		if ( pFV  )
		    s += Chain(", ");
	    }
	    
	    s += Chain(" )"); 
	}
	s += Chain("\n") + indent;
	s += _pSelect->toChain(indent);

	break;
    }
    case DELETE_QUERY:
    {
	s = Chain("delete from ") + _tableName;

	if ( _tableName != _tableAlias )
	    s += Chain(" ") + _tableAlias;

	if (_pPred)
	{
	    s += Chain(" where ");
	    
	    Chain addIndent;
	    for ( int i=1; i<s.length(); i++ )
		addIndent += Chain(" ");
	   
	    s += _pPred->toChain(indent + addIndent);
	}
	break;
    }
    case UPDATE_QUERY:
    {
	s = Chain("update ") + _tableName;

	if ( _tableName != _tableAlias )
	    s += Chain(" ") + _tableAlias;
	
	s += Chain(" set ");

	CegoField* pFV = _schema.First();
	CegoExpr** pExpr = _updList.First();
		    
	while ( pFV && pExpr)
	{	    
	    s += pFV->getAttrName() + Chain("=") + (*pExpr)->toChain();
	    pFV = _schema.Next();
	    pExpr = _updList.Next();
	    if ( pFV && pExpr )
		s += Chain(", ");
	}
	
	if (_pPred)
	{
	    s += Chain(" where ");
	    s += _pPred->toChain(indent);
	}

	if ( _retVarList.isEmpty() == false )
	{ 
	    s += Chain(" return ");
	    CegoReturnVar **pRV = _retVarList.First();
	    while ( pRV ) 
	    {
		s += (*pRV)->toChain();
		pRV = _retVarList.Next();
		if ( pRV ) 
		    s += Chain(",");
	    }
	}
	break;
    }
    case ALTER_QUERY:
    {
	s = Chain("alter ") + _tableName + Chain(" ");
	CegoAlterDesc *pAD = _alterList.First();
	while ( pAD ) 
	{
	    s += pAD->toChain();
	    pAD = _alterList.Next();
	    if ( pAD ) 
		s += ",";
	}
	break;
    }
    case RENAME_QUERY:
    {
	s = Chain("rename ");
	switch ( _objType )
	{
	case CegoObject::TABLE:
	{
	    s += Chain("table ");
	    break;
	}
	case CegoObject::AVLTREE:
	case CegoObject::PAVLTREE:
	case CegoObject::UAVLTREE:
	{
	    s += Chain("avl ");
	    break;
	}
	case CegoObject::BTREE:
	case CegoObject::PBTREE:
	case CegoObject::UBTREE:
	{
	    s += Chain("btree ");
	    break;	    
	}
	case CegoObject::FKEY:
	{
	    s += Chain("key ");
	    break;
	}
	case CegoObject::PROCEDURE:
	{
	    s += Chain("procedure ");
	    break;
	}
	case CegoObject::VIEW:
	{
	    s += Chain("view ");
	    break;
	}
	case CegoObject::CHECK:
	{
	    s += Chain("check ");
	    break;	    
	}
	case CegoObject::SYSTEM:
	case CegoObject::TRIGGER:
	case CegoObject::ALIAS:
	case CegoObject::RBSEG:
	case CegoObject::JOIN:
	case CegoObject::UNDEFINED:	    	    	   	 	    
	    throw Exception(EXLOC, "Invalid object");
	}
	s += _objName + Chain(" to ") + _newObjName;

	break;
    }
    }
    return s;    
}

Chain CegoQuery::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatQuery(_mode, _tableName, _tableAlias, _schema,  _exprListArray,  _updList, _retVarList, _alterList, _pPred, _pSelect, _objName, _objType, _newObjName);
}
