///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAttrCond.cc
// ---------------
// Cego database table field implementation
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAttrCond
//
// Description: Attribute condition description used for index cursors
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoAttrCond.h"

// POSIX INCLUDES
#include <stdlib.h>
#include <string.h>

CegoAttrCond::CegoAttrCond()
{
}

CegoAttrCond::~CegoAttrCond()
{
}

Chain CegoAttrCond::getId() const
{
    Chain s;
    
    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {	
	s += pAC->getId();
	pAC = _attrCompSet.Next();
	if ( pAC )
	    s += Chain("&");
    }
    return s;
}

void CegoAttrCond::add(const CegoAttrComp& attrComp)
{
    CegoAttrComp *pAC;
    if ( (pAC = _attrCompSet.Find(attrComp)) == 0 )
    {	
	_attrCompSet.Insert(attrComp);
    }
    /*
    else
    {
	
	cout << "Adding " << attrComp << " failed, already exists  " << *pAC << endl;
	cout << "T1 = " << attrComp.getTableName() << endl;
	cout << "T2 = " << pAC->getTableName() << endl;
	
    }
    */
}

int CegoAttrCond::numComp() const
{
    return _attrCompSet.Size();
}

int CegoAttrCond::getStrength() const
{
    CegoAttrComp *pAC = _attrCompSet.First();
    if ( pAC )
    {	
	if ( pAC->getCompMode() == CegoAttrComp::BTWN )
	    return 4;
	switch ( pAC->getComparison() )
	{
	case EQUAL:
	    return 5;
	case NOT_EQUAL:
	    return 1;
	case LESS_THAN:
	    return 3;
	case MORE_THAN:
	    return 3;
	case LESS_EQUAL_THAN:
	    return 2;
	case MORE_EQUAL_THAN:
	    return 2;
	}
    }
    return 0;
}

const TreeT<CegoAttrComp>& CegoAttrCond::getAttrCompSet() const
{
    return _attrCompSet;
}

CegoAttrCond CegoAttrCond::getFilterCond(const ListT<CegoField>& fl, bool ignoreNullComp) const
{
    CegoAttrCond ac;

    CegoField *pF = fl.First();
    while ( pF)
    {	
	CegoAttrComp *pAC = _attrCompSet.First();
	while ( pAC )
	{
	    if ( pF->getTableAlias() == pAC->getTableName() && pF->getAttrName() == pAC->getAttrName() )
	    {
		if ( ! ( ignoreNullComp && pAC->isNullComp() ) )
		    ac.add(*pAC);
		/*
		else
		    cout << "Ignoring comp " << *pAC << endl;
		*/
	    }
	    pAC = _attrCompSet.Next();
	}

	pF = fl.Next();
    }
    return ac;
}

CegoAttrCond CegoAttrCond::getIndexCond(const ListT<CegoField>& fl) const
{ 
    CegoAttrCond ac;
 
    int pos = 0;
    CegoField *pF = fl.First();
 
    while ( pF  )
    {
        CegoAttrComp *pAC = _attrCompSet.First();
        while ( pAC )
        {
            if ( pF->getAttrName() == pAC->getAttrName()
                 && ( pAC->getCompMode() == CegoAttrComp::VAL
                      || pAC->getCompMode() == CegoAttrComp::ATTR
                      || pAC->getCompMode() == CegoAttrComp::BTWN ) )
            {
                pAC->setPos(pos);
                ac.add(*pAC);
            }
            pAC = _attrCompSet.Next();
        }
        pF = fl.Next();
        pos++;
    }
 
    return ac;
}

bool CegoAttrCond::setup(const ListT<CegoField>& fl)
{   
    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {
	if ( pAC->getCompMode() == CegoAttrComp::ATTR )
	    if ( pAC->setup(fl) == false )
		return false;
	pAC = _attrCompSet.Next();
    }
    return true;
}

bool CegoAttrCond::setup(ListT<CegoField>** pJoinBuf, int offset)
{
    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {
	pAC->reset();
	if ( pAC->getCompMode() == CegoAttrComp::ATTR || pAC->getCompMode() == CegoAttrComp::BTWN )
	{	  	    
	    if ( pAC->setup(pJoinBuf, offset) == false )
	    {
		return false;
	    }
	   
	}
	pAC = _attrCompSet.Next();
    }
    return true;
}

bool CegoAttrCond::setup(ListT<CegoField>** pParentJoinBuf, int parentOffset, ListT<CegoField>** pJoinBuf, int offset)
{    
    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {
	pAC->reset();
	if ( pAC->getCompMode() == CegoAttrComp::ATTR || pAC->getCompMode() == CegoAttrComp::BTWN )
	{
	    if ( pAC->isParentSetup() )
	    {
		if ( pAC->setup(pParentJoinBuf, parentOffset) == false )
		{
		    return false;
		}
	    }
	    else
	    {
		if ( pAC->setup(pJoinBuf, offset) == false )
		{
		    if ( pParentJoinBuf )
		    {
			if ( pAC->setup(pParentJoinBuf, parentOffset) == false )
			{
			    return false;
			}
			else
			{
			    pAC->setParentSetup();	    
			}
		    }
		    else
		    {
			return false;
		    }
		}
	    }
	}
	pAC = _attrCompSet.Next();
    }
    return true;
}

void CegoAttrCond::asConjunctionList(const ListT<CegoExpr*>& exprList, ListT<CegoPredDesc*>& conjunctionList, ListT<CegoField>** pFLA) const
{
    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {
	CegoPredDesc *pP;

	CegoExpr *pExpr = getExpressionForAlias(exprList, pAC->getAttrName() );

	if ( pExpr == 0 )
	{
	    pExpr = new CegoExpr(new CegoTerm(new CegoFactor( new CegoAttrDesc(pAC->getAttrName()))));
	    pExpr->setExternalFieldListArray(pFLA);
	}

	if ( pAC->getCompMode() == CegoAttrComp::ATTR && pAC->isSetup() == false )   
	{
	    CegoExpr* pExternExpr = new CegoExpr(new CegoTerm(new CegoFactor( pAC->getAttrDesc().clone())));
	    pExternExpr->setExternalFieldListArray(pFLA);
	    pP = new CegoPredDesc(pExpr,
				  pExternExpr,
				  pAC->getComparison());
	    
	}
	else if ( pAC->getCompMode() == CegoAttrComp::VAL || 
		  ( pAC->getCompMode() == CegoAttrComp::ATTR && pAC->isSetup() ) )
	{
	    pP = new CegoPredDesc(pExpr,
				  new CegoExpr(new CegoTerm(new CegoFactor( pAC->getFieldValue()))),
				  pAC->getComparison());
	}
	else if ( pAC->getCompMode() == CegoAttrComp::BTWN )
	{
	    pP = new CegoPredDesc(pExpr,
				  new CegoExpr(new CegoTerm(new CegoFactor( pAC->getFieldValue()))),
				  new CegoExpr(new CegoTerm(new CegoFactor( pAC->getFieldValue2()))));
	}
	else if ( pAC->getCompMode() == CegoAttrComp::ISLIKE )
	{
	    pP = new CegoPredDesc(pExpr,
				  pAC->getPattern(),
				  false);
	}
	else if ( pAC->getCompMode() == CegoAttrComp::ISNOTLIKE )
	{
	    pP = new CegoPredDesc(pExpr,
				  pAC->getPattern(),
				  true);
	}

	conjunctionList.Insert(pP);
	
	pAC = _attrCompSet.Next();
    }	
}

CegoExpr* CegoAttrCond::getExpressionForAlias(const ListT<CegoExpr*>& exprList, const Chain& alias) const
{
    CegoExpr **pExpr = exprList.First();
    while ( pExpr )
    {
	if ( (*pExpr)->getAlias() == alias )
	    return (*pExpr)->clone();
	pExpr = exprList.Next();
    }
    return 0;
}

CegoAttrCond::IndexMatch CegoAttrCond::checkIndex(const ListT<CegoField>& schema) const
{
    int numFound=0;

    bool hasLeak=false;
    CegoField *pF = schema.First();

    while ( pF && hasLeak == false )
    {
	bool isFound=false;	
	CegoAttrComp *pComp = _attrCompSet.First();	
	while ( pComp ) 
	{
	    if ( pComp->getAttrName() == pF->getAttrName() 
		 && ( pComp->getCompMode() == CegoAttrComp::VAL 
		      || pComp->getCompMode() == CegoAttrComp::ATTR
		      || pComp->getCompMode() == CegoAttrComp::BTWN ) )
	    {		
		numFound++;
		isFound=true;
		pComp = _attrCompSet.Next();
	    }
	    else
	    {		
		pComp = _attrCompSet.Next();
	    }
	}
	
	if ( isFound == true )
	    hasLeak=false;
	else
	    hasLeak=true;

	pF = schema.Next();
    }

    if ( numFound == _attrCompSet.Size() )
	return FULL;

    if ( numFound < _attrCompSet.Size() && numFound > 0 )
	return PART;
    
    return INAPP;
}
    
void CegoAttrCond::setIdxSchema(ListT<CegoField>& schema)
{
    _idxSchema = schema;
}

void CegoAttrCond::setPrimaryComparison(CegoComparison comp)
{
    CegoAttrComp* pAC = _attrCompSet.First();
    if ( pAC )
    {
	pAC->setComparison(comp); 
    }
    else
    {
	throw Exception(EXLOC, "Cannot set primary comparison");
    }
}

CegoAttrComp::CompMode CegoAttrCond::getPrimaryCompMode() const
{
    CegoAttrComp* pAC = _attrCompSet.First();
    if ( pAC )
    {
	return pAC->getCompMode();
    }
    else
    {
	throw Exception(EXLOC, "Cannot get primary comp mode");
    }
}

CegoComparison CegoAttrCond::getPrimaryComparison() const
{
    CegoAttrComp* pAC = _attrCompSet.First();
    if ( pAC )
    {
	return pAC->getComparison();
    }
    else
    {
	throw Exception(EXLOC, "Cannot get primary comparison");
    }
}

bool CegoAttrCond::diff( const CegoAttrCond& ac) const
{
    if ( ac._attrCompSet.Size() != _attrCompSet.Size() )
	return false;

    CegoAttrComp* pAC = ac._attrCompSet.First();
    while ( pAC  )
    {
	CegoAttrComp* pAC2 = findComp(pAC);
	if ( pAC2  )
	{
	    if ( pAC->getFieldValue() != pAC2->getFieldValue() ) 
		return false;
	}
	else
	{
	    return false;
	}
	pAC = ac._attrCompSet.Next();
    }

    pAC = _attrCompSet.First();
    while ( pAC  )
    {
	CegoAttrComp* pAC2 = ac.findComp(pAC);
	if ( pAC2  )
	{
	    if ( pAC->getFieldValue() != pAC2->getFieldValue() ) 
		return false;
	}
	else
	{
	    return false;
	}
	pAC = _attrCompSet.Next();
    }
    return true;   
}

CegoAttrComp* CegoAttrCond::findComp(CegoAttrComp* pAttrComp) const
{
    // for searching attribute comparison, we use a dedicated loop, since for TreeT, just < and > operators are used.
    // for comparison we have to use the = operator
    
    CegoAttrComp* pAC = _attrCompSet.First();
    while ( pAC )
    {
	if ( *pAC == *pAttrComp )
	    return pAC;
	pAC = _attrCompSet.Next();
    }
    return 0;
}

void CegoAttrCond::update( const CegoAttrCond& ac )
{    
    CegoAttrComp* pAC = _attrCompSet.First();
    while ( pAC  )
    {
	CegoAttrComp* pAC2 = ac.findComp(pAC);
	if ( pAC2  )
	{
	    pAC->setFieldValue(pAC2->getFieldValue());
	    if ( pAC->getCompMode() == CegoAttrComp::BTWN )
		pAC->setFieldValue2(pAC2->getFieldValue2());
	}
	else
	{
	    throw Exception(EXLOC, Chain("Cannot set up diff for attribute condition"));
	}
	pAC = _attrCompSet.Next();
    }
}

CegoAttrCond& CegoAttrCond::operator = ( const CegoAttrCond& ac)
{
    _idxSchema = ac._idxSchema;
    _attrCompSet = ac._attrCompSet;
    return (*this);
}

bool CegoAttrCond::operator == ( const CegoAttrCond& ac) const
{
    if ( ac._attrCompSet.Size() != _attrCompSet.Size() )
	return false;

    CegoAttrComp* pAC = ac._attrCompSet.First();
    while ( pAC  )
    {
	if ( findComp(pAC) == 0 )
	    return false;
	pAC = ac._attrCompSet.Next();
    }

    pAC = _attrCompSet.First();
    while ( pAC  )
    {
	if ( ac.findComp(pAC) == 0 )
	    return false;	
	pAC = _attrCompSet.Next();
    }
    return true;   
}

bool CegoAttrCond::operator != ( const CegoAttrCond& ac) const
{
    if ( *this == ac )
	return false;
    return true;
}

CegoAttrCond operator + ( const CegoAttrCond& ac1, const CegoAttrCond& ac2 )
{
    CegoAttrCond ac;
    CegoAttrComp *pComp;
    pComp = ac1._attrCompSet.First();    
    while ( pComp )
    {
	ac.add(*pComp);
	pComp = ac1._attrCompSet.Next();
    }
    pComp = ac2._attrCompSet.First();    
    while ( pComp )
    {
	ac.add(*pComp);
	pComp = ac2._attrCompSet.Next();
    }
    return ac;
}

Chain CegoAttrCond::toChain() const
{
    Chain s;

    CegoAttrComp *pAC = _attrCompSet.First();
    while ( pAC )
    {	
	s += pAC->toChain();
	pAC = _attrCompSet.Next();
	if ( pAC )
	    s += Chain(" and ");
    }
    return s;
}

ostream& operator << (ostream& s, const CegoAttrCond& ac)
{
    s << ac.toChain();
    return s;
}
