/* Copyright (c) 2002-2005  Michael Stumpf  <mistumpf@de.pepperl-fuchs.com>
   Copyright (c) 2006,2008  Dmitry Xmelkov

   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
   * Neither the name of the copyright holders nor the names of
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

#if !defined(__AVR_TINY__)

#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdlib.h>
#include <stdint.h>

#include <avr/pgmspace.h>
#include <avr/flash.h>
#include <bits/attribs.h>

#include "sectionname.h"
#include "alias.h"
#include "best_as.h"

#define ARRAY_SIZE(X) (sizeof(X) / sizeof(*X))

#define AS __BEST_AS

#define INFINITYf (__builtin_inff())
#define NANf      (__builtin_nanf(""))

/* Only GCC 4.2 calls the library function to convert an unsigned long
   to float.  Other GCC-es (including 4.3) use a signed long to float
   conversion along with a large inline code to correct the result.	*/
extern float __floatunsisf (uint32_t);

#ifdef __AVR_HAVE_ELPM__
#define READ_FL(x) pgm_read_float_far ((uintptr24_t) x)
#define S_STRNCASECMP "strncasecmp_PF"
#else
#define READ_FL(x) (*(x))
#define S_STRNCASECMP "strncasecmp_P"
#endif

int strncasecmp_AS (const char*, const AS char*, size_t) __asm(S_STRNCASECMP);

/* In libc/stdio/pwr_10.c */
extern const AS float __avrlibc_pwr_p10[6];
extern const AS float __avrlibc_pwr_m10[6];

#define LAST_PWR (ARRAY_SIZE (__avrlibc_pwr_p10) - 1)

static __ATTR_ALWAYS_INLINE__ int
mulhi10 (int i)
{
    register int r24 __asm("24") = i;
    __asm ("%~call __mulhi_const_10"
           : "+r" (r24) :: "22", "23");
    return r24;
}

static __ATTR_ALWAYS_INLINE__ uint32_t
mulsi10 (uint32_t i)
{
    register uint32_t r22 __asm("22") = i;
    __asm ("%~call __mulsi_const_10"
           : "+r" (r22) :: "20", "21", "27"
#ifndef __AVR_HAVE_MUL__
           , "26"
#endif
        );
    return r22;
}

// libc/stdlib/pstr_inf.c
extern const AS char __pstr_inf[3];
extern const AS char __pstr_inity[5];
extern const AS char __pstr_nan[3];

/**  The strtof() function converts the initial portion of the string pointed
     to by \a nptr to \c float representation.

     The expected form of the string is an optional plus ( \c '+' ) or minus
     sign ( \c '-' ) followed by a sequence of digits optionally containing
     a decimal-point character, optionally followed by an exponent.  An
     exponent consists of an \c 'E' or \c 'e', followed by an optional plus
     or minus sign, followed by a sequence of digits.

     Leading white-space characters in the string are skipped.

     The strtof() function returns the converted value, if any.

     If \a endptr is not \c NULL, a pointer to the character after the last
     character used in the conversion is stored in the location referenced by
     \a endptr.

     If no conversion is performed, zero is returned and the value of
     \a nptr is stored in the location referenced by \a endptr.

     If the correct value would cause overflow, plus or minus \c INFINITY is
     returned (according to the sign of the value), and \c ERANGE is stored
     in \c errno.  If the correct value would cause underflow, zero is
     returned and \c ERANGE is stored in \c errno.
 */
ATTRIBUTE_CLIB_SECTION
float
strtof (const char *nptr, char **endptr)
{
    union
    {
	uint32_t u32;
	float flt;
    } x;
    uint8_t c;
    uint8_t flag = 0;

#define FL_MINUS    0x01	/* number is negative	*/
#define FL_ANY	    0x02	/* any digit was read	*/
#define FL_OVFL	    0x04	/* overflow was		*/
#define FL_DOT	    0x08	/* decimal '.' was	*/
#define FL_MEXP	    0x10	/* exponent 'e' is neg.	*/

    if (endptr)
	*endptr = (char *)nptr;

    do
	c = *nptr++;
    while (isspace (c));

    if (c == '-')
    {
	flag = FL_MINUS;
	c = *nptr++;
    }
    else if (c == '+')
	c = *nptr++;

    if (!strncasecmp_AS (nptr - 1, __pstr_inf, sizeof (__pstr_inf)))
    {
	nptr += sizeof (__pstr_inf) - 1;
	if (!strncasecmp_AS (nptr, __pstr_inity, sizeof (__pstr_inity)))
	    nptr += sizeof (__pstr_inity);
	if (endptr)
	    *endptr = (char*) nptr;
	return flag & FL_MINUS ? -INFINITYf : +INFINITYf;
    }

    /* NAN() construction is not realised.
       Length would be 3 characters only. */
    if (!strncasecmp_AS (nptr - 1, __pstr_nan, sizeof (__pstr_nan)))
    {
	if (endptr)
	    *endptr = (char*) nptr + sizeof (__pstr_nan) - 1;
	return NANf;
    }

    x.u32 = 0;
    int exp = 0;

    for (;;)
    {
	c -= '0';

	if (c <= 9)
	{
	    flag |= FL_ANY;
	    if (flag & FL_OVFL)
	    {
		if (!(flag & FL_DOT))
		    exp += 1;
	    }
	    else
	    {
		if (flag & FL_DOT)
		    exp -= 1;
		/* x.u32 = x.u32 * 10 + c	*/
		x.u32 = mulsi10 (x.u32) + c;
		if (x.u32 >= (UINT32_MAX - 9) / 10)
		    flag |= FL_OVFL;
	    }
	}
	else if (c == (('.'-'0') & 0xff) && !(flag & FL_DOT))
	    flag |= FL_DOT;
	else
	    break;

	c = *nptr++;
    }

    if (c == (('e'-'0') & 0xff) || c == (('E'-'0') & 0xff))
    {
	c = *nptr++;
	int i = 2;
	if (c == '-')
	{
	    flag |= FL_MEXP;
	    c = *nptr++;
	}
	else if (c == '+')
	    c = *nptr++;
	else
	    i = 1;
	c -= '0';
	if (c > 9)
	    nptr -= i;
	else
	{
	    i = 0;
	    do
	    {
		if (i < 3200)
		    i = mulhi10 (i) + c;	/* i = 10*i + c	*/
		c = *nptr++ - '0';
	    } while (c <= 9);
	    if (flag & FL_MEXP)
		i = -i;
	    exp += i;
	}
    }

    if ((flag & FL_ANY) && endptr)
	*endptr = (char *)nptr - 1;

    x.flt = __floatunsisf (x.u32);		/* manually */
    if ((flag & FL_MINUS) && (flag & FL_ANY))
	x.flt = -x.flt;

    if (x.flt != 0)
    {
	const AS float *f_pwr;
	if (exp < 0)
	{
	    f_pwr = __avrlibc_pwr_m10 + LAST_PWR;
	    exp = -exp;
	}
	else
	    f_pwr = __avrlibc_pwr_p10 + LAST_PWR;

	for (int pwr = 1 << LAST_PWR; pwr; pwr >>= 1, --f_pwr)
	    for (; exp >= pwr; exp -= pwr)
		x.flt *= READ_FL (f_pwr);

	if (!isfinitef(x.flt) || x.flt == 0)
	    errno = ERANGE;
    }

    return x.flt;
}

DALIAS (strtof)
double strtod (const char*, char**);

LALIAS (strtof)
long double strtold (const char*, char**);

#endif // AVR_TINY
