%feature("docstring") OT::Interval
"Numerical interval.

Available constructors:
    Interval(*dim*)

    Interval(*lowerBound, upperBound, finiteLowerBound, finiteUpperBound*)

Parameters
----------
dim : int, :math:`dim \\geq 0`
    Dimension of the interval. If only *dim* is mentioned, it leads to create
    the finite interval :math:`[0, 1]^{dim}`.
    By default, an empty interval is created.
lowerBound, upperBound : float or sequence of float of dimension *dim*
    Define the interval
    :math:`[lowerBound_0, upperBound_0]\\times \\dots \\times [lowerBound_{dim-1}, upperBound_{dim-1}]`.
    It is allowed to have :math:`lowerBound_i \\geq upperBound_i` for some
    :math:`i`: it simply defines an empty interval.
    The *lowerBound* and the *upperBound* must be of the same type. If
    *finiteLowerBound* and *finiteUpperBound* are mentioned, they must be
    sequences.
finiteLowerBound : sequence of bool of dimension *dim*
    Flags telling for each component of the lower bound whether it is finite or
    not.
finiteUpperBound : sequence of bool of dimension *dim*
    Flags telling for each component of the upper bound whether it is finite or
    not.

Notes
-----
The meaning of a flag is: if flag :math:`i` is *True*, the corresponding
component of the given bound is finite and its value is given by bound
:math:`i`. If not, the corresponding component is infinite and its value is
either :math:`-\\infty` if bound :math:`i < 0` or :math:`+\\infty` if bound
:math:`i \\geq 0`.

It is possible to add or subtract two intervals and multiply an interval by a
scalar according to the following rules: 

.. math::
   I_1 + I_2 & = \\{ \\vect{x} + \\vect{y} \\, |\\, \\vect{x} \\in I_1, \\vect{y} \\in I_2\\} \\\\
   I_1 - I_2 & = \\{ \\vect{x} - \\vect{y} \\, |\\, \\vect{x} \\in I_1, \\vect{y} \\in I_2\\} \\\\
   \\lambda I_1 & = \\{ \\lambda \\vect{x} \\, |\\, \\vect{x} \\in I_1 \\} \\\\


Examples
--------
>>> import openturns as ot
>>> # The finite interval [2,4] * [3,5]
>>> print(ot.Interval([2.0, 3.0], [4.0, 5.0]))
[2, 4]
[3, 5]
>>> # The right-unbounded interval [2, +inf[
>>> a = 2.0
>>> print(ot.Interval([a], [1], [True], [False]))
[2, (1) +inf[
>>> print(ot.Interval([1], [a], [False], [True]))
]-inf (1), 2]
>>> # Operations with intervals:
>>> interval1 = ot.Interval([2.0, 3.0], [5.0, 8.0])
>>> interval2 = ot.Interval([1.0, 4.0], [6.0, 13.0])
>>> # Addition
>>> print(interval1 + interval2)
[3, 11]
[7, 21]
>>> # Subtraction
>>> print(interval1 - interval2)
[-4, 4]
[-10, 4]
>>> # Multiplication
>>> print(interval1 * 3)
[6, 15]
[9, 24]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::isEmpty
"Check if the interval is empty.

Returns
-------
isEmpty : bool
    *True* if the interior of the interval is empty.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([1.0, 2.0], [1.0, 2.0])
>>> interval.setFiniteLowerBound([True, False])
>>> print(interval.isEmpty())
False"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::isNumericallyEmpty
"Check if the interval is numerically empty.

Returns
-------
isEmpty : bool
    Flag telling whether the interval is numerically empty, i.e. if its numerical
    volume is inferior or equal to :math:`\\epsilon` (defined in the
    :class:`~openturns.ResourceMap`:
    :math:`\\epsilon` = Domain-SmallVolume).

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([1.0, 2.0], [1.0, 2.0])
>>> print(interval.isNumericallyEmpty())
True"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::numericallyContains
"Check if the given point is inside of the discretization of the interval.

Parameters
----------
point : sequence of float
    Point with the same dimension as the current domain's dimension.

Returns
-------
isInside : bool
    Flag telling whether the point is inside the interval bounds, not taking
    into account whether bounds are finite or not."

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getVolume
"Get the volume of the interval.

Returns
-------
volume : float
    Volume contained within interval bounds.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([2.0, 3.0], [4.0, 5.0], [True, False], [True, True])
>>> print(interval.getVolume())
4.0"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getFiniteLowerBound
"Tell for each component of the lower bound whether it is finite or not.

Returns
-------
flags : :class:`~openturns.BoolCollection`
    If the :math:`i^{th}` element is *False*, the corresponding component of
    the lower bound is infinite. Otherwise, it is finite.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([2.0, 3.0], [4.0, 5.0], [True, False], [True, True])
>>> print(interval.getFiniteLowerBound())
[1,0]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::setFiniteLowerBound
"Tell for each component of the lower bound whether it is finite or not.

Parameters
----------
flags : sequence of bool
    If the :math:`i^{th}` element is *False*, the corresponding component of
    the lower bound is infinite. Otherwise, it is finite.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval(2)
>>> interval.setFiniteLowerBound([True, False])
>>> print(interval)
[0, 1]
]-inf (0), 1]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getFiniteUpperBound
"Tell for each component of the upper bound whether it is finite or not.

Returns
-------
flags : :class:`~openturns.BoolCollection`
    If the :math:`i^{th}` element is *False*, the corresponding component of
    the upper bound is infinite. Otherwise, it is finite.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([2.0, 3.0], [4.0, 5.0], [True, False], [True, True])
>>> print(interval.getFiniteUpperBound())
[1,1]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::setFiniteUpperBound
"Tell for each component of the upper bound whether it is finite or not.

Parameters
----------
flags : sequence of bool
    If the :math:`i^{th}` element is *False*, the corresponding component of
    the upper bound is infinite. Otherwise, it is finite.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval(2)
>>> interval.setFiniteUpperBound([True, False])
>>> print(interval)
[0, 1]
[0, (1) +inf["

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getLowerBound
"Get the lower bound.

Returns
-------
lowerBound : :class:`~openturns.Point`
    Value of the lower bound.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([2.0, 3.0], [4.0, 5.0], [True, False], [True, True])
>>> print(interval.getLowerBound())
[2,3]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::setLowerBound
"Set the lower bound.

Parameters
----------
lowerBound : sequence of float
    Value of the lower bound.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval(2)
>>> interval.setLowerBound([-4, -5])
>>> print(interval)
[-4, 1]
[-5, 1]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getUpperBound
"Get the upper bound.

Returns
-------
upperBound : :class:`~openturns.Point`
    Value of the upper bound.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval([2.0, 3.0], [4.0, 5.0], [True, False], [True, True])
>>> print(interval.getUpperBound())
[4,5]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::setUpperBound
"Set the upper bound.

Parameters
----------
upperBound : sequence of float
    Value of the upper bound.

Examples
--------
>>> import openturns as ot
>>> interval = ot.Interval(2)
>>> interval.setUpperBound([4, 5])
>>> print(interval)
[0, 4]
[0, 5]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::intersect
"Get the intersection with another interval.

Parameters
----------
otherInterval : :class:`~openturns.Interval`
    Interval of the same dimension.

Returns
-------
interval : :class:`~openturns.Interval`
    An interval corresponding to the intersection of the current interval with
    *otherInterval*.

Examples
--------
>>> import openturns as ot
>>> interval1 = ot.Interval([2.0, 3.0], [5.0, 8.0])
>>> interval2 = ot.Interval([1.0, 4.0], [6.0, 13.0])
>>> print(interval1.intersect(interval2))
[2, 5]
[4, 8]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::join
"Get the smallest interval containing both the current interval and another one.

Parameters
----------
otherInterval : :class:`~openturns.Interval`
    Interval of the same dimension.

Returns
-------
interval : :class:`~openturns.Interval`
    Smallest interval containing both the current interval and
    *otherInterval*.

Examples
--------
>>> import openturns as ot
>>> interval1 = ot.Interval([2.0, 3.0], [5.0, 8.0])
>>> interval2 = ot.Interval([1.0, 4.0], [6.0, 13.0])
>>> print(interval1.join(interval2))
[1, 6]
[3, 13]"

// ---------------------------------------------------------------------

%feature("docstring") OT::Interval::getMarginal
"Marginal accessor.

Parameters
----------
index : int or sequence of int
    Index or indices of the selected components.

Returns
-------
interval : :class:`~openturns.Interval`
    The marginal interval."
