%feature("docstring") OT::ConditionalRandomVector
"Conditional random vector.

Helper class for defining the random vector :math:`\\vect{X}` such that  :math:`\\vect{X}|\\vect{\\Theta}` follows the distribution :math:`\\mathcal{L}_{\\vect{X}|\\vect{\\Theta}}`, with :math:`\\vect{\\Theta}` a random vector of dimension the dimension of :math:`\\vect{\\Theta}`.

Parameters
----------

conditionedDist : :class:`~openturns.Distribution`, the distribution of :math:`\\vect{X}|\\vect{\\Theta}`, whose parameters will be overwritten by :math:`\\vect{\\Theta}`.
    
randomParameters : :class:`~openturns.RandomVector`, the random parameters :math:`\\vect{\\Theta}` of the `conditionedDist` distribution. 

Notes
-----
Its probability density function is defined as:

.. math::

    f_{\\vect{X}}(\\vect{x}) = \\int f_{\\vect{X}|\\vect{\\Theta}=\\vect{\\theta}}(\\vect{x}|\\vect{\\theta}) f_{\\vect{\\Theta}}(\\vect{\\theta})\\di{\\vect{\\theta}}

with  :math:`f_{\\vect{X}|\\vect{\\Theta}=\\vect{\\theta}}` the PDF of the distribution of :math:`\\vect{X}|\\vect{\\Theta}`, where :math:`\\vect{\\Theta}` has been replaced by :math:`\\vect{\\theta}`, :math:`f_{\\vect{\\Theta}}` the PDF of :math:`\\vect{\\Theta}`.

Note that there exist other (quasi) equivalent modellings using a combination of the classes :class:`~openturns.DeconditionedDistribution` and :class:`~openturns.RandomVector` (see the Use Cases Guide).

Examples
--------
Create a random vector:

>>> import openturns as ot
>>> distXgivenT = ot.Exponential()
>>> distGamma = ot.Uniform(1.0, 2.0)
>>> distAlpha = ot.Uniform(0.0, 0.1)
>>> distTheta = ot.JointDistribution([distGamma, distAlpha])
>>> rvTheta = ot.RandomVector(distTheta)
>>> rvX = ot.ConditionalRandomVector(distXgivenT, rvTheta)

Draw a sample:

>>> sample = rvX.getSample(5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::ConditionalRandomVector::getRandomParameters
"Accessor to the distribution's random parameter `randomParameters`.

Returns
-------
randomParameters : :class:`~openturns.RandomVector`, the random parameters :math:`\\vect{\\Theta}`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::ConditionalRandomVector::getDistribution
"Accessor to the distribution's conditioned distribution parameter `conditionedDistribution`.

Returns
-------
conditionedDistribution : :class:`~openturns.Distribution`, the distribution of :math:`\\vect{X}|\\vect{\\Theta}=\\vect{\\theta}`, where the parameters :math:`\\vect{\\theta}` are equal to the  values used to generate the last realization of :math:`\\vect{X}`.
"

