%feature("docstring") OT::Ceres
"Interface to Ceres Solver.

This class exposes the solvers from the non-linear least squares optimization library [ceres2012]_.

More details about least squares algorithms are available `here <http://ceres-solver.org/nnls_solving.html>`_.

Algorithms are also available for general unconstrained optimization.

Parameters
----------
problem : :class:`~openturns.OptimizationProblem`
    Optimization problem to solve, either least-squares or general (unconstrained).
algoName : str
    The identifier of the algorithm.
    Use :func:`GetAlgorithmNames()` to list available names.

Notes
-----
Solvers use first order derivative information.

As for constraint support, only the trust-region solvers allow for bound constraints:

+------------------------------+---------------+------------------------+--------------------+
| Algorithm                    | Method type   | Problem type support   | Constraint support |
+==============================+===============+========================+====================+
| LEVENBERG_MARQUARDT          | trust-region  | least-squares          | bounds             |
+------------------------------+---------------+------------------------+--------------------+
| DOGLEG                       | trust-region  | least-squares          | bounds             |
+------------------------------+---------------+------------------------+--------------------+
| STEEPEST_DESCENT             | line-search   | least-squares, general | none               |
+------------------------------+---------------+------------------------+--------------------+
| NONLINEAR_CONJUGATE_GRADIENT | line-search   | least-squares, general | none               |
+------------------------------+---------------+------------------------+--------------------+
| LBFGS                        | line-search   | least-squares, general | none               |
+------------------------------+---------------+------------------------+--------------------+
| BFGS                         | line-search   | least-squares, general | none               |
+------------------------------+---------------+------------------------+--------------------+


Ceres least squares solver can be further tweaked thanks to the following
:class:`~openturns.ResourceMap` parameters, refer to
`nlls solver options <http://ceres-solver.org/nnls_solving.html#solver-options>`_
for more details.

+------------------------------------------------------------+-------+
| Key                                                        | Type  |
+============================================================+=======+
| Ceres-minimizer_type                                       | str   |
+------------------------------------------------------------+-------+
| Ceres-line_search_direction_type                           | str   |
+------------------------------------------------------------+-------+
| Ceres-line_search_type                                     | str   |
+------------------------------------------------------------+-------+
| Ceres-nonlinear_conjugate_gradient_type                    | str   |
+------------------------------------------------------------+-------+
| Ceres-max_lbfgs_rank                                       | int   |
+------------------------------------------------------------+-------+
| Ceres-use_approximate_eigenvalue_bfgs_scaling              | bool  |
+------------------------------------------------------------+-------+
| Ceres-line_search_interpolation_type                       | str   |
+------------------------------------------------------------+-------+
| Ceres-min_line_search_step_size                            | float |
+------------------------------------------------------------+-------+
| Ceres-line_search_sufficient_function_decrease             | float |
+------------------------------------------------------------+-------+
| Ceres-max_line_search_step_contraction                     | float |
+------------------------------------------------------------+-------+
| Ceres-min_line_search_step_contraction                     | float |
+------------------------------------------------------------+-------+
| Ceres-max_num_line_search_step_size_iterations             | int   |
+------------------------------------------------------------+-------+
| Ceres-max_num_line_search_direction_restarts               | int   |
+------------------------------------------------------------+-------+
| Ceres-line_search_sufficient_curvature_decrease            | float |
+------------------------------------------------------------+-------+
| Ceres-max_line_search_step_expansion                       | float |
+------------------------------------------------------------+-------+
| Ceres-trust_region_strategy_type                           | str   |
+------------------------------------------------------------+-------+
| Ceres-dogleg_type                                          | str   |
+------------------------------------------------------------+-------+
| Ceres-use_nonmonotonic_steps                               | bool  |
+------------------------------------------------------------+-------+
| Ceres-max_consecutive_nonmonotonic_steps                   | int   |
+------------------------------------------------------------+-------+
| Ceres-max_num_iterations                                   | int   |
+------------------------------------------------------------+-------+
| Ceres-max_solver_time_in_seconds                           | float |
+------------------------------------------------------------+-------+
| Ceres-num_threads                                          | int   |
+------------------------------------------------------------+-------+
| Ceres-initial_trust_region_radius                          | float |
+------------------------------------------------------------+-------+
| Ceres-max_trust_region_radius                              | float |
+------------------------------------------------------------+-------+
| Ceres-min_trust_region_radius                              | float |
+------------------------------------------------------------+-------+
| Ceres-min_relative_decrease                                | float |
+------------------------------------------------------------+-------+
| Ceres-min_lm_diagonal                                      | float |
+------------------------------------------------------------+-------+
| Ceres-max_lm_diagonal                                      | float |
+------------------------------------------------------------+-------+
| Ceres-max_num_consecutive_invalid_steps                    | int   |
+------------------------------------------------------------+-------+
| Ceres-function_tolerance                                   | float |
+------------------------------------------------------------+-------+
| Ceres-gradient_tolerance                                   | float |
+------------------------------------------------------------+-------+
| Ceres-parameter_tolerance                                  | float |
+------------------------------------------------------------+-------+
| Ceres-preconditioner_type                                  | str   |
+------------------------------------------------------------+-------+
| Ceres-visibility_clustering_type                           | str   |
+------------------------------------------------------------+-------+
| Ceres-dense_linear_algebra_library_type                    | str   |
+------------------------------------------------------------+-------+
| Ceres-sparse_linear_algebra_library_type                   | str   |
+------------------------------------------------------------+-------+
| Ceres-use_explicit_schur_complement                        | bool  |
+------------------------------------------------------------+-------+
| Ceres-use_postordering                                     | bool  |
+------------------------------------------------------------+-------+
| Ceres-dynamic_sparsity                                     | bool  |
+------------------------------------------------------------+-------+
| Ceres-min_linear_solver_iterations                         | int   |
+------------------------------------------------------------+-------+
| Ceres-max_linear_solver_iterations                         | int   |
+------------------------------------------------------------+-------+
| Ceres-eta                                                  | float |
+------------------------------------------------------------+-------+
| Ceres-jacobi_scaling                                       | bool  |
+------------------------------------------------------------+-------+
| Ceres-use_inner_iterations                                 | bool  |
+------------------------------------------------------------+-------+
| Ceres-inner_iteration_tolerance                            | float |
+------------------------------------------------------------+-------+
| Ceres-logging_type                                         | str   |
+------------------------------------------------------------+-------+
| Ceres-minimizer_progress_to_stdout                         | bool  |
+------------------------------------------------------------+-------+
| Ceres-trust_region_problem_dump_directory                  | str   |
+------------------------------------------------------------+-------+
| Ceres-trust_region_problem_dump_format_type                | str   |
+------------------------------------------------------------+-------+
| Ceres-check_gradients                                      | bool  |
+------------------------------------------------------------+-------+
| Ceres-gradient_check_relative_precision                    | float |
+------------------------------------------------------------+-------+
| Ceres-gradient_check_numeric_derivative_relative_step_size | float |
+------------------------------------------------------------+-------+
| Ceres-update_state_every_iteration                         | bool  |
+------------------------------------------------------------+-------+


Ceres unconstrained solver can be further tweaked using the following :class:`~openturns.ResourceMap` parameters,
refer to `gradient solver options <http://ceres-solver.org/gradient_solver.html#solving>`_ for more details.

+------------------------------------------------------------+-------+
| Key                                                        | Type  |
+============================================================+=======+
| Ceres-line_search_direction_type                           | str   |
+------------------------------------------------------------+-------+
| Ceres-line_search_type                                     | str   |
+------------------------------------------------------------+-------+
| Ceres-nonlinear_conjugate_gradient_type                    | str   |
+------------------------------------------------------------+-------+
| Ceres-max_lbfgs_rank                                       | int   |
+------------------------------------------------------------+-------+
| Ceres-use_approximate_eigenvalue_bfgs_scaling              | bool  |
+------------------------------------------------------------+-------+
| Ceres-line_search_interpolation_type                       | str   |
+------------------------------------------------------------+-------+
| Ceres-min_line_search_step_size                            | float |
+------------------------------------------------------------+-------+
| Ceres-line_search_sufficient_function_decrease             | float |
+------------------------------------------------------------+-------+
| Ceres-max_line_search_step_contraction                     | float |
+------------------------------------------------------------+-------+
| Ceres-min_line_search_step_contraction                     | float |
+------------------------------------------------------------+-------+
| Ceres-max_num_line_search_step_size_iterations             | int   |
+------------------------------------------------------------+-------+
| Ceres-max_num_line_search_direction_restarts               | int   |
+------------------------------------------------------------+-------+
| Ceres-line_search_sufficient_curvature_decrease            | float |
+------------------------------------------------------------+-------+
| Ceres-max_line_search_step_expansion                       | float |
+------------------------------------------------------------+-------+
| Ceres-max_num_iterations                                   | int   |
+------------------------------------------------------------+-------+
| Ceres-max_solver_time_in_seconds                           | float |
+------------------------------------------------------------+-------+
| Ceres-function_tolerance                                   | float |
+------------------------------------------------------------+-------+
| Ceres-gradient_tolerance                                   | float |
+------------------------------------------------------------+-------+
| Ceres-parameter_tolerance                                  | float |
+------------------------------------------------------------+-------+
| Ceres-logging_type                                         | str   |
+------------------------------------------------------------+-------+
| Ceres-minimizer_progress_to_stdout                         | bool  |
+------------------------------------------------------------+-------+

See also
--------
AbdoRackwitz, Cobyla, CMinpack, NLopt, SQP, TNC

Examples
--------
List available algorithms:

>>> import openturns as ot
>>> print(ot.Ceres.GetAlgorithmNames())
[LEVENBERG_MARQUARDT,DOGLEG,...

Solve a least-squares problem:

>>> dim = 2
>>> residualFunction = ot.SymbolicFunction(['x0', 'x1'], ['10*(x1-x0^2)', '1-x0'])
>>> problem = ot.LeastSquaresProblem(residualFunction)
>>> problem.setBounds(ot.Interval([-3.0] * dim, [5.0] * dim))
>>> ot.ResourceMap.AddAsScalar('Ceres-gradient_tolerance', 1e-5)  # doctest: +SKIP
>>> algo = ot.Ceres(problem, 'LEVENBERG_MARQUARDT')  # doctest: +SKIP
>>> algo.setStartingPoint([0.0] * dim)  # doctest: +SKIP
>>> algo.run()  # doctest: +SKIP
>>> result = algo.getResult()  # doctest: +SKIP
>>> x_star = result.getOptimalPoint()  # doctest: +SKIP
>>> y_star = result.getOptimalValue()  # doctest: +SKIP

Or, solve a general optimization problem:

>>> dim = 4
>>> linear = ot.SymbolicFunction(['x1', 'x2', 'x3', 'x4'], ['(x1-1)^2+(x2-2)^2+(x3-3)^2+(x4-4)^2'])
>>> problem = ot.OptimizationProblem(linear)
>>> ot.ResourceMap.AddAsScalar('Ceres-gradient_tolerance', 1e-5)  # doctest: +SKIP
>>> algo = ot.Ceres(problem, 'BFGS')  # doctest: +SKIP
>>> algo.setStartingPoint([0.0] * 4)  # doctest: +SKIP
>>> algo.run()  # doctest: +SKIP
>>> result = algo.getResult()  # doctest: +SKIP
>>> x_star = result.getOptimalPoint()  # doctest: +SKIP
>>> y_star = result.getOptimalValue()  # doctest: +SKIP"

// ---------------------------------------------------------------------

%feature("docstring") OT::Ceres::setAlgorithmName
"Accessor to the algorithm name.

Parameters
----------
algoName : str
    The identifier of the algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::Ceres::getAlgorithmName
"Accessor to the algorithm name.

Returns
-------
algoName : str
    The identifier of the algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::Ceres::GetAlgorithmNames
"Accessor to the list of algorithms provided, by names.

Returns
-------
names : :class:`~openturns.Description`
    List of algorithm names provided, according to its naming convention.

    The trust region methods are not able to solve general optimization
    problems, in that case a warning is printed and the default line search
    method is used instead.

Examples
--------
>>> import openturns as ot
>>> print(ot.Ceres.GetAlgorithmNames())
[LEVENBERG_MARQUARDT,DOGLEG,STEEPEST_DESCENT,NONLINEAR_CONJUGATE_GRADIENT,LBFGS,BFGS]"


