// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
#ifndef COMPONENTS_AUTOFILL_IOS_FORM_UTIL_FORM_ACTIVITY_PARAMS_H_
#define COMPONENTS_AUTOFILL_IOS_FORM_UTIL_FORM_ACTIVITY_PARAMS_H_

#include <string>

#include "components/autofill/core/common/unique_ids.h"

namespace autofill {

// HTML password field type.
constexpr char kPasswordFieldType[] = "password";

// Wraps information about form activity.
struct BaseFormActivityParams {
  BaseFormActivityParams();
  BaseFormActivityParams(const BaseFormActivityParams& other);
  virtual ~BaseFormActivityParams();

  std::string form_name;
  FormRendererId unique_form_id;
  std::string frame_id;

  // |input_missing| is set to true if at least one of the members above isn't
  // set.
  bool input_missing = false;

  // |is_main_frame| is true when the activity was registered in the main frame.
  bool is_main_frame = false;
};

// Wraps information about event happening on an input field.
// Example HTML
// <form name="np" id="np1" action="https://example.com/" method="post">
// <input type="text" name="name" id="password_name"><br>
// <input type="password" name="password" id="password_field"><br>
// <input type="reset" value="Reset">
// <input type="submit" value="Submit" id="password_submit">
// </form>
// A tap on the password field will produce
// form_name:  "np"
// field_identifier:  "password_field"
// unique_field_id: will be the unique numeric ID generated for the field
// field_type:  "password"
// type: "focus"
// value: "LouisLane" (assuming that was the password typed)
// has_user_gesture:  true
// input_missing:  false
// frame_id: will be the unique ID generated in for the frame containing the
// form (see __gCrWeb.message.getFrameId for details).
struct FormActivityParams : public BaseFormActivityParams {
  FormActivityParams();
  FormActivityParams(const FormActivityParams& other);
  ~FormActivityParams() override;

  // Generated by __gCrWeb.form.getFieldIdentifier in form.js.
  std::string field_identifier;
  // Generated by __gCrWeb.fill.getUniqueID in fill.js.
  FieldRendererId unique_field_id;
  std::string field_type;
  std::string value;
  std::string type;
  // |has_user_gesture| is true when the activity was registered as a result of
  // a user action, and not by an event created and dispatched by JavaScript.
  bool has_user_gesture = false;
};

// Wraps information about the form removal.
struct FormRemovalParams : public BaseFormActivityParams {
  FormRemovalParams();
  FormRemovalParams(const FormRemovalParams& other);
  ~FormRemovalParams() override;

  // Renderer ids of removed unowned fields.
  std::vector<FieldRendererId> removed_unowned_fields;
};

}  // namespace autofill

#endif  // COMPONENTS_AUTOFILL_IOS_FORM_UTIL_FORM_ACTIVITY_PARAMS_H_
