// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/signin/sync_confirmation_ui.h"

#include <string>

#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "base/values.h"
#include "build/buildflag.h"
#include "build/chromeos_buildflags.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_attributes_storage.h"
#include "chrome/browser/profiles/profile_avatar_icon_util.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/signin/account_consistency_mode_manager.h"
#include "chrome/browser/signin/identity_manager_factory.h"
#include "chrome/browser/signin/signin_features.h"
#include "chrome/browser/sync/sync_service_factory.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/signin/profile_colors_util.h"
#include "chrome/browser/ui/webui/signin/signin_url_utils.h"
#include "chrome/browser/ui/webui/signin/sync_confirmation_handler.h"
#include "chrome/browser/ui/webui/webui_util.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "chrome/common/url_constants.h"
#include "chrome/grit/generated_resources.h"
#include "chrome/grit/signin_resources.h"
#include "components/signin/public/base/avatar_icon_util.h"
#include "components/signin/public/identity_manager/identity_manager.h"
#include "components/strings/grit/components_strings.h"
#include "content/public/browser/web_ui.h"
#include "content/public/browser/web_ui_data_source.h"
#include "services/network/public/mojom/content_security_policy.mojom.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/webui/resource_path.h"
#include "ui/base/webui/web_ui_util.h"
#include "ui/gfx/color_utils.h"
#include "ui/native_theme/native_theme.h"
#include "ui/resources/grit/webui_generated_resources.h"

SyncConfirmationUI::SyncConfirmationUI(content::WebUI* web_ui)
    : SigninWebDialogUI(web_ui), profile_(Profile::FromWebUI(web_ui)) {
  const GURL& url = web_ui->GetWebContents()->GetVisibleURL();
  const bool is_sync_allowed = SyncServiceFactory::IsSyncAllowed(profile_);

  content::WebUIDataSource* source =
      content::WebUIDataSource::Create(chrome::kChromeUISyncConfirmationHost);
  webui::SetJSModuleDefaults(source);

  static constexpr webui::ResourcePath kResources[] = {
      {"signin_shared.css.js", IDR_SIGNIN_SIGNIN_SHARED_CSS_JS},
      {"signin_vars.css.js", IDR_SIGNIN_SIGNIN_VARS_CSS_JS},
      {"sync_confirmation_browser_proxy.js",
       IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_CONFIRMATION_BROWSER_PROXY_JS},
      {"sync_confirmation.js",
       IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_CONFIRMATION_JS},
      {chrome::kChromeUISyncConfirmationLoadingPath,
       IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_LOADING_CONFIRMATION_HTML},
  };
  source->AddResourcePaths(kResources);

  AddStringResource(source, "syncLoadingConfirmationTitle",
                    IDS_SYNC_LOADING_CONFIRMATION_TITLE);

  if (is_sync_allowed) {
    InitializeForSyncConfirmation(source, GetSyncConfirmationStyle(url));
  } else {
    InitializeForSyncDisabled(source);
  }

  base::Value::Dict strings;
  webui::SetLoadTimeDataDefaults(
      g_browser_process->GetApplicationLocale(), &strings);
  source->AddLocalizedStrings(strings);

  content::WebUIDataSource::Add(profile_, source);
}

SyncConfirmationUI::~SyncConfirmationUI() = default;

void SyncConfirmationUI::InitializeMessageHandlerWithBrowser(Browser* browser) {
  web_ui()->AddMessageHandler(std::make_unique<SyncConfirmationHandler>(
      profile_, js_localized_string_to_ids_map_, browser));
}

void SyncConfirmationUI::InitializeForSyncConfirmation(
    content::WebUIDataSource* source,
    SyncConfirmationStyle style) {
  int title_id = IDS_SYNC_CONFIRMATION_TITLE;
  int info_title_id = IDS_SYNC_CONFIRMATION_SYNC_INFO_TITLE;
  int confirm_label_id = IDS_SYNC_CONFIRMATION_CONFIRM_BUTTON_LABEL;
#if BUILDFLAG(IS_CHROMEOS_LACROS)
  title_id = IDS_SYNC_CONFIRMATION_TITLE_LACROS_NON_FORCED;
#endif
  AddStringResource(source, "syncConfirmationSyncInfoDesc",
                    IDS_SYNC_CONFIRMATION_SYNC_INFO_DESC);
  AddStringResource(source, "syncConfirmationSettingsInfo",
                    IDS_SYNC_CONFIRMATION_SETTINGS_INFO);

  source->AddResourcePath(
      "sync_confirmation_app.js",
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_CONFIRMATION_APP_JS);
  source->AddResourcePath(
      "sync_confirmation_app.html.js",
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_CONFIRMATION_APP_HTML_JS);
  source->SetDefaultResource(
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_CONFIRMATION_HTML);

  source->AddBoolean("isModalDialog",
                     style == SyncConfirmationStyle::kDefaultModal ||
                         style == SyncConfirmationStyle::kSigninInterceptModal);
  source->AddBoolean("isSigninInterceptFre",
                     style == SyncConfirmationStyle::kSigninInterceptModal);

  source->AddString("accountPictureUrl",
                    profiles::GetPlaceholderAvatarIconUrl());
  switch (style) {
    case SyncConfirmationStyle::kSigninInterceptModal: {
      DCHECK(base::FeatureList::IsEnabled(kSyncPromoAfterSigninIntercept));
      ProfileAttributesEntry* entry =
          g_browser_process->profile_manager()
              ->GetProfileAttributesStorage()
              .GetProfileAttributesWithPath(profile_->GetPath());
      DCHECK(entry);
      std::u16string gaia_name = entry->GetGAIANameToDisplay();
      if (gaia_name.empty())
        gaia_name = entry->GetLocalProfileName();
      AddStringResourceWithPlaceholder(
          source, "syncConfirmationTitle",
          IDS_SYNC_CONFIRMATION_WELCOME_TITLE_SIGNIN_INTERCEPT, gaia_name);
      AddStringResource(source, "syncConfirmationSyncInfoTitle",
                        IDS_SYNC_CONFIRMATION_SYNC_INFO_SIGNIN_INTERCEPT);
      AddStringResource(source, "syncConfirmationConfirmLabel",
                        IDS_SYNC_CONFIRMATION_TURN_ON_SYNC_BUTTON_LABEL);
      AddStringResource(source, "syncConfirmationUndoLabel", IDS_NO_THANKS);
      AddStringResource(source, "syncConfirmationSettingsLabel",
                        IDS_SYNC_CONFIRMATION_SETTINGS_BUTTON_LABEL);

      source->AddResourcePath(
          "images/sync_confirmation_signin_intercept_illustration.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_SIGNIN_INTERCEPT_ILLUSTRATION_SVG);
      source->AddResourcePath(
          "images/sync_confirmation_signin_intercept_illustration_dark.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_SIGNIN_INTERCEPT_ILLUSTRATION_DARK_SVG);
      break;
    }
    case SyncConfirmationStyle::kDefaultModal:
      AddStringResource(source, "syncConfirmationTitle", title_id);
      AddStringResource(source, "syncConfirmationSyncInfoTitle", info_title_id);
      AddStringResource(source, "syncConfirmationConfirmLabel",
                        confirm_label_id);
      AddStringResource(source, "syncConfirmationUndoLabel", IDS_CANCEL);
      AddStringResource(source, "syncConfirmationSettingsLabel",
                        IDS_SYNC_CONFIRMATION_SETTINGS_BUTTON_LABEL);

      source->AddResourcePath(
          "images/sync_confirmation_illustration.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_ILLUSTRATION_SVG);
      source->AddResourcePath(
          "images/sync_confirmation_illustration_dark.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_ILLUSTRATION_DARK_SVG);
      break;
    case SyncConfirmationStyle::kWindow:
      AddStringResource(source, "syncConfirmationTitle", title_id);
      AddStringResource(source, "syncConfirmationSyncInfoTitle", info_title_id);
      AddStringResource(source, "syncConfirmationConfirmLabel",
                        confirm_label_id);
      AddStringResource(source, "syncConfirmationUndoLabel", IDS_NO_THANKS);
      AddStringResource(source, "syncConfirmationSettingsLabel",
                        IDS_SYNC_CONFIRMATION_REFRESHED_SETTINGS_BUTTON_LABEL);
      source->AddResourcePath(
          "images/sync_confirmation_refreshed_illustration.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_REFRESHED_ILLUSTRATION_SVG);
      source->AddResourcePath(
          "images/sync_confirmation_refreshed_illustration_dark.svg",
          IDR_SIGNIN_SYNC_CONFIRMATION_IMAGES_SYNC_CONFIRMATION_REFRESHED_ILLUSTRATION_DARK_SVG);
      break;
  }
}

void SyncConfirmationUI::InitializeForSyncDisabled(
    content::WebUIDataSource* source) {
  source->SetDefaultResource(
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_DISABLED_CONFIRMATION_HTML);
  source->AddResourcePath(
      "sync_disabled_confirmation_app.js",
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_DISABLED_CONFIRMATION_APP_JS);
  source->AddResourcePath(
      "sync_disabled_confirmation_app.html.js",
      IDR_SIGNIN_SYNC_CONFIRMATION_SYNC_DISABLED_CONFIRMATION_APP_HTML_JS);

  AddStringResource(source, "syncDisabledConfirmationTitle",
                    IDS_SYNC_DISABLED_CONFIRMATION_CHROME_SYNC_TITLE);
  AddStringResource(source, "syncDisabledConfirmationDetails",
                    IDS_SYNC_DISABLED_CONFIRMATION_DETAILS);
  AddStringResource(source, "syncDisabledConfirmationConfirmLabel",
                    IDS_SYNC_DISABLED_CONFIRMATION_CONFIRM_BUTTON_LABEL);
  AddStringResource(source, "syncDisabledConfirmationUndoLabel",
                    IDS_SYNC_DISABLED_CONFIRMATION_UNDO_BUTTON_LABEL);
}

void SyncConfirmationUI::AddStringResource(content::WebUIDataSource* source,
                                           const std::string& name,
                                           int ids) {
  source->AddLocalizedString(name, ids);
  AddLocalizedStringToIdsMap(l10n_util::GetStringUTF8(ids), ids);
}

void SyncConfirmationUI::AddStringResourceWithPlaceholder(
    content::WebUIDataSource* source,
    const std::string& name,
    int ids,
    const std::u16string& parameter) {
  std::string localized_string = l10n_util::GetStringFUTF8(ids, parameter);
  source->AddString(name, localized_string);
  AddLocalizedStringToIdsMap(localized_string, ids);
}

void SyncConfirmationUI::AddLocalizedStringToIdsMap(
    const std::string& localized_string,
    int ids) {
  // When the strings are passed to the HTML, the Unicode NBSP symbol (\u00A0)
  // will be automatically replaced with "&nbsp;". This change must be mirrored
  // in the string-to-ids map. Note that "\u00A0" is actually two characters,
  // so we must use base::ReplaceSubstrings* rather than base::ReplaceChars.
  // TODO(msramek): Find a more elegant solution.
  std::string sanitized_string = localized_string;
  base::ReplaceSubstringsAfterOffset(&sanitized_string, 0, "\u00A0" /* NBSP */,
                                     "&nbsp;");
  js_localized_string_to_ids_map_[sanitized_string] = ids;
}
