// Copyright (C) 2019 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only

#ifndef QSSGSCENEMANAGER_P_H
#define QSSGSCENEMANAGER_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <QtCore/QObject>
#include <QtCore/QSet>

#include <QtQuick3D/private/qtquick3dglobal_p.h>

#include "qquick3dobject_p.h"
#include "qquick3dnode_p.h"

QT_BEGIN_NAMESPACE

class QSGDynamicTexture;
class QQuickWindow;
class QSSGBufferManager;
class QSSGRenderContextInterface;

class Q_QUICK3D_PRIVATE_EXPORT QQuick3DWindowAttachment : public QObject
{
    Q_OBJECT
public:
    explicit QQuick3DWindowAttachment(QQuickWindow *window);
    ~QQuick3DWindowAttachment() override;

    Q_INVOKABLE void preSync();
    Q_INVOKABLE void cleanupResources();
    Q_INVOKABLE bool synchronize(QSet<QSSGRenderGraphObject *> &resourceLoaders);
    Q_INVOKABLE void requestUpdate();

    QQuickWindow *window() const;

    const std::shared_ptr<QSSGRenderContextInterface> &rci() const { return m_rci; }
    void setRci(const std::shared_ptr<QSSGRenderContextInterface> &rciptr);

    void registerSceneManager(QQuick3DSceneManager &manager);
    void unregisterSceneManager(QQuick3DSceneManager &manager);

    void queueForCleanup(QSSGRenderGraphObject *obj);
    void queueForCleanup(QQuick3DSceneManager *manager);

Q_SIGNALS:
    void releaseCachedResources();
    void renderContextInterfaceChanged();

private:
    Q_INVOKABLE void onReleaseCachedResources();
    Q_INVOKABLE void onInvalidated();

    QPointer<QQuickWindow> m_window;
    std::shared_ptr<QSSGRenderContextInterface> m_rci;
    QList<QQuick3DSceneManager *> sceneManagers;
    QList<QQuick3DSceneManager *> sceneManagerCleanupQueue;
    QList<QSSGRenderGraphObject *> pendingResourceCleanupQueue;
    QSet<QSSGRenderGraphObject *> resourceCleanupQueue;
};

class Q_QUICK3D_PRIVATE_EXPORT QQuick3DSceneManager : public QObject
{
    Q_OBJECT
public:
    explicit QQuick3DSceneManager(QObject *parent = nullptr);
    ~QQuick3DSceneManager() override;

    void setWindow(QQuickWindow *window);
    QQuickWindow *window();

    void dirtyItem(QQuick3DObject *item);
    void requestUpdate();
    void cleanup(QSSGRenderGraphObject *item);

    void polishItems();
    void forcePolish();
    void sync();
    void preSync();

    bool cleanupNodes();
    bool updateDirtyResourceNodes();
    void updateDirtySpatialNodes();

    void updateDirtyResource(QQuick3DObject *resourceObject);
    void updateDirtySpatialNode(QQuick3DNode *spatialNode);
    void updateBoundingBoxes(QSSGBufferManager &mgr);

    QQuick3DObject *lookUpNode(const QSSGRenderGraphObject *node) const;

    // Where the enumerator is placed will decide the priority it gets.
    // NOTE: Place new list types before 'Count'.
    // NOTE: InstanceNodes are nodes that have an instance root set, we'll process these
    // after the other nodes but before light nodes; this implies that lights are not good candidates
    // for being instance roots...
    enum class NodePriority { Skeleton, Other, ModelWithInstanceRoot, Lights, Count };
    enum class ResourcePriority { TextureData, Texture, Other, Count };

    static inline size_t resourceListIndex(QSSGRenderGraphObject::Type type)
    {
        Q_ASSERT(!QSSGRenderGraphObject::isNodeType(type));

        if (QSSGRenderGraphObject::isTexture(type))
            return size_t(ResourcePriority::Texture);

        if (type == QSSGRenderGraphObject::Type::TextureData)
            return size_t(ResourcePriority::TextureData);

        return size_t(ResourcePriority::Other);
    }

    static inline size_t nodeListIndex(QSSGRenderGraphObject::Type type)
    {
        Q_ASSERT(QSSGRenderGraphObject::isNodeType(type));

        if (QSSGRenderGraphObject::isLight(type))
            return size_t(NodePriority::Lights);

        if (type == QSSGRenderGraphObject::Type::Skeleton)
            return size_t(NodePriority::Skeleton);

        return size_t(NodePriority::Other);
    }

    static QQuick3DWindowAttachment *getOrSetWindowAttachment(QQuickWindow &window);

    QQuick3DObject *dirtyResources[size_t(ResourcePriority::Count)] {};
    QQuick3DObject *dirtyNodes[size_t(NodePriority::Count)] {};

    QList<QQuick3DObject *> dirtyBoundingBoxList;
    QList<QSSGRenderGraphObject *> cleanupNodeList;
    QList<QSSGRenderGraphObject *> resourceCleanupQueue;

    QSet<QQuick3DObject *> parentlessItems;
    QVector<QSGDynamicTexture *> qsgDynamicTextures;
    QHash<const QSSGRenderGraphObject *, QQuick3DObject *> m_nodeMap;
    QSet<QSSGRenderGraphObject *> resourceLoaders;
    QQuickWindow *m_window = nullptr;
    QPointer<QQuick3DWindowAttachment> wattached;
    int inputHandlingEnabled = 0; // Holds the count of active item2Ds, input disabled if zero.
    bool sharedResourceRemoved = false;
    friend QQuick3DObject;

Q_SIGNALS:
    void needsUpdate();
    void windowChanged();

private Q_SLOTS:
    bool updateResources(QQuick3DObject **listHead);
    void updateNodes(QQuick3DObject **listHead);
};

QT_END_NAMESPACE

QML_DECLARE_TYPE(QQuick3DSceneManager)

#endif // QSSGSCENEMANAGER_P_H
