/*
 * h235dh.cxx
 *
 * H.235 Diffie-Hellman key exchange PDU's
 *
 * Open H323 Library
 *
 * Copyright (c) 1998-2013 Vox Lucida Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Revision: 33786 $
 * $Author: rjongbloed $
 * $Date: 2015-07-31 10:14:18 +0100 (Fri, 31 Jul 2015) $
 */

#include <ptlib.h>

#include <opal_config.h>

#if OPAL_H235_6

#ifdef __GNUC__
#pragma implementation "h235dh.h"
#endif

#include <h323/h235dh.h>
#include <h323/h323pdu.h>
#include <opal/mediasession.h>


#define new PNEW
#define PTraceModule() "H235"


// Magic numbers for DH modulus: H.235.6 section 8.5, table 4
const uint8_t DH1024_P[1024/8] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68, 0xC2, 0x34,
  0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74,
  0x02, 0x0B, 0xBE, 0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
  0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14, 0x37,
  0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6,
  0xF4, 0x4C, 0x42, 0xE9, 0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
  0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
  0x49, 0x28, 0x66, 0x51, 0xEC, 0xE6, 0x53, 0x81, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

#if 0
// H.235.6 is confusing, one part says this is needed for AES-128, and another part says it isn't
// Then again it says the 168 bit triple DES can use DH1024, so 128 bit AES should work too.
const uint8_t DH1536_P[1536/8] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68, 0xC2, 0x34,
  0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74,
  0x02, 0x0B, 0xBE, 0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
  0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14, 0x37,
  0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6,
  0xF4, 0x4C, 0x42, 0xE9, 0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
  0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
  0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05,
  0x98, 0xDA, 0x48, 0x36, 0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
  0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB,
  0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D, 0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04,
  0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x23, 0x73, 0x27, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};
#endif

const uint8_t DH2048_P[2048/8] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68, 0xC2, 0x34,
  0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74,
  0x02, 0x0B, 0xBE, 0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
  0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14, 0x37,
  0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6,
  0xF4, 0x4C, 0x42, 0xE9, 0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
  0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
  0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05,
  0x98, 0xDA, 0x48, 0x36, 0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
  0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB,
  0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D, 0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04,
  0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
  0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2, 0xEC, 0x07, 0xA2, 0x8F,
  0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
  0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
  0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAC, 0xAA, 0x68, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

const uint8_t DH4096_P[4096] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68, 0xC2, 0x34,
  0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74,
  0x02, 0x0B, 0xBE, 0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
  0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14, 0x37,
  0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6,
  0xF4, 0x4C, 0x42, 0xE9, 0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
  0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
  0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05,
  0x98, 0xDA, 0x48, 0x36, 0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
  0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB,
  0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D, 0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04,
  0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
  0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2, 0xEC, 0x07, 0xA2, 0x8F,
  0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
  0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
  0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D, 0x04, 0x50, 0x7A, 0x33,
  0xA8, 0x55, 0x21, 0xAB, 0xDF, 0x1C, 0xBA, 0x64, 0xEC, 0xFB, 0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A,
  0x8A, 0xEA, 0x71, 0x57, 0x5D, 0x06, 0x0C, 0x7D, 0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4, 0xC7,
  0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09, 0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0, 0x4A, 0x25, 0x61, 0x9D,
  0xCE, 0xE3, 0xD2, 0x26, 0x1A, 0xD2, 0xEE, 0x6B, 0xF1, 0x2F, 0xFA, 0x06, 0xD9, 0x8A, 0x08, 0x64,
  0xD8, 0x76, 0x02, 0x73, 0x3E, 0xC8, 0x6A, 0x64, 0x52, 0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
  0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D, 0x6C, 0x77, 0x09, 0x88, 0xC0, 0xBA, 0xD9, 0x46, 0xE2,
  0x08, 0xE2, 0x4F, 0xA0, 0x74, 0xE5, 0xAB, 0x31, 0x43, 0xDB, 0x5B, 0xFC, 0xE0, 0xFD, 0x10, 0x8E,
  0x4B, 0x82, 0xD1, 0x20, 0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72, 0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7,
  0x88, 0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26, 0x99, 0xC3, 0x27, 0x18, 0x6A, 0xF4, 0xE2, 0x3C,
  0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15, 0x0B, 0xDA, 0x25, 0x83, 0xE9, 0xCA, 0x2A, 0xD4, 0x4C, 0xE8,
  0xDB, 0xBB, 0xC2, 0xDB, 0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC, 0x14, 0x1F, 0xBE, 0xCA, 0xA6,
  0x28, 0x7C, 0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99, 0xB2, 0x96, 0x4F, 0xA0, 0x90, 0xC3, 0xA2,
  0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7, 0xED, 0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2, 0xD7, 0xAF,
  0xB8, 0x1B, 0xDD, 0x76, 0x21, 0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27, 0xD5, 0xB0, 0x5A, 0xA9,
  0x93, 0xB4, 0xEA, 0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF, 0xB7, 0xDC, 0x90, 0xA6, 0xC0, 0x8F,
  0x4D, 0xF4, 0x35, 0xC9, 0x34, 0x06, 0x31, 0x99, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};


static struct DHInfo {
  const char    * m_algOID;
  const char    * m_dhOID;
  PINDEX          m_groupBits;
  const uint8_t * m_modulus;
} const DHGroup[] = {
  { "2.16.840.1.101.3.4.1.2",  "0.0.8.235.0.3.43", 1024, DH1024_P },
  { "2.16.840.1.101.3.4.1.22", "0.0.8.235.0.3.45", 2048, DH2048_P },
  { "2.16.840.1.101.3.4.1.42", "0.0.8.235.0.3.47", 4096, DH4096_P }
};

static PConstString const H235Version3("0.0.8.235.0.3.24");


// Find DH group
static const DHInfo * FindDH(const char * oid)
{
  for (PINDEX group = 0; group < PARRAYSIZE(DHGroup); ++group) {
    if (strcmp(DHGroup[group].m_algOID, oid) == 0)
      return &DHGroup[group];
  }

  PTRACE(2, "No Diffie-Hellman group available for " << oid);
  return NULL;
}


/////////////////////////////////////////////////////////////////////////////

bool H235DiffieHellman::AddForAlgorithm(const OpalMediaCryptoSuite & cryptoSuite)
{
  // Find DH group (H.235.6 section 11, table 6)
  const DHInfo * group = FindDH(cryptoSuite.GetOID());
  if (group == NULL)
    return false;

  if (Contains(group->m_dhOID))
    return true;

  // From H.235.6 section 8.5, table 4, generator is "2"
  PINDEX groupBytes = group->m_groupBits/8;
  BYTE * generator = (BYTE *)alloca(groupBytes);
  memset(generator, 0, groupBytes-1);
  generator[groupBytes-1] = 2;

  PSSLDiffieHellman * dh = new PSSLDiffieHellman(group->m_groupBits, group->m_modulus, generator);
  if (dh->IsValid()) {
    SetAt(group->m_dhOID, dh);
    return true;
  }

  delete dh;
  return false;
}


PBYTEArray H235DiffieHellman::FindMasterKey(const OpalMediaCryptoSuite & cryptoSuite) const
{
  PBYTEArray key;

  const DHInfo * group = FindDH(cryptoSuite.GetOID());
  if (group != NULL) {
    PSSLDiffieHellman * dh = GetAt(group->m_dhOID);
    if (dh != NULL) {
      PTRACE(4, "Using  Diffie-Hellman group " << group->m_dhOID << " for " << cryptoSuite.GetDescription());
      return dh->GetSessionKey();
    }

    PTRACE(2, "No Diffie-Hellman group exchanged for " << cryptoSuite.GetDescription());
  }
  return key;
}


bool H235DiffieHellman::ToTokens(H225_ArrayOf_ClearToken & tokens) const
{
  if (IsEmpty())
    return false;

  if (m_version3) {
    PINDEX last = tokens.GetSize();
    tokens.SetSize(last+1);
    tokens[last].m_tokenOID = H235Version3; // Indicate version 3 support.
  }

  for (const_iterator it = begin(); it != end(); ++it) {
    PINDEX last = tokens.GetSize();
    tokens.SetSize(last+1);
    H235_ClearToken & token = tokens[last];
    token.m_tokenOID = it->first;
    if (it->second.GetNumBits() > 2048) {
      token.IncludeOptionalField(H235_ClearToken::e_dhkeyext);
      token.m_dhkeyext.m_halfkey.SetData(it->second.GetHalfKey());
      token.m_dhkeyext.IncludeOptionalField(H235_DHsetExt::e_modSize);
      token.m_dhkeyext.m_modSize.SetData(it->second.GetModulus());
      token.m_dhkeyext.IncludeOptionalField(H235_DHsetExt::e_generator);
      token.m_dhkeyext.m_generator.SetData(it->second.GetGenerator());
    }
    else {
      token.IncludeOptionalField(H235_ClearToken::e_dhkey);
      token.m_dhkey.m_halfkey.SetData(it->second.GetHalfKey());
      token.m_dhkey.m_modSize.SetData(it->second.GetModulus());
      token.m_dhkey.m_generator.SetData(it->second.GetGenerator());
    }
  }

  return true;
}


bool H235DiffieHellman::FromTokens(const H225_ArrayOf_ClearToken & tokens)
{
  m_version3 = false;

  for (PINDEX i = 0; i < tokens.GetSize(); ++i) {
    H235_ClearToken & token = tokens[i];
    PString oid = token.m_tokenOID.AsString();
    if (oid == H235Version3)
      m_version3 = true;
    else {
      for (PINDEX group = 0; group < PARRAYSIZE(DHGroup); ++group) {
        if (oid == DHGroup[group].m_dhOID) {
          PBYTEArray halfKey, modSize, generator;
          if (token.HasOptionalField(H235_ClearToken::e_dhkeyext)) {
            halfKey = token.m_dhkeyext.m_halfkey.GetData();
            if (token.m_dhkeyext.HasOptionalField(H235_DHsetExt::e_modSize))
              modSize = token.m_dhkeyext.m_modSize.GetData();
            if (token.m_dhkeyext.HasOptionalField(H235_DHsetExt::e_generator))
              generator = token.m_dhkeyext.m_generator.GetData();
          }
          else if (token.HasOptionalField(H235_ClearToken::e_dhkey)) {
            halfKey = token.m_dhkey.m_halfkey.GetData();
            modSize = token.m_dhkey.m_modSize.GetData();
            generator = token.m_dhkey.m_generator.GetData();
          }
          else {
            PTRACE(3, "No Diffie-Hellman key information in token");
            continue;
          }

          PSSLDiffieHellman * dh;
          if ((dh = GetAt(oid)) == NULL) {
            PTRACE(4, "Adding Diffie-Hellman group " << oid);
            dh = new PSSLDiffieHellman(modSize, generator);
            SetAt(oid, dh);
          }
          else {
            PTRACE(4, "Found offered Diffie-Hellman group " << oid);
            PTRACE_IF(3, !modSize.IsEmpty() && dh->GetModulus() != modSize,
                        "Reply Diffie-Hellman group has different modulus to offer");
          }

          if (dh->ComputeSessionKey(halfKey)) {
            PTRACE(4, "Computed key for Diffie-Hellman group " << oid
                    << "\n" << hex << fixed << setfill('0') << setprecision(4) << dh->GetSessionKey());
          }

          break;
        }
      }
    }
  }

  // Only keep replies
  for (iterator it = begin(); it != end(); ++it) {
    if (it->second.GetSessionKey().IsEmpty()) {
      PTRACE(4, "Dropping Diffie-Hellman group " << it->first);
      erase(it);
      it = begin();
    }
  }

  return !IsEmpty();
}


#endif // OPAL_H235_6

/////////////////////////////////////////////////////////////////////////////
