/*
Copyright 1990-2006 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <locale.h>

#include <gdk/gdk.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "iiim-properties.h"
#include "iiim-properties-trigger.h"

#include "kbltrans.h"

static gchar *default_langs[] = {
  "en", N_("English/European"),
  "x-kbl-ru", N_("Russian"),
  "x-kbl-el", N_("Greek"),
  "th_TH", N_("Thai"),
  "x-kbl-ar", N_("Arabic"),
  "hi_IN", N_("Indic"),
  "he", N_("Hebrew"),
  "ja", N_("Japanese"),
  "ko", N_("Korean"),
  "zh_CN", N_("S-Chinese"),
  "zh_TW", N_("T-Chinese"),
  "zh_HK", N_("T-Chinese (Hong Kong)"),
  "UNICODE-HEX", N_("UNICODE (HEX)"),
  "UNICODE-OCTAL", N_("UNICODE (OCTAL)"),
  NULL,
  /* above 14 entries are Solaris 10 default */
};

static gchar *additional_langs[] = {
  /* "- " at the top indicates that it is replaced with x-kbl- version */
  "fr_FR", ("- French"),
  "de_DE", ("- Germany"),
  "it_IT", ("- Italian"),
  "es_ES", ("- Spanish"),
  "sv_SE", ("- Swedish"),
  "pt_BR", ("- Portuguese (Brazil)"),
  "am_ET", N_("Amharic Ethiopia"),
  "as", N_("Assamese"),
  "bn", N_("Bengali (India)"),
  "cs_CS", ("- Czech"),
  "da_DK", ("- Danish"),
  "es_MX", ("- Spanish (Mexico)"),
  "fi_FI", ("- Finnish"),
  "gu_IN", N_("Gujarati (India)"),
  "kn_IN", N_("Kannada (India)"),
  "lv_LV", ("- Latvian"),
  "ml_IN", N_("Malayalam (India)"),
  "nl_NL", ("- Dutch"),
  "nn_NO", ("- Norwegian Nynorsk (Norway)"),
  "no_NO", ("- Norwegian (Norway)"),
  "pa_IN", N_("Punjabi (India)"),
  "pt_PT", ("- Portuguese (Portugal)"),
  "sk_SK", ("- Slovak (Slovakia)"),
  /*"sr", N_("Serbian"),*/
  "sr_CS", ("- Serbian (Serbia And Montenegro)"),
  "sr_YU", ("- Serbian (YU)"),
  "sv_FI", ("- Swedish (Finland)"),
  "ta_IN", N_("Tamil (India)"),
  "te_IN", N_("Telugu (India)"),
  "ti_ER", N_("Tigrinya (Eritrea)"),
  "tr_TR", ("- Turkish (Turkey)"),
  "vi_VN", N_("Vietnamese (Vietnam)"),
  "bg_BG", ("- Cyrillic (Bulgarian)"),
  /* keyboard layout emulation type */
  "x-kbl-al", N_("Albanian"),
  /* "x-kbl-ar", N_("Arabic"),*/
  "x-kbl-by", N_("Belarusian"),
  "x-kbl-be", N_("Belgian"),
  "x-kbl-bs", N_("Bosnian/Herzegovina"),
  "x-kbl-br", N_("Brazilian/Portuguese"),
  "x-kbl-bg", N_("Bulgarian"),
  "x-kbl-ca_enhanced", N_("Canadian/French"),
  "x-kbl-ca", N_("Canadian"),
  "x-kbl-hr", N_("Croatian"),
  "x-kbl-cz", N_("Czech"),
  "x-kbl-cz_qwerty", N_("Czech/Qwerty"),
  "x-kbl-dk", N_("Danish"),
  "x-kbl-nl", N_("Dutch"),
  "x-kbl-ee", N_("Estonian"),
  "x-kbl-fi", N_("Finnish"),
  "x-kbl-fr", N_("French"),
  "x-kbl-ge", N_("Georgian"),
  "x-kbl-de", N_("German"),
  /*"x-kbl-el", N_("Greek"),*/
  "x-kbl-hu", N_("Hungarian"),
  "x-kbl-is", N_("Icelandic"),
  "x-kbl-it", N_("Italian"),
  "x-kbl-ja", N_("Japanese"),
  "x-kbl-kz", N_("Kazakhstan"),
  "x-kbl-ko", N_("Korean"),
  "x-kbl-lv", N_("Latvian"),
  "x-kbl-lt", N_("Lithuanian"),
  "x-kbl-mk", N_("Macedonian"),
  "x-kbl-mt", N_("Maltese/UK"),
  "x-kbl-mt_us", N_("Maltese/US"),
  "x-kbl-no", N_("Norwegian"),
  "x-kbl-pl", N_("Polish"),
  "x-kbl-pt", N_("Portuguese"),
  "x-kbl-ro", N_("Romanian"),
  /*"x-kbl-ru", N_("Russian"),*/
  "x-kbl-sr", N_("Serbian"),
  "x-kbl-sk", N_("Slovak"),
  "x-kbl-sk_qwerty", N_("Slovak/Qwerty"),
  "x-kbl-si", N_("Slovenian"),
  "x-kbl-es", N_("Spanish"),
  "x-kbl-se", N_("Swedish"),
  "x-kbl-fr_CH", N_("Swiss/French"),
  "x-kbl-de_CH", N_("Swiss/German"),
  "x-kbl-tw", N_("Traditional/Chinese"),
  "x-kbl-tr_f", N_("Turkish/F"),
  "x-kbl-tr", N_("Turkish/Q"),
  "x-kbl-ua", N_("Ukrainian"),
  "x-kbl-pk", N_("Urdu"),
  "x-kbl-gb", N_("UK"),
  "x-kbl-dvorak", N_("US/Dvorak"),
  "x-kbl-en", N_("US/English"),
  NULL,
};  

static gchar *alias_langs[] = {
  "el", "el_GR",
  "zh", "zh_CN",
  "fr", "fr_FR",
  "de", "de_DE",
  "it", "it_IT",
  "es", "es_ES",
  "sv", "sv_SE",
  "th", "th_TH",
  NULL,
};

static gboolean no_cf = FALSE;
static gboolean on_CDE = FALSE;

static IIIMProps *system_default;
static GSList *override_list = NULL;

GSList *cf_get_supported_langs ();
GSList *cf_get_engines (gchar *);
GSList *cf_get_hotkeys (gchar *);
GSList *cf_get_user_layout_engine ();
GSList *sendlayout_get_use_layout_list ();
gchar *sendlayout_get_ui_from_id (gchar *);
gchar *sendlayout_get_id_from_ui (gchar *);

/*
 * no_cf == TRUE means no iiimd is running.
 * (ex: non UTF-8 locale)
 */
void setget_set_no_cf ()
{
  no_cf = TRUE;
}

void setget_set_on_CDE ()
{
  on_CDE = TRUE;
}

gboolean setget_get_on_CDE ()
{
  return on_CDE;
}

gboolean is_kbl (void *id) {
  if (strncmp ((gchar *)id, "x-kbl-", 6)) {
    return FALSE;
  }
  return TRUE;
}

GSList *setget_get_user_defined_kbl_list ();
/*
 * get lang_id from localized language string
 */
LanguageID *setget_langstr_to_lang_id (gchar *l10nstr)
{
  LanguageID *lid = NULL;
  gint i;

  for (i = 0; default_langs[i] != NULL; i += 2)
    {
      if (!strcmp (_(default_langs[i + 1]), l10nstr))
        {
	  lid = g_new0 (LanguageID, 1);
	  lid->id = default_langs[i];
	  lid->ui = l10nstr;
	  /* TODO extract engine info */
        }
    }
  if (lid == NULL)
    {
      for (i = 0; additional_langs[i] != NULL; i += 2)
	{
	  if (!strcmp (_(additional_langs[i + 1]), l10nstr))
	    {
	      lid = g_new0 (LanguageID, 1);
	      lid->id = additional_langs[i];
	      lid->ui = l10nstr;
	      /* TODO extract engine info */
	    }
	}
    }
  if (lid == NULL)
    {
      GSList *user_defined = setget_get_user_defined_kbl_list ();
      GSList *lp;
      for (lp = user_defined; lp; lp = g_slist_next (lp))
	{
	  if (!strcmp (((LanguageID *)(lp->data))->ui, l10nstr))
	    {
	      lid = lp->data;
	    }
	}
    }

  return lid;
}

/*
 * get lang_id from localized language string
 */
gchar *setget_lang_to_id (gchar *l10nstr)
{
  gint i;

  for (i = 0; default_langs[i] != NULL; i += 2)
    {
      if (!strcmp (_(default_langs[i + 1]), l10nstr))
        {
          return default_langs[i];
        }
    }
  for (i = 0; additional_langs[i] != NULL; i += 2)
    {
      if (!strcmp (_(additional_langs[i + 1]), l10nstr))
        {
          return additional_langs[i];
        }
    }

  /* try user defined */
  return sendlayout_get_id_from_ui (l10nstr);
}

gchar *setget_nol10nlang_to_id (gchar *lang)
{
  gint i;

  for (i = 0; default_langs[i] != NULL; i += 2)
    {
      if (!strcmp (default_langs[i + 1], lang))
	{
	  return default_langs[i];
	}
    }
  for (i =0; additional_langs[i] != NULL; i += 2)
    {
      if (!strcmp (additional_langs[i + 1], lang))
	{
	  return additional_langs[i];
	}

    }
  return NULL;
}
    

/*
 * return UI string for lang_id
 */
static gchar *get_ui (gchar *id)
{
  int i;

  for (i = 0; default_langs[i] != NULL; i += 2)
    {
      if (!strcmp (id, default_langs[i]))
	{
	  return _(default_langs[i + 1]);
	}
    }
  for (i = 0; additional_langs[i] != NULL; i += 2)
    {
      if (!strcmp (id, additional_langs[i]))
	{
	  return _(additional_langs[i + 1]);
	}
    }
}

/*
 * This function returns AND list of two list
 * with keeping the order of second list.
 */
GSList *setget_and_list_of_lang (GSList *list_a, GSList *list_b)
{
  GSList *ret = NULL;
  LanguageID *l1, *l2;

  GSList *p1, *p2;

  for (p2 = list_b; p2; p2 = g_slist_next (p2))
    {
      l1 = (LanguageID *)p2->data;
      for (p1 = list_a; p1; p1 = g_slist_next (p1))
	{
	  l2 = (LanguageID *)p1->data;
	  if (!strcmp (l1->id, l2->id))
	    {
	      ret = g_slist_append (ret, l1);
	    }
	}
    }
  return ret;
}

/*
 * This function returns Language list
 * which current installed LE's supported langauages
 * with UI representation
 */
GSList *setget_get_available_lang_list ()
{
  GSList *list = NULL;
  GSList *ret = NULL;
  GSList *lp;
  LanguageID *lid;
  
  list = cf_get_supported_langs ();

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      lid = g_new0 (LanguageID, 1);
      lid->id = strdup (lp->data);
      lid->ui = get_ui (lp->data);
      if (is_kbl (lp->data)) {
	lid->kbl = TRUE;
      } else {
	lid->kbl = FALSE;
      }
      lid->user_defined = FALSE;
      ret = g_slist_append (ret, lid);
    }
  g_slist_free (list);
  
  return ret;
}

/*
 * This function returns Language list
 * which current user's custom keyboard layout.
 *
 * user's defined layout name would be x-xkl-XX-YYYY
 */
static GSList *user_defined_kbl_list = NULL;
GSList *setget_get_user_defined_kbl_list ()
{
  GSList *list = NULL;
  GSList *ret = NULL;
  GSList *lp;
  LanguageID *lid;
  CustomLayout *cl;

  if (user_defined_kbl_list != NULL)
    {
      return user_defined_kbl_list;
    }

  list = sendlayout_get_use_layout_list();
  for (lp = list; lp; lp = g_slist_next (lp))
    {
      cl = (CustomLayout *)lp->data;
      lid = g_new0 (LanguageID, 1);
      lid->id = cl->id;
      lid->ui = cl->name;
      lid->kbl = TRUE;
      lid->user_defined = TRUE;
      lid->engines = cf_get_user_layout_engine ();
      ret = g_slist_append (ret, lid);
    }

  return ret;
}

gboolean
is_user_defined (gchar *id)
{
  if (g_str_has_prefix (id, "x-kbl-") && g_str_has_suffix (id, USER_DEFINED_ID_SUFFIX)) {
    return TRUE;
  }
  return FALSE;
}

/*
 * Get LE list (EngineDesc list)
 */
GSList *setget_get_available_engine_list (gchar *lang_id)
{
  GSList *engines = NULL;
  GSList *engines2 = NULL;
  gint i;

  if (is_user_defined (lang_id))
    {
      return cf_get_user_layout_engine ();
    }

  /* this list contains IIIMCF_input_method list */
  engines = cf_get_engines (lang_id);
  for (i = 0; alias_langs[i] != NULL; i += 2)
    {
      if (!strcmp (lang_id, alias_langs[i + 1]))
	{
	  engines2 = cf_get_engines (alias_langs[i]);
	  engines = g_slist_concat (engines, engines2);
	  break;
	}
    }

  return engines;
}

/*
 * Get default language list from system config file - TODO
 */
static GSList *get_default_lang_list ()
{
  GSList *list = NULL;
  gint i;
  LanguageID *lid;

  if (system_default != NULL)
    {
      GSList *lp;
      for (lp = system_default->l_languages->langs; lp; lp = g_slist_next (lp))
	{
	  lid = g_new0 (LanguageID, 1);
	  lid->id = strdup ((gchar *)lp->data);
	  lid->ui = get_ui (lid->id);
	  if (is_kbl (lp->data)) {
	    lid->kbl = TRUE;
	  } else {
	    lid->kbl = FALSE;
	  }
	  list = g_slist_append (list, lid);
	}
    }
  else
    {
      for (i = 0; default_langs[i] != NULL; i += 2)
	{
	  lid = g_new0 (LanguageID, 1);
	  lid->id = strdup (default_langs[i]);
	  lid->ui = _(default_langs[i + 1]);
	  if (is_kbl (lid->id)) {
	    lid->kbl = TRUE;
	  } else {
	    lid->kbl = FALSE;
	  }
	  list = g_slist_append (list, lid);
	}
    }
  return list;
}

#define DELIMITER ";"
#define HOTKEY_START "["
#define HOTKEY_END "]"

/*
 * make property format string from lang list and engine list
 * _IIIM_SWITCHER_INPUT_LANGUAGE_LIST is the string representation of
 * language/script/engine list. One element consists of three part
 * lang_id, lang_ui, and engine_id. If engine is not needed
 * to be specified, then engine_id would be empty.
 * " - " is the special separator between language and it's engine
 * in UI string.
 * Ex: en;English/European;;ja;Japanese - ATOK;atokle;ja;Japanese - Wnn8;wnn8;...
 */
static gchar *make_lang_string (LanguageList *llist)
{
  GSList *lp;
  gchar buf[MAX_LANG_LIST_LEN];
  gchar *p;
  gint len = 0;
  gint add_len = 0;

  p = buf;
  for (lp = llist->langs; lp; lp = g_slist_next (lp))
    {
      LanguageID* langid = (LanguageID *)lp->data;

      if (langid->engines != NULL /* && g_slist_length (langid->engines) > 1 */)
	{
	  GSList *lp2;

	  for (lp2 = langid->engines; lp2; lp2 = g_slist_next (lp2))
	    {
	      gchar *ui;
	      EngineDesc *desc = (EngineDesc *)lp2->data;
	      if (g_slist_length (langid->engines) > 1)
		{
		  ui = g_strconcat (langid->ui, " - ", desc->ui, NULL);
		}
	      else
		{
		  ui = strdup (langid->ui);
		}
	      add_len = (strlen (langid->id) + strlen (ui) +
			 strlen (desc->id) + strlen (DELIMITER) * 3);
	      if (len + add_len >= MAX_LANG_LIST_LEN)
		{
		  g_free (ui);
		  break;
		}
	      p = g_stpcpy (p, langid->id);
	      p = g_stpcpy (p, DELIMITER);
	      p = g_stpcpy (p, ui);
	      p = g_stpcpy (p, DELIMITER);
	      p = g_stpcpy (p, desc->id);
	      p = g_stpcpy (p, DELIMITER);
	      g_free (ui);
	      len += add_len;
	    }
	}
      else
	{
	  add_len = (strlen (langid->id) + strlen (langid->ui) + strlen (DELIMITER) * 3);
	  if (len + add_len >= MAX_LANG_LIST_LEN)
	    {
	      break;
	    }
	  p = g_stpcpy (p, langid->id);
	  p = g_stpcpy (p, DELIMITER);
	  p = g_stpcpy (p, langid->ui);
	  p = g_stpcpy (p, DELIMITER);
	  p = g_stpcpy (p, DELIMITER);
	  len += add_len;
	}
    }
  return g_strndup (buf, len);
}

/*
 * format is lang_id;lang_ui;engine_id;
 */
static GSList *make_lang_string_list (LanguageList *llist)
{
  LanguageID *lid;
  GSList *lp;
  GSList *ret = NULL;
  gchar buf[MAX_LANG_LIST_LEN];
  gchar *p;
  gint len = 0;
  gchar *langstr;

  for (lp = llist->langs; lp; lp = g_slist_next (lp))
    {
      lid = (LanguageID *)lp->data;
      if (lid->engines != NULL && g_slist_length (lid->engines) > 1)
	{
	  GSList *lp2;
	  for (lp2 = lid->engines; lp2; lp2 = g_slist_next (lp2))
	    {
	      EngineDesc *desc = (EngineDesc *)lp2->data;
	      gchar *ui = g_strconcat (lid->ui, " - ", desc->ui, NULL);
	      len = (strlen (lid->id) + strlen (ui) + strlen (desc->id) +
		     strlen (DELIMITER) * 3);
	      if (len >= MAX_LANG_LIST_LEN)
		{
		  /* ignore too long name (must be corrupted data) */
		  continue;
		}
	      p = buf;
	      p = g_stpcpy (p, lid->id);
	      p = g_stpcpy (p, DELIMITER);
	      p = g_stpcpy (p, ui);
	      p = g_stpcpy (p, DELIMITER);
	      p = g_stpcpy (p, desc->id);
	      p = g_stpcpy (p, DELIMITER);
	      ret = g_slist_append (ret, g_strndup (buf, len));
	      g_free (ui);
	    }
	}
      else
	{
	  len = (strlen (lid->id) + strlen (lid->ui) +
		 strlen (DELIMITER) * 3);
	  if (len >= MAX_LANG_LIST_LEN)
	    {
	      /* ignore too long name (must be corrupted data) */
	      continue;
	    }
	  p = buf;
	  p = g_stpcpy (p, lid->id);
	  p = g_stpcpy (p, DELIMITER);
	  p = g_stpcpy (p, lid->ui);
	  p = g_stpcpy (p, DELIMITER);
	  p = g_stpcpy (p, DELIMITER);
	  ret = g_slist_append (ret, g_strndup (buf, len));
	}
    }
  return ret;
}

static gint lidcmp (gconstpointer a, gconstpointer b)
{
  const char *ac, *bc;
  gchar *temp, *p;
  bc = (const char *)b;

  ac = (const char *)((LanguageID *)a)->id;
  temp = g_strconcat (ac, ";", NULL);
  p = strstr (bc, (const char *)temp);
  g_free (temp);

  return (p == bc);
}
 
/*
 * make string to keep for KEY_LANG_TO_SAVE_LIST
 */
static GSList *make_lang_save_string_list (IIIMProps *props)
{
  GSList *saved = props->l_saved_languages;
  GSList *available = setget_get_available_lang_list();
  GSList *selected = props->l_languages->langs;
  GSList *lp, *lp2;
  GSList *ret = NULL;
  gchar buf[MAX_LANG_LIST_LEN];
  gchar *p;
  gint len;

  for (lp = selected; lp; lp = g_slist_next (lp))
    {
      p = buf;
      len = 0;
      LanguageID *lid = (LanguageID *)lp->data;
      if (is_user_defined (lid->id))
	{
	  continue;
	}
      p = g_stpcpy (p, lid->id);
      p = g_stpcpy (p, DELIMITER);
      len = strlen (lid->id) + strlen (DELIMITER);
      if (lid->engines != NULL)
	{
	  for (lp2 = lid->engines; lp2; lp2 = g_slist_next (lp2))
	    {
	      EngineDesc *desc = (EngineDesc *)lp2->data;
	      p = g_stpcpy (p, desc->id);
	      p = g_stpcpy (p, DELIMITER);
	      len += (strlen (desc->id) + strlen (DELIMITER));
	    }
	}
      ret = g_slist_append (ret, g_strndup (buf, len));
    }
  /* append languages which were saved before but
   * not listed in current sessiong due to configuration change
   * -- saved is the string list and available is LanguageID list --
   */
  for (lp = saved; lp; lp = g_slist_next (lp))
    {
      gchar *saved_id = (gchar *)lp->data;
      GSList *found = g_slist_find_custom (available, saved_id, lidcmp);
      if (found == NULL)
	{
	  ret = g_slist_append (ret, saved_id);
	}
    }
  return ret;
}

gchar *make_key_string (Hotkeys *list)
{
  GSList *lp;
  gchar buf[MAX_HOTKEY_STRING_SIZE];
  gchar *p;

  if (!list) return NULL;

  memset (buf, 0, MAX_HOTKEY_STRING_SIZE);
  p = buf;

  for (lp = list->hotkeys; lp; lp = g_slist_next (lp))
    {
      gchar *key = (gchar *)lp->data;

      if (lp != list->hotkeys)
	p = g_stpcpy (p, HOTKEY_DELIMITER);
      p = g_stpcpy (p, key);
    }

  return g_strdup (buf);
}

Hotkeys *make_hotkey_list (gchar *hotkey_str)
{
  Hotkeys *list;
  gchar **keys, **p;

  list = g_new0 (Hotkeys, 1);
  if (!hotkey_str || !*hotkey_str)
    {
      list->hotkeys = g_slist_append (list->hotkeys, "");
    }
  else
    {
      keys = g_strsplit ((const gchar *)hotkey_str, HOTKEY_DELIMITER, -1);

      for (p = keys; *p; p++)
	list->hotkeys = g_slist_append (list->hotkeys, *p);
    }

  return list;
}

static void set_gconf (GConfClient *client, const gchar *key, const gchar *value)
{
  if (!gconf_client_set_string (client, key, value, NULL))
    {
      g_message ("GConf writer error : %s/%s\n", key, value);
    }
}

static void set_gconf_int (GConfClient *client, const gchar *key, const gint value)
{
  if (!gconf_client_set_int (client, key, value, NULL))
    {
      g_message ("GConf writer error : %s/%d\n", key, value);
    }
}

static void set_gconf_bool (GConfClient *client, const gchar *key, const gboolean value)
{
  if (!gconf_client_set_bool (client, key, value, NULL))
    {
      g_message ("GConf writer error : %s/%d\n", key, value);
    }
}

static gint desc_strcmp (gconstpointer a, gconstpointer b)
{
  EngineDesc *desc = (EngineDesc *)a;
  return (gint)strcmp((const char *)desc->id, (const char *)b);
}

static gint my_strcmp (gconstpointer a, gconstpointer b)
{
  return (gint)strcmp((const char *)a, (const char *)b);
}

static gint cmp_lid_name (gconstpointer a, gconstpointer b)
{
  LanguageID *lid = (LanguageID *)b;
  return (gint)strcmp((const char *)a, (const char *)lid->ui);
}

/*
 * sort function which puts default engine at the top of list
 */
static gint sortWithDefaultEngine (gconstpointer a, gconstpointer b, gpointer user_data)
{
  const gchar *default_engine = (const gchar *)user_data;
  EngineDesc *desc_a = (EngineDesc *)a;
  EngineDesc *desc_b = (EngineDesc *)b;

  if (!strcmp (desc_a->id, default_engine))
    return -1;
  if (!strcmp (desc_b->id, default_engine))
    return 1;
  return 0;
}

/*
 * add engine info to default LanguageID list
 */
static GSList *add_default_engine_info (GSList *list)
{
  GSList *lp;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      LanguageID *lid = (LanguageID *)lp->data;
      lid->engines = setget_get_available_engine_list (lid->id);
      if (g_slist_length (lid->engines) > 1 &&
	  system_default != NULL && system_default->l_language_engine != NULL)
	{
	  GHashTable *engine_table = system_default->l_language_engine->engine_table;
	  gchar *default_engine = g_hash_table_lookup (engine_table, lid->id);
	  if (default_engine != NULL)
	    {
	      lid->engines = g_slist_sort_with_data (lid->engines,
						     sortWithDefaultEngine,
						     default_engine);
	    }
	}
    }
  return list;
}

static GSList *delete_non_exist_entry (GSList *list)
{
  GSList *current_list = sendlayout_get_use_layout_list ();
  GSList *lp;
  GSList *del_list = NULL;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      if (!g_slist_find_custom (current_list, lp->data, my_strcmp)) {
	g_slist_append (del_list, lp->data);
      }
    }
  for (lp = del_list; lp; lp = g_slist_next (lp))
    {
      g_slist_remove (list, lp->data);
    }
  
  return list;
}

/*
 * convert lang_id list to LanguageID list which contains
 * currently available languages
 */
static GSList *convert_to_langid_list (GSList *list)
{
  GSList *ret = NULL;
  GSList *lp;
  LanguageID *lid;
  gint i;
  GSList *available = cf_get_supported_langs ();
  GSList *engines;
  gchar **elements;
  gchar *lang_id;
  GSList *found_engine;

  for (lp = list; lp; lp = g_slist_next (lp))
    {
      elements = g_strsplit ((const gchar *)lp->data, DELIMITER, -1);
      lang_id = elements[0];
      GSList *found = g_slist_find_custom (available, lang_id, my_strcmp);
      if (found != NULL)
	{
	  GSList *valid_engines = NULL;
	  lid = g_new0 (LanguageID, 1);
	  lid->id = lang_id;
	  lid->ui = get_ui (lang_id);
	  if (is_kbl (lang_id)) {
	    lid->kbl = TRUE;
	  } else {
	    lid->kbl = FALSE;
	  }
	  engines = setget_get_available_engine_list (lang_id);
	  for (i = 0;; i++)
	    {
	      if (elements[i] == NULL || !strcmp (elements[i], ""))
		{
		  break;
		}
	      found_engine = g_slist_find_custom (engines, elements[i], desc_strcmp);
	      if (found_engine != NULL)
		{
		  valid_engines = g_slist_append (valid_engines, found_engine->data);
		}
	    }
	  if (valid_engines != NULL /* && g_slist_length (valid_engines) > 1*/)
	    {
	      lid->engines = valid_engines;
	    }
	  else
	    {
	      /* if only one engine is available for specific language,
	       * then language list does not need to show engine name
	       */
	      lid->engines = NULL;
	    }
	  ret = g_slist_append (ret, lid);
	} else {
	  /* lang_id may be user defined one */
	  if (is_user_defined (lang_id)) {
	    lid = g_new0 (LanguageID, 1);
	    lid->id = lang_id;
	    lid->ui = sendlayout_get_ui_from_id (lang_id);
	    lid->kbl = TRUE;
	    lid->user_defined = TRUE;
	    lid->engines = cf_get_user_layout_engine ();
	  }
	}
    }

  return ret;
}

static void set_gconf_list (GConfClient *client, const gchar *key, GSList *value)
{
  if (!gconf_client_set_list (client, key, GCONF_VALUE_STRING, value, NULL))
    {
      g_message ("GConf writer error : %s\n", key);
    }
}

void setget_write_back_to_gconf_and_x_props (IIIMProps *props)
{
  GConfClient *client;
  client = gconf_client_get_default ();
  gchar *value;
  gchar *gimlet_value;
  gint gimlet_value_int;
  gboolean gimlet_value_bool;
  gchar *use_language_list;
  gchar *use_hotkey_list = NULL;
  GSList *use_language_gslist;
  GSList *save_language_gslist;
  GdkDisplay *display;
  gint screen_count;
  gint n;

  IMSettings *gimlet_setting = g_new0 (IMSettings, 1);
  IIIMSettings *iiim_props_setting = g_new0 (IIIMSettings, 1);

  /*--------- im enabled --------*/
  if (props->b_im_enabled)
    {
      value = V_TRUE;
      gimlet_setting->im_enabled = IM_ON;
      gimlet_value = "enabled";
    }
  else
    {
      value = V_FALSE;
      gimlet_setting->im_enabled = IM_OFF;
      gimlet_value = "disabled";
    }
  set_gconf (client, KEY_ENABLE_IM, value);
  set_gconf (client, KEY_ENABLE_IM_GIMLET, gimlet_value);

  /*--------- switcher autostart -------*/
  if (props->b_switcher_autostart)
    {
      value = V_TRUE;
      gimlet_setting->switcher_autostart = IM_ON;
    }
  else
    {
      gimlet_setting->switcher_autostart = IM_OFF;
      value = V_FALSE;
    }

  set_gconf (client, KEY_ENABLE_GIMLET_AUTOSTART, value);

  /*--------- status enabled -------*/
  if (props->b_status_enabled)
    {
      value = V_TRUE;
      gimlet_setting->status_enabled = IM_ON;
      gimlet_value = "enabled";
    }
  else
    {
      value = V_FALSE;
      gimlet_setting->status_enabled = IM_OFF;
      gimlet_value = "disabled";
    }
  set_gconf (client, KEY_STATUS_DISPLAY, value);
  set_gconf (client, KEY_STATUS_DISPLAY_GIMLET, gimlet_value);

  /*--------- status placement -------*/
  switch (props->e_status_placement)
    {
    case S_DESKTOP_DEFAULT:
      value = V_DESKTOP_DEFAULT;
      if (on_CDE)
	{
	  gimlet_setting->status_placement = ATTACH_TO_APP_FRAME;
	  gimlet_value = "application";
	}
      else
	{
	  gimlet_setting->status_placement = ON_DESKTOP_PANEL;
	  gimlet_value = "panel";
	}
      break;
    case S_PANEL:
      value = V_PANEL;
      gimlet_setting->status_placement = ON_DESKTOP_PANEL;
      gimlet_value = "panel";
      break;
    case S_ATTACH_FRAME:
      value = V_ATTACH_FRAME;
      gimlet_setting->status_placement = ATTACH_TO_APP_FRAME;
      gimlet_value = "applicaton";
      break;
    case S_NONE:
      value = V_NONE;
      gimlet_setting->status_placement = NONE;
      gimlet_value = "none";
      break;
    default:
      value = V_DESKTOP_DEFAULT;
      gimlet_setting->status_placement = ON_DESKTOP_PANEL;      
      gimlet_value = "panel";
    }
  set_gconf (client, KEY_STATUS_PLACEMENT, value);
  set_gconf (client, KEY_STATUS_PLACEMENT_GIMLET, gimlet_value);

  /*------------- sync activation -----------------*/
  if (props->b_sync_activation)
    {
      value = V_TRUE;
      iiim_props_setting->sync_activation = IM_ON;
    }
  else
    {
      value = V_FALSE;
      iiim_props_setting->sync_activation = IM_OFF;
    }
  set_gconf (client, KEY_SYNC_ACTIVATION, value);

  /*------------ lang switch policy ---------------*/
  switch (props->e_lang_switch_policy)
    {
    case LS_IC:
      value = V_IC;
      iiim_props_setting->lang_switch_policy = LS_IC;
      gimlet_value_int = FOLLOW_APPLICATION;
      break;
    case LS_DESKTOP:
      value = V_DESKTOP;
      iiim_props_setting->lang_switch_policy = LS_DESKTOP;
      gimlet_value_int = FOLLOW_QUICK_ACCESS_MENU;
      break;
    default:
      value = V_IC;
      iiim_props_setting->lang_switch_policy = LS_IC;
      gimlet_value_int = FOLLOW_APPLICATION;
    }
  set_gconf (client, KEY_LANG_SWITCH_POLICY, value);
  set_gconf_int (client, KEY_INPUT_LANG_POLICY_GIMLET, gimlet_value_int);
  gconf_client_notify (client, KEY_INPUT_LANG_POLICY_GIMLET);


  /*--------- on/off at swtching policy -----------*/
  switch (props->e_on_off_policy)
    {
    case OO_ACTIVATE:
      value = V_ACTIVATE;
      iiim_props_setting->on_off_policy = OO_ACTIVATE;
      gimlet_value_bool = TRUE;
      break;
    case OO_DONOT_CHANGE:
      value = V_DONOT_CHANGE;
      iiim_props_setting->on_off_policy = OO_DONOT_CHANGE;
      gimlet_value_bool = FALSE;
      break;
    default:
      value = V_ACTIVATE;
      iiim_props_setting->on_off_policy = OO_ACTIVATE;
      gimlet_value_bool = TRUE;
    }
  set_gconf (client, KEY_ON_OFF_POLICY, value);
  set_gconf_bool (client, KEY_CONVERSION_ON_GIMLET, gimlet_value_bool);

  /*------------- remember last le -----------------*/
  if (props->b_remember_last_le)
    {
      value = V_TRUE;
      iiim_props_setting->remember_last_le= IM_ON;
    }
  else
    {
      value = V_FALSE;
      iiim_props_setting->remember_last_le = IM_OFF;
    }
  set_gconf (client, KEY_REMEMBER_LAST_LE, value);

  /*------------ language list & engine list ----------*/ 
  /* use_language_list is for X properties */

  {
    GSList *lp;
    for (lp = props->l_languages->langs; lp; lp = g_slist_next (lp)) {
      LanguageID *lid = (LanguageID *)lp->data;
    }
  }

  use_language_list = make_lang_string (props->l_languages);
  use_language_gslist = make_lang_string_list (props->l_languages);
  /* element in use list : lang_id;lang_ui;engine_id; */
  set_gconf_list (client, KEY_LANG_TO_USE_LIST, use_language_gslist);
  save_language_gslist = make_lang_save_string_list (props);
  /* element in save list : lang_id;engine_id;engine_id;...; */
  set_gconf_list (client, KEY_LANG_TO_SAVE_LIST, save_language_gslist);
  /* non use user defined layout */
  set_gconf_list (client, KEY_USER_DEFINED_NON_USE_LIST, props->l_user_defined_non_use);

  /*------------------ trigger keys -------------------*/
  if (props->l_trigger_keys)
    {
      value = make_key_string (props->l_trigger_keys);
      set_gconf (client, KEY_TRIGGER, value);
      
      use_hotkey_list = g_strconcat (TRIGGER_L, LABEL_DELIMITER, value, NULL);
      g_free (value);
    }

  /*----------------- keyborad -------------------*/
  value = props->b_kbd_check ? V_TRUE : V_FALSE;
  set_gconf (client, KEY_KBD_CHECK, value);
  set_gconf_int (client, KEY_KBD_LAYOUT, props->i_kbd_layout);
  iiim_props_setting->kbd_layout = props->i_kbd_layout;

  /*---------------- lookup  ------------------*/
  /* This property is always true for now */
  set_gconf (client, KEY_ENABLE_LOOKUP, V_TRUE);
  gimlet_setting->lookup_enabled = IM_ON;

  /*----------------- misc --------------------*/
  value = props->b_lang_list_enabled ? V_TRUE : V_FALSE;
  set_gconf (client, KEY_ENABLE_LANG_LIST, value);

  switch (props->e_lang_list_placement)
    {
    case LL_INPUT_LOCATION:
      value = V_INPUT_LOCATION;
      break;
    case LL_PANEL:
      value = V_PANEL;
      break;
    case LL_STATUS:
      value = V_STATUS;
      break;
    default:
      value = V_INPUT_LOCATION;
    }
  set_gconf (client, KEY_LANG_LIST_PLACEMENT, value);

  if (props->l_lang_list_hotkey)
    {
      value = make_key_string (props->l_lang_list_hotkey);
      set_gconf (client, KEY_LANG_LIST_HOTKEY, value);

      if (!props->b_lang_list_enabled)
	{
	  g_free (value);
	  value = g_strdup ("");
	}

      if (use_hotkey_list)
	{
	  char *tmp_list = g_strconcat (use_hotkey_list, LABEL_DELIMITER,
					LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	  g_free (use_hotkey_list);
	  use_hotkey_list = tmp_list;
	}
      else
	{
	  use_hotkey_list = g_strconcat (LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	}
      g_free (value);
    }
  
  value = props->b_misc_hotkey_enabled ? V_TRUE : V_FALSE;
  set_gconf (client, KEY_ENABLE_MISC_HOTKEY, value);

  if (props->l_cycle_lang_hotkey)
    {
      value = make_key_string (props->l_cycle_lang_hotkey);
      set_gconf (client, KEY_CYCLE_LANG_HOTKEY, value);

      if (!props->b_misc_hotkey_enabled)
	{
	  g_free (value);
	  value = g_strdup ("");
	}

      if (use_hotkey_list)
	{
	  char *tmp_list = g_strconcat  (use_hotkey_list, LABEL_DELIMITER,
					 CYCLE_LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	  g_free (use_hotkey_list);
	  use_hotkey_list = tmp_list;
	}
      else
	{
	  use_hotkey_list = g_strconcat (CYCLE_LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	}
      g_free (value);
    }

  if (props->l_reverse_cycle_lang_hotkey)
    {
      value = make_key_string (props->l_reverse_cycle_lang_hotkey);
      set_gconf (client, KEY_REVERSE_CYCLE_LANG_HOTKEY, value);

      if (!props->b_misc_hotkey_enabled)
	{
	  g_free (value);
	  value = g_strdup ("");
	}

      if (use_hotkey_list)
	{
	  char *tmp_list = g_strconcat (use_hotkey_list, LABEL_DELIMITER,
					RE_CYCLE_LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	  g_free (use_hotkey_list);
	  use_hotkey_list = tmp_list;
	}
      else
	{
	  use_hotkey_list = g_strconcat (RE_CYCLE_LE_SWITCH_L, LABEL_DELIMITER, value, NULL);
	}
      g_free (value);
    }

  /* x property setting */
  display = gdk_display_get_default ();
  screen_count = gdk_display_get_n_screens (display);
  for (n = 0; n < screen_count; n++) {
    /* gimlet compatibility */
    GdkWindow *root;
    GdkAtom im_settings_atom, language_list_atom, iiim_settings_atom,
            hotkey_list_atom, desktop_lang_atom;
    GdkScreen *screen;

    screen = gdk_display_get_screen (display, n);
    root = gdk_screen_get_root_window (screen);

    im_settings_atom = gdk_atom_intern ("_IM_SETTINGS", FALSE);
    gdk_property_change (root, im_settings_atom, im_settings_atom,
			 32, GDK_PROP_MODE_REPLACE,
			 (guchar *)gimlet_setting,
			 (sizeof (IMSettings)) / (sizeof (int)));

    /* use language list */
    language_list_atom = gdk_atom_intern ("_IIIM_SWITCHER_INPUT_LANGUAGE_LIST", FALSE);
    gdk_property_change (root, language_list_atom, GDK_TARGET_STRING,
			 8, GDK_PROP_MODE_REPLACE,
			 (guchar *)use_language_list, strlen (use_language_list));

    /* use hotkey list */
    if (use_hotkey_list)
      {
	hotkey_list_atom = gdk_atom_intern ("_IIIM_SWITCHER_HOTKEY_LIST", FALSE);
	gdk_property_change (root, hotkey_list_atom, GDK_TARGET_STRING,
			     8, GDK_PROP_MODE_REPLACE,
			     (guchar *)use_hotkey_list, strlen (use_hotkey_list));
      }

    /* other properties */
    iiim_settings_atom = gdk_atom_intern ("_IIIM_SETTINGS", FALSE);
    gdk_property_change (root, iiim_settings_atom, iiim_settings_atom,
			 32, GDK_PROP_MODE_REPLACE,
			 (guchar *)iiim_props_setting,
			 (sizeof (IIIMSettings)) / (sizeof (int)));

    /* set the last le as the desktop input language property */

    if (props->b_remember_last_le) {
        gchar * last_le = gconf_client_get_string (client, KEY_LANG_LAST_SELECTION, NULL);
        if (last_le) {
          desktop_lang_atom = gdk_atom_intern ("_IIIM_SWITCHER_DESKTOP_INPUT_LANGUAGE", FALSE);
          gdk_property_change (gdk_get_default_root_window (),
                               desktop_lang_atom, desktop_lang_atom,
                               8, GDK_PROP_MODE_REPLACE,
                               (guchar *)last_le, strlen (last_le) + 1); /* including last NULL */
          g_free (last_le);
        }
    }
  }
  g_free (use_language_list);
  g_free (use_hotkey_list);
  g_free (gimlet_setting);
  g_free (iiim_props_setting);
}

/*
 * get system default value from $sysconfigdir/iiim/client.xml
 */
static void set_boolean (xmlChar *val, gboolean override, gpointer ptr)
{
  gboolean b = xmlStrcmp (val, (const xmlChar *)"True") ? FALSE : TRUE;

  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((gboolean *)ptr) = b;
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      *((gboolean *)ptr) = b;
    }
}

static void set_status_placement (xmlChar *val, gboolean override, gpointer ptr)
{
  StatusPlacement place = S_DESKTOP_DEFAULT;
  if (!xmlStrcmp (val, (const xmlChar *)"Panel"))
    place = S_PANEL;
  else if (!xmlStrcmp (val, (const xmlChar *)"AttachFrame"))
    place = S_ATTACH_FRAME;
  else if (!xmlStrcmp (val, (const xmlChar *)"None"))
    place = S_NONE;

  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((StatusPlacement *)ptr) = place;
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      *((StatusPlacement *)ptr) = place;
    }
}

static void set_switch_policy (xmlChar *val, gboolean override, gpointer ptr)
{
  LangSwitchPolicy policy = LS_DESKTOP;
  if (!xmlStrcmp (val, (const xmlChar *)"InputContext"))
    policy = LS_IC;

  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((LangSwitchPolicy *)ptr) = policy;
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      *((LangSwitchPolicy *)ptr) = policy;
    }
}

static void set_onoff_policy (xmlChar *val, gboolean override, gpointer ptr)
{
  OnOffPolicy policy = OO_ACTIVATE;
  if (!xmlStrcmp (val, (const xmlChar *)"DonotChange"))
    policy = OO_DONOT_CHANGE;

  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((OnOffPolicy *)ptr) = policy;
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      *((OnOffPolicy *)ptr) = policy;
    }
}

static void process_general_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  xmlChar *val;
  while (cur != NULL)
    {
      xmlChar *props = xmlGetProp (cur, (const xmlChar *)"props");
      xmlChar *val = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
      
      if (!xmlStrcmp (props, (const xmlChar *)"enabled"))
	set_boolean (val, override, &system_default->b_im_enabled);
      else if (!xmlStrcmp (props, (const xmlChar *)"status enabled"))
	set_boolean (val, override, &system_default->b_status_enabled);
      else if (!xmlStrcmp (props, (const xmlChar *)"status location"))
	set_status_placement (val, override, &system_default->e_status_placement);
      else if (!xmlStrcmp (props, (const xmlChar *)"lang switch policy"))
	set_switch_policy (val, override, &system_default->e_lang_switch_policy);
      else if (!xmlStrcmp (props, (const xmlChar *)"on off policy"))
	set_onoff_policy (val, override, &system_default->e_on_off_policy);
      else if (!xmlStrcmp (props, (const xmlChar *)"sync activation"))
	set_boolean (val, override, &system_default->b_sync_activation);
      else if (!xmlStrcmp (props, (const xmlChar *)"remember last le"))
	set_boolean (val, override, &system_default->b_remember_last_le);

      cur = cur->next;
    }
}

static LanguageList *get_language_list (xmlDocPtr doc, xmlNodePtr cur)
{
  LanguageList *lang_list = g_new0 (LanguageList, 1);
  GSList *slist = NULL;

  while (cur != NULL)
    {
      xmlChar *val;

      if (!xmlStrcmp (cur->name, (xmlChar *)"li"))
	{
	  val = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
	  slist = g_slist_append (slist, val);
	}
      cur = cur->next;
    }
  lang_list->langs = slist;

  return lang_list;
}

static void set_language_list (LanguageList *lang_list, gboolean override, gpointer ptr)
{
  LanguageList **langlist_ptr = (LanguageList **)ptr;
  if (override)
    {
      if (*langlist_ptr != NULL)
	{
	  g_slist_free ((*langlist_ptr)->langs);
	  g_free (*langlist_ptr);
	}
      override_list = g_slist_append (override_list, ptr);
      *langlist_ptr = lang_list;
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      if (*langlist_ptr != NULL)
	{
	  g_slist_free ((*langlist_ptr)->langs);
	  g_free (*langlist_ptr);
	}
      *langlist_ptr = lang_list;
    }
}

static EngineList *default_engine_list = NULL;
static void update_engine_list (xmlDocPtr doc, xmlNodePtr cur,
				gboolean override, gpointer ptr)
{
  EngineList **enginelist_ptr = (EngineList **)ptr;
  
  if (default_engine_list == NULL)
    {
      default_engine_list = g_new0 (EngineList, 1);
      default_engine_list->engine_table = g_hash_table_new (g_str_hash, g_str_equal);
      *enginelist_ptr = default_engine_list;
    }

  xmlChar *lang = xmlGetProp (cur, (const xmlChar *)"lang");
  if (lang != NULL)
    {
      xmlChar *engine = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
      if (engine != NULL && xmlStrcmp (engine, (xmlChar *)""))
	{
	  if (!override &&
	      (g_hash_table_lookup (default_engine_list->engine_table, lang) != NULL))
	    return;

	  g_hash_table_insert (default_engine_list->engine_table, lang, engine);
	}
    }
}

static void process_language_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  while (cur != NULL)
    {
      xmlChar *props = xmlGetProp (cur, (const xmlChar *)"props");
      if (!xmlStrcmp (props, (const xmlChar *)"default languages"))
	{
	  LanguageList *lang_list = get_language_list (doc, cur->xmlChildrenNode);
	  set_language_list (lang_list, override, &system_default->l_languages);
	}
      else if (!xmlStrcmp (props, (const xmlChar *)"default engine"))
	{
	  update_engine_list (doc, cur, override, &system_default->l_language_engine);
	}
      cur = cur->next;
    }
}

static void set_hotkeys (xmlChar *trigger_list, gboolean override, gpointer ptr)
{
  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((Hotkeys **)ptr) = make_hotkey_list ((gchar *)trigger_list);
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      /* locale specific data is not set */
      *((Hotkeys **)ptr) = make_hotkey_list ((gchar *)trigger_list);
    }
}

static void set_secondary_hotkeys (xmlChar *trigger_list, gboolean override, gpointer ptr)
{
  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((Hotkeys **)ptr) = make_hotkey_list ((gchar *)trigger_list);
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      *((Hotkeys **)ptr) = make_hotkey_list ((gchar *)trigger_list);
    }
}

static void process_trigger_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  while (cur != NULL)
    {
      xmlChar *props = xmlGetProp (cur, (const xmlChar *)"props");
      if (!xmlStrcmp (props, (const xmlChar *)"trigger keys"))
	{
	  xmlChar *trigger_list = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
	  set_hotkeys (trigger_list, override, &system_default->l_trigger_keys);
	}
      else if (!xmlStrcmp (props, (const xmlChar *)"secondary trigger keys"))
	{
	  xmlChar *secondary_trigger_list = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
	  set_secondary_hotkeys (secondary_trigger_list, override, &system_default->l_secondary_trigger_keys);
	}
      cur = cur->next;
    }
}

static void process_misc_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  while (cur != NULL)
    {
      xmlChar *props = xmlGetProp (cur, (const xmlChar *)"props");
      xmlChar *val = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
      
      if (!xmlStrcmp (props, (const xmlChar *)"popup lang menu"))
	set_boolean (val, override, &system_default->b_lang_list_enabled);
      else if (!xmlStrcmp (props, (const xmlChar *)"lang menu hotkeys"))
	set_hotkeys (val, override, &system_default->l_lang_list_hotkey);
      else if (!xmlStrcmp (props, (const xmlChar *)"cycle lang switch"))
	set_boolean (val, override, &system_default->b_misc_hotkey_enabled);
      else if (!xmlStrcmp (props, (const xmlChar *)"cycle lang hotkeys"))
	set_hotkeys (val, override, &system_default->l_cycle_lang_hotkey);
      else if (!xmlStrcmp (props, (const xmlChar *)"reverse cycle lang hotkeys"))
	set_hotkeys (val, override, &system_default->l_reverse_cycle_lang_hotkey);
      cur = cur->next;
    }
}

static void set_keyboard_layout (xmlChar *val, gboolean override, gpointer ptr)
{
  if (override)
    {
      override_list = g_slist_append (override_list, ptr);
      *((gint *)ptr) = (gint)atoi((char *)val);
    }
  else if (g_slist_index (override_list, ptr) == -1)
    {
      *((gint *)ptr) = (gint)atoi((char *)val);
    }
}

static void process_keyboard_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  xmlChar *val;
  while (cur != NULL)
    {
      xmlChar *props = xmlGetProp (cur, (const xmlChar *)"props");
      xmlChar *val = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
      
      if (!xmlStrcmp (props, (const xmlChar *)"auto detect"))
	set_boolean (val, override, &system_default->b_kbd_check);
      else if (!xmlStrcmp (props, (const xmlChar *)"keyboard layout"))
	set_keyboard_layout (val, override, &system_default->i_kbd_layout);

      cur = cur->next;
    }
}

static void process_categories_default (xmlDocPtr doc, xmlNodePtr cur, gboolean override)
{
  while (cur != NULL)
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *)"general"))
	{
	  process_general_default (doc, cur->xmlChildrenNode, override);
	}
      else if (!xmlStrcmp (cur->name, (const xmlChar *)"language"))
	{
	  process_language_default (doc, cur->xmlChildrenNode, override);
	}
      else if (!xmlStrcmp (cur->name, (const xmlChar *)"trigger"))
	{
	  process_trigger_default (doc, cur->xmlChildrenNode, override);
	}
      else if (!xmlStrcmp (cur->name, (const xmlChar *)"misc"))
	{
	  process_misc_default (doc, cur->xmlChildrenNode, override);
	}
      else if (!xmlStrcmp (cur->name, (const xmlChar *)"keyboard"))
	{
	  process_keyboard_default (doc, cur->xmlChildrenNode, override);
	}
      cur = cur->next;
    }
}

static GSList *add_names_to_list (xmlDocPtr doc, xmlNodePtr cur, GSList *list)
{
  while (cur != NULL)
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *)"item"))
	{
	  xmlChar *name;
	  xmlChar *beginwith;

	  name = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
	  if (name == NULL || strlen ((const char *)name) == 0)
	    {
	      cur = cur->next;
	      continue;
	    }
	  beginwith = xmlGetProp (cur, (const xmlChar*)"beginwith");
	  if (beginwith != NULL && !xmlStrcmp (beginwith, (const xmlChar *)"True"))
	    {
	      /* "t" means if target name begins with aliase name, then
	       * it is match. "f" means only exactry same alias name is match.
	       * The content of list is "[t|f]", "alias name", "[t|f], "alias name"....
	       */
	      list = g_slist_append (list, "t");
	    }
	  else
	    {
	      list = g_slist_append (list, "f");
	    }
	  list = g_slist_append (list, name);
	}
      cur = cur->next;
    }
  return list;
}

gboolean apply_locale_alias (xmlChar *locale, xmlChar *user_locale, GHashTable *table)
{
  GSList *list;
  
  if (!xmlStrcmp (locale, user_locale))
    {
      return TRUE;
    }
  list = g_hash_table_lookup (table, locale);
  if (list != NULL)
    {
      GSList *lp;
      for (lp = list; lp; lp = g_slist_next (lp))
	{
	  gboolean beginwith = !xmlStrcmp ((const xmlChar *)lp->data, (const xmlChar *)"t");
	  xmlChar *alias;
	  lp = g_slist_next (lp);
	  if (lp == NULL)
	    {
	      break;
	    }
	  alias = (xmlChar *)lp->data;
	  if (beginwith)
	    {
	      if (strstr ((const char *)user_locale, (const char *)alias) == (char *)user_locale)
		{
		  return TRUE;
		}
	    }
	  else
	    {
	      if (!xmlStrcmp (user_locale, alias))
		{
		  return TRUE;
		}
	    }
	}
    }

  return FALSE;
}

static void get_system_default_values ()
{
  xmlDocPtr doc;
  xmlNodePtr cur;
  xmlChar *locale;
  xmlChar *user_locale = (xmlChar *)setlocale (LC_CTYPE, NULL);
  GHashTable *locale_alias_table = NULL;

  doc = xmlParseFile (XMLCONFDIR "/client.xml");
  if (doc == NULL)
    {
      system_default = NULL;
      return;
    }

  cur = xmlDocGetRootElement (doc);
  if (cur == NULL)
    {
      system_default = NULL;
      return;
    }
  cur = cur->xmlChildrenNode;

  system_default = g_new0 (IIIMProps, 1);
  while (cur != NULL)
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *)"localealias"))
	{
	  xmlChar *aname;
	  GSList *name_list = NULL;
	  aname = xmlGetProp (cur, (const xmlChar*)"aname");
	  if (locale_alias_table == NULL)
	    {
	      locale_alias_table = g_hash_table_new (g_str_hash, g_str_equal);
	    }
	  name_list = g_hash_table_lookup (locale_alias_table, aname);
	  name_list = add_names_to_list (doc, cur->xmlChildrenNode, name_list);
	  g_hash_table_insert (locale_alias_table, aname, name_list);
	}
      if (!xmlStrcmp (cur->name, (const xmlChar *)"client"))
	{
	  locale = xmlGetProp (cur, (const xmlChar *)"locale");
	  if (!xmlStrcmp (locale, (const xmlChar *)"all"))
	    {
	      process_categories_default (doc, cur->xmlChildrenNode, FALSE);
	    }
	  else if (apply_locale_alias (locale, user_locale, locale_alias_table))
	    {
	      process_categories_default (doc, cur->xmlChildrenNode, TRUE);
	    }
	}
      cur = cur->next;
    }
}

static gboolean check_keys_availability (Hotkeys *keys) {
  GSList *p;

  for (p = keys->hotkeys; p; p = g_slist_next (p)) {
    guint keyval;
    gchar *key_str = (gchar *)p->data;
    /* remove modifier part */
    gchar *keypart = g_strrstr (key_str, "+");
    if (keypart == NULL)
      keypart = key_str;
    else
      keypart = keypart++;
	
    keyval = gdk_keyval_from_name (keypart);
    if (keyval != GDK_VoidSymbol && keyval != 0)
      {
	GdkKeymapKey *map = g_malloc (sizeof (GdkKeymapKey));
	gint num;
	if (gdk_keymap_get_entries_for_keyval (NULL, keyval, &map, &num)) {
	  return TRUE;
	}
      }
  }
  return FALSE;
}

/*
 * get properties from user's gconf registry
 */
void setget_fetch_gconf_settings (IIIMProps *iiim_props)
{
  GConfClient *client;

  /* string representation of each properties */
  gchar
    *im_enabled,
    *switcher_autostart,
    *status_enabled,
    *sync_activation,
    *status_placement,
    *remember_last_le,
    *language_engine,
    *trigger_keys,
    *use_trigger_keys_to_deactivate,
    *use_same_keys_to_deactivate,
    *deactivate_trigger_keys,
    *lang_switch_policy,
    *on_off_policy,
    *lookup,
    *lang_list,
    *lang_list_placement,
    *lang_list_hotkey,
    *misc_hotkey,
    *cycle_lang_hotkey,
    *reverse_cycle_lang_hotkey,
    *kbd_check;

  gint kbd_layout;

  GSList *language_slist;
  GSList *user_defined_slist;
  GSList *user_defined_non_use;
  GSList *language_engine_slist;
  GSList *trigger_list;
  GSList *switch_lang_list;
  GSList *cycle_lang_list;
  GSList *reverse_cycle_lang_list;
  GSList *lp;

  /* logical representation of each properties
     b_* : boolean
     e_* : enum
     l_* : list of appropriate struct
   */
  gboolean b_im_enabled, b_switcher_autostart, b_status_enabled, b_use_trigger_keys_to_deactivate,
    b_sync_activation, b_use_same_keys_to_deactivate, b_lookup, b_lang_list, b_misc_hotkey, b_kbd_check, b_remember_last_le;
  StatusPlacement e_status_placement;
  LanguageList *l_languages;
  EngineList *l_language_engines;
  Hotkeys *l_trigger_keys;
  Hotkeys *l_deactivate_trigger_keys;
  LangSwitchPolicy e_lang_switch_policy;
  OnOffPolicy e_on_off_policy;
  LangListPlacement e_lang_list_placement;
  Hotkeys *l_lang_list_hotkey;
  Hotkeys *l_cycle_lang_hotkey;
  Hotkeys *l_reverse_cycle_lang_hotkey;

  /*
   * get system default value (fallback value)
   */
  get_system_default_values ();

  client = gconf_client_get_default ();

  /*
   * get all of iiim properties from gconf registry.
   * if some keys are not available use default.
   *
   * convert string representation to logical value
   * default value should be regrieved from /etc/iiim/iiim.xml.conf file (TODO)
   */
  /*********** IM ENABLED ***********/
  im_enabled = gconf_client_get_string (client, KEY_ENABLE_IM, NULL);
  if (im_enabled == NULL)
    b_im_enabled = (system_default != NULL ?
		    system_default->b_im_enabled : TRUE);
  else if (!strcmp (im_enabled, V_TRUE))
    b_im_enabled = TRUE;
  else
    b_im_enabled = FALSE;
  g_free (im_enabled);
  iiim_props->b_im_enabled = b_im_enabled;

  /*********** SWITCHER AUTOSTART ***********/
  switcher_autostart = gconf_client_get_string (client, KEY_ENABLE_GIMLET_AUTOSTART, NULL);
  if (switcher_autostart == NULL)
    b_switcher_autostart = (system_default != NULL ?
		          system_default->b_switcher_autostart : TRUE);
  else if (!strcmp (switcher_autostart, V_TRUE))
    b_switcher_autostart = TRUE;
  else
    b_switcher_autostart = FALSE;
  g_free (switcher_autostart);
  iiim_props->b_switcher_autostart = b_switcher_autostart;
  /********** STATUS ENBLED ********/
  status_enabled = gconf_client_get_string (client, KEY_STATUS_DISPLAY, NULL);

  if (status_enabled == NULL)
    b_status_enabled = (system_default != NULL ?
			system_default->b_status_enabled : TRUE);
  else if (!strcmp (status_enabled, V_TRUE))
    b_status_enabled = TRUE;
  else
    b_status_enabled = FALSE;
  g_free (status_enabled);
  iiim_props->b_status_enabled = b_status_enabled;

  /********* SYNC ACTIVATION *******/
  sync_activation = gconf_client_get_string (client, KEY_SYNC_ACTIVATION, NULL);
  if (sync_activation == NULL)
    b_sync_activation = (system_default != NULL ?
			 system_default->b_sync_activation : TRUE);
  else if (!strcmp (sync_activation, V_TRUE))
    b_sync_activation = TRUE;
  else
    b_sync_activation = FALSE;
  g_free (sync_activation);
  iiim_props->b_sync_activation = b_sync_activation;

  /******** STATUS PLACEMENT *******/
  status_placement = gconf_client_get_string (client, KEY_STATUS_PLACEMENT, NULL);
  if (status_placement == NULL)
    e_status_placement = (system_default != NULL ?
			  system_default->e_status_placement : S_DESKTOP_DEFAULT);
  else if (!strcmp (status_placement, V_PANEL))
    e_status_placement = S_PANEL;
  else if (!strcmp (status_placement, V_ATTACH_FRAME))
    e_status_placement = S_ATTACH_FRAME;
  else if (!strcmp (status_placement, V_NONE))
    e_status_placement = S_NONE;
  else
    e_status_placement = S_DESKTOP_DEFAULT;
  g_free (status_placement);
  iiim_props->e_status_placement = e_status_placement;
  if (e_status_placement == S_DESKTOP_DEFAULT)
    {
      if (on_CDE)
	{
	  iiim_props->b_switcher_autostart = FALSE;
	}
      else
	{
	  iiim_props->b_switcher_autostart = TRUE;
	}
    }
  /********** LANG SWITCH POLICY *************/
  lang_switch_policy = gconf_client_get_string (client, KEY_LANG_SWITCH_POLICY, NULL);
  if (lang_switch_policy == NULL)
    {
      e_lang_switch_policy = (system_default != NULL ?
			      system_default->e_lang_switch_policy : LS_DESKTOP);
    }
  else if (!strcmp (lang_switch_policy, V_DESKTOP))
    {
      e_lang_switch_policy = LS_DESKTOP;
    }
  else
    {
      e_lang_switch_policy = LS_IC;
    }
  g_free (lang_switch_policy);
  iiim_props->e_lang_switch_policy = e_lang_switch_policy;

  /******** ON/OFF POLICY ************/
  on_off_policy = gconf_client_get_string (client, KEY_ON_OFF_POLICY, NULL);
  if (on_off_policy == NULL)
    {
      e_on_off_policy = (system_default != NULL ?
			 system_default->e_on_off_policy : OO_ACTIVATE);
    }
  else if (!strcmp (on_off_policy, V_DONOT_CHANGE))
    {
      e_on_off_policy = OO_DONOT_CHANGE;
    }
  else
    {
      e_on_off_policy = OO_ACTIVATE;
    }
  g_free (on_off_policy);
  iiim_props->e_on_off_policy = e_on_off_policy;

  /********* SYNC ACTIVATION *******/
  remember_last_le = gconf_client_get_string (client, KEY_REMEMBER_LAST_LE, NULL);
  if (remember_last_le == NULL)
    b_remember_last_le = (system_default != NULL ?
			  system_default->b_remember_last_le : TRUE);
  else if (!strcmp (remember_last_le, V_TRUE))
    b_remember_last_le = TRUE;
  else
    b_remember_last_le = FALSE;
  g_free (remember_last_le);
  iiim_props->b_remember_last_le = b_remember_last_le;

  /********* LANGUAGE TO USE LIST *********/
  language_slist = gconf_client_get_list (client, KEY_LANG_TO_SAVE_LIST,
					  GCONF_VALUE_STRING, NULL);
  iiim_props->l_saved_languages = language_slist;

  /* language_slist contains user selected languages. but the availability of it
     depends on server status. if we maintain only use_lang list, then temporaliy
     unavailable language will be lost at saving, so we need to maintain save_list
     and use_list. */

  l_languages = g_new0 (LanguageList, 1);
  if (no_cf)
    {
      l_languages->langs = NULL;
    }
  else if (language_slist == NULL)
    {
      /* GET DEFAULT */
      GSList *def = get_default_lang_list (); /* LanguageID list */
      GSList *valid = setget_and_list_of_lang (setget_get_available_lang_list(), def);
      l_languages->langs = add_default_engine_info (valid);
    }
  else
    {
      /* check language_slist is not broken.
       * The list which has the prulal same entries must be broken, then
       * use default.
       */
      GSList *check_list = g_slist_sort (g_slist_copy (language_slist),
					(GCompareFunc) g_utf8_collate);
      gchar *prev = NULL;
      gboolean broken_list = FALSE;

      for (lp = check_list; lp; lp = g_slist_next(lp))
	{
	  if (prev != NULL && !strcmp (lp->data, prev))
	    {
	      broken_list = TRUE;
	      break;
	    }
	  prev = lp->data;
	}
      if (broken_list)
	{
	  GSList *def = get_default_lang_list (); /* LanguageID list */
	  GSList *valid = setget_and_list_of_lang (setget_get_available_lang_list(), def);
	  l_languages->langs = add_default_engine_info (valid);
	}
      else
	l_languages->langs = convert_to_langid_list (language_slist);
    }

  /* Add user defined list here. The user defined list is not saved in gconf, but
     they are always added to use list except ones which were excluded explicitly
     in iiim-properites. This is because user defined layout is made outside of
     iiim-properties, and they are always intended to be used. */
  user_defined_slist = setget_get_user_defined_kbl_list ();

  /********* USER DEFINED NON USE *******/
  user_defined_non_use = gconf_client_get_list (client, KEY_USER_DEFINED_NON_USE_LIST,
						GCONF_VALUE_STRING, NULL);
  if (user_defined_non_use != NULL)
    {
      GSList *remove_list = NULL;
      iiim_props->l_user_defined_non_use = delete_non_exist_entry (user_defined_non_use);
      /* remove non use entry from user_defined_list */
      for (lp = user_defined_slist; lp; lp = g_slist_next (lp))
	{
	  LanguageID *lid = (LanguageID *)lp->data;
	  if (g_slist_find_custom (user_defined_non_use, lp->data, cmp_lid_name)) {
	    remove_list = g_slist_append (remove_list, lid);
	  }
	}
      for (lp = remove_list; lp; lp = g_slist_next (lp))
	{
	  user_defined_slist = g_slist_remove (user_defined_slist, lp->data);
	}
      g_slist_free (remove_list);
    }
  l_languages->langs = g_slist_concat (l_languages->langs, user_defined_slist);
  iiim_props->l_languages = l_languages;

  /********* TRIGGER KEYS *********/

  trigger_keys = gconf_client_get_string (client, KEY_TRIGGER, NULL);
  if (trigger_keys == NULL)
    {
      if (system_default != NULL)
	{
	  gboolean valid;
	  l_trigger_keys = system_default->l_trigger_keys;
	  valid = check_keys_availability (l_trigger_keys);
	  if (!valid)
	    {
	      l_trigger_keys->hotkeys = g_slist_concat (system_default->l_secondary_trigger_keys->hotkeys, l_trigger_keys->hotkeys);
	    }
	}
      else
	{
	  trigger_list = cf_get_hotkeys (TRIGGER_L);
	  l_trigger_keys = g_new0(Hotkeys, 1);
	  l_trigger_keys->hotkeys = trigger_list;
	}
    }
  else
    {
      l_trigger_keys = make_hotkey_list (trigger_keys);
    }
  iiim_props->l_trigger_keys = l_trigger_keys;
  g_free (trigger_keys);
  
  /******* LOOKUP POLICY **********/
  iiim_props->b_lookup_enabled = TRUE;

  /******* USE POPUP LANG LISE ********/
  lang_list = gconf_client_get_string (client, KEY_ENABLE_LANG_LIST, NULL);
  if (lang_list == NULL)
    b_lang_list = (system_default != NULL ?
		   system_default->b_lang_list_enabled : TRUE);
  else if (!strcmp (lang_list, V_TRUE))
    b_lang_list = TRUE;
  else
    b_lang_list = FALSE;
  g_free (lang_list);
  iiim_props->b_lang_list_enabled = b_lang_list;

  /******* LANG LIST PLACEMENT ********/
  lang_list_placement = gconf_client_get_string (client, KEY_LANG_LIST_PLACEMENT, NULL);
  if (lang_list_placement == NULL)
    ;
  g_free (lang_list_placement);
  iiim_props->e_lang_list_placement = e_lang_list_placement;

  /******* LANG LIST HOTKEY *********/
  lang_list_hotkey = gconf_client_get_string (client, KEY_LANG_LIST_HOTKEY, NULL);
  if (lang_list_hotkey == NULL)
    {
      if (system_default != NULL)
	{
	  l_lang_list_hotkey = system_default->l_lang_list_hotkey;
	}
      else
	{
	  switch_lang_list = cf_get_hotkeys (LE_SWITCH_L);
	  l_lang_list_hotkey = g_new0 (Hotkeys, 1);
	  l_lang_list_hotkey->hotkeys = switch_lang_list;
	}
    }
  else
    {
      l_lang_list_hotkey = make_hotkey_list (lang_list_hotkey);
    }
  iiim_props->l_lang_list_hotkey = l_lang_list_hotkey;
  g_free (lang_list_hotkey);

  /******** KEYBOARD *************/
  kbd_check = gconf_client_get_string (client, KEY_KBD_CHECK, NULL);
  if (kbd_check == NULL)
    b_kbd_check = (system_default != NULL ?
		   system_default->b_kbd_check : TRUE);
  else if (!strcmp (kbd_check, V_TRUE))
    b_kbd_check = TRUE;
  else
    b_kbd_check = FALSE;

  iiim_props->b_kbd_check = b_kbd_check;

  kbd_layout = gconf_client_get_int (client, KEY_KBD_LAYOUT, NULL);
 
  if (b_kbd_check || kbd_layout == -1)
    {
      iiim_props->i_kbd_layout = kbltrans_get_platform_id ();
    }
  else
    {
      iiim_props->i_kbd_layout = kbd_layout;
    }

  /******* MISC HOTKEY ***********/
  misc_hotkey = gconf_client_get_string (client, KEY_ENABLE_MISC_HOTKEY, NULL);
  if (misc_hotkey == NULL)
    b_misc_hotkey = (system_default != NULL ?
		     system_default->b_misc_hotkey_enabled : TRUE);
  else if (!strcmp (misc_hotkey, V_TRUE))
    b_misc_hotkey = TRUE;
  else
    b_misc_hotkey = FALSE;
  g_free (misc_hotkey);
  iiim_props->b_misc_hotkey_enabled = b_misc_hotkey;

  /******** CYCLE LANG HOTKEY *********/
  cycle_lang_hotkey = gconf_client_get_string (client, KEY_CYCLE_LANG_HOTKEY, NULL);
  if (cycle_lang_hotkey == NULL)
    {
      if (system_default != NULL)
	{
	  l_cycle_lang_hotkey = system_default->l_cycle_lang_hotkey;
	}
      else
	{
	  cycle_lang_list = cf_get_hotkeys (CYCLE_LE_SWITCH_L);
	  l_cycle_lang_hotkey= g_new0 (Hotkeys, 1);
	  l_cycle_lang_hotkey->hotkeys = cycle_lang_list;
	}
    }
  else
    {
      l_cycle_lang_hotkey = make_hotkey_list (cycle_lang_hotkey);
    }
  iiim_props->l_cycle_lang_hotkey = l_cycle_lang_hotkey;
  g_free (cycle_lang_hotkey);

  /******* REVERSE CYCLE LANG HOTKEY *******/
  reverse_cycle_lang_hotkey =
    gconf_client_get_string (client, KEY_REVERSE_CYCLE_LANG_HOTKEY, NULL);
  if (reverse_cycle_lang_hotkey == NULL)
    {
      if (system_default != NULL)
	{
	  l_reverse_cycle_lang_hotkey = system_default->l_reverse_cycle_lang_hotkey;
	}
      else
	{
	  reverse_cycle_lang_list = cf_get_hotkeys (RE_CYCLE_LE_SWITCH_L);
	  l_reverse_cycle_lang_hotkey= g_new0 (Hotkeys, 1);
	  l_reverse_cycle_lang_hotkey->hotkeys = reverse_cycle_lang_list;
	}
    }
  else
    {
      l_reverse_cycle_lang_hotkey = make_hotkey_list (reverse_cycle_lang_hotkey);
    }
  iiim_props->l_reverse_cycle_lang_hotkey = l_reverse_cycle_lang_hotkey;
  g_free (reverse_cycle_lang_hotkey);
}
