/*
Copyright 1990-2008 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
/*
 * KeyModel.java
 *
 * Created on 2007/12/03, 17:50
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package com.sun.g11n.vkb;

import java.util.*;

import java.awt.dnd.*;
import java.awt.datatransfer.*;

/**
 *
 * @author naoyuki
 */
public class KeyModel implements Transferable {

    private String[] labelArray;
    private int code;
    private int altCode;
    private String name;
    protected KeyboardLayout layout;
    private String identicalName;
    protected IMInterface imInterface;
    protected Type type;
    
    public static enum Type {
        NORMAL, MODIFIER, FUNCTION, CONTROL;
        
        @Override public String toString() {
            // string representation of Type is recorded in
            // customized layout data file
            switch(this) {
                case MODIFIER:
                    return "MOD";
                case FUNCTION:
                    return "FUNC";
                case CONTROL:
                    return "CTL";
            }
            return "NORM";
        }
    }

    /** Creates a new instance of KeyModel */
    private KeyModel(String name) {
        this.name = name;
        labelArray = new String[4];
        identicalName = null;
        imInterface = Main.getIMInterface();
        type = Type.NORMAL;
    }
    
    public Type getType() {
        return type;
    }
    
    public void setType(Type type) {
        this.type = type;
    }
    
    public String getName() {
        return name;
    }
    
    public void setName(String name) {
        this.name = name;
    }
    
    public void setLayout(KeyboardLayout layout) {
        this.layout = layout;
    }
    
    public KeyboardLayout getLayout() {
        return layout;
    }
    
    public int getLevel() {
        if (layout != null) {
            return layout.getLevel();
        }
        return 0;
    }
    
    /* 
     * KeyModel is instanciated with only this factory method, so that
     * different class would be used depends on Key Name 
     */
    public static KeyModel getInstance(String name) {
        KeyModel model = null;
        if (shiftSet.contains(name)) {
            model = new ShiftKeyModel(name);
        } else if (altgrSet.contains(name)) {
            model = new AltGrKeyModel(name);
        } else if (capsSet.contains(name)) {
            model = new CapsKeyModel(name);
        } else if (modifierSet.contains(name)) {
            model = new ModifierKeyModel(name);
        } else if (functionSet.contains(name)) {
            model = new FunctionKeyModel(name);
        } else if (controlSet.contains(name)) {
            model = new ControlKeyModel(name);
        } else if (numlockSet.contains(name)) {
            model = new NumLockKeyModel(name);
        }
        if (model != null) {
            model.setLabel(Main.getKeyLabel(name), 0);
        } else {
            model = new KeyModel(name);
        }
        
        /*
         * Identical model support for Right and Left modifier (Shift or Meta)
         * so that they act at the same if the other is pressed.
         */
        Set<Map.Entry<String, String>> identicalPairs = identicalMap.entrySet();
        for (Map.Entry<String, String> pair : identicalPairs) {
            String p1 = pair.getKey();
            String p2 = pair.getValue();
            if (p1.equals(name)) {
                model.setIdentical(p2);
            } else if (p2.equals(name)) {
                model.setIdentical(p1);
            }
        }
        
        return model;
    }
    public KeyModel copy() {
        KeyModel model = getInstance(name);
        for (int i = 0; i < 4; i++) {
            model.setLabel(getLabel(i), i);
        }
        model.setLayout(layout);
        model.setCode(getCode());
        model.setAltCode(getAltCode());
        
        return model;
    }
    
    public void setLabel(String l, int level) {
        if (level < 0 || level > 3) {
            return;
        }
        labelArray[level] = l;
    }
    public String getLabel() {
        int level = getLevel();
        String ret = getLabel(level);
        if (ret != null) 
            return ret;
        
        switch(level) {
            case 0:
                return null;
            case 1:
                return getLabel(0);
            case 2:
                return getLabel(0);
            case 3:
                ret = getLabel(2);
                if (ret != null) 
                    return ret;
                ret = getLabel(1);
                if (ret != null)
                    return ret;
        }
        return getLabel(0);
    }
    public String getLabel(int level) {
        if (level < 0 || level > 3) {
            return null;
        }
        String value = labelArray[level];
        // Caps lock support:
        // caps lock does not change level, but just capitalize characters
        // and if level is 1 or 3 (shift), then it lowered characters
        if (layout != null) {
            boolean caps = layout.isCaps();
            if (value != null && caps && !(this instanceof NonCharKeyModel) && value.length() == 1) {
                Character c = value.charAt(0);
                if (Character.isLowerCase(c) && (level == 0 || level == 2)) {
                    value = value.toUpperCase();
                } else if (Character.isUpperCase(c) && (level == 1 || level == 3)) {
                    value = value.toLowerCase();
                }
            }
        }
        return value;
    }
    public void setCode(int code) {
        this.code = code;
    }
    public int getCode() {
        return code;
    }
    public int getChar() {
        String label = getLabel();
        if (label.length() > 1) {
            return 0;
        }
        return label.charAt(0);
    }
    public void setAltCode(int altCode) {
        this.altCode = altCode;
    }
    public int getAltCode() {
        return altCode;
    }
    public boolean isModifier() {
        return false;
    }
    public boolean isCharacter() {
        return true;
    }
    public void pressed() {
        if (layout == null)
            return;

        if (layout.getControl()) {
            // send control code to client
            imInterface.sendKeyEvent(getCode(), getChar(), layout.getModifierState(), System.currentTimeMillis());
        } else if (layout.getAlt()) {
            imInterface.sendKeyEvent(getCode(), 0, layout.getModifierState(), System.currentTimeMillis());
        } else {
            // send character(s) to client
            String l = getLabel();
            if (l != null) {
                imInterface.sendText(l);
            }
        }
    }
    public void released() {
    }
    public void clicked() {
    }
    
    /* 
     * Whether this key has the same key
     * This is for Right and Left modifier handling
     */
    public boolean hasIdentical() {
        return identicalName != null;
    }
    
    /*
     * Returns identical model
     * This is for Right and Left modifier handling
     */
    public String getIdentical() {
        return identicalName;
    }
    
    /*
     * Set identical model
     */
    public void setIdentical(String name) {
        identicalName = name;
    }
    

    /*
     * Used by KeyView's pressed() method for identical key handling
     */
    KeyView getOtherView(KeyView view) {
        if (layout != null) {
            return layout.getOtherView(view, this);
        }
        return null;
    }
    
    // Transferable implementation
    @Override public Object getTransferData(DataFlavor flavor) {
        if (flavor.isMimeTypeEqual(DataFlavor.javaJVMLocalObjectMimeType)) {
            return this;
        }
        return null;
    }
    private static DataFlavor[] supportedFlavors;
    static {
        supportedFlavors = new DataFlavor[1];
        try {
            supportedFlavors[0] = new DataFlavor(DataFlavor.javaJVMLocalObjectMimeType);
        } catch(Exception e) {
            e.printStackTrace();
        }
    }
    @Override public DataFlavor[] getTransferDataFlavors() {
        return supportedFlavors;
    }
    @Override public boolean isDataFlavorSupported(DataFlavor flavor) {
        for (DataFlavor flv : supportedFlavors) {
            if (flavor.isMimeTypeEqual(flv)) {
                return true;
            }
        }
        return false;
    }
    
    /*
     * base class for not normal character keys
     */
    private static abstract class NonCharKeyModel extends KeyModel {
        private NonCharKeyModel(String name) {
           super(name);
           
        }
        @Override public boolean isCharacter() {
            return false;
        }
    }
    
    /*
     * class represents misc modifier keys
     */
    private static class ModifierKeyModel extends NonCharKeyModel {
        private ModifierKeyModel(String name) {
            super(name);
            type = Type.MODIFIER;
        }
        @Override public boolean isModifier() {
            return true;
        }
        @Override public void pressed() {
            int mask = modifierMaskMap.get(getName());
            if (mask != 0) {
                layout.setModifierState(mask);
            }
        }
    }
    
    /*
     * class represents Shift key
     */
    private static class ShiftKeyModel extends ModifierKeyModel {
        private ShiftKeyModel(String name) {
            super(name);
        }
        @Override public void pressed() {
            if (layout != null) {
                int lev = layout.getLevel();
                if (lev == 0 || lev == 2) {
                    layout.setLevel(++lev);
                } else if (lev == 1 || lev == 3) {
                    layout.setLevel(--lev);
                }
            }
        }
    }
    
    /*
     * class represens AltGrph key
     */
    private static class AltGrKeyModel extends ModifierKeyModel {
        private AltGrKeyModel(String name) {
            super(name);
        }
        @Override public void pressed() {
            if (layout != null) {
                int lev = layout.getLevel();
                if (lev == 0 || lev == 1) {
                    layout.setLevel(lev + 2);
                } else if (lev == 2 || lev == 3) {
                    layout.setLevel(lev - 2);
                }
            }
        }
    }
    
    /*
     * class represents Caps key
     */
    private static class CapsKeyModel extends ModifierKeyModel {
        private CapsKeyModel(String name) {
            super(name);
            lock = false;
        }
        private boolean lock;
        @Override public void pressed() {
            lock = !lock;
            if (layout != null) {
                layout.setCaps(lock);
            }
        }
    }

    /*
     * class represents Function keys
     */
    private static class FunctionKeyModel extends NonCharKeyModel {
        private FunctionKeyModel(String name) {
            super(name);
            type = Type.FUNCTION;
        }
    }
    
    /*
     * class represents Control character keys (ESC, Tab etc...)
     */ 
    private static class ControlKeyModel extends NonCharKeyModel {
        private ControlKeyModel(String name) {
            super(name);
            type = Type.CONTROL;
        }
        @Override public void pressed() {
            imInterface.sendKeyEvent(getCode(), 0, layout.getModifierState(), System.currentTimeMillis());      
        }
    }
    
    /*
     * class represents Num Lock key
     */
    private static class NumLockKeyModel extends ModifierKeyModel {
        private NumLockKeyModel(String name) {
            super(name);
        }
    }
           
    private static String[][] IDENTICAL_PAIR = {
        {"LFSH", "RTSH"},
        {"LMTA", "RMTA"},
        {"LCTL", "RCTL"},
        {"LWIN", "RWIN"},
    };
    private static String[] DEFAULT_MODIFIERS = {
        "CAPS", "LFSH", "LCTL", "RTSH", "LALT", 
        "RCTL", "RALT", "RMTA", "LMTA", "ALGR",
        "LWIN", "RWIN", "NMLK",
    };
    private static String[] DEFAULT_SHIFT = {
        "LFSH", "RTSH",
    };
    private static String[] DEFAULT_ALTGR = {
        "ALGR", "RALT",
    };
    private static String[] DEFAULT_CAPS = {
        "CAPS",
    };
    private static String[] DEFAULT_NUMLOCK = {
        "NMLK",
    };
    private static String[] DEFAULT_FUNCTIONS = {
        "FK01", "FK02", "FK03", "FK04", "FK05", 
        "FK06", "FK07", "FK08", "FK09", "FK10",
        "FK11", "FK12", "FK13",
        "PRSC", "SCLK", "PAUS", "INS", "HOME",
        "PGUP", "END", "PGDN", "UP", "LEFT", 
        "DOWN", "RGHT",
        "KPDV", "KPMU", "KPSU", "KPAD", "KPDL",
        "KP1", "KP2", "KP3", "KP4", "KP5",
        "KP6", "KP7", "KP8", "KP9", "KP0",
        "STOP", "AGAI", "PROP", "UNDO", "FRNT",
        "COPY", "OPEN", "PAST", "FIND", "CUT",
        "HELP", "COMP", 
        "HZTG",
        "MENU",
    };
    private static String[] DEFAULT_CONTROLS = {
        "BKSP", "TAB", "RTRN", "DELE", "ESC",
    };
    private static Map<String, String> identicalMap;
    private static Set<String> modifierSet;
    private static Map<String, Integer> modifierMaskMap;
    private static Set<String> numlockSet;
    private static Set<String> functionSet;
    private static Set<String> controlSet;
    private static Set<String> shiftSet;
    private static Set<String> altgrSet;
    private static Set<String> capsSet;
    static {
        identicalMap = new HashMap<String, String>();
        for (String[] pair : IDENTICAL_PAIR) {
            identicalMap.put(pair[0], pair[1]);
        }
        modifierSet = new HashSet<String>();
        modifierMaskMap = new HashMap<String, Integer>();
        for (String name : DEFAULT_MODIFIERS) {
            if (name.equals("LFSH") || name.equals("RTSH"))
                modifierMaskMap.put(name, java.awt.event.InputEvent.SHIFT_MASK);
            else if (name.equals("LALT") || name.equals("RALT"))
                modifierMaskMap.put(name, java.awt.event.InputEvent.ALT_MASK);
            else if (name.equals("LCTL") || name.equals("RCTL"))
                modifierMaskMap.put(name, java.awt.event.InputEvent.CTRL_MASK);
            else if (name.equals("ALGR"))
                modifierMaskMap.put(name, java.awt.event.InputEvent.ALT_GRAPH_MASK);
            
            modifierSet.add(name);
        }
        numlockSet = new HashSet<String>();
        for (String name : DEFAULT_NUMLOCK) {
            numlockSet.add(name);
        }
        functionSet = new HashSet<String>();
        for (String name: DEFAULT_FUNCTIONS) {
            functionSet.add(name);
        }
        controlSet = new HashSet<String>();
        for (String name : DEFAULT_CONTROLS) {
            controlSet.add(name);
        }
        shiftSet = new HashSet<String>();
        for (String name : DEFAULT_SHIFT) {
            shiftSet.add(name);
        }
        altgrSet = new HashSet<String>();
        for (String name: DEFAULT_ALTGR) {
            altgrSet.add(name);
        }
        capsSet = new HashSet<String>();
        for (String name : DEFAULT_CAPS) {
            capsSet.add(name);
        }
    }
}
