#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <iconv.h>
#include <sys/types.h>
#if defined(lint) && defined(HAVE_NOTE_H)
#include <note.h>
#endif /* lint && HAVE_NOTE_H */

#include <iiimp.h>
#include <iiimp-data.h>

#include "iiimp-dataP.h"


static iconv_t
iiimp_string_iconv_descriptor_utf8_utf16()
{
    static int		iconv_initialized;
    static iconv_t	iconv_descriptor = (iconv_t)(-1);

    if (0 == iconv_initialized) {
	iconv_initialized = 1;
	iconv_descriptor = iconv_open("UTF-8", "UTF-16");
    }

    return iconv_descriptor;
}


static iconv_t
iiimp_string_iconv_descriptor_utf16_utf8()
{
    static int		iconv_initialized;
    static iconv_t	iconv_descriptor = (iconv_t)(-1);

    if (0 == iconv_initialized) {
	iconv_initialized = 1;
	iconv_descriptor = iconv_open("UTF-16", "UTF-8");
    }

    return iconv_descriptor;
}


static IIIMP_string *
iiimp_string_utf8_sync(
    IIIMP_data_s *		data_s,
    IIIMP_string *		str)
{
    char *		utf8;
    size_t		nbyte;
    iconv_t		cd;
    size_t		ret;
    const char *	inbuf;
    size_t		inbytesleft;
    char *		outbuf;
    size_t		outbytesleft;
    size_t		outbuflen;
    IIIMP_card16 *	utf16_ptr;
    size_t		utf16_len;

    if ((NULL == str) || (0 == str->len) || (NULL == str->ptr)) return str;

    if (0 == iiimp_data_s_option_get(data_s, IIIMP_DATA_S_OPTION_STRING_UTF8)) {
	return str;
    }

    cd = iiimp_string_iconv_descriptor_utf8_utf16();
    if ((iconv_t)(-1) == cd) {
	iiimp_string_delete(data_s, str);
	data_s->status = IIIMP_DATA_ICONV_ERROR;
	return NULL;
    }

    outbuflen = ((str->len * 6) + 1);

    utf8 = (char *)malloc(outbuflen);
    if (NULL == utf8) {
	iiimp_string_delete(data_s, str);
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    inbuf = (char *)(str->ptr);
    inbytesleft = (str->len);
    outbuf = (char *)utf8;
    outbytesleft = outbuflen;

    ret = iconv(cd, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
    if (((size_t)(-1)) == ret) {
	free(utf8);
	iiimp_string_delete(data_s, str);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    str->utf8_len = (outbuflen - outbytesleft);
    str->utf8_ptr = utf8;
    *(utf8 + str->utf8_len) = '\0';

    return str;
}


static IIIMP_string *
iiimp_string_utf16_new(
    IIIMP_data_s *		data_s,
    size_t			len,
    const IIIMP_card16 *	ptr)
{
    IIIMP_string *	str;
    size_t		nbyte;

    str = (IIIMP_string *)malloc(sizeof (IIIMP_string));
    if (NULL == str) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = (2 * len);

    str->nbyte = (2 + nbyte + PAD(2 + nbyte));
    str->len = len;
    str->next = NULL;
    str->utf8_len = 0;
    str->utf8_ptr = NULL;

    nbyte = ((sizeof (IIIMP_card16)) * len);

    if (0 == len) {
	str->ptr = NULL;
    } else {
	str->ptr = (IIIMP_card16 *)malloc(str->nbyte);
	if (NULL == str->ptr) {
	    free(str);
	    data_s->status = IIIMP_DATA_MALLOC_ERROR;
	    return NULL;
	}
    }

    if (NULL != ptr) {
	(void)memcpy(str->ptr, ptr, nbyte);
    }

    return str;
}


IIIMP_string *
iiimp_string_new(
    IIIMP_data_s *		data_s,
    size_t			len,
    const IIIMP_card16 *	ptr)
{
    IIIMP_string *	str;

    str = iiimp_string_utf16_new(data_s, len, ptr);
    str = iiimp_string_utf8_sync(data_s, str);

    return str;
}


IIIMP_string *
iiimp_string_utf8_new(
    IIIMP_data_s *	data_s,
    size_t		len,
    const char *	ptr)
{
    IIIMP_string *	str;
    IIIMP_card16 *	utf16;
    iconv_t		cd;
    size_t		ret;
    const char *	inbuf;
    size_t		inbytesleft;
    char *		outbuf;
    size_t		outbytesleft;
    size_t		outbuflen;
    IIIMP_card16 *	utf16_ptr;
    size_t		utf16_len;

    if (0 == len) return iiimp_string_utf16_new(data_s, 0, NULL);

    cd = iiimp_string_iconv_descriptor_utf16_utf8();
    if ((iconv_t)(-1) == cd) {
	data_s->status = IIIMP_DATA_ICONV_ERROR;
	return NULL;
    }

    outbuflen = ((sizeof (IIIMP_card16)) * (len + 1)); /* 1 for BOM */

    utf16 = (IIIMP_card16 *)malloc(outbuflen);
    if (NULL == utf16) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    inbuf = (char *)ptr;
    inbytesleft = len;
    outbuf = (char *)utf16;
    outbytesleft = outbuflen;

    ret = iconv(cd, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
    if (((size_t)(-1)) == ret) {
	data_s->status = IIIMP_DATA_INVALID;
	free(utf16);
	return NULL;
    }

    if (((IIIMP_card16)0xfeff == *utf16) || ((IIIMP_card16)0xfffe == *utf16)) {
	utf16_ptr = (utf16 + 1);
	utf16_len = (outbuflen - outbytesleft - sizeof(IIIMP_card16)) / sizeof(IIIMP_card16);
    } else {
	utf16_ptr = utf16;
	utf16_len = (outbuflen - outbytesleft) / sizeof(IIIMP_card16);
    }

    str = iiimp_string_utf16_new(data_s, utf16_len, utf16_ptr);

    free(utf16);

    if (NULL == str) return NULL;

    str->utf8_ptr = (char *)malloc(len);
    if (NULL == str->utf8_ptr) {
	iiimp_string_delete(data_s, str);
	return NULL;
    }

    str->utf8_len = len;
    memcpy(str->utf8_ptr, ptr, len);

    return str;
}


void
iiimp_string_delete(IIIMP_data_s * data_s, IIIMP_string * str)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */
    if (NULL == str) return;
    free(str->ptr);
    free(str->utf8_ptr);
    free(str);
    return;
}


void
iiimp_string_list_delete(IIIMP_data_s * data_s, IIIMP_string * str)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */
    IIIMP_string *	str_next;
    for (; NULL != str; str = str_next) {
	str_next = str->next;
	free(str->ptr);
	free(str);
    }
    return;
}


void
iiimp_string_pack(
    IIIMP_data_s *	data_s,
    IIIMP_string *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUTU16((2 * m->len), rest, p, data_s->byte_swap);

    for (i = 0; i < m->len; i++) {
	PUTU16(*(m->ptr + i), rest, p, data_s->byte_swap);
    }

    if (0 == (1 & m->len)) {
	PUTU16(0, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_string_list_pack(
    IIIMP_data_s *	data_s,
    IIIMP_string *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    for (; NULL != m; m = m->next) {
	iiimp_string_pack(data_s, m, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


IIIMP_string *
iiimp_string_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_string *	str;
    const uchar_t *	p;
    size_t		rest;
    size_t		len;
    int			data_size;
    int			i;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < 4)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    GET16(len, rest, p, data_s->byte_swap);

    data_size = (len + PAD(2 + len));
    if ((0 != (len & 0x01)) || (rest < data_size)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    str = (IIIMP_string *)malloc(sizeof (IIIMP_string));
    if (NULL == str) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    str->len = (len / 2);
    str->nbyte = (2 + len + PAD(2 + len));
    str->next = NULL;
    if (0 == len) {
	str->ptr = NULL;
    } else {
	str->ptr = (IIIMP_card16 *)malloc(len);
	if (NULL == str->ptr) {
	    iiimp_string_delete(data_s, str);
	    data_s->status = IIIMP_DATA_MALLOC_ERROR;
	    return NULL;
	}

	for (i = 0; i < str->len; i++) {
	    GETU16(*(str->ptr + i), rest, p, data_s->byte_swap);
	}
    }

    *nbyte -= (2 + data_size);
    *ptr += (2 + data_size);

    str->utf8_len = 0;
    str->utf8_ptr = NULL;

    str = iiimp_string_utf8_sync(data_s, str);

    return str;
}


IIIMP_string *
iiimp_string_list_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_string *	str;
    size_t		rest;
    const uchar_t *	p;
    IIIMP_string *	str_first;
    IIIMP_string *	str_last;

    rest = nbyte_max;
    p = *ptr;
    str = NULL;
    str_first = NULL;
    str_last = NULL;

    if (((*nbyte) < nbyte_max) || (0 != (rest & 0x01)) || (0 == rest)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    while (0 < rest) {
	str = iiimp_string_unpack(data_s, &rest, &p, rest);
	if (NULL == str) {
	    iiimp_string_list_delete(data_s, str_first);
	    return NULL;
	} else {
	    if (NULL == str_first) {
		str_first = str;
	    } else {
		str_last->next = str;
	    }
	    str_last = str;
	}
    }

    *nbyte -= (nbyte_max - rest);
    *ptr = p;

    return str_first;
}


void
iiimp_string_print(
    IIIMP_data_s *	data_s,
    IIIMP_string *	m)
{
    int			i;
    const uchar_t *	p;
    int			byte_len;

    if ((NULL == m) || (0 == m->len)) {
	return;
    }

    /* ASCII or UTF-8 */
    for (i = 0; i < m->len; i++) {
	if ((*(m->ptr + i) < 0x20) || (0x7f <= *(m->ptr + i))) {
	    break;
	}
    }

    if (i == m->len) { /* ASCII */
	(void)fputc('"', data_s->print_fp);
	for (i = 0; i < m->len; i++) {
	    (void)fprintf(data_s->print_fp, "%c", *(m->ptr + i));
	}
	(void)fputc('"', data_s->print_fp);
    } else { /* UTF-16 */
	p = (uchar_t *)(m->ptr);
	byte_len = (m->len * 2);

	for (i = 0; i < byte_len; i += 2) {
	    (void)fprintf(data_s->print_fp, " U+%04x ", ((*p << 8) | *(p + 1)));
	    (void)fprintf(data_s->print_fp, "%c", isprint(*p) ? *p : ' ');
	    p++;
	    (void)fprintf(data_s->print_fp, "%c", isprint(*p) ? *p : ' ');
	    p++;
	}
    }
}


void
iiimp_string_list_print(
    IIIMP_data_s *	data_s,
    IIIMP_string *	m)
{
    if (NULL == m) return;
    iiimp_string_print(data_s, m);
    for (m = m->next; NULL != m; m = m->next) {
	(void)fputc(' ', data_s->print_fp);
	iiimp_string_print(data_s, m);
    }
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
