#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ime.h"
#include "chewing_im.h"
#include "chewing_properties.h"

#ifndef LIBCHEWING_PREFIX
#define LIBCHEWING_PREFIX       "/usr"
#endif

#define CHEWING_LIB             LIBCHEWING_PREFIX"/lib/libchewing.so"
#define CHEWING_DATADIR         LIBCHEWING_PREFIX"/share/chewing"
#define CHEWING_NAME_UTF8       "新酷音"
#define CHEWING_UUID            "chewing-9d3878a2-ca6a-4dbb-9e81-62b3774716c3"
#define CHEWING_VERSION         300
#define AUTHOR \
	"Chewing core team <http://chewing.csie.net>\n" \
    "Gavin Tu\n" \
	"Jeffrey Chen"
#define COPYRIGHT		"Copyright (c) 2005 Chewing core team"
#define HINTING			"Chewing Input Method"
#define CHEWING_ICONPATH        "chewing.xpm"

ImeResult chewing_Initialize(ImeInfo ime_info);
ImeResult chewing_Destroy(ImeInfo ime_info);
ImeResult chewing_Process_Key_Event(ImeInputContext ic, ImeKey key_event);
ImeResult chewing_Create_Session(ImeInputContext ic);
ImeResult chewing_Destroy_Session(ImeInputContext ic);
ImeResult chewing_FocusOut(ImeInputContext ic);
ImeResult chewing_FocusIn(ImeInputContext ic);

ImmServices imm_services;

ImeMethodsRec chewing_methods = {
    1,				/* version */
    chewing_Initialize,		/* ImeInitialize */
    chewing_Destroy,		/* ImeDestroy  */
    chewing_Process_Key_Event,	/* ImeProcessKeyEvent */
    NULL,			/* ImeProcessAuxEvent  */
    chewing_Create_Session,	/* ImeAttachSession */
    chewing_Destroy_Session,	/* ImeDetachSession */
    chewing_FocusIn,		/* ImeFocusIn  */
    chewing_FocusOut,		/* ImeFocusOut */
    NULL,			/* ImeAttachUser */
    NULL,			/* ImeDetachUser */
    NULL,			/* ImeAttachDesktop */
    NULL,			/* ImeDetachDesktop */
    NULL,			/* ImeGetErrorMessage */
#if 0
    NULL,			/* ImeDoConfig */
#endif
};

#ifdef	WIN32
#define EXPORT extern __declspec(dllexport)
EXPORT
#endif
ImeResult RegisterIME(ImmServices srvs, ImeInfo * ppinfo,
		      ImeMethods * pmthds, int argc, char **argv)
{
    ImeInfoRec *chewing_info = NULL;
    char *base_dir = NULL;
    int i;

    DEBUG_printf("Register Chewing IM: argc: %d\n", argc);
    for (i = 0; i < argc; i++) {
	if (!strcasecmp(argv[i], "-basedir")) {
	    if (argv[i + 1]) {
		base_dir = argv[i + 1];
		DEBUG_printf("       setting base dir to: %s\n",
			     argv[i + 1]);
	    }
	    i++;
	}
    }

    if (access(CHEWING_LIB, R_OK) != 0) {
	fprintf(stderr, "Error: Chewing: %s not found !!\n", CHEWING_LIB);
	return (IME_FAIL);
    }

    if (access(CHEWING_DATADIR "/dict.dat", R_OK) != 0) {
	fprintf(stderr, "Error: Chewing: data directory: %s not found !!\n", CHEWING_DATADIR);
	return (IME_FAIL);
    }

    chewing_info = (ImeInfoRec *) calloc(1, sizeof(ImeInfoRec));
    DEBUG_printf("chewing_info: %p\n", chewing_info);
    if (chewing_info == NULL) {
	return (IME_FAIL);
    }

    chewing_info->version = CHEWING_VERSION;
    chewing_info->mt_safe = 0;
    chewing_info->encoding = ENCODE_UTF8;
    chewing_info->uuid = CHEWING_UUID;
    chewing_info->name = CHEWING_NAME_UTF8;
    chewing_info->author = AUTHOR;
    chewing_info->hinting = HINTING;
    chewing_info->copyright = COPYRIGHT;
    chewing_info->icon_file = CHEWING_ICONPATH;
    chewing_info->support_locales = "zh_TW.UTF-8,zh_TW.BIG5";
    chewing_info->pl = NULL;
    chewing_info->specific_data = NULL;

    chewing_Init_Ime_Properties(chewing_info);

    *ppinfo = chewing_info;
    *pmthds = &chewing_methods;

    imm_services = srvs;

    DEBUG_printf("begin leave Register IME\n");
    return (IME_OK);
}

ImeResult chewing_Initialize(ImeInfo chewing_info)
{
    DEBUG_printf("chewing_Initialize\n");

    if (chewing_info == NULL)
	return (IME_FAIL);

    const char *prefix = CHEWING_DATADIR;
    const char *hash_postfix = "/.chewing/";
    char *home = getenv ("HOME");
    char path[1024];

    snprintf (path, sizeof(path), "%s/%s", home, hash_postfix);
    chewing_Init(prefix, path);

    DEBUG_printf("Chewing initialize done\n");

    return (IME_OK);
}

ImeResult chewing_Destroy(ImeInfo chewing_info)
{
    DEBUG_printf("chewing_Destroy\n");

    if (chewing_info != NULL) {
	chewing_Destroy_Ime_Properties(chewing_info);
	free((char *) chewing_info);
    }

    return (IME_OK);
}

ImeResult chewing_Create_Session(ImeInputContext ic)
{
    int i;
    ImmResult imm_result;
    ImeInfoRec *chewing_info = NULL;
    
    ChewingContext *ctx = NULL;
    ctx = (ChewingContext *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    DEBUG_printf ("chewing_Create_Session ======= begin get ime_session_data: %p\n", ctx);

    if (ctx == NULL) {
        int i, j=1;
	int default_selectionKeys[] = {'1','2','3','4','5','6','7','8','9','0'};
        
        ctx = chewing_new();
        chewing_set_KBType (ctx, 0);

        chewing_set_candPerPage (ctx, 9);
        chewing_set_maxChiSymbolLen (ctx, 16);
        chewing_set_addPhraseDirection (ctx, 1);
        chewing_set_phraseChoiceRearward( ctx, 1);
        chewing_set_autoShiftCur (ctx, 1);
        chewing_set_selKey (ctx, default_selectionKeys, sizeof(default_selectionKeys));
        chewing_set_spaceAsSelection (ctx, 1);
        chewing_set_escCleanAllBuf( ctx, 1);
        
	imm_result = imm_services->ImmSetData(ic, IME_SCOPE_SESSION, ctx);
	if (imm_result == IMM_FAIL) {
            chewing_delete( ctx );
	    return (IME_FAIL);
	}
    }

    return (IME_OK);
}

ImeResult chewing_Destroy_Session(ImeInputContext ic)
{
    ChewingContext *ctx;

    ctx = (ChewingContext *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    DEBUG_printf (" ====>chewing_Destroy_Session ======= begin get ime_session_data: %p\n",
		  ctx);

    if (ctx != NULL) {
        chewing_delete( ctx );
    }

    imm_services->ImmSetData(ic, IME_SCOPE_SESSION, NULL);
    return (IME_OK);
}

ImeResult chewing_FocusIn(ImeInputContext ic)
{
    ChewingContext *ctx = (ChewingContext *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION); 
    if (ctx) chewing_Set_Ime_Properties(ic, ctx);
    return (IME_OK);
}

ImeResult chewing_FocusOut(ImeInputContext ic)
{
    DEBUG_printf("codetable: call chewing_FocusOut()\n");
    return (IME_OK);
}

ImmResult chewing_session_beep(ImeInputContext ic, ChewingContext *ctx)
{
    return (imm_services->ImmBeep(ic, ImeBeepWarning));
}

ImmResult chewing_session_commit(ImeInputContext ic,  ChewingContext *ctx)
{
    if ( chewing_commit_Check( ctx ) ) {
        char* str = chewing_commit_String(ctx);
        if ( str ) {
            imm_services->ImmCommit(ic, (unsigned char*)str);
            chewing_free( str );
        }
    }

    return (IMM_OK);
}

ImmResult chewing_session_show_preedit(ImeInputContext ic,  ChewingContext *ctx)
{
    ImePreeditRec ime_preedit;

    int i, caret_pos;
    int len = 0;
    char buf[1024];

    memset (buf, 0, sizeof(buf));

    if( chewing_buffer_Check( ctx ) ) {
        char* chibuf = chewing_buffer_String( ctx );
        if ( chibuf ) {
            strcat (buf, chibuf);
            chewing_free (chibuf);
        }
    }

    int zuin_count;
    char* zuin_str = chewing_zuin_String( ctx, &zuin_count );
    if( zuin_str ) {
        strcat (buf, zuin_str);
        chewing_free (zuin_str);
    }

    len = strlen(buf);
    if (len == 0) {
        return (imm_services->ImmHidePreedit(ic));
    }

    imm_services->ImmShowPreedit(ic);

    caret_pos = chewing_cursor_Current( ctx );
    if (caret_pos < 0 || caret_pos > 100)
        caret_pos = 0;

    memset(&ime_preedit, 0, sizeof(ImePreeditRec));
    ime_preedit.caret = caret_pos;
    ime_preedit.preedit.text = (unsigned char*)buf;

    DEBUG_printf("Show Preedit  caret:%d text:%s\n", caret_pos, buf);
    return (imm_services->ImmUpdatePreedit(ic, &ime_preedit));
}

ImmResult chewing_session_handle_candidates(ImeInputContext ic,  ChewingContext *ctx)
{
    ImmResult imm_result;
    ImeCandidatesRec ime_candidates;

    int i, fc, num_candidates;
    int page_state = 0;

    int total_choices = chewing_cand_TotalChoice (ctx);
    if (total_choices <= 0) {
        return (imm_services->ImmHideCandidates(ic));
    }

    int choices_perpage = chewing_cand_ChoicePerPage (ctx);
    int page_no = chewing_cand_CurrentPage (ctx);

    fc = page_no * choices_perpage;
    if (page_no == 0)
        page_state |= ImeCandidatesFirstPage;  /* disable PageUp icon */

    num_candidates = total_choices - fc;
    if (num_candidates > choices_perpage) {
        num_candidates = choices_perpage;
    } else {
        page_state |= ImeCandidatesLastPage;    /* disable PageDown icon */
    }

    memset(&ime_candidates, 0, sizeof(ImeCandidatesRec));

    ime_candidates.title = NULL;
    ime_candidates.numbers = NULL;
    ime_candidates.focus = 0;
    ime_candidates.page_state = page_state;
    ime_candidates.count = num_candidates;
    ime_candidates.candidates = (ImeTextRec *) calloc(num_candidates, sizeof(ImeTextRec));
    if (ime_candidates.candidates == NULL)
        return (IMM_FAIL);

    DEBUG_printf("Candidate ");

    chewing_cand_Enumerate (ctx);
    for (i = 0; i < num_candidates; ++i) {
        if ( chewing_cand_hasNext( ctx ) ) {
            char* str = chewing_cand_String( ctx );
            if( str ) {
                ime_candidates.candidates[i].text = (unsigned char*) strdup (str);
                DEBUG_printf("   %s   ", ime_candidates.candidates[i].text);
                chewing_free( str );
            }
        }
    }

    DEBUG_printf(" \n");

    imm_result = imm_services->ImmUpdateCandidates(ic, &ime_candidates);
    imm_services->ImmShowCandidates(ic);
    free ((char *)ime_candidates.candidates);
    return (imm_result);
}

/* process key input event */
/* return value:  IME_UNUSED_KEY:  if IME not use this key, return this key to systerm directly */
/*                IME_OK:      if IME has used this key */
ImeResult chewing_Process_Key_Event(ImeInputContext ic, ImeKey key_event)
{
    ChewingContext *ctx = NULL;

    long kcode, kstate;
    unsigned short kchar;

    DEBUG_printf("chewing_Process_Key_Event: ic: %p\n", ic);
    ctx = (ChewingContext *) imm_services->ImmGetData(ic, IME_SCOPE_SESSION); 
    if (ctx == NULL)
	return (IME_UNUSED_KEY);

    kcode = key_event->keycode;
    kchar = key_event->keychar;
    kstate = key_event->modifier;

    DEBUG_printf ("  ====> Chewing processing key (0X%X - 0X%X - 0X%X) ...\n", kcode, kchar, kstate);

    if (kstate == 0) {
        switch (kcode) {
        case IME_VK_PAGE_UP:
        case IME_VK_LEFT:
        chewing_handle_Left( ctx );
	    break;

        case IME_VK_PAGE_DOWN:
        case IME_VK_RIGHT:
        chewing_handle_Right( ctx );
	    break;

        case IME_VK_UP:
        chewing_handle_Up(ctx);
	    break;

        case IME_VK_DOWN:
        chewing_handle_Down(ctx);
	    break;

        case IME_VK_SPACE:
        chewing_handle_Space(ctx);
	    break;

        case IME_VK_CAPS_LOCK:
        chewing_handle_Capslock(ctx);
	    break;

        case IME_VK_ENTER:
        chewing_handle_Enter(ctx);
	    break;

        case IME_VK_BACK_SPACE:
        chewing_handle_Backspace(ctx);
	    break;

        case IME_VK_ESCAPE:
        chewing_handle_Esc(ctx);
	    break;

        case IME_VK_DELETE:
        chewing_handle_Del(ctx);
	    break;

        case IME_VK_HOME:
        chewing_handle_Home(ctx);
	    break;

        case IME_VK_END:
        chewing_handle_End(ctx);
	    break;

        case IME_VK_TAB:
        chewing_handle_Tab(ctx);
	    break;

        case IME_VK_CONTROL:
        case IME_VK_ALT:
        case IME_VK_SHIFT:
            return IME_UNUSED_KEY;

        default:
	    if (kchar)
            chewing_handle_Default(ctx, kchar);
	    break;
        }
    } else if (kstate == IME_SHIFT_MASK) {
        switch (kcode) {
        case IME_VK_PAGE_UP:
        case IME_VK_LEFT:
        chewing_handle_Left(ctx);
	    break;

        case IME_VK_PAGE_DOWN:
        case IME_VK_RIGHT:
        chewing_handle_Right(ctx);
	    break;

        default:
	    if (kchar)
            chewing_handle_Default(ctx, kchar);
	    break;
        }
    } else if (kstate == IME_CTRL_MASK) {
        /* Handle user-phrase addition */
        if (kchar <= '9' && kchar >= '0') {
            chewing_handle_CtrlNum(ctx, kchar);
        }
    }

    DEBUG_printf("  ---->return bitmask 0x%x\n", op->keystrokeRtn);
    chewing_session_commit(ic, ctx);

    DEBUG_printf("    ---->show preedit...\n");
    chewing_session_show_preedit(ic, ctx);

    DEBUG_printf("    ---->handle candidates...\n");
    chewing_session_handle_candidates(ic, ctx);

#if 0 /* looks like no beep since 0.3.2 */
    if (op->keystrokeRtn & KEYSTROKE_BELL)
	chewing_session_beep(ic, ctx);
#endif

    if (chewing_keystroke_CheckAbsorb (ctx))
	return (IME_OK);

    if (chewing_keystroke_CheckIgnore (ctx))
	return (IME_UNUSED_KEY);

    return (IME_OK);
}
