#include <stdio.h>

#include "le_info.h"
#include "imbean.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

extern LeInfoRec *le_info;

/***************************************************************************************/
/*                         LE User Identity                                            */
/***************************************************************************************/
LeUserIdRec *le_user_id_init()
{
    LeUserIdRec *le_user_id = NULL;

    le_user_id = (LeUserIdRec *) calloc(1, sizeof(LeUserIdRec));
    return (le_user_id);
}

LeResult le_user_id_destroy(LeUserIdRec * le_user_id)
{
    if (le_user_id == NULL)
	return (LE_FAIL);

    if (le_user_id->user_name)
	free((char *) le_user_id->user_name);
    if (le_user_id->host_name)
	free((char *) le_user_id->host_name);

    free((char *) le_user_id);

    return (LE_OK);
}

LeResult le_user_id_print(LeUserIdRec * le_user_id)
{
#if DEBUG
    if (le_user_id == NULL)
	return (LE_FAIL);

    if (le_user_id->user_name)
	printf(" user_name: %s\n", le_user_id->user_name);
    if (le_user_id->host_name)
	printf(" host_name: %s\n", le_user_id->host_name);

    return (LE_OK);
#endif
}

LeResult le_user_id_set_name(LeUserIdRec * le_user_id,
			     char *user_name, char *host_name)
{
    if (le_user_id == NULL)
	return (LE_FAIL);

    if (user_name != NULL) {
	if (le_user_id->user_name)
	    free((char *) le_user_id->user_name);
	le_user_id->user_name = (char *) strdup(user_name);
    }

    if (host_name != NULL) {
	if (le_user_id->host_name)
	    free((char *) le_user_id->host_name);
	le_user_id->host_name = (char *) strdup(host_name);
    }

    return (LE_OK);
}

#define COMPARE_HOST_NAME 1

int le_user_id_compare(LeUserIdRec * user_id1, LeUserIdRec * user_id2)
{
    if (user_id1 == NULL && user_id2 == NULL)
	return (0);

    if (user_id1 == NULL || user_id2 == NULL)
	return (1);

    if (user_id1->user_name && user_id2->user_name) {
	if (strcmp(user_id1->user_name, user_id2->user_name))
	    return (1);
#if COMPARE_HOST_NAME
	if (user_id1->host_name && user_id2->host_name) {
	    if (strcmp(user_id1->host_name, user_id2->host_name))
		return (1);

	    return (0);
	}
#else
	return (0);
#endif
    }
    return (1);
}

/***************************************************************************************/
/*                         ImeModuleContext                                            */
/***************************************************************************************/
ImeModuleContextRec *ime_module_context_init()
{
    ImeModuleContextRec *ime_module_context = NULL;

    ime_module_context =
	(ImeModuleContextRec *) calloc(1, sizeof(ImeModuleContextRec));
    return (ime_module_context);
}

LeResult ime_module_context_destroy(ImeModuleContextRec *
				    ime_module_context)
{
    if (ime_module_context == NULL)
	return (LE_OK);

    /* FIXME:  need free ime properties data */

    free((char *) ime_module_context);
    return (LE_OK);
}

LeResult ime_module_context_print(ImeModuleContextRec * ime_module_context)
{
#if DEBUG
    if (ime_module_context == NULL)
	return (LE_OK);

    printf("ime_module: %p, enabled: %d\n", ime_module_context->ime_module,
	   ime_module_context->enabled);

    return (LE_OK);
#endif
}

/***************************************************************************************/
/*                         LE User Context                                             */
/***************************************************************************************/
LeUserContextRec *le_user_context_init()
{
    int i;

    LeUserContextRec *le_user_context = NULL;

    le_user_context =
	(LeUserContextRec *) calloc(1, sizeof(LeUserContextRec));

    if (le_user_context == NULL)
	return (NULL);

    if (le_info != NULL &&
	le_info->ime_modules != NULL && le_info->num_ime_modules > 0) {
	int num_ime_modules = le_info->num_ime_modules;

	le_user_context->ime_modules =
	    (ImeModuleContextRec **) calloc(num_ime_modules,
					    sizeof(ImeModuleContextRec *));

	if (le_user_context->ime_modules != NULL) {
	    le_user_context->num_ime_modules = num_ime_modules;
	    for (i = 0; i < num_ime_modules; i++) {
		le_user_context->ime_modules[i] =
		    (ImeModuleContextRec *) ime_module_context_init();

		if (le_user_context->ime_modules[i] == NULL)
		    continue;

		le_user_context->ime_modules[i]->ime_module =
		    le_info->ime_modules[i];
		le_user_context->ime_modules[i]->enabled = 1;
		le_user_context->ime_modules[i]->pl =
		    le_info->ime_modules[i]->info->pl;
	    }
	}
    }

    return (le_user_context);
}

LeResult le_user_context_destroy(LeUserContextRec * le_user_context)
{
    int i;

    if (le_user_context == NULL)
	return (LE_OK);

    if (le_user_context->user_id)
	le_user_id_destroy(le_user_context->user_id);

    if (le_user_context->ime_modules) {
	for (i = 0; i < le_user_context->num_ime_modules; i++) {
	    if (le_user_context->ime_modules[i])
		ime_module_context_destroy(le_user_context->
					   ime_modules[i]);
	}

	free((char *) le_user_context->ime_modules);
    }

    free((char *) le_user_context);
    return (LE_OK);
}

LeResult le_user_context_print(LeUserContextRec * le_user_context)
{
    int i;

    if (le_user_context == NULL)
	return (LE_OK);

    if (le_user_context->user_id)
	le_user_id_print(le_user_context->user_id);

    if (le_user_context->ime_modules) {
	for (i = 0; i < le_user_context->num_ime_modules; i++) {
	    if (le_user_context->ime_modules[i])
		ime_module_context_print(le_user_context->ime_modules[i]);
	}
    }

    return (LE_OK);
}

LeResult le_user_context_set_user_id(LeUserContextRec * le_user_context,
				     char *user_name, char *host_name)
{
    if (le_user_context == NULL)
	return (LE_OK);

    if (le_user_context->user_id)
	le_user_id_destroy(le_user_context->user_id);

    le_user_context->user_id = le_user_id_init();
    le_user_id_set_name(le_user_context->user_id, user_name, host_name);

    return (LE_OK);
}

/***************************************************************************************/
/*                         LE User Manager                                             */
/***************************************************************************************/
LeUserManagerRec *le_user_manager_init()
{
    LeUserManagerRec *le_user_manager = NULL;

    le_user_manager =
	(LeUserManagerRec *) calloc(1, sizeof(LeUserManagerRec));
    return (le_user_manager);
}

LeResult le_user_manager_destroy(LeUserManagerRec * le_user_manager)
{
    int i;
    LeUserContextRec *le_user_context;

    if (le_user_manager == NULL)
	return (LE_OK);
    if (le_user_manager->le_users == NULL)
	return (LE_OK);

    /* free ime modules */
    for (i = 0; i < le_user_manager->num_le_users; i++) {
	le_user_context = le_user_manager->le_users[i];
	if (le_user_context != NULL)
	    le_user_context_destroy(le_user_context);
    }

    free((char *) le_user_manager->le_users);
    free((char *) le_user_manager);

    return (LE_OK);
}

LeResult le_user_manager_print(LeUserManagerRec * le_user_manager)
{
#if DEBUG
    int i;
    LeUserContextRec *le_user_context;

    if (le_user_manager == NULL)
	return (LE_OK);
    if (le_user_manager->le_users == NULL)
	return (LE_OK);

    printf("num_le_users:  %d\n", le_user_manager->num_le_users);
    for (i = 0; i < le_user_manager->num_le_users; i++) {
	printf("  le_user_context id: %d\n", i);
	le_user_context = le_user_manager->le_users[i];
	if (le_user_context != NULL)
	    le_user_context_print(le_user_context);
    }

    return (LE_OK);
#endif
}

LeUserContextRec *le_user_manager_create_user_with_name(LeUserManagerRec *
							le_user_manager,
							char *user_name,
							char *host_name)
{
#define NUM_ALLOC_USERS 5
    int i;
    LeUserIdRec user_id;
    LeUserContextRec *le_user_context;

    if (le_user_manager == NULL)
	return (NULL);

    user_id.user_name = user_name;
    user_id.host_name = host_name;

    for (i = 0; i < le_user_manager->num_le_users; i++) {
	if (le_user_manager->le_users == NULL)
	    break;
	if (le_user_manager->le_users[i] == NULL)
	    continue;
	if (le_user_manager->le_users[i]->user_id == NULL)
	    continue;

	if (!le_user_id_compare
	    (le_user_manager->le_users[i]->user_id, &user_id)) {
	    le_user_manager->le_users[i]->ref_num++;
	    return (le_user_manager->le_users[i]);
	}
    }

    if (le_user_manager->num_le_users_alloced == 0) {
	le_user_manager->le_users =
	    (LeUserContextRec **) calloc(NUM_ALLOC_USERS,
					 sizeof(LeUserContextRec *));

	if (le_user_manager->le_users == NULL) {
	    le_user_manager->num_le_users = 0;
	    return (NULL);
	}

	le_user_manager->num_le_users_alloced = NUM_ALLOC_USERS;
    }

    if (le_user_manager->num_le_users >=
	le_user_manager->num_le_users_alloced) {
	le_user_manager->le_users =
	    (LeUserContextRec **) realloc(le_user_manager->le_users,
					  (le_user_manager->
					   num_le_users_alloced +
					   NUM_ALLOC_USERS) *
					  sizeof(LeUserContextRec *));

	if (le_user_manager->le_users == NULL) {
	    le_user_manager->num_le_users = 0;
	    le_user_manager->num_le_users_alloced = 0;
	    return (NULL);
	}

	le_user_manager->num_le_users_alloced += NUM_ALLOC_USERS;

	for (i = le_user_manager->num_le_users;
	     i < le_user_manager->num_le_users_alloced; i++)
	    le_user_manager->le_users[i] = NULL;
    }

    le_user_context = (LeUserContextRec *) le_user_context_init();
    le_user_context_set_user_id(le_user_context, user_name, host_name);

    le_user_manager->le_users[le_user_manager->num_le_users] =
	le_user_context;
    le_user_manager->num_le_users++;

    return (le_user_context);
}

LeResult le_user_manager_destroy_user_with_name(LeUserManagerRec *
						le_user_manager,
						char *user_name,
						char *host_name)
{
    int i, j;
    LeUserIdRec user_id;

    if (le_user_manager == NULL)
	return (LE_FAIL);
    if (le_user_manager->le_users == NULL)
	return (LE_FAIL);

    user_id.user_name = user_name;
    user_id.host_name = host_name;

    for (i = 0; i < le_user_manager->num_le_users; i++) {
	if (le_user_manager->le_users[i] == NULL)
	    continue;
	if (le_user_manager->le_users[i]->user_id == NULL)
	    continue;

	if (!le_user_id_compare
	    (le_user_manager->le_users[i]->user_id, &user_id)) {
	    le_user_manager->le_users[i]->ref_num--;
	    if (le_user_manager->le_users[i]->ref_num < 0) {
		le_user_context_destroy(le_user_manager->le_users[i]);
		for (j = i; j < le_user_manager->num_le_users - 1; j++)
		    le_user_manager->le_users[j] =
			le_user_manager->le_users[j + 1];
		le_user_manager->num_le_users--;
	    }
	}
    }

    return (LE_OK);
}

LeUserContextRec *le_user_manager_get_user_with_name(LeUserManagerRec *
						     le_user_manager,
						     char *user_name,
						     char *host_name)
{
    int i;
    LeUserIdRec user_id;

    if (le_user_manager == NULL)
	return (NULL);
    if (le_user_manager->le_users == NULL)
	return (NULL);

    user_id.user_name = user_name;
    user_id.host_name = host_name;

    for (i = 0; i < le_user_manager->num_le_users; i++) {
	if (le_user_manager->le_users[i] == NULL)
	    continue;
	if (le_user_manager->le_users[i]->user_id == NULL)
	    continue;

	if (!le_user_id_compare
	    (le_user_manager->le_users[i]->user_id, &user_id)) {
	    return (le_user_manager->le_users[i]);
	}
    }

    return (NULL);
}

/***************************************************************************************/
/*                         LE User Profile                                             */
/***************************************************************************************/
int le_user_profile_new_from_ibml_category(LeUserContextRec * uc,
					   IbmlCategory * ibml_category)
{
    DEBUG_printf("Before le_user_profile_new_from_ibml_category()\n");
    IbmlElement *ibml_element;
    IbmlProperty *ibml_property;

    int i, j, k;
    int current_ime_module_id;
    ImeModuleContextRec *current_ime_module_context;
    ImePropertyListRec *pl;

    char *id, *scope;
    char *name, *value, *options;

    if (uc == NULL || uc->ime_modules == NULL)
	return (LE_FAIL);

    if (ibml_category == NULL)
	return (LE_FAIL);

    if (ibml_category->num_elements <= 0)
	return (LE_FAIL);

    current_ime_module_id = 0;
    for (i = 0; i < ibml_category->num_elements; i++) {
	char *uuid = NULL;
	char *enable_str = NULL;
	int enable_value;

	ibml_element = ibml_category->elements[i];
	if (!ibml_element)
	    continue;

	id = (char *) ibml_element->id;
	scope = (char *) ibml_element->scope;
	DEBUG_printf("id:%s, scope:%s\n", id ? id : "NULL",
		     scope ? scope : "NULL");
	if (!id || !*id)
	    continue;

	uuid = id;

	current_ime_module_context = NULL;
	for (j = current_ime_module_id; j < uc->num_ime_modules; j++) {
	    ImeModuleContext ime_module_context = uc->ime_modules[j];

	    if (!ime_module_context || !ime_module_context->ime_module)
		continue;

	    if (!strcasecmp
		(uuid, ime_module_context->ime_module->info->uuid)) {
		for (k = j; k > current_ime_module_id; k--)
		    uc->ime_modules[k] = uc->ime_modules[k - 1];

		uc->ime_modules[current_ime_module_id] =
		    ime_module_context;
		current_ime_module_id++;
		current_ime_module_context = ime_module_context;
		break;
	    }
	}

	if (current_ime_module_context == NULL)
	    continue;

	pl = current_ime_module_context->ime_module->info->pl;

	for (j = 0; j < ibml_element->num_properties; j++) {
	    ibml_property = ibml_element->properties[j];
	    if (!ibml_property)
		continue;
	    name = (char *) ibml_property->name;
	    if (!name || !*name)
		continue;

	    value = (char *) ibml_property->value;
	    if (!value || !*value)
		continue;

	    DEBUG_printf("value is %s.\n", value);
	    if (!strcasecmp(name, "enabled")) {
		enable_str = (char *) value;

		enable_value = 1;
		if (enable_str && *enable_str) {
		    if (!strcasecmp(enable_str, "0")
			|| !strcasecmp(enable_str, "false"))
			enable_value = 0;
		}

		current_ime_module_context->enabled = enable_value;
		continue;
	    }

	    if (pl == NULL)
		continue;

	    for (k = 0; k < pl->count; k ++) {
		ImePropertyRec *pi = &(pl->properties[k]);
		if (pi == NULL || pi->key == NULL)
		    continue;

		if (!strcmp(name, pi->key)) {
		    pi->value.int_value = atoi(value);
		}
	    }
	}
    }

    DEBUG_printf("End le_user_profile_new_from_ibml_category()\n");
    return (LE_OK);
}

#define ALL_ENGINE_CATAGORY   "all_engines"

int le_user_profile_new_from_memory(LeUserContextRec * uc, char *buf,
				    int size)
{
    IbmlData *ibml_data;
    IbmlCategory *ibml_category;
    int i;
    int ret = LE_OK;

    if (uc == NULL)
	return (LE_FAIL);
    if (buf == NULL || size <= 0)
	return (LE_FAIL);

    ibml_data = (IbmlData *) imbean_config_new_from_memory(buf, size);
    if (ibml_data == NULL)
	return (LE_FAIL);

    for (i = 0; i < ibml_data->num_categories; i++) {
	ibml_category = ibml_data->categories[i];

	if (!ibml_category->scope || !*ibml_category->scope)
	    continue;

	if (!strcasecmp(ibml_category->scope, ALL_ENGINE_CATAGORY)) {
	    ret =
		le_user_profile_new_from_ibml_category(uc, ibml_category);
	}
    }

    ibml_data_free(ibml_data);
    return (ret);
}

char *le_user_profile_write_to_memory(LeUserContextRec * uc)
{
    xmlDocPtr doc;
    xmlNodePtr root;
    xmlNodePtr im_node;
    xmlNodePtr imbean_node;
    xmlNodePtr property_node;
    xmlChar *xmlbuf = NULL;
    int size = 0;
    int i;

    if (uc == NULL || uc->ime_modules == NULL)
	return (NULL);

    doc = xmlNewDoc("1.0");

    root = xmlNewNode(NULL, IBML_ELEMENT_ROOT);
    xmlSetProp(root, IBML_ELEMENT_SCOPE, LE_NAME);
    xmlDocSetRootElement(doc, root);

    doc->children = root;

    im_node = xmlNewChild(root, NULL, IBML_ELEMENT_CATAGORY, NULL);
    xmlSetProp(im_node, IBML_ELEMENT_SCOPE, ALL_ENGINE_CATAGORY);

    for (i = 0; i < uc->num_ime_modules; i++) {
	ImeModuleRec *ime_module;
	ImeInfoRec *ime_info;
	ImeModuleContextRec *ime_module_context = uc->ime_modules[i];
	ImePropertyListRec *pl;

	int encoding;
	int enabled_value;
	char *enabled_str;

	char utf8_buf[1024];
	char *tmp_ptr, *name_str;
	int from_len, to_left, ret;

	if (ime_module_context == NULL)
	    continue;

	ime_module = ime_module_context->ime_module;
	if (ime_module == NULL)
	    continue;

	ime_info = ime_module->info;
	if (ime_info == NULL ||
	    ime_info->uuid == NULL ||
	    ime_info->name == NULL || ime_info->author == NULL)
	    continue;

	encoding = ime_info->encoding;
	enabled_value = ime_module_context->enabled;

	imbean_node =
	    xmlNewChild(im_node, NULL, IBML_ELEMENT_IMBEAN, NULL);
	xmlSetProp(imbean_node, IBML_ELEMENT_IMBEAN_ID,
		   (char *) ime_info->uuid);

	name_str = (char *) ime_info->name;
	from_len = strlen(name_str);
	to_left = 1024;
	tmp_ptr = utf8_buf;
	memset(utf8_buf, 0, 1024);
	ret = Convert_Native_To_UTF8(encoding,
				     name_str,
				     from_len,
				     (char **) &tmp_ptr,
				     (size_t *) & to_left);

	if (ret == -1)
	    continue;

	property_node =
	    xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY, NULL);
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME, "name");
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE, utf8_buf);

	name_str = (char *) ime_info->author;
	from_len = strlen(name_str);
	to_left = 1024;
	tmp_ptr = utf8_buf;
	memset(utf8_buf, 0, 1024);
	ret = Convert_Native_To_UTF8(encoding,
				     name_str,
				     from_len,
				     (char **) &tmp_ptr,
				     (size_t *) & to_left);

	if (ret == -1)
	    continue;

	property_node =
	    xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY, NULL);
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME, "author");
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE, utf8_buf);

	sprintf(utf8_buf, "%d", ime_info->version);
	property_node =
	    xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY, NULL);
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME, "version");
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE, utf8_buf);

	if (ime_info->copyright) {
	    name_str = (char *) ime_info->copyright;
	    from_len = strlen(name_str);
	    to_left = 1024;
	    tmp_ptr = utf8_buf;
	    memset(utf8_buf, 0, 1024);
	    ret = Convert_Native_To_UTF8(encoding,
					 name_str,
					 from_len,
					 (char **) &tmp_ptr,
					 (size_t *) & to_left);

	    if (ret == -1)
		continue;

	    property_node =
		xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY,
			    NULL);
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME,
		       "copyright");
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE,
		       utf8_buf);
	}

	if (ime_info->hinting) {
	    name_str = (char *) ime_info->hinting;
	    from_len = strlen(name_str);
	    to_left = 1024;
	    tmp_ptr = utf8_buf;
	    memset(utf8_buf, 0, 1024);
	    ret = Convert_Native_To_UTF8(encoding,
					 name_str,
					 from_len,
					 (char **) &tmp_ptr,
					 (size_t *) & to_left);

	    if (ret == -1)
		continue;

	    property_node =
		xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY,
			    NULL);
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME,
		       "hinting");
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE,
		       utf8_buf);
	}

	if (ime_info->icon_file) {
	    property_node =
		xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY,
			    NULL);
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME,
		       "icon_file");
	    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE,
		       ime_info->icon_file);
	}

	property_node =
	    xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY, NULL);
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME, "enabled");
	enabled_str = "true";
	if (enabled_value == 0)
	    enabled_str = "false";
	xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE,
		   enabled_str);

	DEBUG_printf("ime_name: %s\n", ime_info->uuid);
	pl = ime_module_context->pl;
	if (pl != NULL) {
	    int j;
	    for (j = 0; j < pl->count; j++) {
		ImeProperty ime_property;
		char options_str[1024];
		char options_type[56];
		char options_value[10];

		ime_property = &(pl->properties[j]);
		if (ime_property->name == NULL
		    || ime_property->key == NULL)
		    continue;

		property_node =
		    xmlNewChild(imbean_node, NULL, IBML_ELEMENT_PROPERTY,
				NULL);
		xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_NAME,
			   ime_property->key);

		options_str[0] = 0;
		options_type[0] = 0;
		options_value[0] = '0';
		options_value[1] = 0;
		if (ime_property->type == ImeProperty_Selection) {
		    char **value_ptr;
		    int k;
		    strcat(options_str, ime_property->name);
		    value_ptr = ime_property->range.multiString_range;
		    if (value_ptr != NULL) {
			for (k = 0; value_ptr[k] != NULL; k++) {
			    strcat(options_str, "/");
			    strcat(options_str, value_ptr[k]);
			}
		    }
		    strcat(options_type, "ImeProperty_Selection");
		} else if (ime_property->type == ImeProperty_Toggle) {
		    strcat(options_str, ime_property->name);
		    strcat(options_type, "ImeProperty_Toggle");
		} else if (ime_property->type == ImeProperty_Int) {
		    char value[12];
		    strcat(options_str, ime_property->name);
		    strcat(options_str, "/");
		    sprintf(value, "%d/%d",
			    ime_property->range.int_range.min,
			    ime_property->range.int_range.max);
		    strcat(options_str, value);
		    strcat(options_type, "ImeProperty_Int");
		}

		snprintf(options_value, 10, "%d", ime_property->value.int_value);

		DEBUG_printf("options_str: %s, options_type: %s, options_value: %s\n",
			     options_str, options_type, options_value);
		if (*options_str && *options_type) {
		    encoding = ime_property->encode;
		    name_str = (char *) options_str;
		    from_len = strlen(name_str);
		    to_left = 1024;
		    tmp_ptr = utf8_buf;
		    memset(utf8_buf, 0, 1024);
		    ret = Convert_Native_To_UTF8(encoding,
						 name_str,
						 from_len,
						 (char **) &tmp_ptr,
						 (size_t *) & to_left);

		    if (ret == -1)
			continue;

		    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_OPTIONS,
			       utf8_buf);
		    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_TYPE,
			       options_type);
		    xmlSetProp(property_node, IBML_ELEMENT_PROPERTY_VALUE,
			       options_value);
		}
	    }
	}
    }

    xmlDocDumpFormatMemoryEnc(doc, &xmlbuf, &size, "UTF-8", 1);

    xmlFreeDoc(doc);

    return (xmlbuf);
}
