/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "../sh4lt/c/cfollower.h"
#include "../sh4lt/c/clogger.h"
#include "../sh4lt/c/cshtype.h"
#include "../sh4lt/c/cwriter.h"

static char my_user_data[6] = "hello";
static char label[16] = "check-c-wrapper";
static char group_label[5] = "test";
int server_interactions = 0;
int num_successive_write = 5;
static Sh4ltShType shtype = nullptr;

// a struct with contiguous data storage 
typedef struct _Frame {
  size_t count = 0;
  const int tab[3] = {1, 2, 3};
} Frame;

void mylog(void *user_data, const char *str) {
  assert(user_data == my_user_data);
  printf("%s\n", str);
}

void on_server_connect(void* user_data, const char* /*type_descr*/) {
  assert(user_data == my_user_data);
  ++server_interactions;
}

void on_server_disconnect(void *user_data) {
  assert(user_data == my_user_data);
  ++server_interactions;
}

void on_data(void* user_data, void* data, size_t size, const sh4lt_time_info_t* info) {
  assert(user_data == my_user_data);
  Frame *frame = (Frame *)data;
  printf("new data for client: %zu size %zu ptr %p\n",
         frame->count,
         size,
         data);
         assert(info->buffer_duration == 1000000/*1ms*/);
         assert(info->buffer_date != -1);
         assert(info->buffer_number != -1);
         assert(info->system_clock_date > 0);
         assert(info->steady_clock_date > 0);
}

int main () {
  // create a logger
  Sh4ltLogger logger = sh4lt_make_logger(&mylog, &mylog, &mylog, &mylog, &mylog,
                                         &mylog, my_user_data);
  assert(NULL != logger);
  // create a spec for the sh4lt
  shtype = sh4lt_make_shtype("application/x-check-c-wrapper", label, group_label);
  assert(shtype);
  // get and test the sh4lt path for the unix socket
  char *path = sh4lt_get_path(label, group_label);
  if(NULL == path) {
    char *msg = sh4lt_get_log_for_path_issue();
    assert(msg);
    free(msg);
  }
  assert(path);
  // create a follower
  Sh4ltFollower follower = sh4lt_make_follower(
      path, &on_data, &on_server_connect,
      &on_server_disconnect, my_user_data, logger);
  free(path);
  assert(follower);
  // create somes writers
  int i = num_successive_write;
  while (-1 != --i) {
    Sh4ltWriter writer = sh4lt_make_writer(shtype, sizeof(Frame), NULL, NULL,
                                           NULL, logger, 0600);
    assert(writer); 
    usleep(50000);
    Frame frame;
    frame.count = 0;
    for (size_t j = 0; j < 10; ++j) {
      // first option: copy (time information are fake, see check_timecode.cpp for correct use)
      ++frame.count;
      assert(0 != sh4lt_copy_to_shm(
                      writer, &frame, sizeof(Frame), frame.count * 1000000, 1000000 /*1ms*/));
    }
    for (size_t j = 0; j < 10; ++j) {
      // second option: direct write
      Sh4ltWriterAccess access = sh4lt_get_one_write_access(writer);
      Frame *shared_frame = (Frame *)sh4lt_get_mem(access);
      shared_frame->count = ++frame.count;
      sh4lt_notify_clients(access, sizeof(Frame), frame.count * 1000000, 1000000 /*1ms*/);
      sh4lt_release_one_write_access(access);
    }
    sh4lt_delete_writer(writer);
    usleep(50000);
    }

  // clean before quit
  sh4lt_delete_follower(follower);
  sh4lt_delete_shtype(shtype);
  sh4lt_delete_logger(logger);

  // test success
  if (server_interactions == 2 * num_successive_write)
     return 0;
  return 1;  // fail
}

