%define OT_FieldToPointFunction_doc
"Function mapping a field into a point.

Parameters
----------
inputMesh : :class:`~openturns.Mesh`
    The input mesh
inputDim : positive int
    Dimension of the input field values d
outputDim : positive int
    Dimension of the output vector d'

Notes
-----
Field to point functions  act on fields to produce points:

.. math::

    f: \\left| \\begin{array}{rcl}
                \\cM_N \\times (\\Rset^d)^N & \\rightarrow &\\Rset^{d'} \\\\
                F & \\mapsto & \\vect{v}'
              \\end{array} \\right.

with  :math:`\\cM_N` a mesh of :math:`\\cD \\subset \\Rset^n`.

A field is represented by a collection :math:`(\\vect{t}_i, \\vect{v}_i)_{1 \\leq i \\leq N}`
of elements of :math:`\\cM_N \\times (\\Rset^d)^N` where :math:`\\vect{t}_i` is a vertex of :math:`\\cM_N` and :math:`\\vect{v}_i` the associated value in :math:`\\Rset^d`.

The two first constructors build an object which evaluation operator is not defined (it throws a *NotYetImplementedException*).
The instantiation of such an object is used to extract an actual :class:`~openturns.FieldToPointFunction` from a  :class:`~openturns.Study`.

See also
--------
PythonFieldToPointFunction, OpenTURNSPythonFieldToPointFunction, PointToFieldFunction 

Examples
--------
>>> import openturns as ot

Use the class :class:`~openturns.OpenTURNSPythonFieldToPointFunction` to create a function
that acts on fields of input dimension :math:`n=1` and input dimension :math:`d=2`
and returns their spatial mean vector of dimension :math:`d'=2`.

Using the class :class:`~openturns.OpenTURNSPythonFieldToPointFunction` allows one to define a persistent state between the evaluations of the function.

>>> class FUNC(ot.OpenTURNSPythonFieldToPointFunction):
...     def __init__(self):
...         # first argument:
...         mesh = ot.RegularGrid(0.0, 0.1, 11)
...         super(FUNC, self).__init__(mesh, 2, 2)
...         self.setInputDescription(['R', 'S'])
...         self.setOutputDescription(['T', 'U'])  
...     def _exec(self, X):
...         Xs = ot.Sample(X)
...         Y = Xs.computeMean()
...         return Y
>>> F = FUNC()

Create the associated FieldToPointFunction: 

>>> myFunc = ot.FieldToPointFunction(F)

It is also possible to create a FieldToPointFunction from a python function as follows:

>>> def myPyFunc(X):
...     Xs = ot.Sample(X)
...     Y = Xs.computeMean()
...     return Y
>>> inputDim = 2
>>> outputDim = 2
>>> mesh = ot.RegularGrid(0.0, 0.1, 11)
>>> myFunc = ot.PythonFieldToPointFunction(mesh, inputDim, outputDim, myPyFunc)

Evaluate the function on a field:

>>> myField = ot.Field(mesh, ot.Normal(2).getSample(11))
>>> Y = myFunc(myField)
"
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation
OT_FieldToPointFunction_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getCallsNumber_doc
"Get the number of calls of the function.

Returns
-------
callsNumber : int
    Counts the number of times the function has been called since its
    creation."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getCallsNumber
OT_FieldToPointFunction_getCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getInputDescription_doc
"Get the description of the input field values.

Returns
-------
inputDescription : :class:`~openturns.Description`
    Description of the input field values."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getInputDescription
OT_FieldToPointFunction_getInputDescription_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getInputDimension_doc
"Get the dimension of the input field values.

Returns
-------
d : int
    Dimension :math:`d` of the input field values."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getInputDimension
OT_FieldToPointFunction_getInputDimension_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getMarginal_doc
"Get the marginal(s) at given indice(s).

Parameters
----------
i : int or list of ints, :math:`0 \\leq i < d`
    Indice(s) of the marginal(s) to be extracted.

Returns
-------
function : :class:`~openturns.FieldToPointFunction`
    The initial function restricted to the concerned marginal(s) at the indice(s)
    :math:`i`."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getMarginal
OT_FieldToPointFunction_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getInputMesh_doc
"Get the mesh associated to the input domain.

Returns
-------
inputMesh : :class:`~openturns.Mesh`
    The input mesh :math:`\\cM_{N'}`."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getInputMesh
OT_FieldToPointFunction_getInputMesh_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getOutputDescription_doc
"Get the description of the output vector.

Returns
-------
outputDescription : :class:`~openturns.Description`
    Description of the output vector."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getOutputDescription
OT_FieldToPointFunction_getOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_getOutputDimension_doc
"Get the dimension of the output vector.

Returns
-------
d' : int
    Dimension :math:`d'` of the output vector."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::getOutputDimension
OT_FieldToPointFunction_getOutputDimension_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_setInputDescription_doc
"Set the description of the input field values.

Parameters
----------
inputDescription : sequence of str
    Description of the input field values."
%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::setInputDescription
OT_FieldToPointFunction_setInputDescription_doc

// ---------------------------------------------------------------------

%define OT_FieldToPointFunction_setOutputDescription_doc
"Set the description of the output vector.

Parameters
----------
outputDescription : sequence of str
    Description of the output vector."%enddef
%feature("docstring") OT::FieldToPointFunctionImplementation::setOutputDescription
OT_FieldToPointFunction_setOutputDescription_doc
