%feature("docstring") OT::LinearLeastSquaresCalibration
"Linear least squares calibration algorithm.

Available constructors:
    LinearLeastSquaresCalibration(*model, inputObservations, outputObservations, startingPoint, methodName*)

    LinearLeastSquaresCalibration(*modelObservations, gradientObservations, outputObservations, startingPoint, methodName*)

Parameters
----------
model : :class:`~openturns.Function`
    The parametric function to be calibrated.
inputObservations : 2-d sequence of float
    The sample of input observations.
    Can have dimension 0 to specify no observations.
outputObservations : 2-d sequence of float
    The sample of output observations.
startingPoint : sequence of float
    The reference value of the parameter, where the linearization
    of the parametric model is evaluated.
methodName : str
    The name of the least-squares method to use for the calibration.
    By default, equal to *QR*. Possible values are *SVD*, *QR*, *Cholesky*.
modelObservations : 2-d sequence of float
    The sample of output values of the model.
gradientObservations : 2-d sequence of float
    The Jacobian matrix of the model with respect to the parameter.

Notes
-----
LinearLeastSquaresCalibration is the minimum variance estimator of the parameter of a given model under the assumption that this parameter acts linearly in the model.

The prior distribution of the parameter is a noninformative prior
emulated using a flat :class:`~openturns.Normal` centered on the `startingPoint` and with a variance equal to SpecFunc.MaxScalar.

The posterior distribution of the parameter is :class:`~openturns.Normal` and reflects the 
variability of the optimum parameter depending on the observation sample. 
The associated covariance matrix may be regularized depending on the value of the 
key `LinearLeastSquaresCalibration-Regularization` in the :class:`~openturns.ResourceMap`. 
Let us denote by :math:`\\sigma_1` the largest singular value of the covariance matrix. 
The default value of the `LinearLeastSquaresCalibration-Regularization`, zero, 
ensures that the singular values of the covariance matrix are left unmodified.  
If this parameter is set to a nonzero, relatively small, value denoted by :math:`\\epsilon`, 
then all singular values of the covariance matrix are increased by :math:`\\epsilon \\sigma_1`. 

The resulting distribution of the output error is :class:`~openturns.Normal` with a zero mean
and a diagonal covariance matrix computed from the residuals.
The residuals are computed based on the linearization of the model,
where the Jacobian matrix is evaluated at the `startingPoint`.
The diagonal of the covariance matrix of the output error
is constant and is estimated with the unbiased variance estimator.

Please read :ref:`code_calibration` for more details.

See also
--------
GaussianLinearCalibration, NonLinearLeastSquaresCalibration, GaussianNonLinearCalibration

Examples
--------
Calibrate a nonlinear model using linear least-squares:

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> m = 10
>>> x = [[0.5 + i] for i in range(m)]
>>> inVars = ['a', 'b', 'c', 'x']
>>> formulas = ['a + b * exp(c * x)']
>>> model = ot.SymbolicFunction(inVars, formulas)
>>> p_ref = [2.8, 1.2, 0.5]
>>> params = [0, 1, 2]
>>> modelX = ot.ParametricFunction(model, params, p_ref)
>>> y = modelX(x)
>>> y += ot.Normal(0.0, 0.05).getSample(m)
>>> startingPoint = [1.0]*3
>>> method = 'SVD'
>>> algo = ot.LinearLeastSquaresCalibration(modelX, x, y, startingPoint, method)
>>> algo.run()
>>> print(algo.getResult().getParameterMAP())
[8.24019,0.0768046,0.992957]"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearLeastSquaresCalibration::getModelObservations
"Accessor to the model evaluation at the `startingPoint`.

Returns
-------
modelObservation : :class:`~openturns.Sample`
    Evaluation of the model at the `startingPoint`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearLeastSquaresCalibration::getGradientObservations
"Accessor to the model gradient at the startingPoint.

Returns
-------
gradientObservation : :class:`~openturns.Matrix`
    Gradient of the model at the startingPoint point."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearLeastSquaresCalibration::getStartingPoint
"Accessor to the parameter startingPoint.

Returns
-------
startingPoint : :class:`~openturns.Point`
    Parameter startingPoint."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearLeastSquaresCalibration::getMethodName
"Accessor to the name of least-squares method used for the resolution.

Returns
-------
name : :class:`~openturns.String`
    Name of least-squares method used for the resolution."

