﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appsync/AppSync_EXPORTS.h>
#include <aws/appsync/model/SourceApiAssociationConfig.h>
#include <aws/appsync/model/SourceApiAssociationStatus.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace AppSync {
namespace Model {

/**
 * <p>Describes the configuration of a source API. A source API is a GraphQL API
 * that is linked to a merged API. There can be multiple source APIs attached to
 * each merged API. When linked to a merged API, the source API's schema, data
 * sources, and resolvers will be combined with other linked source API data to
 * form a new, singular API. </p> <p>Source APIs can originate from your account or
 * from other accounts via Amazon Web Services Resource Access Manager. For more
 * information about sharing resources from other accounts, see <a
 * href="https://docs.aws.amazon.com/ram/latest/userguide/what-is.html">What is
 * Amazon Web Services Resource Access Manager?</a> in the <i>Amazon Web Services
 * Resource Access Manager</i> guide.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/appsync-2017-07-25/SourceApiAssociation">AWS
 * API Reference</a></p>
 */
class SourceApiAssociation {
 public:
  AWS_APPSYNC_API SourceApiAssociation() = default;
  AWS_APPSYNC_API SourceApiAssociation(Aws::Utils::Json::JsonView jsonValue);
  AWS_APPSYNC_API SourceApiAssociation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_APPSYNC_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID generated by the AppSync service for the source API association.</p>
   */
  inline const Aws::String& GetAssociationId() const { return m_associationId; }
  inline bool AssociationIdHasBeenSet() const { return m_associationIdHasBeenSet; }
  template <typename AssociationIdT = Aws::String>
  void SetAssociationId(AssociationIdT&& value) {
    m_associationIdHasBeenSet = true;
    m_associationId = std::forward<AssociationIdT>(value);
  }
  template <typename AssociationIdT = Aws::String>
  SourceApiAssociation& WithAssociationId(AssociationIdT&& value) {
    SetAssociationId(std::forward<AssociationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the source API association.</p>
   */
  inline const Aws::String& GetAssociationArn() const { return m_associationArn; }
  inline bool AssociationArnHasBeenSet() const { return m_associationArnHasBeenSet; }
  template <typename AssociationArnT = Aws::String>
  void SetAssociationArn(AssociationArnT&& value) {
    m_associationArnHasBeenSet = true;
    m_associationArn = std::forward<AssociationArnT>(value);
  }
  template <typename AssociationArnT = Aws::String>
  SourceApiAssociation& WithAssociationArn(AssociationArnT&& value) {
    SetAssociationArn(std::forward<AssociationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the AppSync source API.</p>
   */
  inline const Aws::String& GetSourceApiId() const { return m_sourceApiId; }
  inline bool SourceApiIdHasBeenSet() const { return m_sourceApiIdHasBeenSet; }
  template <typename SourceApiIdT = Aws::String>
  void SetSourceApiId(SourceApiIdT&& value) {
    m_sourceApiIdHasBeenSet = true;
    m_sourceApiId = std::forward<SourceApiIdT>(value);
  }
  template <typename SourceApiIdT = Aws::String>
  SourceApiAssociation& WithSourceApiId(SourceApiIdT&& value) {
    SetSourceApiId(std::forward<SourceApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the AppSync source API.</p>
   */
  inline const Aws::String& GetSourceApiArn() const { return m_sourceApiArn; }
  inline bool SourceApiArnHasBeenSet() const { return m_sourceApiArnHasBeenSet; }
  template <typename SourceApiArnT = Aws::String>
  void SetSourceApiArn(SourceApiArnT&& value) {
    m_sourceApiArnHasBeenSet = true;
    m_sourceApiArn = std::forward<SourceApiArnT>(value);
  }
  template <typename SourceApiArnT = Aws::String>
  SourceApiAssociation& WithSourceApiArn(SourceApiArnT&& value) {
    SetSourceApiArn(std::forward<SourceApiArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the AppSync Merged API.</p>
   */
  inline const Aws::String& GetMergedApiArn() const { return m_mergedApiArn; }
  inline bool MergedApiArnHasBeenSet() const { return m_mergedApiArnHasBeenSet; }
  template <typename MergedApiArnT = Aws::String>
  void SetMergedApiArn(MergedApiArnT&& value) {
    m_mergedApiArnHasBeenSet = true;
    m_mergedApiArn = std::forward<MergedApiArnT>(value);
  }
  template <typename MergedApiArnT = Aws::String>
  SourceApiAssociation& WithMergedApiArn(MergedApiArnT&& value) {
    SetMergedApiArn(std::forward<MergedApiArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the AppSync Merged API.</p>
   */
  inline const Aws::String& GetMergedApiId() const { return m_mergedApiId; }
  inline bool MergedApiIdHasBeenSet() const { return m_mergedApiIdHasBeenSet; }
  template <typename MergedApiIdT = Aws::String>
  void SetMergedApiId(MergedApiIdT&& value) {
    m_mergedApiIdHasBeenSet = true;
    m_mergedApiId = std::forward<MergedApiIdT>(value);
  }
  template <typename MergedApiIdT = Aws::String>
  SourceApiAssociation& WithMergedApiId(MergedApiIdT&& value) {
    SetMergedApiId(std::forward<MergedApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description field.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  SourceApiAssociation& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>SourceApiAssociationConfig</code> object data.</p>
   */
  inline const SourceApiAssociationConfig& GetSourceApiAssociationConfig() const { return m_sourceApiAssociationConfig; }
  inline bool SourceApiAssociationConfigHasBeenSet() const { return m_sourceApiAssociationConfigHasBeenSet; }
  template <typename SourceApiAssociationConfigT = SourceApiAssociationConfig>
  void SetSourceApiAssociationConfig(SourceApiAssociationConfigT&& value) {
    m_sourceApiAssociationConfigHasBeenSet = true;
    m_sourceApiAssociationConfig = std::forward<SourceApiAssociationConfigT>(value);
  }
  template <typename SourceApiAssociationConfigT = SourceApiAssociationConfig>
  SourceApiAssociation& WithSourceApiAssociationConfig(SourceApiAssociationConfigT&& value) {
    SetSourceApiAssociationConfig(std::forward<SourceApiAssociationConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the source API association.</p>
   */
  inline SourceApiAssociationStatus GetSourceApiAssociationStatus() const { return m_sourceApiAssociationStatus; }
  inline bool SourceApiAssociationStatusHasBeenSet() const { return m_sourceApiAssociationStatusHasBeenSet; }
  inline void SetSourceApiAssociationStatus(SourceApiAssociationStatus value) {
    m_sourceApiAssociationStatusHasBeenSet = true;
    m_sourceApiAssociationStatus = value;
  }
  inline SourceApiAssociation& WithSourceApiAssociationStatus(SourceApiAssociationStatus value) {
    SetSourceApiAssociationStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The detailed message related to the current state of the source API
   * association.</p>
   */
  inline const Aws::String& GetSourceApiAssociationStatusDetail() const { return m_sourceApiAssociationStatusDetail; }
  inline bool SourceApiAssociationStatusDetailHasBeenSet() const { return m_sourceApiAssociationStatusDetailHasBeenSet; }
  template <typename SourceApiAssociationStatusDetailT = Aws::String>
  void SetSourceApiAssociationStatusDetail(SourceApiAssociationStatusDetailT&& value) {
    m_sourceApiAssociationStatusDetailHasBeenSet = true;
    m_sourceApiAssociationStatusDetail = std::forward<SourceApiAssociationStatusDetailT>(value);
  }
  template <typename SourceApiAssociationStatusDetailT = Aws::String>
  SourceApiAssociation& WithSourceApiAssociationStatusDetail(SourceApiAssociationStatusDetailT&& value) {
    SetSourceApiAssociationStatusDetail(std::forward<SourceApiAssociationStatusDetailT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The datetime value of the last successful merge of the source API
   * association. The result will be in UTC format and your local time zone.</p>
   */
  inline const Aws::Utils::DateTime& GetLastSuccessfulMergeDate() const { return m_lastSuccessfulMergeDate; }
  inline bool LastSuccessfulMergeDateHasBeenSet() const { return m_lastSuccessfulMergeDateHasBeenSet; }
  template <typename LastSuccessfulMergeDateT = Aws::Utils::DateTime>
  void SetLastSuccessfulMergeDate(LastSuccessfulMergeDateT&& value) {
    m_lastSuccessfulMergeDateHasBeenSet = true;
    m_lastSuccessfulMergeDate = std::forward<LastSuccessfulMergeDateT>(value);
  }
  template <typename LastSuccessfulMergeDateT = Aws::Utils::DateTime>
  SourceApiAssociation& WithLastSuccessfulMergeDate(LastSuccessfulMergeDateT&& value) {
    SetLastSuccessfulMergeDate(std::forward<LastSuccessfulMergeDateT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_associationId;

  Aws::String m_associationArn;

  Aws::String m_sourceApiId;

  Aws::String m_sourceApiArn;

  Aws::String m_mergedApiArn;

  Aws::String m_mergedApiId;

  Aws::String m_description;

  SourceApiAssociationConfig m_sourceApiAssociationConfig;

  SourceApiAssociationStatus m_sourceApiAssociationStatus{SourceApiAssociationStatus::NOT_SET};

  Aws::String m_sourceApiAssociationStatusDetail;

  Aws::Utils::DateTime m_lastSuccessfulMergeDate{};
  bool m_associationIdHasBeenSet = false;
  bool m_associationArnHasBeenSet = false;
  bool m_sourceApiIdHasBeenSet = false;
  bool m_sourceApiArnHasBeenSet = false;
  bool m_mergedApiArnHasBeenSet = false;
  bool m_mergedApiIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_sourceApiAssociationConfigHasBeenSet = false;
  bool m_sourceApiAssociationStatusHasBeenSet = false;
  bool m_sourceApiAssociationStatusDetailHasBeenSet = false;
  bool m_lastSuccessfulMergeDateHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppSync
}  // namespace Aws
