﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/fsx/FSxRequest.h>
#include <aws/fsx/FSx_EXPORTS.h>
#include <aws/fsx/model/RestoreOpenZFSVolumeOption.h>

#include <utility>

namespace Aws {
namespace FSx {
namespace Model {

/**
 */
class RestoreVolumeFromSnapshotRequest : public FSxRequest {
 public:
  AWS_FSX_API RestoreVolumeFromSnapshotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RestoreVolumeFromSnapshot"; }

  AWS_FSX_API Aws::String SerializePayload() const override;

  AWS_FSX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  RestoreVolumeFromSnapshotRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the volume that you are restoring.</p>
   */
  inline const Aws::String& GetVolumeId() const { return m_volumeId; }
  inline bool VolumeIdHasBeenSet() const { return m_volumeIdHasBeenSet; }
  template <typename VolumeIdT = Aws::String>
  void SetVolumeId(VolumeIdT&& value) {
    m_volumeIdHasBeenSet = true;
    m_volumeId = std::forward<VolumeIdT>(value);
  }
  template <typename VolumeIdT = Aws::String>
  RestoreVolumeFromSnapshotRequest& WithVolumeId(VolumeIdT&& value) {
    SetVolumeId(std::forward<VolumeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the source snapshot. Specifies the snapshot that you are restoring
   * from.</p>
   */
  inline const Aws::String& GetSnapshotId() const { return m_snapshotId; }
  inline bool SnapshotIdHasBeenSet() const { return m_snapshotIdHasBeenSet; }
  template <typename SnapshotIdT = Aws::String>
  void SetSnapshotId(SnapshotIdT&& value) {
    m_snapshotIdHasBeenSet = true;
    m_snapshotId = std::forward<SnapshotIdT>(value);
  }
  template <typename SnapshotIdT = Aws::String>
  RestoreVolumeFromSnapshotRequest& WithSnapshotId(SnapshotIdT&& value) {
    SetSnapshotId(std::forward<SnapshotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The settings used when restoring the specified volume from snapshot.</p> <ul>
   * <li> <p> <code>DELETE_INTERMEDIATE_SNAPSHOTS</code> - Deletes snapshots between
   * the current state and the specified snapshot. If there are intermediate
   * snapshots and this option isn't used, <code>RestoreVolumeFromSnapshot</code>
   * fails.</p> </li> <li> <p> <code>DELETE_CLONED_VOLUMES</code> - Deletes any
   * dependent clone volumes created from intermediate snapshots. If there are any
   * dependent clone volumes and this option isn't used,
   * <code>RestoreVolumeFromSnapshot</code> fails.</p> </li> </ul>
   */
  inline const Aws::Vector<RestoreOpenZFSVolumeOption>& GetOptions() const { return m_options; }
  inline bool OptionsHasBeenSet() const { return m_optionsHasBeenSet; }
  template <typename OptionsT = Aws::Vector<RestoreOpenZFSVolumeOption>>
  void SetOptions(OptionsT&& value) {
    m_optionsHasBeenSet = true;
    m_options = std::forward<OptionsT>(value);
  }
  template <typename OptionsT = Aws::Vector<RestoreOpenZFSVolumeOption>>
  RestoreVolumeFromSnapshotRequest& WithOptions(OptionsT&& value) {
    SetOptions(std::forward<OptionsT>(value));
    return *this;
  }
  inline RestoreVolumeFromSnapshotRequest& AddOptions(RestoreOpenZFSVolumeOption value) {
    m_optionsHasBeenSet = true;
    m_options.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  Aws::String m_volumeId;
  bool m_volumeIdHasBeenSet = false;

  Aws::String m_snapshotId;
  bool m_snapshotIdHasBeenSet = false;

  Aws::Vector<RestoreOpenZFSVolumeOption> m_options;
  bool m_optionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FSx
}  // namespace Aws
