﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ram/RAMRequest.h>
#include <aws/ram/RAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace RAM {
namespace Model {

/**
 */
class AssociateResourceShareRequest : public RAMRequest {
 public:
  AWS_RAM_API AssociateResourceShareRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateResourceShare"; }

  AWS_RAM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specifies the <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of the resource share that you want to add principals or
   * resources to.</p>
   */
  inline const Aws::String& GetResourceShareArn() const { return m_resourceShareArn; }
  inline bool ResourceShareArnHasBeenSet() const { return m_resourceShareArnHasBeenSet; }
  template <typename ResourceShareArnT = Aws::String>
  void SetResourceShareArn(ResourceShareArnT&& value) {
    m_resourceShareArnHasBeenSet = true;
    m_resourceShareArn = std::forward<ResourceShareArnT>(value);
  }
  template <typename ResourceShareArnT = Aws::String>
  AssociateResourceShareRequest& WithResourceShareArn(ResourceShareArnT&& value) {
    SetResourceShareArn(std::forward<ResourceShareArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> of the resources that you want to share. This can be
   * <code>null</code> if you want to add only principals.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceArns() const { return m_resourceArns; }
  inline bool ResourceArnsHasBeenSet() const { return m_resourceArnsHasBeenSet; }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  void SetResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns = std::forward<ResourceArnsT>(value);
  }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  AssociateResourceShareRequest& WithResourceArns(ResourceArnsT&& value) {
    SetResourceArns(std::forward<ResourceArnsT>(value));
    return *this;
  }
  template <typename ResourceArnsT = Aws::String>
  AssociateResourceShareRequest& AddResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns.emplace_back(std::forward<ResourceArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of principals to whom you want to the resource share. This
   * can be <code>null</code> if you want to add only resources.</p> <p>What the
   * principals can do with the resources in the share is determined by the RAM
   * permissions that you associate with the resource share. See
   * <a>AssociateResourceSharePermission</a>.</p> <p>You can include the following
   * values:</p> <ul> <li> <p>An Amazon Web Services account ID, for example:
   * <code>123456789012</code> </p> </li> <li> <p>An <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of an organization in Organizations, for example:
   * <code>organizations::123456789012:organization/o-exampleorgid</code> </p> </li>
   * <li> <p>An ARN of an organizational unit (OU) in Organizations, for example:
   * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
   * </p> </li> <li> <p>An ARN of an IAM role, for example:
   * <code>iam::123456789012:role/rolename</code> </p> </li> <li> <p>An ARN of an IAM
   * user, for example: <code>iam::123456789012user/username</code> </p> </li> </ul>
   *  <p>Not all resource types can be shared with IAM roles and users. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types">Sharing
   * with IAM roles and users</a> in the <i>Resource Access Manager User
   * Guide</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPrincipals() const { return m_principals; }
  inline bool PrincipalsHasBeenSet() const { return m_principalsHasBeenSet; }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  void SetPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals = std::forward<PrincipalsT>(value);
  }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  AssociateResourceShareRequest& WithPrincipals(PrincipalsT&& value) {
    SetPrincipals(std::forward<PrincipalsT>(value));
    return *this;
  }
  template <typename PrincipalsT = Aws::String>
  AssociateResourceShareRequest& AddPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals.emplace_back(std::forward<PrincipalsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. This lets you safely retry the request without
   * accidentally performing the same operation a second time. Passing the same value
   * to a later call to an operation requires that you also pass the same value for
   * all other parameters. We recommend that you use a <a
   * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID type of
   * value.</a>.</p> <p>If you don't provide this value, then Amazon Web Services
   * generates a random one for you.</p> <p>If you retry the operation with the same
   * <code>ClientToken</code>, but with different parameters, the retry fails with an
   * <code>IdempotentParameterMismatch</code> error.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  AssociateResourceShareRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies from which source accounts the service principal has access to the
   * resources in this resource share.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSources() const { return m_sources; }
  inline bool SourcesHasBeenSet() const { return m_sourcesHasBeenSet; }
  template <typename SourcesT = Aws::Vector<Aws::String>>
  void SetSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources = std::forward<SourcesT>(value);
  }
  template <typename SourcesT = Aws::Vector<Aws::String>>
  AssociateResourceShareRequest& WithSources(SourcesT&& value) {
    SetSources(std::forward<SourcesT>(value));
    return *this;
  }
  template <typename SourcesT = Aws::String>
  AssociateResourceShareRequest& AddSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources.emplace_back(std::forward<SourcesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceShareArn;

  Aws::Vector<Aws::String> m_resourceArns;

  Aws::Vector<Aws::String> m_principals;

  Aws::String m_clientToken;

  Aws::Vector<Aws::String> m_sources;
  bool m_resourceShareArnHasBeenSet = false;
  bool m_resourceArnsHasBeenSet = false;
  bool m_principalsHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
  bool m_sourcesHasBeenSet = false;
};

}  // namespace Model
}  // namespace RAM
}  // namespace Aws
