﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoTSiteWise {
namespace Model {

/**
 */
class ListAssetModelCompositeModelsRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API ListAssetModelCompositeModelsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAssetModelCompositeModels"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  AWS_IOTSITEWISE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the asset model. This can be either the actual ID in UUID format,
   * or else <code>externalId:</code> followed by the external ID, if it has one. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetModelId() const { return m_assetModelId; }
  inline bool AssetModelIdHasBeenSet() const { return m_assetModelIdHasBeenSet; }
  template <typename AssetModelIdT = Aws::String>
  void SetAssetModelId(AssetModelIdT&& value) {
    m_assetModelIdHasBeenSet = true;
    m_assetModelId = std::forward<AssetModelIdT>(value);
  }
  template <typename AssetModelIdT = Aws::String>
  ListAssetModelCompositeModelsRequest& WithAssetModelId(AssetModelIdT&& value) {
    SetAssetModelId(std::forward<AssetModelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to be used for the next set of paginated results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAssetModelCompositeModelsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return for each paginated request.</p>
   * <p>Default: 50</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAssetModelCompositeModelsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version alias that specifies the latest or active version of the asset
   * model. The details are returned in the response. The default value is
   * <code>LATEST</code>. See <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/model-active-version.html">
   * Asset model versions</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetModelVersion() const { return m_assetModelVersion; }
  inline bool AssetModelVersionHasBeenSet() const { return m_assetModelVersionHasBeenSet; }
  template <typename AssetModelVersionT = Aws::String>
  void SetAssetModelVersion(AssetModelVersionT&& value) {
    m_assetModelVersionHasBeenSet = true;
    m_assetModelVersion = std::forward<AssetModelVersionT>(value);
  }
  template <typename AssetModelVersionT = Aws::String>
  ListAssetModelCompositeModelsRequest& WithAssetModelVersion(AssetModelVersionT&& value) {
    SetAssetModelVersion(std::forward<AssetModelVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assetModelId;

  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::String m_assetModelVersion;
  bool m_assetModelIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_assetModelVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
