/* valaintegerliteral.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaintegerliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaintegertype.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valastruct.h>




struct _ValaIntegerLiteralPrivate {
	char* _value;
};

#define VALA_INTEGER_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteralPrivate))
enum  {
	VALA_INTEGER_LITERAL_DUMMY_PROPERTY
};
static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_integer_literal_real_to_string (ValaCodeNode* base);
static gboolean vala_integer_literal_real_is_pure (ValaExpression* base);
static gboolean vala_integer_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_integer_literal_parent_class = NULL;
static void vala_integer_literal_finalize (ValaCodeNode* obj);



/**
 * Creates a new integer literal.
 *
 * @param i      literal value
 * @param source reference to source code
 * @return       newly created integer literal
 */
ValaIntegerLiteral* vala_integer_literal_construct (GType object_type, const char* i, ValaSourceReference* source) {
	ValaIntegerLiteral* self;
	g_return_val_if_fail (i != NULL, NULL);
	self = (ValaIntegerLiteral*) g_type_create_instance (object_type);
	vala_integer_literal_set_value (self, i);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaIntegerLiteral* vala_integer_literal_new (const char* i, ValaSourceReference* source) {
	return vala_integer_literal_construct (VALA_TYPE_INTEGER_LITERAL, i, source);
}


static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaIntegerLiteral * self;
	self = (ValaIntegerLiteral*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_integer_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static char* vala_integer_literal_real_to_string (ValaCodeNode* base) {
	ValaIntegerLiteral * self;
	const char* _tmp0;
	self = (ValaIntegerLiteral*) base;
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
}


/**
 * Returns the type name of the value this literal represents.
 *
 * @return the name of literal type
 */
char* vala_integer_literal_get_type_name (ValaIntegerLiteral* self) {
	const char* _tmp0;
	char* number;
	gint l;
	gboolean u;
	gboolean _tmp3;
	gint64 n;
	gboolean _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	number = (_tmp0 = self->priv->_value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	l = 0;
	while (TRUE) {
		gboolean _tmp1;
		char* _tmp2;
		_tmp1 = FALSE;
		if (g_str_has_suffix (number, "l")) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = g_str_has_suffix (number, "L");
		}
		if (!_tmp1) {
			break;
		}
		l++;
		_tmp2 = NULL;
		number = (_tmp2 = g_strndup (number, (gsize) (strlen (number) - 1)), number = (g_free (number), NULL), _tmp2);
	}
	u = FALSE;
	_tmp3 = FALSE;
	if (g_str_has_suffix (number, "u")) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = g_str_has_suffix (number, "U");
	}
	if (_tmp3) {
		char* _tmp4;
		u = TRUE;
		_tmp4 = NULL;
		number = (_tmp4 = g_strndup (number, (gsize) (strlen (number) - 1)), number = (g_free (number), NULL), _tmp4);
	}
	n = g_ascii_strtoll (number, NULL, 0);
	_tmp5 = FALSE;
	if (!u) {
		_tmp5 = n > 0x7fffffff;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		/* value doesn't fit into signed 32-bit*/
		l = 2;
	} else {
		gboolean _tmp6;
		_tmp6 = FALSE;
		if (u) {
			_tmp6 = n > 0xffffffff;
		} else {
			_tmp6 = FALSE;
		}
		if (_tmp6) {
			/* value doesn't fit into unsigned 32-bit*/
			l = 2;
		}
	}
	if (l == 0) {
		if (u) {
			char* _tmp7;
			_tmp7 = NULL;
			return (_tmp7 = g_strdup ("uint"), number = (g_free (number), NULL), _tmp7);
		} else {
			char* _tmp8;
			_tmp8 = NULL;
			return (_tmp8 = g_strdup ("int"), number = (g_free (number), NULL), _tmp8);
		}
	} else {
		if (l == 1) {
			if (u) {
				char* _tmp9;
				_tmp9 = NULL;
				return (_tmp9 = g_strdup ("ulong"), number = (g_free (number), NULL), _tmp9);
			} else {
				char* _tmp10;
				_tmp10 = NULL;
				return (_tmp10 = g_strdup ("long"), number = (g_free (number), NULL), _tmp10);
			}
		} else {
			if (u) {
				char* _tmp11;
				_tmp11 = NULL;
				return (_tmp11 = g_strdup ("uint64"), number = (g_free (number), NULL), _tmp11);
			} else {
				char* _tmp12;
				_tmp12 = NULL;
				return (_tmp12 = g_strdup ("int64"), number = (g_free (number), NULL), _tmp12);
			}
		}
	}
	number = (g_free (number), NULL);
}


static gboolean vala_integer_literal_real_is_pure (ValaExpression* base) {
	ValaIntegerLiteral * self;
	self = (ValaIntegerLiteral*) base;
	return TRUE;
}


static gboolean vala_integer_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaIntegerLiteral * self;
	ValaIntegerType* _tmp4;
	char* _tmp3;
	ValaStruct* _tmp2;
	char* _tmp1;
	self = (ValaIntegerLiteral*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp4 = vala_integer_type_new (_tmp2 = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (analyzer->root_symbol), _tmp1 = vala_integer_literal_get_type_name (self))), self->priv->_value, _tmp3 = vala_integer_literal_get_type_name (self))));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
	_tmp3 = (g_free (_tmp3), NULL);
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
	_tmp1 = (g_free (_tmp1), NULL);
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


const char* vala_integer_literal_get_value (ValaIntegerLiteral* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_integer_literal_set_value (ValaIntegerLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_value = (g_free (self->priv->_value), NULL), _tmp2);
}


static void vala_integer_literal_class_init (ValaIntegerLiteralClass * klass) {
	vala_integer_literal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_integer_literal_finalize;
	g_type_class_add_private (klass, sizeof (ValaIntegerLiteralPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_integer_literal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_integer_literal_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_integer_literal_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_integer_literal_real_check;
}


static void vala_integer_literal_instance_init (ValaIntegerLiteral * self) {
	self->priv = VALA_INTEGER_LITERAL_GET_PRIVATE (self);
}


static void vala_integer_literal_finalize (ValaCodeNode* obj) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	VALA_CODE_NODE_CLASS (vala_integer_literal_parent_class)->finalize (obj);
}


GType vala_integer_literal_get_type (void) {
	static GType vala_integer_literal_type_id = 0;
	if (vala_integer_literal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaIntegerLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_integer_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaIntegerLiteral), 0, (GInstanceInitFunc) vala_integer_literal_instance_init, NULL };
		vala_integer_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaIntegerLiteral", &g_define_type_info, 0);
	}
	return vala_integer_literal_type_id;
}




