/* valascope.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valascope.h>
#include <gee/map.h>
#include <gee/hashmap.h>
#include <vala/valasymbol.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>




struct _ValaScopePrivate {
	ValaSymbol* _owner;
	ValaScope* _parent_scope;
	GeeMap* symbol_table;
};
#define VALA_SCOPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SCOPE, ValaScopePrivate))
enum  {
	VALA_SCOPE_DUMMY_PROPERTY,
	VALA_SCOPE_OWNER,
	VALA_SCOPE_PARENT_SCOPE
};
static gpointer vala_scope_parent_class = NULL;
static void vala_scope_dispose (GObject * obj);



/**
 * Creates a new scope.
 *
 * @return newly created scope
 */
ValaScope* vala_scope_new (ValaSymbol* owner) {
	ValaScope * self;
	g_return_val_if_fail (owner == NULL || VALA_IS_SYMBOL (owner), NULL);
	self = g_object_newv (VALA_TYPE_SCOPE, 0, NULL);
	vala_scope_set_owner (self, owner);
	return self;
}


/**
 * Adds the specified symbol with the specified name to the symbol table
 * of this scope.
 *
 * @param name name for the specified symbol
 * @param sym  a symbol
 */
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	g_return_if_fail (VALA_IS_SYMBOL (sym));
	if (name != NULL) {
		if (self->priv->symbol_table == NULL) {
			GeeMap* _tmp0;
			_tmp0 = NULL;
			self->priv->symbol_table = (_tmp0 = GEE_MAP (gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->symbol_table == NULL ? NULL : (self->priv->symbol_table = (g_object_unref (self->priv->symbol_table), NULL))), _tmp0);
		} else {
			ValaSymbol* _tmp1;
			gboolean _tmp2;
			_tmp1 = NULL;
			if ((_tmp2 = (_tmp1 = vala_scope_lookup (self, name)) != NULL, (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), _tmp2)) {
				char* _tmp4;
				char* _tmp3;
				vala_code_node_set_error (VALA_CODE_NODE (self->priv->_owner), TRUE);
				_tmp4 = NULL;
				_tmp3 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (sym)), (_tmp4 = g_strdup_printf ("`%s' already contains a definition for `%s'", (_tmp3 = vala_symbol_get_full_name (self->priv->_owner)), name)));
				_tmp4 = (g_free (_tmp4), NULL);
				_tmp3 = (g_free (_tmp3), NULL);
				return;
			}
		}
		gee_map_set (((GeeMap*) self->priv->symbol_table), ((char*) name), sym);
	}
	vala_symbol_set_owner (sym, self);
}


void vala_scope_remove (ValaScope* self, const char* name) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	g_return_if_fail (name != NULL);
	gee_map_remove (self->priv->symbol_table, name);
}


/**
 * Returns the symbol stored in the symbol table with the specified
 * name.
 *
 * @param name name of the symbol to be returned
 * @return     found symbol or null
 */
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name) {
	ValaSymbol* sym;
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	if (self->priv->symbol_table == NULL) {
		return NULL;
	}
	sym = ((ValaSymbol*) gee_map_get (((GeeMap*) self->priv->symbol_table), name));
	if (sym != NULL && !vala_symbol_get_active (sym)) {
		ValaSymbol* _tmp1;
		_tmp1 = NULL;
		sym = (_tmp1 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp1);
	}
	return sym;
}


/**
 * Returns whether the specified scope is an ancestor of this scope.
 *
 * @param scope a scope or null for the root scope
 * @return      true if this scope is a subscope of the specified
 *              scope, false otherwise
 */
gboolean vala_scope_is_subscope_of (ValaScope* self, ValaScope* scope) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), FALSE);
	g_return_val_if_fail (scope == NULL || VALA_IS_SCOPE (scope), FALSE);
	if (scope == self) {
		return TRUE;
	}
	/* null scope is the root scope*/
	if (scope == NULL) {
		return TRUE;
	}
	if (self->priv->_parent_scope != NULL) {
		return vala_scope_is_subscope_of (self->priv->_parent_scope, scope);
	}
	return FALSE;
}


ValaSymbol* vala_scope_get_owner (ValaScope* self) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	return self->priv->_owner;
}


void vala_scope_set_owner (ValaScope* self, ValaSymbol* value) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	self->priv->_owner = value;
}


ValaScope* vala_scope_get_parent_scope (ValaScope* self) {
	g_return_val_if_fail (VALA_IS_SCOPE (self), NULL);
	return self->priv->_parent_scope;
}


void vala_scope_set_parent_scope (ValaScope* self, ValaScope* value) {
	g_return_if_fail (VALA_IS_SCOPE (self));
	self->priv->_parent_scope = value;
}


static void vala_scope_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaScope * self;
	self = VALA_SCOPE (object);
	switch (property_id) {
		case VALA_SCOPE_OWNER:
		g_value_set_object (value, vala_scope_get_owner (self));
		break;
		case VALA_SCOPE_PARENT_SCOPE:
		g_value_set_object (value, vala_scope_get_parent_scope (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_scope_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaScope * self;
	self = VALA_SCOPE (object);
	switch (property_id) {
		case VALA_SCOPE_OWNER:
		vala_scope_set_owner (self, g_value_get_object (value));
		break;
		case VALA_SCOPE_PARENT_SCOPE:
		vala_scope_set_parent_scope (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_scope_class_init (ValaScopeClass * klass) {
	vala_scope_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaScopePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_scope_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_scope_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_scope_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SCOPE_OWNER, g_param_spec_object ("owner", "owner", "owner", VALA_TYPE_SYMBOL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SCOPE_PARENT_SCOPE, g_param_spec_object ("parent-scope", "parent-scope", "parent-scope", VALA_TYPE_SCOPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_scope_init (ValaScope * self) {
	self->priv = VALA_SCOPE_GET_PRIVATE (self);
}


static void vala_scope_dispose (GObject * obj) {
	ValaScope * self;
	self = VALA_SCOPE (obj);
	(self->priv->symbol_table == NULL ? NULL : (self->priv->symbol_table = (g_object_unref (self->priv->symbol_table), NULL)));
	G_OBJECT_CLASS (vala_scope_parent_class)->dispose (obj);
}


GType vala_scope_get_type (void) {
	static GType vala_scope_type_id = 0;
	if (G_UNLIKELY (vala_scope_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaScopeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_scope_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaScope), 0, (GInstanceInitFunc) vala_scope_init };
		vala_scope_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaScope", &g_define_type_info, 0);
	}
	return vala_scope_type_id;
}




