///
// Copyright (C) 2003, 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "postscriptviewent.h"
#include "config.h"
#include "document/basicframe.h" // For Gen_Pic_Error; Todo: remove!
#include "util/tempfile.h"
#include "util/processman.h"
#include "util/os.h"
#include "util/filesys.h"
#include "util/warning.h"
#include "util/stringutil.h" // tostr
#include <sstream>

/// \todo Remote the following line!
typedef BasicFrame::GenPicError GenPicError;

PostscriptViewent::PostscriptViewent(View& view, const FrameRef psframe,
				     int _scale_factor)
  : Viewent(view, psframe), frame(psframe), 
    pid(-1), tmp_file(Tempfile::find_new_name()),
    scale_factor(_scale_factor)
{
  ProcessManager::instance().process_stopped.connect
    (slot(*this, &PostscriptViewent::process_stopped));
  /// \todo  Only regenerate when necessary, i.e. not when just moving ...
  frame->object_changed_signal.connect
    (SigC::hide<Pagent*>(slot(*this, &PostscriptViewent::regenerate)));
  view.connect_zoom_change
    (SigC::hide<float>(slot(*this, &PostscriptViewent::regenerate)));

  canvas_pixbuf.reset(new Gnome::Canvas::Pixbuf(*content_group));
  canvas_pixbuf->property_anchor() = Gtk::ANCHOR_SOUTH_WEST;
  canvas_pixbuf->property_x() = 0; canvas_pixbuf->property_y() = 0;
  on_geometry_changed(psframe);
  
  try {
    set_state(WAIT);
    frame->generate_picture(SigC::slot(*this, &PostscriptViewent::set_image));
  } catch(const GenPicError &e) {
    set_state(e.type == BasicFrame::ASSOCIATION ? MISSING : BROKEN);
    warning << frame->get_name() << ": " << e.what() << std::endl;
  }
}

PostscriptViewent::~PostscriptViewent() {
  try { unlink(tmp_file); }
  catch(std::exception& err) {
    warning << "Failed to delete postscript view render file." << err.what()
	    << std::endl;
  }
  if(pid != -1) {
    ProcessManager::instance().stop(pid);
  }
}

void PostscriptViewent::set_state(State state) {
  Viewent::set_state(state);
  // make sure old images don't linger
  if(state == BROKEN || state == MISSING) {
    canvas_pixbuf->property_pixbuf() = Glib::RefPtr<Gdk::Pixbuf>();
  }
}

void PostscriptViewent::on_geometry_changed(Pagent *pagent) {
  Viewent::on_geometry_changed(pagent);

  Vector size = frame->get_inherent_size();

  canvas_pixbuf->property_width() = size.x;
  canvas_pixbuf->property_height() = size.y;
  canvas_pixbuf->property_width_set() = true;
  canvas_pixbuf->property_height_set() = true;
  canvas_pixbuf->property_width_in_pixels() = false;
  canvas_pixbuf->property_height_in_pixels() = false;
}

void PostscriptViewent::regenerate() {
  pixbuf.clear();
  if(pid != -1) {
    verbose << "PostscriptViewent stopping gs " << pid << std::endl;
    ProcessManager::instance().stop(pid);
    pid = -1;
  }
  try {
    set_state(WAIT);
    frame->generate_picture(SigC::slot(*this, &PostscriptViewent::set_image));
  } catch(const GenPicError &e) {
    set_state(e.type == BasicFrame::ASSOCIATION ? MISSING : BROKEN);
    warning << frame->get_name() << ": " << e.what() << std::endl;
  }
}

void PostscriptViewent::set_image(const std::string& psfile) {
  try {
    std::string &psinterpreter = config.PSInterpreter.values.front();
    if(psinterpreter.empty())
      throw GenPicError(BasicFrame::GENERATION,
                        "No PostScript interpreter specified");
  
    const Matrix &m = frame->get_matrix();
    const Vector framesize = frame->get_inherent_size();
    
    // Size of frame as pixels in this view.
    const Gdk::Point pv(int(view.pt2scr(framesize.x * m.sc_x())),
                        int(view.pt2scr(framesize.y * m.sc_y())));
    if(pv.get_x() == 0 || pv.get_y() == 0) 
      throw GenPicError(BasicFrame::ZEROSIZE, "Picture has zero size");

    std::ostringstream tmp;
    /// \todo -sFONTPATH parameter ?
    tmp << psinterpreter //assuming it is ghostscript
        << " -q -dSAFER -dNOPAUSE -dBATCH -sDEVICE=ppmraw -r" 
        << view.get_scrres() * scale_factor
        << " -g" << scale_factor * pv.get_x() << 'x' 
        << scale_factor * pv.get_y()
        << " -sOutputFile=- " << psfile
        << " > " << tmp_file;
    pid = ProcessManager::instance().start(tmp.str());
  
    if(pid == -1)
      throw GenPicError(BasicFrame::GENERATION,
                        "Failed to run " + psinterpreter);

    verbose << pid << ": " << tmp.str() << std::endl;
    set_state(WAIT);
  } catch(const std::exception& e) {
    set_state(BROKEN);
    throw;
  }
}

void PostscriptViewent::process_stopped(pid_t _pid, bool exited_normally, 
					int exit_code)
{
  if(_pid != pid)
    return;

  try {
    debug << _pid << ", exit code: " << exit_code << std::endl;

    if(!exited_normally)
      throw std::runtime_error("Process " + tostr(_pid)
                               + " exited abnormally");

    pixbuf.clear();
    if(!exists(tmp_file)) 
      throw std::runtime_error("\"" + tmp_file + "\" does not exist");

    pixbuf = Gdk::Pixbuf::create_from_file(tmp_file);

    // Make all white pixels transparent.
    if(config.FakeTransparency.values.front()
       && pixbuf && !pixbuf->get_has_alpha())
      pixbuf = pixbuf->add_alpha(true, 255, 255, 255); 
    // had some trouble with some images becoming totally transparent

    // scale
    if(scale_factor != 1)
      pixbuf = pixbuf->scale_simple(pixbuf->get_width() / scale_factor,
                                    pixbuf->get_height() / scale_factor,
                                    Gdk::INTERP_BILINEAR);

    canvas_pixbuf->property_pixbuf() = pixbuf;
    on_geometry_changed(frame);
    set_state(NORMAL);
  
    pid = -1;
  } catch(const std::exception& e) {
    set_state(BROKEN);
    warning << frame->get_name() << ": " << e.what() << std::endl; 
  } catch(const Glib::Error& e) {
    set_state(BROKEN);
    warning << frame->get_name() << ": " << e.what() << std::endl; 
  }
}
