// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - main-window.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <iostream>
#include <vector>
#include <glib/gi18n.h>
#include <glibmm/markup.h>
#include <gtkmm-utils/tile.h>
#include "browser.hh"
#include "file-utils.hh"
#include "document-tag-cloud-model.hh"
#include "document-tile.hh"
#include "main-window.hh"
#include "paths.hh"

namespace paperbox {

    using std::vector;
    using boost::shared_ptr;

    MainWindow::MainWindow(GtkWindow* cobject,
                           const Glib::RefPtr<Gnome::Glade::Xml>& glade)
        :
        Gtk::Window(cobject),
        glade_(glade),
        hpane_(0),
        left_top_vbox_(0),
        tile_view_(0),
        right_top_vbox_(0),
        tag_box_(false, 4),
        button_view_all_(Gtk::Stock::INDEX)
    {
        init_gui();
        set_default_size(800, 670);
        set_pane_pos();

        browser_ = Browser::instance();
        tiles_.reset(new TileSet());

        connect_signals();
        
        //browser_->dump_document_data();
    }

    MainWindow::~MainWindow()
    {
    }

    MainWindow*
    MainWindow::create()
    {
        Glib::RefPtr<Gnome::Glade::Xml> glade_xml =
            Gnome::Glade::Xml::create(glade_window_main);

        MainWindow* p = 0;
        glade_xml->get_widget_derived("MainWindow", p);
        return p;
    }

    void
    MainWindow::init_gui()
    {
        get_widgets_from_ui_file();
        set_up_tiles();

        model_.reset(new DocumentTagCloudModel(10, 18));
        tag_cloud_.set_model(model_);
        right_top_vbox_->pack_start(tag_box_);

        tag_box_.pack_start(button_view_all_, false, false);
        tag_box_.pack_start(tag_cloud_, true, true);

        show_all_children();

        button_view_all_.set_sensitive(false);
    }

    void
    MainWindow::get_widgets_from_ui_file()
    {
        // make sure the glade object has been initialized properly in the ctor
        g_assert(glade_);

        glade_->get_widget("hpane", hpane_);
        g_assert(hpane_);

        glade_->get_widget("left_top_vbox", left_top_vbox_);
        g_assert(left_top_vbox_);

        glade_->get_widget("right_top_vbox", right_top_vbox_);
        g_assert(right_top_vbox_);
    }

    void
    MainWindow::set_up_tiles()
    {
        tile_view_ = Gtk::manage(new DocumentTileView());
        tile_view_->set_navigator_title(_("Showing documents"));
        tile_view_->set_tiles_per_page(7);

        left_top_vbox_->pack_start(*tile_view_);
    }

    void
    MainWindow::set_pane_pos()
    {
        int width, height;
        get_size(width, height);
        hpane_->set_position(3 * (width / 4));
    }

    void
    MainWindow::connect_signals()
    {
        browser_->signal_new_document().connect(
            sigc::mem_fun(*this, &MainWindow::on_new_document));

        tile_view_->signal_tile_activated().connect(
            sigc::mem_fun(*this, &MainWindow::on_document_tile_selected));

        button_view_all_.signal_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_view_all));

        tile_view_->signal_tag_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_tag_clicked));

        tag_cloud_.signal_tag_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_tag_clicked));
    }

    // Invoked from Browser during idle time.
    // All this should be re-thought when we get xesam dbus api in tracker.
    void
    MainWindow::on_new_document(const boost::shared_ptr<Document>& doc)
    {
        shared_ptr<DocumentTile> tile(new DocumentTile(thumbnailer_, doc));

        bool ok = tiles_->add(tile);
        if (! ok) return; // maybe not?

        tile_view_->add_tile(*tile);

        std::vector<Glib::ustring> tags = doc->get_tags();
        std::vector<Glib::ustring>::iterator it(tags.begin());
        std::vector<Glib::ustring>::iterator end(tags.end());

        for ( ; it != end; ++it)
            model_->add_tag(*it);

        show_all_children();
    }

    void
    MainWindow::on_document_tile_selected(/*Document*/Gtk::Util::Tile& t)
    {
        DocumentTile* dt = dynamic_cast<DocumentTile*>(&t);
        open_file_with_xdg(dt->get_document_uri());
    }

    void
    MainWindow::render_new_tile_set(const vector<shared_ptr<Document> >& docs)
    {
        tile_view_->clear();

        vector<shared_ptr<Document> >::const_iterator it(docs.begin());
        vector<shared_ptr<Document> >::const_iterator end(docs.end());

        for ( ; it != end; ++it) {
            std::string uri = (*it)->get_uri().raw();
            shared_ptr<DocumentTile> tile = tiles_->get_tile(uri);
            if (tile) tile_view_->add_tile(*tile);
        }
    }

    void
    MainWindow::on_tag_clicked(const Glib::ustring& tag)
    {
        //g_debug("selected tag %s", tag.c_str());

        vector<shared_ptr<Document> > docs;
        browser_->get_documents_for_tag(tag, docs);
        //g_debug("that's %d documents", (int) docs.size());

        render_new_tile_set(docs);
        button_view_all_.set_sensitive(true);
    }

    void
    MainWindow::on_view_all()
    {
        vector<shared_ptr<Document> > docs;
        browser_->get_all_documents(docs);
        render_new_tile_set(docs);
        button_view_all_.set_sensitive(false);
    }

} // namespace paperbox
