/* gnome-db-matrix.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <libgda/libgda.h>
#include <libgnomedb/libgnomedb.h>
#include "gnome-db-matrix.h" 
#include "gnome-db-table.h"
#include "gnome-db-constraint.h"
#include "gnome-db-table-field.h"
#include "gnome-db-data-widget.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-condition.h"
#include "gnome-db-server.h"
#include "gnome-db-database.h"
#include "gnome-db-renderer.h"
#include "gnome-db-result-set.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-data-set.h"
#include "gnome-db-parameter.h"
#include "gnome-db-basic-form.h"
#include "utility.h"

static void gnome_db_matrix_class_init (GnomeDbMatrixClass * class);
static void gnome_db_matrix_init (GnomeDbMatrix * wid);
static void gnome_db_matrix_dispose (GObject   * object);

static void gnome_db_matrix_set_property (GObject              *object,
					 guint                 param_id,
					 const GValue         *value,
					 GParamSpec           *pspec);
static void gnome_db_matrix_get_property (GObject              *object,
					 guint                 param_id,
					 GValue               *value,
					 GParamSpec           *pspec);

static void     gnome_db_matrix_compute_queries (GnomeDbMatrix *matrix);
static void     gnome_db_matrix_initialize      (GnomeDbMatrix *matrix);
static void     gnome_db_matrix_clean_all       (GnomeDbMatrix *matrix);

static void model_update (GnomeDbMatrix *matrix, gboolean keep_user_modifs);
static void model_commit_changes (GnomeDbMatrix *matrix);
static void model_update_complete (GnomeDbMatrix *matrix, gboolean keep_old_modifications);


static GtkWidget *modif_buttons_make   (GnomeDbMatrix *matrix);
static void       modif_buttons_update (GnomeDbMatrix *matrix);

/* GnomeDbDataWidget interface */
static void            gnome_db_matrix_widget_init         (GnomeDbDataWidgetIface *iface);
static void            gnome_db_matrix_run                 (GnomeDbDataWidget *iface, guint mode);
static void            gnome_db_matrix_set_mode            (GnomeDbDataWidget *iface, guint mode);
static void            gnome_db_matrix_set_entry_editable  (GnomeDbDataWidget *iface, GnomeDbQfield *field, gboolean editable);
static void            gnome_db_matrix_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions);
static GnomeDbParameter    *gnome_db_matrix_get_param_for_field (GnomeDbDataWidget *iface, GnomeDbQfield *field, const gchar *field_name, 
							   gboolean in_exec_context);
static gboolean        gnome_db_matrix_has_been_changed    (GnomeDbDataWidget *iface);
static GnomeDbDataSet      *gnome_db_matrix_get_exec_context    (GnomeDbDataWidget *iface);
static GtkActionGroup *gnome_db_matrix_get_actions_group   (GnomeDbDataWidget *iface);

/* structure for each type of view */
typedef struct _ViewData {
	GnomeDbMatrix     *matrix;
	GtkWidget      *(*init_view)   (GnomeDbMatrix *matrix, struct _ViewData *vd);
	void            (*update_view) (GnomeDbMatrix *matrix, struct _ViewData *vd);
	void            (*clean_data)  (GnomeDbMatrix *matrix, struct _ViewData *vd);
	gpointer          private;
} ViewData;
#define NB_TYPES 3 /* same number as of GnomeDbMatrixType enum options */
#define VIEW_DATA(x) ((ViewData*)(x))


struct _GnomeDbMatrixPriv
{
	GnomeDbDict            *dict;
	gboolean           has_run;
	gboolean           cleaned;
	
	/* widgets and related */
	GtkWidget         *title;
	GtkWidget         *notebook;         /* main notebook to display the various kind of views */
	ViewData          *view_data[NB_TYPES];  /* ViewData structures for each view type*/
	gint               view_pages[NB_TYPES]; /* page number for each view type */
	gint               gtk_table_view_nb_cols; /* number of columns when using a view with a GtkTable layout */

	guint              mode;             /* ORed values of GnomeDbActionMode enum */
	guint              view_mode;        /* GnomeDbMatrixType, index of views when no error */
	gboolean           assoc_data_only;
	GtkTooltips       *tooltips;

	GtkUIManager      *uimanager;
	GtkActionGroup    *actions_group;
	GtkWidget         *modif_all;

	/* queries and related */
	GnomeDbQuery           *query_select_rows;
	GnomeDbQuery           *query_select_cols;
	GnomeDbTarget          *rows_target;
        GnomeDbTarget          *cols_target;

	GnomeDbTable         *modif_table;
	GSList            *modif_table_fields;
	GnomeDbQuery           *query_select_contents;

	GnomeDbQuery           *query_update;
        GnomeDbQuery           *query_delete;
        GnomeDbQuery           *query_insert;

	GnomeDbDataSet         *args_context;
	GnomeDbDataSet         *work_context;

	GSList            *rows_cond_params;
	GSList            *cols_cond_params;

	GHashTable        *replacements; /* replacements during the copy of rows_select_query and cols_select_query */

	/* data resultsets and model */
	GtkTreeModel      *model;
	GdaDataModel      *rs_rows;
	GdaDataModel      *rs_cols;
	GdaDataModel      *rs_contents;
	GSList            *assoc_chuncks; /* list of AssocChunck structures */

        GHashTable        *hash_rows; /* key=col no in rs_contents for FK value, value=col no in rs_rows for PK value */
        GHashTable        *hash_cols; /* key=col no in rs_contents for FK value, value=col no in rs_cols for PK value */
	GSList            *rows_pk_no;          /* list of the positions, in rs_rows of the PK fields */
	GSList            *cols_pk_no;          /* list of the positions, in rs_cols of the PK fields */
	GSList            *contents_rows_fk_no; /* list of the positions, in rs_contents of the FK values linked to rs_rows */
	GSList            *contents_cols_fk_no; /* list of the positions, in rs_contents of the FK values linked to rs_cols */
};


/* structure to hold information about a particular association */
typedef struct {
	gint       rows_row;        /* row number in priv->rs_rows, MUST always be valid */
	GdaValue  *pk_value;        /* a GdaValue of type GDA_VALUE_TYPE_LIST referencing the PK value identified at 'rows_row'*/

	GSList    *assoc_data_list; /* list of AssocData structures */
} AssocChunck;
#define ASSOC_CHUNCK(x) ((AssocChunck *)(x))

typedef struct {
	gint         cols_row;      /* row number in priv->rs_cols, MUST always be valid */
	GdaValue    *pk_value;      /* a GdaValue of type GDA_VALUE_TYPE_LIST referencing the PK value identified at 'cols_row'*/

	gboolean     req_ins_del;   /* TRUE = insert or delete to be performed, FALSE = only UPDATE to be done */
	gint         contents_row;  /* -1 if association does not exist and row number of priv->rs_contents otherwise */
	GdaValue    *data;          /* data for that association, as a GdaValueTypeList, (data is owned here) */
	GdaValue    *data_orig;     /* original data in the association */
	gboolean     data_valid;    /* TRUE if 'data' is valid, and FALSE otherwise */

	AssocChunck *ac;
} AssocData;
#define ASSOC_DATA(x) ((AssocData *)(x))

static AssocData *assoc_data_create           (GnomeDbMatrix *matrix, AssocChunck *ac, gint cols_row);
static void       assoc_data_free             (AssocData *ad);
static gboolean   assoc_data_is_enabled       (AssocData *ad);
static void       assoc_data_enable           (AssocData *ad);
static AssocData *assoc_data_disable          (AssocData *ad);
static gboolean   assoc_data_has_been_changed (AssocData *ad);
static gboolean   assoc_data_is_valid         (AssocData *ad);

static void set_params_from_assoc_data (GnomeDbMatrix *matrix, gint rows_row, gint cols_row, AssocData *ad);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


/* column enum for the GtkTreeModel */
enum {
        COLUMN_ROW_NUM,
        COLUMN_ASSOC_CHUNCK
};


/* properties */
enum
{
        PROP_0,
	PROP_TITLE_VISIBLE,
	PROP_TITLE_STRING,
	PROP_NB_COLUMNS,
	PROP_ASSOC_DATA_ONLY
};

GType
gnome_db_matrix_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbMatrixClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_matrix_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbMatrix),
			0,
			(GInstanceInitFunc) gnome_db_matrix_init
		};		
		
		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) gnome_db_matrix_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbMatrix", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_DATA_WIDGET_TYPE, &work_widget_info);
	}

	return type;
}

static void
gnome_db_matrix_widget_init (GnomeDbDataWidgetIface *iface)
{
	iface->set_mode = gnome_db_matrix_set_mode;
	iface->set_column_editable = gnome_db_matrix_set_entry_editable;
	iface->show_column_actions = NULL; /* method non implemented */
	iface->show_global_actions = gnome_db_matrix_show_global_actions;
	iface->get_actions_group = gnome_db_matrix_get_actions_group;
}


static void
gnome_db_matrix_class_init (GnomeDbMatrixClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = gnome_db_matrix_dispose;

	/* Properties */
        object_class->set_property = gnome_db_matrix_set_property;
        object_class->get_property = gnome_db_matrix_get_property;
	g_object_class_install_property (object_class, PROP_TITLE_VISIBLE,
                                         g_param_spec_boolean ("title_visible", NULL, 
							       "Show of hide the title of the widget", FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_STRING,
                                         g_param_spec_string ("title_string", NULL, 
							      "String to display in the widget's title", NULL,
							      G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_NB_COLUMNS,
                                         g_param_spec_int ("layout_nb_columns", NULL, 
							   "Number of columns when the chosen view "
							   "involves a table layout", 0, G_MAXINT, 4, 
							   G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_ASSOC_DATA_ONLY,
                                         g_param_spec_boolean ("assoc_data_only", NULL, 
							       "Disallow INSERT and DELETE on the modified table", 
							       FALSE,
                                                               G_PARAM_WRITABLE));
}

static void
gnome_db_matrix_init (GnomeDbMatrix * wid)
{
	wid->priv = g_new0 (GnomeDbMatrixPriv, 1);
	wid->priv->dict = NULL;
	wid->priv->has_run = FALSE;
	wid->priv->cleaned = FALSE;
	
	wid->priv->gtk_table_view_nb_cols = 4; /* default is 4 columns */

	wid->priv->mode = 0;
	wid->priv->tooltips = NULL;
	wid->priv->view_mode = GNOME_DB_MATRIX_TABULAR_SYNTHETIC;
	wid->priv->assoc_data_only = FALSE;

	wid->priv->assoc_chuncks = NULL;
	/* Model creation: 
	 * --> 1 column which refers to the considered row number from priv->rs_rows
	 * --> 1 column which holds a AssocChunck structure (which itself holds a list of AssocData structure(s))
	 *     all the data in each of these structures is allocated on demand, 
	 *     and managed in priv->assoc_chuncks. There is always an AssocChunck.
	 */
	wid->priv->model = GTK_TREE_MODEL (gtk_list_store_new (2, G_TYPE_INT, G_TYPE_POINTER));
	wid->priv->hash_rows = g_hash_table_new (g_direct_hash, g_direct_equal);
	wid->priv->hash_cols = g_hash_table_new (g_direct_hash, g_direct_equal);
}

static void nullified_main_object_cb (GnomeDbQuery *query, GnomeDbMatrix *matrix);


/**
 * gnome_db_matrix_new
 * @dict: a #GnomeDbDict object
 * @rows_select_query:
 * @rows_target:
 * @cols_select_query:
 * @cols_target:
 * @modif_table:
 * @modif_table_fields:
 *
 * Creates a new #GnomeDbMatrix widget.
 *
 * @rows_select_query and @cols_select_query must be SELECT queries, from their results depend what
 * will be displayed in each row and column of the associated matrix widget.
 *
 * The @rows_target and @cols_target targets are used as reference for primary key values.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_matrix_new (GnomeDbDict *dict,
		    GnomeDbQuery *rows_select_query, GnomeDbTarget *rows_target,
		    GnomeDbQuery *cols_select_query, GnomeDbTarget *cols_target,
		    GnomeDbTable *modif_table, GSList *modif_table_fields)
{
	GObject *obj;
	GnomeDbMatrix *matrix;
	GHashTable *repl;
	GSList *list;
	GnomeDbQuery *query;
	GnomeDbTarget *target;

	/* checks */
	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	g_return_val_if_fail (rows_select_query && IS_GNOME_DB_QUERY (rows_select_query), NULL);
	g_return_val_if_fail (gnome_db_query_get_query_type (rows_select_query) == GNOME_DB_QUERY_TYPE_SELECT, NULL);
	g_return_val_if_fail (gnome_db_base_get_dict (GNOME_DB_BASE (rows_select_query)) == ASSERT_DICT (dict), NULL);
	g_return_val_if_fail (rows_target && IS_GNOME_DB_TARGET (rows_target), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (rows_target) == rows_select_query, NULL);

	g_return_val_if_fail (cols_select_query && IS_GNOME_DB_QUERY (cols_select_query), NULL);
	g_return_val_if_fail (gnome_db_query_get_query_type (cols_select_query) == GNOME_DB_QUERY_TYPE_SELECT, NULL);
	g_return_val_if_fail (gnome_db_base_get_dict (GNOME_DB_BASE (cols_select_query)) == ASSERT_DICT (dict), NULL);
	g_return_val_if_fail (cols_target && IS_GNOME_DB_TARGET (cols_target), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (cols_target) == cols_select_query, NULL);
	
	g_return_val_if_fail (modif_table && IS_GNOME_DB_TABLE (modif_table), NULL);
	g_return_val_if_fail (gnome_db_base_get_dict (GNOME_DB_BASE (modif_table)) == ASSERT_DICT (dict), NULL);

	if (modif_table_fields) {
		GSList *list = modif_table_fields;
		
		while (list) {
			g_return_val_if_fail (list->data && IS_GNOME_DB_TABLE_FIELD (list->data), NULL);
			g_return_val_if_fail (gnome_db_field_get_entity (GNOME_DB_FIELD (list->data)) == GNOME_DB_ENTITY (modif_table),
					      NULL);
			list = g_slist_next (list);
		}
	}
	
	/* object creation */
	obj = g_object_new (GNOME_DB_MATRIX_TYPE, NULL);
	matrix = GNOME_DB_MATRIX (obj);

	/* initialization */
	matrix->priv->dict = ASSERT_DICT (dict);
	matrix->priv->modif_table = modif_table;
	gnome_db_base_connect_nullify (matrix->priv->modif_table,
				 G_CALLBACK (nullified_main_object_cb), matrix);

	repl = g_hash_table_new (NULL, NULL);
	matrix->priv->replacements = repl;
	matrix->priv->query_select_rows = GNOME_DB_QUERY (gnome_db_query_new_copy (rows_select_query, repl));
	gnome_db_base_connect_nullify (matrix->priv->query_select_rows,
				 G_CALLBACK (nullified_main_object_cb), matrix);	
	matrix->priv->rows_target = g_hash_table_lookup (repl, rows_target);
	gnome_db_base_connect_nullify (matrix->priv->rows_target,
				 G_CALLBACK (nullified_main_object_cb), matrix);

	matrix->priv->query_select_cols = GNOME_DB_QUERY (gnome_db_query_new_copy (cols_select_query, repl));
	gnome_db_base_connect_nullify (matrix->priv->query_select_cols,
				 G_CALLBACK (nullified_main_object_cb), matrix);	
	matrix->priv->cols_target = g_hash_table_lookup (repl, cols_target);
	gnome_db_base_connect_nullify (matrix->priv->cols_target,
				 G_CALLBACK (nullified_main_object_cb), matrix);
	
	query = GNOME_DB_QUERY (gnome_db_query_new (matrix->priv->dict));
	matrix->priv->query_select_contents = query;
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_SELECT);
	gnome_db_base_connect_nullify (matrix->priv->query_select_contents, 
				 G_CALLBACK (nullified_main_object_cb), matrix);
	
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (matrix->priv->modif_table)));
	gnome_db_query_add_target (query, target, NULL);
	g_object_unref (G_OBJECT (target));

	list = modif_table_fields;
	while (list) {
		GnomeDbField *field;
		field = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (matrix->priv->query_select_contents, 
								target, GNOME_DB_FIELD (list->data)));
		gnome_db_entity_add_field (GNOME_DB_ENTITY (matrix->priv->query_select_contents), field);
		matrix->priv->modif_table_fields = g_slist_append (matrix->priv->modif_table_fields, field);
		gnome_db_base_set_name (GNOME_DB_BASE (field), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
		g_object_unref (G_OBJECT (field));
		list = g_slist_next (list);
	}

	/* build queries */
	gnome_db_matrix_compute_queries (matrix);

	/* finish init with widgets part */
	gnome_db_matrix_initialize (matrix);

	return GTK_WIDGET (obj);
}

static void
gnome_db_matrix_dispose (GObject *object)
{
	GnomeDbMatrix *matrix;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_MATRIX (object));
	matrix = GNOME_DB_MATRIX (object);

	if (matrix->priv) {
		/* global clean */
		gnome_db_matrix_clean_all (matrix);

		/* the private area itself */
		g_free (matrix->priv);
		matrix->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_matrix_set_property (GObject              *object,
			     guint                 param_id,
			     const GValue         *value,
			     GParamSpec           *pspec)
{
	GnomeDbMatrix *matrix;

        matrix = GNOME_DB_MATRIX (object);
        if (matrix->priv) {
                switch (param_id) {
                case PROP_TITLE_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (matrix->priv->title);
			else
				gtk_widget_hide (matrix->priv->title);
                        break;
                case PROP_TITLE_STRING:
			gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (matrix->priv->title), g_value_get_string (value));
			gtk_widget_show (matrix->priv->title);
                        break;
		case PROP_NB_COLUMNS:
			if (g_value_get_int (value) == matrix->priv->gtk_table_view_nb_cols)
				return;
			else {
				ViewData *vd;
				matrix->priv->gtk_table_view_nb_cols = g_value_get_int (value);

				vd = matrix->priv->view_data [GNOME_DB_MATRIX_LIST_DETAILLED];
				if (matrix->priv->view_pages [GNOME_DB_MATRIX_LIST_DETAILLED] != 0)
					(vd->update_view) (matrix, vd);
				vd = matrix->priv->view_data [GNOME_DB_MATRIX_LIST_SYNTHETIC];
				if (matrix->priv->view_pages [GNOME_DB_MATRIX_LIST_SYNTHETIC] != 0)
					(vd->update_view) (matrix, vd);
			}
			break;
		case PROP_ASSOC_DATA_ONLY:
			if (matrix->priv->assoc_data_only != g_value_get_boolean (value)) {
				gint i;
				matrix->priv->assoc_data_only = g_value_get_boolean (value);
				/* update all the wiews */
				for (i=0; i<NB_TYPES; i++) {
					ViewData *vd = matrix->priv->view_data [i];
					if (matrix->priv->view_pages [i] != 0)
						(vd->update_view) (matrix, vd);
				}
			}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
gnome_db_matrix_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	GnomeDbMatrix *matrix;

        matrix = GNOME_DB_MATRIX (object);
        if (matrix->priv) {
                switch (param_id) {
		case PROP_NB_COLUMNS:
			g_value_set_int (value, matrix->priv->gtk_table_view_nb_cols);
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}

static void
nullified_main_object_cb (GnomeDbQuery *query, GnomeDbMatrix *matrix)
{
	gnome_db_matrix_clean_all (matrix);
}

static void gnome_db_matrix_compute_query_update (GnomeDbMatrix *matrix, GnomeDbQuery *base_query);
static void gnome_db_matrix_compute_query_insert (GnomeDbMatrix *matrix, GnomeDbQuery *base_query);
static void gnome_db_matrix_compute_query_delete (GnomeDbMatrix *matrix, GnomeDbQuery *base_query);

/*
 * Creates all the matrix->priv->query_* GnomeDbQuery objects
 */
static void
gnome_db_matrix_compute_queries (GnomeDbMatrix *matrix)
{
	GSList *list, *fields;
	GnomeDbTarget *contents_target;
	GnomeDbEntity *entity;
	GHashTable *repl;
	GnomeDbQuery *work_query;
	GSList *rows_for_cond_fields = NULL; /* GnomeDbQField objects from query_select_contents, related to query_select_rows */
	GSList *cols_for_cond_fields = NULL; /* GnomeDbQField objects from query_select_contents, related to query_select_rows */
	GSList *for_cond_value_qfields = NULL;
	GnomeDbDataSet *context;

	/*
	 * Creation of an empty query for query_select_contents
	 */
	list = gnome_db_query_get_targets (matrix->priv->query_select_contents);
	contents_target = GNOME_DB_TARGET (list->data);
	g_slist_free (list);

	/* 
	 * in query_select_rows, expand any '*' into all the corresponding fields for the rows_target target
	 */
	fields = gnome_db_query_expand_all_field (matrix->priv->query_select_rows, matrix->priv->rows_target);
	if (fields)
		g_slist_free (fields);

	/* 
	 * in query_select_cols, expand any '*' into all the corresponding fields for the cols_target target
	 */
	fields = gnome_db_query_expand_all_field (matrix->priv->query_select_cols, matrix->priv->cols_target);
	if (fields)
		g_slist_free (fields);
	
	/* 
	 * Use the FK constraints to improve query_select_rows and query_select_contents 
	 * with the PK and FK fields respectively
	 */
	entity = gnome_db_target_get_represented_entity (matrix->priv->rows_target);
	if (IS_GNOME_DB_TABLE (entity)) {
		GSList *fkcons_list;
		GnomeDbTable *table = GNOME_DB_TABLE (entity);
		
		fkcons_list = gnome_db_database_get_tables_fk_constraints (gnome_db_dict_get_database (matrix->priv->dict),
								     matrix->priv->modif_table, table, TRUE);
		if (fkcons_list) {
			GSList *fkfields = gnome_db_constraint_fkey_get_fields (GNOME_DB_CONSTRAINT (fkcons_list->data));
			list = fkfields;
			while (list) {
				gint pk_col, fk_col;
				GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
				GnomeDbQfield *pfield;

				/* PK field */
				pfield = (GnomeDbQfield *) 
					gnome_db_query_get_field_by_ref_field (matrix->priv->query_select_rows,
									 NULL, GNOME_DB_FIELD (pair->ref_pkey),
									 GNOME_DB_FIELD_VISIBLE);
				if (!pfield) {
					pfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (matrix->priv->query_select_rows,
											  matrix->priv->rows_target, 
											  GNOME_DB_FIELD (pair->ref_pkey)));
					gnome_db_entity_add_field (GNOME_DB_ENTITY (matrix->priv->query_select_rows), 
							     GNOME_DB_FIELD (pfield));
					gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pfield), TRUE);
					gnome_db_base_set_name (GNOME_DB_BASE (pfield), gnome_db_base_get_name (GNOME_DB_BASE (pair->ref_pkey)));
					g_object_unref (G_OBJECT (pfield));
				}
				gnome_db_query_set_order_by_field (matrix->priv->query_select_rows, pfield, 
							     G_MAXINT, TRUE);
				pk_col = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (matrix->priv->query_select_rows),
								    GNOME_DB_FIELD (pfield));

				/* FK field */
				pfield = (GnomeDbQfield *) 
					gnome_db_query_get_field_by_ref_field (matrix->priv->query_select_contents,
									 NULL, GNOME_DB_FIELD (pair->fkey), GNOME_DB_FIELD_VISIBLE);
				if (!pfield) {
					pfield = GNOME_DB_QFIELD 
						(gnome_db_qf_field_new_with_objects (matrix->priv->query_select_contents,
									       contents_target, 
									       GNOME_DB_FIELD (pair->fkey)));
					gnome_db_entity_add_field (GNOME_DB_ENTITY (matrix->priv->query_select_contents),
							     GNOME_DB_FIELD (pfield));
					gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pfield), TRUE);
					gnome_db_base_set_name (GNOME_DB_BASE (pfield), gnome_db_base_get_name (GNOME_DB_BASE (pair->fkey)));
					g_object_unref (G_OBJECT (pfield));
				}
				gnome_db_query_set_order_by_field (matrix->priv->query_select_contents, pfield, 
							     G_MAXINT, TRUE);
				fk_col = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (matrix->priv->query_select_contents),
								    GNOME_DB_FIELD (pfield));

				/* hash and lists insertion */
				g_hash_table_insert (matrix->priv->hash_rows, GINT_TO_POINTER (fk_col), GINT_TO_POINTER (pk_col));
				matrix->priv->contents_rows_fk_no = g_slist_append (matrix->priv->contents_rows_fk_no, 
										    GINT_TO_POINTER (fk_col));
				matrix->priv->rows_pk_no = g_slist_append (matrix->priv->rows_pk_no, GINT_TO_POINTER (pk_col));
				rows_for_cond_fields = g_slist_append (rows_for_cond_fields, pfield);
				
				list = g_slist_next (list);
			}

			g_slist_free (fkfields);			
			g_slist_free (fkcons_list);
		}
	}
	else 
		TO_IMPLEMENT;

	/* 
	 * Use the FK constraints to improve query_select_cols and query_select_contents 
	 * with the PK and FK fields respectively
	 */
	entity = gnome_db_target_get_represented_entity (matrix->priv->cols_target);
	if (IS_GNOME_DB_TABLE (entity)) {
		GSList *fkcons_list;
		GnomeDbTable *table = GNOME_DB_TABLE (entity);
		
		fkcons_list = gnome_db_database_get_tables_fk_constraints (gnome_db_dict_get_database (matrix->priv->dict),
								     matrix->priv->modif_table, table, TRUE);
		if (fkcons_list) {
			GSList *fkfields = gnome_db_constraint_fkey_get_fields (GNOME_DB_CONSTRAINT (fkcons_list->data));
			list = fkfields;
			while (list) {
				gint pk_col, fk_col;
				GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
				GnomeDbQfield *pfield;

				/* PK field */
				pfield = (GnomeDbQfield *) 
					gnome_db_query_get_field_by_ref_field (matrix->priv->query_select_cols,
									 NULL, GNOME_DB_FIELD (pair->ref_pkey), 
									 GNOME_DB_FIELD_VISIBLE);
				if (!pfield) {
					pfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (matrix->priv->query_select_cols,
											  matrix->priv->cols_target, 
											  GNOME_DB_FIELD (pair->ref_pkey)));
					gnome_db_entity_add_field (GNOME_DB_ENTITY (matrix->priv->query_select_cols), 
							     GNOME_DB_FIELD (pfield));
					gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pfield), TRUE);
					gnome_db_base_set_name (GNOME_DB_BASE (pfield), gnome_db_base_get_name (GNOME_DB_BASE (pair->ref_pkey)));
					g_object_unref (G_OBJECT (pfield));
				}
				gnome_db_query_set_order_by_field (matrix->priv->query_select_cols, pfield, 
							     G_MAXINT, TRUE);
				pk_col = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (matrix->priv->query_select_cols),
								    GNOME_DB_FIELD (pfield));

				/* FK field */
				pfield = (GnomeDbQfield *) 
					gnome_db_query_get_field_by_ref_field (matrix->priv->query_select_contents,
									 NULL, GNOME_DB_FIELD (pair->fkey), GNOME_DB_FIELD_VISIBLE);
				if (!pfield) {
					pfield = GNOME_DB_QFIELD 
						(gnome_db_qf_field_new_with_objects (matrix->priv->query_select_contents,
									       contents_target, 
									       GNOME_DB_FIELD (pair->fkey)));
					gnome_db_entity_add_field (GNOME_DB_ENTITY (matrix->priv->query_select_contents),
							     GNOME_DB_FIELD (pfield));
					gnome_db_qfield_set_internal (GNOME_DB_QFIELD (pfield), TRUE);
					gnome_db_base_set_name (GNOME_DB_BASE (pfield), gnome_db_base_get_name (GNOME_DB_BASE (pair->fkey)));
					g_object_unref (G_OBJECT (pfield));
				}
				gnome_db_query_set_order_by_field (matrix->priv->query_select_contents, pfield, 
							     G_MAXINT, TRUE);
				fk_col = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (matrix->priv->query_select_contents),
								    GNOME_DB_FIELD (pfield));
				
				/* hash and lists insertion */
				g_hash_table_insert (matrix->priv->hash_cols, GINT_TO_POINTER (fk_col), GINT_TO_POINTER (pk_col));
				matrix->priv->contents_cols_fk_no = g_slist_append (matrix->priv->contents_cols_fk_no, 
										    GINT_TO_POINTER (fk_col));
				matrix->priv->cols_pk_no = g_slist_append (matrix->priv->cols_pk_no, GINT_TO_POINTER (pk_col));
				cols_for_cond_fields = g_slist_append (cols_for_cond_fields, pfield);

				list = g_slist_next (list);
			}

			g_slist_free (fkfields);			
			g_slist_free (fkcons_list);
		}
	}
	else 
		TO_IMPLEMENT;

	/* 
	 * building args_context 
	 */
	matrix->priv->args_context = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (matrix->priv->query_select_rows));
	context = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (matrix->priv->query_select_cols));
	gnome_db_data_set_merge_dataset_params (matrix->priv->args_context, context);
	g_object_unref (G_OBJECT (context));

	gnome_db_base_connect_nullify (matrix->priv->args_context,
				 G_CALLBACK (nullified_main_object_cb), matrix);

#ifdef debug_NO
	{
		gchar *sql;
		
		g_print ("=========== Args context =============\n");
		gnome_db_base_dump (GNOME_DB_BASE (matrix->priv->args_context), 0);
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_rows), NULL, 0, NULL);
		g_print ("Matrix ROWS: %s\n", sql);
		g_free (sql);
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_cols), NULL, 0, NULL);
		g_print ("Matrix COLS: %s\n", sql);
		g_free (sql);
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_contents), NULL, 0, NULL);
		g_print ("Matrix CONTENTS: %s\n", sql);
		g_free (sql);
	}
#endif

	/*
	 * Manipulations on a copy of matrix->priv->query_select_contents to add value providers to all
	 * of the visible fields (as GnomeDbQfValue objects). The copy is discarded afterwards.
	 */
	repl = g_hash_table_new (NULL, NULL);
	work_query = GNOME_DB_QUERY (gnome_db_query_new_copy (matrix->priv->query_select_contents, repl));

	fields = NULL;
	list = rows_for_cond_fields;
	while (list) {
		fields = g_slist_append (fields, g_hash_table_lookup (repl, list->data));
		list = g_slist_next (list);
	}
	g_slist_free (rows_for_cond_fields);
	rows_for_cond_fields = fields;

	fields = NULL;
	list = cols_for_cond_fields;
	while (list) {
		fields = g_slist_append (fields, g_hash_table_lookup (repl, list->data));
		list = g_slist_next (list);
	}
	g_slist_free (cols_for_cond_fields);
	cols_for_cond_fields = fields;

	fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (work_query));
	list = fields;
	while (list) {
		GnomeDbField *field = GNOME_DB_FIELD (list->data);

		GnomeDbQfValue *vfield;
		GnomeDbField *reffield;
		const gchar *plugin;
		GnomeDbDataHandler *default_dh = NULL;
		
		g_assert (IS_GNOME_DB_QF_FIELD (list->data));

		/* Add a GnomeDbQfValue for that field */
		vfield = GNOME_DB_QF_VALUE (gnome_db_qf_value_new (work_query, gnome_db_field_get_data_type (field)));
		gnome_db_entity_add_field (GNOME_DB_ENTITY (work_query), GNOME_DB_FIELD (vfield));
		gnome_db_base_set_name (GNOME_DB_BASE (vfield), gnome_db_base_get_name (GNOME_DB_BASE (field)));
		gnome_db_base_set_description (GNOME_DB_BASE (vfield), gnome_db_base_get_description (GNOME_DB_BASE (field)));
		g_object_unref (G_OBJECT (vfield));
		gnome_db_qf_value_set_is_parameter (vfield, TRUE);
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (vfield), FALSE);
		g_object_set (G_OBJECT (field), "value_provider", vfield, NULL);
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (vfield), TRUE);
		
		/* GnomeDbQfValue properties */
		if (g_slist_find (rows_for_cond_fields, field) || g_slist_find (cols_for_cond_fields, field)) {
			gnome_db_base_set_name (GNOME_DB_BASE (vfield), "_Condition Value");
			g_object_set_data (G_OBJECT (vfield), "for_cond", field);
			for_cond_value_qfields = g_slist_append (for_cond_value_qfields, vfield);
		}
		if (IS_GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field)))) {
			GnomeDbTableField *dbfield = GNOME_DB_TABLE_FIELD (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field)));
			GnomeDbServer *srv;
			
			gnome_db_qf_value_set_not_null (vfield, !gnome_db_table_field_is_null_allowed (dbfield));
			if (gnome_db_table_field_get_default_value (dbfield))
				gnome_db_qf_value_set_default_value (vfield, 
							       gnome_db_table_field_get_default_value (dbfield));
			
			srv = gnome_db_dict_get_server (matrix->priv->dict);
			default_dh = gnome_db_server_get_object_handler (srv, G_OBJECT (dbfield));
		}
		else {
			TO_IMPLEMENT;
		}
		
		/* Entry plugin if available */
		g_object_get (G_OBJECT (list->data), "handler_plugin", &plugin, NULL);
		if (plugin) 
			g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
		else {
			if (default_dh) {
				plugin = gnome_db_base_get_name (GNOME_DB_BASE (default_dh));
				g_object_set (G_OBJECT (vfield), "handler_plugin", plugin, NULL);
			}
		}
		
		reffield = gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field));
		if (IS_GNOME_DB_TABLE_FIELD (reffield)) 
			gnome_db_qf_value_set_not_null (vfield,
						  !gnome_db_table_field_is_null_allowed (GNOME_DB_TABLE_FIELD (reffield)));
		
		
		list = g_slist_next (list);
	}

	/* 
	 * building work_context 
	 */
	matrix->priv->work_context = gnome_db_entity_get_exec_dataset (GNOME_DB_ENTITY (work_query));
	gnome_db_base_connect_nullify (matrix->priv->work_context,
				 G_CALLBACK (nullified_main_object_cb), matrix);
#ifdef debug_NO
	{
		g_print ("=========== Work context =============\n");
		gnome_db_base_dump (GNOME_DB_BASE (matrix->priv->work_context), 0);
	}
#endif


	/*
	 * setting a "wcontext_param" user data to point to the "corresponding" 
	 * GnomeDbParameter in work_context for the fields in matrix->priv->modif_table_fields.
	 */
	list = matrix->priv->modif_table_fields;
	while (list) {
		GnomeDbQfield *field, *value;
		GnomeDbParameter *param;

		field = g_hash_table_lookup (repl, list->data);
		g_object_get (G_OBJECT (field), "value_provider", &value, NULL);
		param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, value);
		g_object_set_data (G_OBJECT (list->data), "wcontext_param", param);
		list = g_slist_next (list);
	}

	/* 
	 * Preparing the list of parameters for which no entry will have to be displayed in the
	 * form for the data associated to each association
	 * computing the priv->rows_cond_params and priv->cols_cond_params list 
	 */
	g_object_get (G_OBJECT (work_query), "really_all_fields", &list, NULL);
	while (list) {
		if (g_slist_find (for_cond_value_qfields, list->data)) {
			GnomeDbParameter *param;
			GnomeDbQfield *for_cond_field;
			gint pos;

			param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, 
								     GNOME_DB_QFIELD (list->data));
			g_assert (param);
			for_cond_field = g_object_get_data (G_OBJECT (list->data), "for_cond");
			g_assert (for_cond_field);
			pos = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (work_query), GNOME_DB_FIELD (for_cond_field));
			if (g_slist_find (rows_for_cond_fields, for_cond_field)) {
				matrix->priv->rows_cond_params = g_slist_append (matrix->priv->rows_cond_params, param);
				pos = GPOINTER_TO_INT (g_hash_table_lookup (matrix->priv->hash_rows, GINT_TO_POINTER (pos)));
			}
			else {
				matrix->priv->cols_cond_params = g_slist_append (matrix->priv->cols_cond_params, param);
				pos = GPOINTER_TO_INT (g_hash_table_lookup (matrix->priv->hash_cols, GINT_TO_POINTER (pos)));
			}
			g_object_set_data (G_OBJECT (param), "rs_col", GINT_TO_POINTER (pos));			
#ifdef debug_NO
			g_print ("NO SHOW param %p (%s) because for cond\n", param, gnome_db_base_get_name (GNOME_DB_BASE (param)));
#endif
		}
		
		list = g_slist_next (list);
	}

	g_slist_free (rows_for_cond_fields);
	g_slist_free (cols_for_cond_fields);
	g_slist_free (for_cond_value_qfields);
	for_cond_value_qfields = NULL;
	
	/* 
	 * building INSERT / UPDATE / DELETE queries 
	 */
	gnome_db_matrix_compute_query_insert (matrix, work_query);
	gnome_db_matrix_compute_query_update (matrix, work_query);
	gnome_db_matrix_compute_query_delete (matrix, work_query);

	g_object_unref (G_OBJECT (work_query));
	g_hash_table_destroy (repl);
}

static void
gnome_db_matrix_compute_query_update (GnomeDbMatrix *matrix, GnomeDbQuery *base_query)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;
	GnomeDbCondition *cond = NULL;
	GHashTable *replacements;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (matrix->priv->dict));
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_UPDATE);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (matrix->priv->modif_table)));
	gnome_db_query_add_target (query, target, NULL);
	replacements = g_hash_table_new (NULL, NULL);

	/* Fields */
	g_object_get (G_OBJECT (base_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_FIELD (list->data)) {
			GnomeDbField *field, *value_prov, *new_value_prov;
			GnomeDbParameter *param;
			
			field = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, target,
									gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data))));
			gnome_db_base_set_name (GNOME_DB_BASE (field), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
			gnome_db_entity_add_field (GNOME_DB_ENTITY (query), field);
			if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), FALSE);

			g_hash_table_insert (replacements, list->data, field);
			g_object_unref (G_OBJECT (field));
			g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);

			if (value_prov) {
				g_assert (IS_GNOME_DB_QF_VALUE (value_prov));
				
				param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, 
									     GNOME_DB_QFIELD (value_prov));
				new_value_prov = GNOME_DB_FIELD (gnome_db_qf_value_new (query, 
							   gnome_db_qf_value_get_server_data_type (GNOME_DB_QF_VALUE (value_prov))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (new_value_prov), TRUE);
				gnome_db_base_set_name (GNOME_DB_BASE (new_value_prov), 
						  gnome_db_base_get_name (GNOME_DB_BASE (value_prov)));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), new_value_prov);
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (new_value_prov), FALSE);
				g_object_unref (G_OBJECT (new_value_prov));
				g_object_set (G_OBJECT (field), "value_provider", new_value_prov, NULL);
				
				gnome_db_parameter_add_dest_field (param, GNOME_DB_QFIELD (new_value_prov));
			}
		}

		list = g_slist_next (list);
	}

	/* Condition */
	g_object_get (G_OBJECT (base_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_VALUE (list->data)) {
			if (g_object_get_data (G_OBJECT (list->data), "for_cond")) {
				GnomeDbQfield *field, *value;
				GnomeDbCondition *subcond;
				GnomeDbParameter *param;
				
				if (!cond) {
					cond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND));
					gnome_db_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}
				
				field = g_hash_table_lookup (replacements, 
							     g_object_get_data (G_OBJECT (list->data), "for_cond"));
				g_assert (field);
				g_assert (GNOME_DB_QF_FIELD (field));
				
				param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, GNOME_DB_QFIELD (list->data));
				g_assert (param);
				value = GNOME_DB_QFIELD (gnome_db_qf_value_new (query, 
						   gnome_db_field_get_data_type (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (field)))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (value), TRUE);
				gnome_db_base_set_name (GNOME_DB_BASE (value), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (value));
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (value), FALSE);
				g_object_unref (G_OBJECT (value));
				gnome_db_parameter_add_dest_field (param, GNOME_DB_QFIELD (value));

				subcond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_LEAF_EQUAL));
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_LEFT, field);
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_RIGHT, value);
				gnome_db_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

	
	matrix->priv->query_update = query;
	gnome_db_base_connect_nullify (matrix->priv->query_update,
				 G_CALLBACK (nullified_main_object_cb), matrix);

	/* Free memory */
	g_hash_table_destroy (replacements);

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Matrix UPDATE: %s\n", sql);
		g_free (sql);
	}
#endif
}

static void
gnome_db_matrix_compute_query_insert (GnomeDbMatrix *matrix, GnomeDbQuery *base_query)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (matrix->priv->dict));
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_INSERT);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (matrix->priv->modif_table)));
	gnome_db_query_add_target (query, target, NULL);

	/* Fields */
	g_object_get (G_OBJECT (base_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_FIELD (list->data)) {
			GnomeDbField *field, *value_prov;
			GnomeDbParameter *param;
				
			field = GNOME_DB_FIELD (gnome_db_qf_field_new_with_objects (query, target,
									gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (list->data))));
			gnome_db_entity_add_field (GNOME_DB_ENTITY (query), field);
			if (!gnome_db_qfield_is_visible (GNOME_DB_QFIELD (list->data)))
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), FALSE);
			
			g_object_unref (G_OBJECT (field));
			g_object_get (G_OBJECT (list->data), "value_provider", &value_prov, NULL);
			if (value_prov) {
				g_assert (IS_GNOME_DB_QF_VALUE (value_prov));
				
				param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, 
									     GNOME_DB_QFIELD (value_prov));
				g_assert (param);
				value_prov = GNOME_DB_FIELD (gnome_db_qf_value_new (query, 
								    gnome_db_qf_value_get_server_data_type (GNOME_DB_QF_VALUE (value_prov))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (value_prov), TRUE);
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), value_prov);
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (value_prov), FALSE);
				g_object_unref (G_OBJECT (value_prov));
				g_object_set (G_OBJECT (field), "value_provider", value_prov, NULL);
				
				gnome_db_parameter_add_dest_field (param, GNOME_DB_QFIELD (value_prov));
			}
		}

		list = g_slist_next (list);
	}

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Matrix INSERT: %s\n", sql);
		g_free (sql);
	}
#endif

	matrix->priv->query_insert = query;
	gnome_db_base_connect_nullify (matrix->priv->query_insert,
				 G_CALLBACK (nullified_main_object_cb), matrix);
}

static void
gnome_db_matrix_compute_query_delete (GnomeDbMatrix *matrix, GnomeDbQuery *base_query)
{
	GSList *list;
	GnomeDbTarget *target;
	GnomeDbQuery *query;
	GnomeDbCondition *cond = NULL;
	GHashTable *replacements;

	/* Query object */
	query = GNOME_DB_QUERY (gnome_db_query_new (matrix->priv->dict));
	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_DELETE);
	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (matrix->priv->modif_table)));
	gnome_db_query_add_target (query, target, NULL);
	replacements = g_hash_table_new (NULL, NULL);

	/* Condition */
	g_object_get (G_OBJECT (base_query), "really_all_fields", &list, NULL);
	while (list) {
		if (IS_GNOME_DB_QF_VALUE (list->data)) {
			if (g_object_get_data (G_OBJECT (list->data), "for_cond")) {
				GnomeDbQfield *fieldv, *fieldf;
				GnomeDbCondition *subcond;
				GnomeDbParameter *param;
				
				if (!cond) {
					cond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND));
					gnome_db_query_set_condition (query, cond);
					g_object_unref (G_OBJECT (cond));
				}
				
				fieldf = g_object_get_data (G_OBJECT (list->data), "for_cond");
				fieldf = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (query, target,
										 gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (fieldf))));
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (fieldf));
				g_object_unref (G_OBJECT (fieldf));
				gnome_db_qfield_set_visible (fieldf, FALSE);

				param = gnome_db_data_set_find_parameter_for_field (matrix->priv->work_context, GNOME_DB_QFIELD (list->data));
				g_assert (param);
				fieldv = GNOME_DB_QFIELD (gnome_db_qf_value_new (query, 
							   gnome_db_field_get_data_type (gnome_db_qf_field_get_ref_field (GNOME_DB_QF_FIELD (fieldf)))));
				gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (fieldv), TRUE);
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), GNOME_DB_FIELD (fieldv));
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fieldv), FALSE);
				g_object_unref (G_OBJECT (fieldv));
				gnome_db_parameter_add_dest_field (param, GNOME_DB_QFIELD (fieldv));

				subcond = GNOME_DB_CONDITION (gnome_db_condition_new (query, GNOME_DB_CONDITION_LEAF_EQUAL));
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_LEFT, fieldf);
				gnome_db_condition_leaf_set_operator (subcond, GNOME_DB_CONDITION_OP_RIGHT, fieldv);
				gnome_db_condition_node_add_child (cond, subcond, NULL);
				g_object_unref (G_OBJECT (subcond));
			}
		}
		list = g_slist_next (list);
	}

	
	matrix->priv->query_delete = query;
	gnome_db_base_connect_nullify (matrix->priv->query_delete,
				 G_CALLBACK (nullified_main_object_cb), matrix);

#ifdef debug_NO
	{
		gchar *sql;
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (query), NULL, 0, NULL);
		g_print ("Matrix DELETE: %s\n", sql);
		g_free (sql);
	}
#endif
}


static void clean_assoc_chuncks (GnomeDbMatrix *matrix);

/* 
 * sets the widget in a safe mode where everything is non-sensitive, and 
 * disconnects all the signals.
 */
static void
gnome_db_matrix_clean_all (GnomeDbMatrix *matrix)
{
	gint i;
	if (matrix->priv->cleaned)
		return;

	matrix->priv->cleaned = TRUE;

	/* view data */
	for (i=0; i<NB_TYPES; i++) {
		ViewData *vd = matrix->priv->view_data [i];
		if (vd) {
			if (vd->clean_data)
				(vd->clean_data) (matrix, vd);
			g_free (vd);
			matrix->priv->view_data [i] = NULL;
		}
	}
	
	/* modif entity */
	if (matrix->priv->modif_table) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->modif_table),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		matrix->priv->modif_table = NULL;
	}
	
	/* contexts */
	if (matrix->priv->rows_cond_params) {
		g_slist_free (matrix->priv->rows_cond_params);
		matrix->priv->rows_cond_params = NULL;
	}
	if (matrix->priv->cols_cond_params) {
		g_slist_free (matrix->priv->cols_cond_params);
		matrix->priv->cols_cond_params = NULL;
	}

	if (matrix->priv->args_context) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->args_context),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->args_context));
		matrix->priv->args_context = NULL;
	}

	if (matrix->priv->work_context) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->work_context),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->work_context));
		matrix->priv->work_context = NULL;
	}

	/* targets */
	if (matrix->priv->rows_target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->rows_target),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		matrix->priv->rows_target = NULL;
	}
	if (matrix->priv->cols_target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->cols_target),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		matrix->priv->cols_target = NULL;
	}

	/* queries */
	if (matrix->priv->query_select_contents) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_select_contents),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_select_contents));
		matrix->priv->query_select_contents  = NULL;
	}

	if (matrix->priv->query_select_cols) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_select_cols),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_select_cols));
		matrix->priv->query_select_cols  = NULL;
	}

	if (matrix->priv->query_select_rows) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_select_rows),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_select_rows));
		matrix->priv->query_select_rows  = NULL;
	}

	if (matrix->priv->modif_table_fields) {
		g_slist_free (matrix->priv->modif_table_fields);
		matrix->priv->modif_table_fields = NULL;
	}

	if (matrix->priv->query_update) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_update),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_update));
		matrix->priv->query_update  = NULL;
	}	

	if (matrix->priv->query_insert) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_insert),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_insert));
		matrix->priv->query_insert  = NULL;
	}

	if (matrix->priv->query_delete) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (matrix->priv->query_delete),
						      G_CALLBACK (nullified_main_object_cb), matrix);
		g_object_unref (G_OBJECT (matrix->priv->query_delete));
		matrix->priv->query_delete  = NULL;
	}	


	/* PK/FK hash tables and lists */
	if (matrix->priv->replacements) {
		g_hash_table_destroy (matrix->priv->replacements);
		matrix->priv->replacements = NULL;
	}

	if (matrix->priv->hash_rows) {
		g_hash_table_destroy (matrix->priv->hash_rows);
		matrix->priv->hash_rows = NULL;
	}

	if (matrix->priv->hash_cols) {
		g_hash_table_destroy (matrix->priv->hash_cols);
		matrix->priv->hash_cols = NULL;
	}

	if (matrix->priv->contents_rows_fk_no) {
		g_slist_free (matrix->priv->contents_rows_fk_no);
		matrix->priv->contents_rows_fk_no = NULL;
	}

	if (matrix->priv->contents_cols_fk_no) {
		g_slist_free (matrix->priv->contents_cols_fk_no);
		matrix->priv->contents_cols_fk_no = NULL;
	}
	
	if (matrix->priv->rows_pk_no) {
		g_slist_free (matrix->priv->rows_pk_no);
		matrix->priv->rows_pk_no = NULL;
	}

	if (matrix->priv->cols_pk_no) {
		g_slist_free (matrix->priv->cols_pk_no);
		matrix->priv->cols_pk_no = NULL;
	}

	/* model */
	if (matrix->priv->model) {
		g_object_unref (G_OBJECT (matrix->priv->model));
		matrix->priv->model = NULL;
	}

	if (matrix->priv->assoc_chuncks)
		clean_assoc_chuncks (matrix);

	/* UI */
	if (matrix->priv->actions_group) {
		g_object_unref (G_OBJECT (matrix->priv->actions_group));
		matrix->priv->actions_group = NULL;
	}
	if (matrix->priv->uimanager) {
		g_object_unref (G_OBJECT (matrix->priv->uimanager));
		matrix->priv->uimanager = NULL;
	}		
}

static void 
clean_assoc_chuncks (GnomeDbMatrix *matrix)
{
	GSList *list = matrix->priv->assoc_chuncks;
	while (list) {
		AssocChunck *ac = ASSOC_CHUNCK (list->data);
		GSList *adlist = ac->assoc_data_list;
	
		while (adlist) {
			assoc_data_free (ASSOC_DATA (adlist->data));
			adlist = g_slist_next (adlist);
		}
		g_slist_free (ac->assoc_data_list);

		if (ac->pk_value)
			gda_value_free (ac->pk_value);

		g_free (ac);

		list = g_slist_next (list);
	}
	g_slist_free (matrix->priv->assoc_chuncks);
	matrix->priv->assoc_chuncks = NULL;
}

#ifdef debug
static void
dump_model (GnomeDbMatrix *matrix)
{
	GtkTreeIter iter;

	g_print ("====================== MODEL DUMP ======================\n");
	if (! gtk_tree_model_get_iter_first (matrix->priv->model, &iter))
		return;

	do {
		gint row;
		AssocChunck *ac;
		GSList *list;

		gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ROW_NUM, &row, COLUMN_ASSOC_CHUNCK, &ac, -1);
		g_print ("=> %d --", row);
		g_assert (ac);

		list = ac->assoc_data_list;
		while (list) {
			AssocData *ad = ASSOC_DATA (list->data);
			g_print (" [PK=%s CR=%d ACT=%d]", gda_value_stringify (ad->pk_value), 
				 ad->contents_row, ad->req_ins_del);
			list = g_slist_next (list);
		}
		g_print ("\n");
	}
	while (gtk_tree_model_iter_next (matrix->priv->model, &iter));
	if (gnome_db_matrix_has_been_changed (GNOME_DB_DATA_WIDGET (matrix)))
		g_print ("Has been changed\n");
	else
		g_print ("Unchanged\n");
}

static gchar *values_list_stringify (GList *values)
{
	GString *string = g_string_new ("{");
	gchar *retval;
	while (values) {
		gchar *str = gda_value_stringify ((GdaValue*) values->data);
		g_string_append (string, " \"");
		g_string_append (string, str);
		g_string_append (string, "\"");
		g_free (str);
		values = values->next;
	}
	g_string_append (string, " }");
	retval = string->str;
	g_string_free (string, FALSE);
	return retval;
}

#endif

static gboolean run_select_queries (GnomeDbMatrix *matrix, GError **error);
static void
arg_param_changed_cb (GnomeDbParameter *param, GnomeDbMatrix *matrix)
{
	model_update_complete (matrix, FALSE);	
}

/*
 * Updates the model after having re-run the select queries and having the result sets updated
 */
static void
model_update_complete (GnomeDbMatrix *matrix, gboolean keep_old_modifications)
{
	GError *error = NULL;

	if (run_select_queries (matrix, &error)) {
		gint i;
		model_update (matrix, keep_old_modifications);
#ifdef debug_NO
		dump_model (matrix);
#endif
		/* set the correct page to display */
		if (matrix->priv->view_pages [matrix->priv->view_mode] == 0) {
			ViewData *vd = matrix->priv->view_data [matrix->priv->view_mode];
			GtkWidget *wid = (vd->init_view) (matrix, vd);
			gtk_widget_show (wid);
			matrix->priv->view_pages [matrix->priv->view_mode] = 
				gtk_notebook_append_page (GTK_NOTEBOOK (matrix->priv->notebook), wid, NULL);
		}

		/* update of the views */
		for (i=0; i<NB_TYPES; i++) {
			ViewData *vd = matrix->priv->view_data [i];
			if (matrix->priv->view_pages [i] != 0)
				(vd->update_view) (matrix, vd);
		}
		
		gtk_notebook_set_current_page (GTK_NOTEBOOK (matrix->priv->notebook), 
					       matrix->priv->view_pages [matrix->priv->view_mode]);
	}
	else {
		/* we don't want non relevant error messages so we check that the args context is valid before */
		if ((matrix->priv->mode & GNOME_DB_ACTION_REPORT_ERROR) && gnome_db_data_set_is_valid (matrix->priv->args_context)) {
			GtkWidget *dlg;
			gchar *message;
			GtkWidget *parent;
			
			/* find the top level window of the matrix */
			parent = gtk_widget_get_parent (GTK_WIDGET (matrix));
			while (parent && !GTK_IS_WINDOW (parent)) 
				parent = gtk_widget_get_parent (parent);
			
			if (error) {
				message = g_strdup (error->message);
				g_error_free (error);
			}
			else
				message = g_strdup_printf (_("An unknown error occurred while executing the query."));
			
			dlg = gtk_message_dialog_new (GTK_WINDOW (parent), 0,
						      GTK_MESSAGE_ERROR,
						      GTK_BUTTONS_CLOSE,
						      message);
			g_free (message);
			gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
		}

		gtk_notebook_set_current_page (GTK_NOTEBOOK (matrix->priv->notebook), 0); /* no data page */
	}

	/* updating the various possible actions */
 	modif_buttons_update (matrix);
}

/*
 * Set the parameters in matrix->priv->work_context from the information
 * contained within @ad; @ad can be NULL
 *
 * @rows_row and @cols_row MUST be coherent with ad->cols_row and ad->ac->rows_row if @ad is not NULL
 */
static void
set_params_from_assoc_data (GnomeDbMatrix *matrix, gint rows_row, gint cols_row, AssocData *ad)
{
	GSList *list;

	/* Condition parameters from the rows */
	list = matrix->priv->rows_cond_params;
	while (list) {
		gint pos;
		
		pos = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (list->data), "rs_col"));
		gnome_db_parameter_set_value (GNOME_DB_PARAMETER (list->data),
					gda_data_model_get_value_at (matrix->priv->rs_rows, pos, rows_row));
		list = g_slist_next (list);
	}

	/* Condition parameters from the cols */
	list = matrix->priv->cols_cond_params;
	while (list) {
		gint pos;
		
		pos = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (list->data), "rs_col"));
		gnome_db_parameter_set_value (GNOME_DB_PARAMETER (list->data),
					gda_data_model_get_value_at (matrix->priv->rs_cols, pos, cols_row));		
		list = g_slist_next (list);
	}
	
	/* setting values for the data parameters */
	if (ad && ad->data) {
		GList *values;
		GSList *list;
		GnomeDbParameter *param;
			
		values = gda_value_get_list (ad->data);
		list = matrix->priv->modif_table_fields;
		while (values && list) {
			param = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			gnome_db_parameter_set_value (param, (GdaValue *) values->data);
			values = g_list_next (values);
			list = g_slist_next (list);
		}
		g_assert (!values && !list);
	}
	else {
		/* set all the para values to NULL */
		GSList *list = matrix->priv->modif_table_fields;
		while (list) {
			GnomeDbParameter *param = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			gnome_db_parameter_set_value (param, NULL);
			list = g_slist_next (list);
		}
	}
}


/*
 * (re)computes the SELECT queries and (re)runs them, resulting
 * in a change in the result sets
 *
 * Returns: TRUE if no error occured and the result sets are filled
 */
static gboolean
run_select_queries (GnomeDbMatrix *matrix, GError **error)
{
	/* discard the previous result sets */
	if (matrix->priv->rs_rows) {
		g_object_unref (G_OBJECT (matrix->priv->rs_rows));
		matrix->priv->rs_rows = NULL;
	}
	if (matrix->priv->rs_cols) {
		g_object_unref (G_OBJECT (matrix->priv->rs_cols));
		matrix->priv->rs_cols = NULL;
	}
	if (matrix->priv->rs_contents) {
		g_object_unref (G_OBJECT (matrix->priv->rs_contents));
		matrix->priv->rs_contents = NULL;
	}

	/* rows */
	if (matrix->priv->query_select_rows) {
		gchar *sql;
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_rows), 
						 matrix->priv->args_context, 0, error);
		if (sql) {
#ifdef debug
			g_print ("ROWS SQL: %s\n", sql);
#endif
			matrix->priv->rs_rows = gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (matrix->priv->dict), sql, 
											GNOME_DB_SERVER_QUERY_SQL, error);
			g_free (sql);
		}
#ifdef debug_NO
		else 
			g_print ("ROWS ERROR: %s\n", (*error)->message);
#endif
	}
	if (!matrix->priv->rs_rows)
		return FALSE;

	/* cols */
	if (matrix->priv->query_select_cols) {
		gchar *sql;
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_cols), 
						 matrix->priv->args_context, 0, error);
		if (sql) {
#ifdef debug
			g_print ("COLS SQL: %s\n", sql);
#endif
			matrix->priv->rs_cols = gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (matrix->priv->dict), sql, 
											GNOME_DB_SERVER_QUERY_SQL, error);
			g_free (sql);
		}
#ifdef debug_NO
		else 
			g_print ("COLS ERROR: %s\n", (*error)->message);
#endif
	}
	
	if (!matrix->priv->rs_cols)
		return FALSE;

	/* contents */
	if (matrix->priv->query_select_contents) {
		gchar *sql;
		/* Actual running query */
		sql = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (matrix->priv->query_select_contents), 
						 matrix->priv->args_context, 0, error);
		if (sql) {
#ifdef debug
			g_print ("CONTENTS SQL: %s\n", sql);
#endif
			matrix->priv->rs_contents = gnome_db_server_do_query_as_data_model (gnome_db_dict_get_server (matrix->priv->dict), sql, 
											    GNOME_DB_SERVER_QUERY_SQL, error);
			g_free (sql);
		}
#ifdef debug_NO
		else 
			g_print ("CONTENTS ERROR: %s\n", (*error)->message);
#endif
	}

	return matrix->priv->rs_contents ? TRUE : FALSE;
}

static gboolean m_gda_value_is_equal (GdaValue *v1, GdaValue *v2);
static gboolean m_gda_values_are_equal (GList *l1, GList *l2);
static GList *  build_values_list (GnomeDbResultSet *rs, gint row_number, GSList *col_numbers);
static gint find_col_for_values (GnomeDbMatrix *matrix, GSList *cols, GList *values, GnomeDbResultSet *rs);
/*
 * Updates the data model according to the result sets.
 * if @keep_user_modifs is TRUE, then the update process tries to keep data as modified by the user
 */
static void
model_update (GnomeDbMatrix *matrix, gboolean keep_user_modifs)
{
	gint rows_cr = 0;
	gint rows_count = 0;
	gint contents_cr = 0;
	gint contents_count = 0;
	gint *used_contents_rows; /* array of used tuples of matrix->priv->rs_contents, indexed by row number */

	GtkTreeIter current_iter;
	gboolean mode_new_data = TRUE; /* TRUE if new rows must be created in the model */

	GSList *new_assoc_list = NULL;

	if (!matrix->priv->rs_rows || !matrix->priv->rs_cols || !matrix->priv->rs_contents) {
		gtk_list_store_clear (GTK_LIST_STORE (matrix->priv->model));

		return;
	}

	/*
	 * Note on the processus:
	 * the idea is to iterate through all the priv->rs_rows tuples, and for each tuple, use the PK
	 * values (a PK can be made by several values which are called PK values). Each new set of PK values
	 * gives way to a new row in the data model (=> a new AssocChunck)
	 *
	 * At the same time, we parse the priv->rs_contents tuples, and where there is a matching, we
	 * set a new AssocData structure
	 *
	 * Also if @keep_user_modifs is TRUE, we try to re-use AssocData structures which already exist
	 */

	/* 
	 * iterate through all the priv->rs_rows result set 
	 */
	if (matrix->priv->rs_contents)
		contents_count = gda_data_model_get_n_rows (matrix->priv->rs_contents);
	else
		contents_count = 0;

	used_contents_rows = g_new0 (gboolean, contents_count);
	rows_count = gda_data_model_get_n_rows (matrix->priv->rs_rows);
	for (rows_cr = 0; rows_cr < rows_count; rows_cr ++) {
		GList *rows_pk_values = NULL;
		GSList *list;
		gboolean found;
		GSList *row_assoc_list = NULL;
		AssocChunck *new_ac;
		AssocChunck *old_ac = NULL;

		/* new row in the data model */
		if (rows_cr == 0) 
			mode_new_data = !gtk_tree_model_get_iter_first (matrix->priv->model, &current_iter);
		else
			mode_new_data = !gtk_tree_model_iter_next (matrix->priv->model, &current_iter);
		if (mode_new_data) 
			gtk_list_store_append (GTK_LIST_STORE (matrix->priv->model), &current_iter);

		new_ac = g_new0 (AssocChunck, 1);
		gtk_list_store_set (GTK_LIST_STORE (matrix->priv->model), &current_iter, 
				    COLUMN_ROW_NUM, rows_cr, COLUMN_ASSOC_CHUNCK, new_ac, -1);

		rows_pk_values = build_values_list (matrix->priv->rs_rows, rows_cr, matrix->priv->rows_pk_no);

		/* Examine the existing AssocData structures where contents_row<0 */
		if (keep_user_modifs) {
			/* find the old_ac */
			list = matrix->priv->assoc_chuncks;
			while (list && !old_ac) {
				old_ac = ASSOC_CHUNCK (list->data);
				if (! m_gda_values_are_equal (gda_value_get_list (old_ac->pk_value), rows_pk_values))
					old_ac = NULL;
				list = g_slist_next (list);
			}

			if (old_ac) {
				/* parse this AssocChunck's list of AssocData structures */
				list = old_ac->assoc_data_list;
				while (list) {
					AssocData *ad = ASSOC_DATA (list->data);
					gint pos;

					if (ad->contents_row >= 0) {
						list = g_slist_next (list);
						continue;
					}
					
					pos = find_col_for_values (matrix, matrix->priv->cols_pk_no,
								   gda_value_get_list (ad->pk_value), 
								   matrix->priv->rs_cols);
					if (pos >= 0) {
						/* the ad's pk_value still exist in rs_cols => keep the ad */
						AssocData *ad = ASSOC_DATA (list->data);
						GSList *next = g_slist_next (list);

						ad->ac = new_ac;
						row_assoc_list = g_slist_append (row_assoc_list, ad);
						old_ac->assoc_data_list = g_slist_delete_link 
							(old_ac->assoc_data_list, list);
						list = next;
					}
					else
						list = g_slist_next (list);
				}
			}
		}

		/* parse matrix->priv->rs_contents for this set of PK values in priv->rs_rows  */
		contents_cr = 0;
		while (contents_cr < contents_count) {
			GList *tst_values;
			
			/* optimization: go to the next row if this one has already been taken into account */
			if (used_contents_rows[contents_cr]) {
				contents_cr++;
				continue;
			}
			
			tst_values = build_values_list (matrix->priv->rs_contents, contents_cr, 
							matrix->priv->contents_rows_fk_no);
			found = m_gda_values_are_equal (rows_pk_values, tst_values);
			g_list_free (tst_values);
			if (found) {
				/* there is a coorespondance => set up an AssocData structure */
				gint cols_cr;
				AssocData *ad = NULL;				

				used_contents_rows[contents_cr] = TRUE;
				tst_values = build_values_list (matrix->priv->rs_contents, contents_cr, 
								matrix->priv->contents_cols_fk_no);
				
				/* list of FK values to priv->rs_cols */
				cols_cr = find_col_for_values (matrix, matrix->priv->cols_pk_no,
							       tst_values, matrix->priv->rs_cols);
				
				/* see if there is already an AssocData for the corresponding PK values */
				if (keep_user_modifs && !old_ac) {
					list = matrix->priv->assoc_chuncks;
					while (list && !old_ac) {
						old_ac = ASSOC_CHUNCK (list->data);
						if (! m_gda_values_are_equal (gda_value_get_list (old_ac->pk_value), 
									      rows_pk_values))
							old_ac = NULL;
						list = g_slist_next (list);
					}
				}
				
				if (keep_user_modifs && old_ac) {
					list = old_ac->assoc_data_list;
					while (list && !ad) {
						ad = ASSOC_DATA (list->data);
						
						if (m_gda_values_are_equal (gda_value_get_list (ad->pk_value), 
									    tst_values)) {
							old_ac->assoc_data_list = g_slist_delete_link 
								(old_ac->assoc_data_list, list);
						}
						else
							ad = NULL;
						list = g_slist_next (list);
					}
				}
				
				if (!ad) {
					/* create an AssocData structure here for the association found in priv->rs_contents */
					ad = g_new0 (AssocData, 1);
					ad->cols_row = cols_cr;
					ad->pk_value = gda_value_new_list (tst_values);
					ad->req_ins_del = FALSE;
					ad->contents_row = contents_cr;
					ad->data = NULL; 
					ad->data_orig = NULL;
					ad->data_valid = TRUE;

					if (matrix->priv->modif_table_fields) {
						/* copy data in priv->rs_contents using the _new()'s modif_table_fields */
						GList *data_values = NULL;
						gint col;
						list = matrix->priv->modif_table_fields;
						while (list) {
							col = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (matrix->priv->query_select_contents),
											 GNOME_DB_FIELD (list->data));
							data_values = g_list_append (data_values,
										     gda_data_model_get_value_at (matrix->priv->rs_contents,
														col, contents_cr));
							list = g_slist_next (list);
						}
						ad->data = gda_value_new_list (data_values);
						ad->data_orig = gda_value_copy (ad->data);
						g_list_free (data_values);
					}
				}
				g_list_free (tst_values);
				
				ad->ac = new_ac;
				
				row_assoc_list = g_slist_append (row_assoc_list, ad);
			}
			contents_cr++;
		}
		
		/* set the model */
		new_ac->rows_row = rows_cr;
		new_ac->pk_value = gda_value_new_list (rows_pk_values);
		new_ac->assoc_data_list = row_assoc_list;
		new_assoc_list = g_slist_append (new_assoc_list, new_ac);

		/* free memory for rows_pk_values */
		g_list_free (rows_pk_values);
	}
	g_free (used_contents_rows);

	/* remove extra remaining rows */
	if (rows_cr == 0)
		gtk_list_store_clear (GTK_LIST_STORE (matrix->priv->model));
	else {
		mode_new_data = !gtk_tree_model_iter_next (matrix->priv->model, &current_iter);			
		if (!mode_new_data) 
			while (gtk_list_store_remove (GTK_LIST_STORE (matrix->priv->model), &current_iter));
	}
	
	clean_assoc_chuncks (matrix);
	matrix->priv->assoc_chuncks = new_assoc_list;
}

static GList *
build_values_list (GnomeDbResultSet *rs, gint row_number, GSList *col_numbers)
{
	GList *values = NULL;
	
	while (col_numbers) {
		values = g_list_append (values, 
					gda_data_model_get_value_at (rs, GPOINTER_TO_INT (col_numbers->data), row_number));
		col_numbers = g_slist_next (col_numbers);
	}

	return values;
}

/*
 * Compares two values to see if they are equal. 
 * --> A NULL value is considered to be equal to a GDA_VALUE_TYPE_NULL GdaValue. 
 * --> FALSE is returned if the two values are of different types (with a warning)
 */
static gboolean
m_gda_value_is_equal (GdaValue *v1, GdaValue *v2)
{
	if ((!v1 || gda_value_is_null (v1)) && (!v2 || gda_value_is_null (v2)))
		return TRUE;

	if (!v1 || !v2)
		return FALSE;

	g_return_val_if_fail (gda_value_get_type (v1) == gda_value_get_type (v2), FALSE);

	if (gda_value_get_type (v1) == GDA_VALUE_TYPE_LIST) {
		GList *l1 = v1->value.v_list;
		GList *l2 = v2->value.v_list;
		gboolean equal = TRUE;

		while (l1 && l2 && equal) {
                        equal = m_gda_value_is_equal ((GdaValue *) l1->data, (GdaValue *) l2->data);

			l1 = l1->next;
			l2 = l2->next;
                }

		g_return_val_if_fail (! (equal && (l1 || l2)), FALSE); /* because l1 and l2 didn't have the same lengths */
		return equal;
	}
	else
		return ! gda_value_compare (v1, v2);
}

/*
 * Compares the values stored in two lists, using the m_gda_value_is_equal()
 * function for each value.
 * If the lists don't have the same lengths, then FALSE is returned (with a warning)
 */
static gboolean
m_gda_values_are_equal (GList *l1, GList *l2)
{
	gboolean equal = TRUE;
	while (l1 && l2 && equal) {
		equal = m_gda_value_is_equal ((GdaValue *) (l1->data), (GdaValue *) (l2->data));
		l1 = l1->next;
		l2 = l2->next;
	}

	g_return_val_if_fail (! (equal && (l1 || l2)), FALSE); /* because l1 and l2 didn't have the same lengths */

	return equal;
}

/*
 * Finds the row number in @rs for which the value at every column number in @cols matches the
 * corresponding value in @values.
 *
 * Retuns: the row number, or -1 if not found.
 */
static gint 
find_col_for_values (GnomeDbMatrix *matrix, GSList *cols, GList *values, GnomeDbResultSet *rs)
{
	/* FIXME: use hash tables on GdaValues for better performances ? */
	gint cr = 0;
	gint count = gda_data_model_get_n_rows (rs);
	gint retval = -1;

	g_return_val_if_fail (cols && values, -1);
	g_return_val_if_fail (g_slist_length (cols) == g_list_length (values), -1);

	while ((cr < count) && (retval == -1)) {
		GSList *list1;
		GList *list2;
		gboolean equal = TRUE;

		list1 = cols;
		list2 = values;
		while (list1 && equal) {
			equal = m_gda_value_is_equal ((GdaValue *) list2->data, 
						      gda_data_model_get_value_at (rs, GPOINTER_TO_INT (list1->data), cr));
			list1 = g_slist_next (list1);
			list2 = g_list_next (list2);
		}

		if (equal)
			retval = cr;

		cr++;
	}

	return retval;
}

/*
 * Real initialization
 */
static GtkWidget *view0_init (GnomeDbMatrix *matrix, ViewData *vd);
static GtkWidget *view1_init (GnomeDbMatrix *matrix, ViewData *vd);
static GtkWidget *view2_init (GnomeDbMatrix *matrix, ViewData *vd);

static void       view0_update (GnomeDbMatrix *matrix, ViewData *vd);
static void       view1_update (GnomeDbMatrix *matrix, ViewData *vd);
static void       view2_update (GnomeDbMatrix *matrix, ViewData *vd);

static void       view1_clean (GnomeDbMatrix *matrix, ViewData *vd);
static void       view2_clean (GnomeDbMatrix *matrix, ViewData *vd);

static void 
gnome_db_matrix_initialize (GnomeDbMatrix *matrix)
{
	GtkWidget *wid, *nb, *vbox;
	ViewData *vd;

	/* title */
	matrix->priv->title = gnome_db_gray_bar_new (_("No title"));
        gtk_box_pack_start (GTK_BOX (matrix), matrix->priv->title, FALSE, TRUE, 2);
	gtk_widget_show (matrix->priv->title);

	/* main notebook */
	nb = gtk_notebook_new ();
	matrix->priv->notebook = nb;
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (matrix), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);

	/* the "No Data" notice */
	vbox = gtk_vbox_new (FALSE, 6);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox, NULL);
	gtk_widget_show (vbox);

	wid = gtk_label_new (_("No data to be displayed"));
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);

	/* structure for the TABULAR_SYNTHETIC view */
	vd = g_new0 (ViewData, 1);
	vd->matrix = matrix;
	vd->init_view = view0_init;
	vd->update_view = view0_update;
	vd->clean_data = NULL;
	matrix->priv->view_data [GNOME_DB_MATRIX_TABULAR_SYNTHETIC] = vd;

	/* structure for the LIST_DETAILLED view */
	if (matrix->priv->modif_table_fields) {
		vd = g_new0 (ViewData, 1);
		vd->init_view = view1_init;
		vd->update_view = view1_update;
		vd->clean_data = view1_clean;
	}
	else {
		vd = g_new0 (ViewData, 1);
		vd->init_view = view2_init;
		vd->update_view = view2_update;
		vd->clean_data = view2_clean;
	}
	vd->matrix = matrix;
	matrix->priv->view_data [GNOME_DB_MATRIX_LIST_DETAILLED] = vd;
	
	/* structure for the LIST_SYNTHETIC view */
	vd = g_new0 (ViewData, 1);
	vd->matrix = matrix;
	vd->init_view = view2_init;
	vd->update_view = view2_update;
	vd->clean_data = view2_clean;
	matrix->priv->view_data [GNOME_DB_MATRIX_LIST_SYNTHETIC] = vd;

	/* attributes */
	if (gnome_db_base_get_name (GNOME_DB_BASE (matrix->priv->query_select_contents)))
		gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (matrix->priv->title),
					    gnome_db_base_get_name (GNOME_DB_BASE (matrix->priv->query_select_contents)));
	else
		gtk_widget_hide (matrix->priv->title);
	
	/* tooltips */
	matrix->priv->tooltips = gtk_tooltips_new ();

	/* the actions part */
	wid = modif_buttons_make (matrix);
	gtk_box_pack_start (GTK_BOX (matrix), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	/* REM: no actual view is initialized here */
}



/* structure attached to the dialog when the association has some data */
typedef struct {
	gboolean     assoc_enabled;
	AssocChunck *ac; /* NOT NULL */
	AssocData   *ad; /* can be NULL */
	gint         cols_row;
	GtkTreePath *path;
	GtkTreeIter  iter;
} DlgPayload;

static void allviews_assoc_button_toggled_cb (GtkToggleButton *button, GtkDialog *dlg);

/* Returns the new status of the association (enabled or not) */
static gboolean
allviews_data_cell_toggled (GtkTreePath *path, gint rows_row, gint cols_row, GnomeDbMatrix *matrix)
{
	GtkTreeIter iter;
	gboolean assoc_enabled = FALSE;
	AssocData *ad = NULL;
	AssocChunck *ac;
	GSList *list;
	GtkWidget *dlg = NULL;
	GtkWidget *form = NULL;

	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);

	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == cols_row)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	if (ad) 
		assoc_enabled = assoc_data_is_enabled (ad);

	if (matrix->priv->modif_table_fields) {
		GtkWidget *vbox, *table, *label, *hbox, *button;
		gchar *str;

		dlg = gtk_dialog_new_with_buttons (_("Association properties"), NULL,
						   GTK_DIALOG_MODAL,
						   GTK_STOCK_OK,
						   GTK_RESPONSE_ACCEPT,
						   GTK_STOCK_CANCEL,
						   GTK_RESPONSE_REJECT,
						   NULL);
		vbox = GTK_DIALOG (dlg)->vbox;
		gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
		gtk_box_set_spacing (GTK_BOX (vbox), 5);

		label = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (label), _("<b>Association:</b>"));
		gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
		gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
		gtk_widget_show (label);

		hbox = gtk_hbox_new (FALSE, 0); /* HIG */
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
		gtk_widget_show (hbox);
		label = gtk_label_new ("    ");
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_widget_show (label);

		table = gtk_table_new (3, 3, FALSE);
		gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);
		gtk_widget_show (table);
		gtk_container_set_border_width (GTK_CONTAINER (table), 5);
		gtk_table_set_row_spacings (GTK_TABLE (table), 5);

		label = gtk_label_new (_("Association between:"));
		gtk_table_attach (GTK_TABLE (table), label, 0, 3, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
		gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
		gtk_widget_show (label);

		str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_rows, matrix->priv->query_select_rows, rows_row, "\n");
		label = gtk_label_new (str);
		g_free (str);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
		gtk_widget_show (label);

		label = gtk_label_new (_("and"));
		gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2, 0, 0, 0, 0);
		gtk_widget_show (label);

		str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_cols, matrix->priv->query_select_cols, cols_row, "\n");
		label = gtk_label_new (str);
		g_free (str);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 2, 3, 1, 2);
		gtk_widget_show (label);

		if (! matrix->priv->assoc_data_only) {
			button = gtk_toggle_button_new ();
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), assoc_enabled);
			gtk_button_set_label (GTK_BUTTON (button), assoc_enabled ? _("Enabled") : _("Disabled"));
			gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 3, 2, 3);
			gtk_widget_show (button);
			g_signal_connect (G_OBJECT (button), "toggled",
					  G_CALLBACK (allviews_assoc_button_toggled_cb), dlg);
			g_object_set_data (G_OBJECT (dlg), "toggle", button);
		}
		else
			g_object_set_data (G_OBJECT (dlg), "toggle", NULL);

		label = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (label), _("<b>Association's data:</b>"));
		gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
		gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
		gtk_widget_show (label);

		g_object_set_data (G_OBJECT (dlg), "data_title", label);
		gtk_widget_set_sensitive (label, assoc_enabled);

		hbox = gtk_hbox_new (FALSE, 0); /* HIG */
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
		gtk_widget_show (hbox);
		label = gtk_label_new ("    ");
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_widget_show (label);

		form = gnome_db_basic_form_new (matrix->priv->dict, matrix->priv->work_context);
		
		gtk_box_pack_start (GTK_BOX (hbox), form, TRUE, TRUE, 0);
		gtk_widget_show (form);

		g_object_set_data (G_OBJECT (dlg), "data_form", form);
		gtk_widget_set_sensitive (form, assoc_enabled);


		/*
		 * hiding the data entries for the condition parameters
		 */
		list = matrix->priv->rows_cond_params;
		while (list) {
			gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (form), GNOME_DB_PARAMETER (list->data), FALSE);
			list = g_slist_next (list);
		}
		list = matrix->priv->cols_cond_params;
		while (list) {
			gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (form), GNOME_DB_PARAMETER (list->data), FALSE);
			list = g_slist_next (list);
		}
	}

	/* fill the parameters with the correct values */
	set_params_from_assoc_data (matrix, rows_row, cols_row, ad);
	
	/* computing the association's new status */
	if (dlg) {
		gint response;

		gnome_db_basic_form_set_entries_auto_default (GNOME_DB_BASIC_FORM (form), TRUE);
		gtk_widget_show (dlg);
		response = gtk_dialog_run (GTK_DIALOG (dlg));
		
		if (response == GTK_RESPONSE_ACCEPT) {
			GtkWidget *toggle = g_object_get_data (G_OBJECT (dlg), "toggle");
			if (toggle)
				assoc_enabled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle));
			gtk_widget_destroy (dlg);
		}
		else {
			gtk_widget_destroy (dlg);
			return assoc_enabled;
		        /* END HERE, nothing more to do */
		}
	}
	else {
		if (!matrix->priv->assoc_data_only)
			assoc_enabled = !assoc_enabled; /* we have a toggle, so change the status of association */
	}

	/* Modifications on AssocData */
	if (assoc_enabled) {
		if (ad) 
			assoc_data_enable (ad);
		else 
			ad = assoc_data_create (matrix, ac, cols_row);
	}
	else
		if (ad)
			ad = assoc_data_disable (ad);

	if (ad && matrix->priv->modif_table_fields) {
		/* Copying values in work_context's parameters back to ad->data */
		GList *values = NULL;
		GSList *list;
		GnomeDbParameter *param;
		
		list = matrix->priv->modif_table_fields;
		while (list) {
			param = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			values = g_list_append (values, gnome_db_parameter_get_value (param));
			list = g_slist_next (list);
		}
		if (ad->data)
			gda_value_free (ad->data);
		ad->data = gda_value_new_list (values);
		g_list_free (values);
	}

	if (ad)
		ad->data_valid = gnome_db_data_set_is_valid (matrix->priv->work_context);

	gtk_tree_model_row_changed (matrix->priv->model, path, &iter);

#ifdef debug_NO
	dump_model (matrix);
#endif
	/* commit right now ? */
	if (matrix->priv->mode & GNOME_DB_ACTION_MODIF_AUTO_COMMIT) {
		model_commit_changes (matrix);
		model_update_complete (matrix, FALSE);
	}

	/* updating the various possible actions */
 	modif_buttons_update (matrix);

	return assoc_enabled;
}

static void
allviews_assoc_button_toggled_cb (GtkToggleButton *button, GtkDialog *dlg)
{
	gboolean state = gtk_toggle_button_get_active (button);
	GtkWidget *wid;

	gtk_button_set_label (GTK_BUTTON (button), state ? _("Enabled") : _("Disabled"));
	wid = g_object_get_data (G_OBJECT (dlg), "data_form");
	gtk_widget_set_sensitive (wid, state);
	wid = g_object_get_data (G_OBJECT (dlg), "data_title");
	gtk_widget_set_sensitive (wid, state);
}






/*
 *
 * View 0
 *
 */

static GtkWidget *
view0_init (GnomeDbMatrix *matrix, ViewData *vd)
{
	GtkWidget *sw, *tv;
	GtkTreeSelection *sel;

	/* GtkTreeView widget itself */
	sw = gtk_scrolled_window_new (NULL, NULL);
	tv = gtk_tree_view_new_with_model (matrix->priv->model);
	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));
	gtk_tree_selection_set_mode (sel, GTK_SELECTION_NONE);
	gtk_widget_show (tv);

	gtk_container_add (GTK_CONTAINER (sw), tv);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	vd->private = tv;

	return sw;
}

static void view0_1stcol_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
					 GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbMatrix *matrix);
static void view0_cols_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
				       GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbMatrix *matrix);
static void view0_data_cell_toggled (GtkCellRenderer *cell, const gchar *path_string, GnomeDbMatrix *matrix);
static void
view0_update (GnomeDbMatrix *matrix, ViewData *vd)
{
	gboolean init_done = FALSE;
	GtkTreeView *treeview = GTK_TREE_VIEW (vd->private);
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;
	gint cr, count;

	init_done = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (treeview), "init_done"));
	if (init_done)
		return;
	g_object_set_data (G_OBJECT (treeview), "init_done", GINT_TO_POINTER (TRUE));

	/* first column: for the values of priv->rs_rows as a textual value */
	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ("", renderer, NULL);
	gtk_tree_view_column_set_cell_data_func (col, renderer, (GtkTreeCellDataFunc) view0_1stcol_cell_data_func, 
						 matrix, NULL);
	gtk_tree_view_append_column (treeview, col);
	
	/* other columns */
	count = gda_data_model_get_n_rows (matrix->priv->rs_cols);
	for (cr = 0; cr < count; cr++) {
		gchar *title;

		title = gnome_db_result_set_get_row_as_string (matrix->priv->rs_cols, matrix->priv->query_select_cols, cr, "\n");

		renderer = gtk_cell_renderer_toggle_new ();
		g_object_set_data (G_OBJECT (renderer), "cols_row", GINT_TO_POINTER (cr));
		g_signal_connect (G_OBJECT (renderer), "toggled",
				  G_CALLBACK (view0_data_cell_toggled), matrix);
		col = gtk_tree_view_column_new_with_attributes (title, renderer, NULL);
		g_free (title);

		g_object_set_data (G_OBJECT (col), "cols_row", GINT_TO_POINTER (cr));
		gtk_tree_view_column_set_cell_data_func (col, renderer, (GtkTreeCellDataFunc) view0_cols_cell_data_func, 
							 matrix, NULL);
		gtk_tree_view_append_column (treeview, col);
	}
}

/*
 * renderer is a GtkCellRendererText
 */
static void
view0_1stcol_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
			     GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbMatrix *matrix)
{
	gchar *str;
	
	if (!matrix->priv->rs_rows)
		str = g_strdup ("???");
	else {
		gint row;
		
		gtk_tree_model_get (matrix->priv->model, iter, COLUMN_ROW_NUM, &row, -1);
		str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_rows, matrix->priv->query_select_rows, row, "\n");
	}
	g_object_set (G_OBJECT (cell), "text", str, NULL);
	g_free (str);
}

/*
 * renderer is a GtkCellRendererToggle
 */
static void
view0_cols_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
			   GtkTreeModel *tree_model, GtkTreeIter *iter, GnomeDbMatrix *matrix)
{
	gint row;
	gboolean set = FALSE;
	gboolean changed = FALSE;
	gboolean valid = TRUE;
	AssocChunck *ac;
	AssocData *ad = NULL;
	GSList *list;
	
	row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (tree_column), "cols_row"));
	
	gtk_tree_model_get (matrix->priv->model, iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == row)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	
	if (ad) {
		set = assoc_data_is_enabled (ad);
		changed = assoc_data_has_been_changed (ad);
		valid = assoc_data_is_valid (ad);
	}

	g_object_set (G_OBJECT (cell), "active", set, 
		      "activatable", TRUE, 
		      "cell-background-set", FALSE,
		      NULL);
	if (!valid)
		g_object_set (G_OBJECT (cell), "cell-background", GNOME_DB_COLOR_NORMAL_INVALID, 
			      "cell-background-set", TRUE, NULL);
	else 
		if (changed)
			g_object_set (G_OBJECT (cell), "cell-background", GNOME_DB_COLOR_NORMAL_MODIF, 
				      "cell-background-set", TRUE, NULL);
}

static void
view0_data_cell_toggled (GtkCellRenderer *cell, const gchar *path_string, GnomeDbMatrix *matrix)
{
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	gint cols_row, rows_row;
	GtkTreeIter iter;

	cols_row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cell), "cols_row"));
	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);
	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ROW_NUM, &rows_row, -1);
	allviews_data_cell_toggled (path, rows_row, cols_row, matrix);
	gtk_tree_path_free (path);	
}







/*
 *
 * View 1
 *
 */

typedef struct {
	gulong     signal_id;
	GtkWidget *sw;
	GtkWidget *viewport;
	GtkWidget *vbox;
	GSList    *buttons;
	GSList    *forms;
	GSList    *contexts;
} View1ListPrivate;
#define VIEW1_LIST_PRIVATE(x) ((View1ListPrivate*)(x))

static void view1_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, ViewData *vd);
static GtkWidget *
view1_init (GnomeDbMatrix *matrix, ViewData *vd)
{
	GtkWidget *sw;

	/* GtkTreeView widget itself */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_NONE);
	vd->private = g_new0 (View1ListPrivate, 1);
	VIEW1_LIST_PRIVATE (vd->private)->sw = sw;

	/* catch the "row_changed" signal from the model */
	VIEW1_LIST_PRIVATE (vd->private)->signal_id = g_signal_connect (G_OBJECT (matrix->priv->model), "row_changed",
								       G_CALLBACK (view1_model_row_changed_cb), vd);

	return sw;
}

static void view1_check_button_toggled (GtkToggleButton *button, GnomeDbMatrix *matrix);
static void view1_update_load_from_model (GnomeDbMatrix *matrix, GtkWidget *button, GnomeDbBasicForm *form);
static void view1_form_param_changed (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean is_user_modif, GnomeDbMatrix *matrix);
static void
view1_update (GnomeDbMatrix *matrix, ViewData *vd)
{
	GtkWidget *viewport, *vbox, *hbox, *label, *table, *box, *wid, *tvbox, *form;
	GnomeDbDataSet *context;
	gint cols, rows;
	gint nbcols, nbrows;
	gchar *str, *header;
	gint tables_nb_rows;
	GSList *list;
	View1ListPrivate *vlp = VIEW1_LIST_PRIVATE (vd->private);
	GHashTable *repl = g_hash_table_new (NULL, NULL);

	viewport = vlp->viewport;
	if (!viewport) {
		viewport = gtk_viewport_new (NULL, NULL);
		gtk_viewport_set_shadow_type (GTK_VIEWPORT (viewport), GTK_SHADOW_NONE);
		gtk_container_add (GTK_CONTAINER (vlp->sw), viewport);
		gtk_widget_show (viewport);
		vlp->viewport = viewport;
	}

	vbox = vlp->vbox;
	if (vbox) {
		gtk_widget_destroy (vbox);
		g_slist_free (vlp->buttons);
		vlp->buttons = NULL;
	}

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (viewport), vbox);
	vlp->vbox = vbox;
	gtk_widget_show (vbox);

	nbrows = gda_data_model_get_n_rows (matrix->priv->rs_rows);
	nbcols = gda_data_model_get_n_rows (matrix->priv->rs_cols);

	for (cols = 0; cols < nbcols; cols ++) {
		gint i, j;
		/* header for the column if more than one column */
		if (nbcols != 1) {
			str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_cols, 
							      matrix->priv->query_select_cols, cols, " / ");
			header = g_strdup_printf ("<big><b>%s:</b></big>", str);
			g_free (str);
			label = gtk_label_new ("");
			gtk_label_set_markup (GTK_LABEL (label), header);
			g_free (header);

			gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
			gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
			gtk_widget_show (label);
			
			hbox = gtk_hbox_new (FALSE, 0); /* HIG */
			gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
			gtk_widget_show (hbox);
			label = gtk_label_new ("    ");
			gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
			gtk_widget_show (label);
			box = hbox;
		}
		else
			box = vbox;

		/* table to hold all the check buttons and forms */
		tables_nb_rows = nbrows / matrix->priv->gtk_table_view_nb_cols + 1;
		table = gtk_table_new (tables_nb_rows, matrix->priv->gtk_table_view_nb_cols, FALSE);
		gtk_table_set_col_spacings (GTK_TABLE (table), 5);
		gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
		gtk_widget_show (table);

		i = 0; j = 0;
		for (rows = 0; rows < nbrows; rows ++) {
			/* vbox for button and form */
			tvbox = gtk_vbox_new (FALSE, 2);
			gtk_table_attach_defaults (GTK_TABLE (table), tvbox, i, i+1, j, j+1);
			gtk_widget_show (tvbox);
			
			/* new context for that form */
			context = GNOME_DB_DATA_SET (gnome_db_data_set_new_copy (matrix->priv->work_context, repl));

			/* check button */
			str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_rows, 
							      matrix->priv->query_select_rows, rows, "\n");
			if (!matrix->priv->assoc_data_only) {
				wid = gtk_check_button_new_with_label (str);
				g_signal_connect (G_OBJECT (wid), "toggled",
						  G_CALLBACK (view1_check_button_toggled), matrix);
			}
			else
				wid = gtk_label_new (str);
			g_free (str);
			vlp->buttons = g_slist_append (vlp->buttons, wid);
			g_object_set_data (G_OBJECT (wid), "context", context);
			gtk_box_pack_start (GTK_BOX (tvbox), wid, FALSE, FALSE, 0);
			gtk_widget_show (wid);
			g_object_set_data (G_OBJECT (wid), "row_ref", GINT_TO_POINTER (rows));
			g_object_set_data (G_OBJECT (wid), "col_ref", GINT_TO_POINTER (cols));
			
			/* form */
			form = gnome_db_basic_form_new (matrix->priv->dict, context);
			gtk_box_pack_start (GTK_BOX (tvbox), form, TRUE, TRUE, 0);
			gtk_widget_show (form);
			vlp->contexts = g_slist_append (vlp->contexts, context);
			vlp->forms = g_slist_append (vlp->forms, form);
			g_object_set_data (G_OBJECT (form), "context", context);
			g_object_set_data (G_OBJECT (form), "row_ref", GINT_TO_POINTER (rows));
			g_object_set_data (G_OBJECT (form), "col_ref", GINT_TO_POINTER (cols));
			g_signal_connect (G_OBJECT (form), "param_changed",
					  G_CALLBACK (view1_form_param_changed), matrix);
			
			/* hiding the data entries for the condition parameters */
			list = matrix->priv->rows_cond_params;
			while (list) {
				gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (form), GNOME_DB_PARAMETER (g_hash_table_lookup (repl, list->data)), FALSE);
				list = g_slist_next (list);
			}
			list = matrix->priv->cols_cond_params;
			while (list) {
				gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (form), GNOME_DB_PARAMETER (g_hash_table_lookup(repl, list->data)), FALSE);
				list = g_slist_next (list);
			}

			view1_update_load_from_model (matrix, wid, GNOME_DB_BASIC_FORM (form));
			i++;
			if (i == matrix->priv->gtk_table_view_nb_cols) {
				j++;
				i = 0;
			}
		}
	}

	g_hash_table_destroy (repl);
}

static void
view1_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, ViewData *vd)
{
	GnomeDbMatrix *matrix = vd->matrix;
	GSList *list, *forms;
	gint iter_row_ref, button_row_ref;
	/* find the toggle buttons which refer to the row identified by iter */

	gtk_tree_model_get (matrix->priv->model, iter, COLUMN_ROW_NUM, &iter_row_ref, -1);
	list = VIEW1_LIST_PRIVATE (vd->private)->buttons;
	forms = VIEW1_LIST_PRIVATE (vd->private)->forms;
	while (list && forms) {
		button_row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (list->data), "row_ref"));
		if (button_row_ref == iter_row_ref)
			view1_update_load_from_model (matrix, list->data, GNOME_DB_BASIC_FORM (forms->data));
		list = g_slist_next (list);
		forms = g_slist_next (forms);
	}

	g_assert (!list && !forms);
}

static void
view1_check_button_toggled (GtkToggleButton *button, GnomeDbMatrix *matrix)
{
	GtkTreePath *path;
	gint row_ref, col_ref;
	gboolean assoc_enabled = FALSE;
	GtkTreeIter iter;
	AssocChunck *ac;
	AssocData *ad = NULL;
	GSList *list;
	GnomeDbDataSet *context = g_object_get_data (G_OBJECT (button), "context");

	/* create a GtkTreePath and get AssocChunck and AssocData */
	row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "row_ref"));
	col_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "col_ref"));

	path = gtk_tree_path_new_from_indices (row_ref, -1);
	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);
	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == col_ref)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	if (ad) 
		assoc_enabled = assoc_data_is_enabled (ad);

	if (assoc_enabled != !gtk_toggle_button_get_active (button))
		g_warning ("Toggle button not in sync with model!");

	assoc_enabled = gtk_toggle_button_get_active (button); /* new association's status */
	
	/* Modifications on AssocData */
	if (assoc_enabled) {
		if (ad) 
			assoc_data_enable (ad);
		else 
			ad = assoc_data_create (matrix, ac, col_ref);
	}
	else
		if (ad)
			ad = assoc_data_disable (ad);
	if (ad)
		ad->data_valid = gnome_db_data_set_is_valid (context);
	gtk_tree_model_row_changed (matrix->priv->model, path, &iter);
	gtk_tree_path_free (path);

	/* commit right now ? */
	if (matrix->priv->mode & GNOME_DB_ACTION_MODIF_AUTO_COMMIT) {
		model_commit_changes (matrix);
		model_update_complete (matrix, FALSE);
	}
	
	/* updating the various possible actions */
 	modif_buttons_update (matrix);
}

static void
view1_form_param_changed (GnomeDbBasicForm *form, GnomeDbParameter *param, gboolean is_user_modif, GnomeDbMatrix *matrix)
{
	GtkTreePath *path;
	gint row_ref, col_ref;
	GtkTreeIter iter;
	AssocChunck *ac;
	AssocData *ad = NULL;
	GSList *list;
	GnomeDbDataSet *context = g_object_get_data (G_OBJECT (form), "context");

	/* create a GtkTreePath and get AssocChunck and AssocData */
	row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (form), "row_ref"));
	col_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (form), "col_ref"));

	path = gtk_tree_path_new_from_indices (row_ref, -1);
	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);
	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == col_ref)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	
	if (ad) {
		/* Copying values in work_context's parameters back to ad->data */
		GList *values = NULL;
		GSList *list;
		GnomeDbParameter *work_param;
		gint index;

		index = g_slist_index (context->parameters, param);
		g_print ("Index: %d\n", index);
		work_param = g_slist_nth_data (matrix->priv->work_context->parameters, index);
		
		list = matrix->priv->modif_table_fields;
		while (list) {
			GnomeDbParameter *tmp = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			if (tmp == work_param)
				tmp = param;
			values = g_list_append (values, gnome_db_parameter_get_value (tmp));
			list = g_slist_next (list);
		}
		if (ad->data)
			gda_value_free (ad->data);
		ad->data = gda_value_new_list (values);
		g_list_free (values);

		ad->data_valid = gnome_db_data_set_is_valid (matrix->priv->work_context);
	}

	gtk_tree_model_row_changed (matrix->priv->model, path, &iter);
	gtk_tree_path_free (path);

	/* updating the various possible actions */
 	modif_buttons_update (matrix);
}

static void 
view1_update_load_from_model (GnomeDbMatrix *matrix, GtkWidget *button, GnomeDbBasicForm *form)
{
	GtkTreePath *path;
	gint row_ref, col_ref;
	AssocChunck *ac;
	AssocData *ad = NULL;
	GtkTreeIter iter;
	GSList *list;
	gboolean set = FALSE;
	gboolean changed = FALSE;
	gboolean valid = TRUE;
	GnomeDbDataSet *context;

	/* create a GtkTreePath and get AssocChunck and AssocData */
	row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "row_ref"));
	col_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "col_ref"));
	
	path = gtk_tree_path_new_from_indices (row_ref, -1);
	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);
	gtk_tree_path_free (path);
	
	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == col_ref)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	
	/* state computations */
	if (ad) {
		set = assoc_data_is_enabled (ad);
		changed = assoc_data_has_been_changed (ad);
		valid = assoc_data_is_valid (ad);
	}

	if (!matrix->priv->assoc_data_only) {
		g_signal_handlers_block_by_func (G_OBJECT (button),
						 G_CALLBACK (view1_check_button_toggled), matrix);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), set);
		/* FIXME: use the changed and valid attributes for visual feedback */
		g_signal_handlers_unblock_by_func (G_OBJECT (button),
						   G_CALLBACK (view1_check_button_toggled), matrix);
	}
	gtk_widget_set_sensitive (GTK_WIDGET (form), set);

	/* fill the form's parameters with the correct values */
	context = g_object_get_data (G_OBJECT (form), "context");
	if (ad && ad->data) {
		GList *values;
		GSList *list;
		GnomeDbParameter *param;
		gint index;
			
		values = gda_value_get_list (ad->data);
		list = matrix->priv->modif_table_fields;
		while (values && list) {
			param = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			index = g_slist_index (matrix->priv->work_context->parameters, param);
			param = g_slist_nth_data (context->parameters, index);

			gnome_db_parameter_set_value (param, (GdaValue *) values->data);
			values = g_list_next (values);
			list = g_slist_next (list);
		}
		g_assert (!values && !list);
	}
	else {
		GnomeDbParameter *param;
		gint index;

		/* set all the para values to NULL */
		GSList *list = matrix->priv->modif_table_fields;
		while (list) {
			param = g_object_get_data (G_OBJECT (list->data), "wcontext_param");
			index = g_slist_index (matrix->priv->work_context->parameters, param);
			param = g_slist_nth_data (context->parameters, index);

			gnome_db_parameter_set_value (param, NULL);
			list = g_slist_next (list);
		}
	}

	gnome_db_basic_form_set_entries_auto_default (form, TRUE);
}

static void
view1_clean (GnomeDbMatrix *matrix, ViewData *vd)
{
	GSList *list;
	View1ListPrivate *vlp = VIEW1_LIST_PRIVATE (vd->private);
	if (!vlp)
		return;

	g_signal_handler_disconnect (G_OBJECT (matrix->priv->model), vlp->signal_id);
	g_slist_free (vlp->buttons);
	g_slist_free (vlp->forms);
	list = vlp->contexts;
	while (list) {
		g_object_unref (G_OBJECT (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (vlp->contexts);
	g_free (vlp);
	vd->private = NULL;
}









/*
 *
 * View 2
 *
 */
typedef struct {
	gulong     signal_id;
	GtkWidget *sw;
	GtkWidget *viewport;
	GtkWidget *vbox;
	GSList    *buttons;
} View2ListPrivate;
#define VIEW2_LIST_PRIVATE(x) ((View2ListPrivate*)(x))

static void view2_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, ViewData *vd);
static GtkWidget *
view2_init (GnomeDbMatrix *matrix, ViewData *vd)
{
	GtkWidget *sw;

	/* GtkTreeView widget itself */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_NONE);
	vd->private = g_new0 (View2ListPrivate, 1);
	VIEW2_LIST_PRIVATE (vd->private)->sw = sw;

	/* catch the "row_changed" signal from the model */
	VIEW2_LIST_PRIVATE (vd->private)->signal_id = g_signal_connect (G_OBJECT (matrix->priv->model), "row_changed",
								       G_CALLBACK (view2_model_row_changed_cb), vd);

	return sw;
}

static void view2_check_button_toggled (GtkToggleButton *button, GnomeDbMatrix *matrix);
static void view2_update_toggle_button_from_model (GnomeDbMatrix *matrix, GtkToggleButton *button);
static void
view2_update (GnomeDbMatrix *matrix, ViewData *vd)
{
	GtkWidget *viewport, *vbox, *hbox, *label, *table, *box, *wid;
	gint cols, rows;
	gint nbcols, nbrows;
	gchar *str, *header;
	gint tables_nb_rows;
	View2ListPrivate *vlp = VIEW2_LIST_PRIVATE (vd->private);

	viewport = vlp->viewport;
	if (!viewport) {
		viewport = gtk_viewport_new (NULL, NULL);
		gtk_viewport_set_shadow_type (GTK_VIEWPORT (viewport), GTK_SHADOW_NONE);
		gtk_container_add (GTK_CONTAINER (vlp->sw), viewport);
		gtk_widget_show (viewport);
		vlp->viewport = viewport;
	}

	vbox = vlp->vbox;
	if (vbox) {
		gtk_widget_destroy (vbox);
		g_slist_free (vlp->buttons);
		vlp->buttons = NULL;
	}

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (viewport), vbox);
	vlp->vbox = vbox;
	gtk_widget_show (vbox);

	nbrows = gda_data_model_get_n_rows (matrix->priv->rs_rows);
	nbcols = gda_data_model_get_n_rows (matrix->priv->rs_cols);

	for (cols = 0; cols < nbcols; cols ++) {
		gint i, j;
		/* header for the column if more than one column */
		if (nbcols != 1) {
			str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_cols, 
							      matrix->priv->query_select_cols, cols, " / ");
			header = g_strdup_printf ("<b>%s:</b>", str);
			g_free (str);
			label = gtk_label_new ("");
			gtk_label_set_markup (GTK_LABEL (label), header);
			g_free (header);

			gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
			gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
			gtk_widget_show (label);
			
			hbox = gtk_hbox_new (FALSE, 0); /* HIG */
			gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
			gtk_widget_show (hbox);
			label = gtk_label_new ("    ");
			gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
			gtk_widget_show (label);
			box = hbox;
		}
		else
			box = vbox;

		/* table to hold all the check buttons */
		tables_nb_rows = nbrows / matrix->priv->gtk_table_view_nb_cols + 1;
		table = gtk_table_new (tables_nb_rows, matrix->priv->gtk_table_view_nb_cols, FALSE);
		gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
		gtk_widget_show (table);

		i = 0; j = 0;
		for (rows = 0; rows < nbrows; rows ++) {
			str = gnome_db_result_set_get_row_as_string (matrix->priv->rs_rows, 
							      matrix->priv->query_select_rows, rows, "\n");
			wid = gtk_check_button_new_with_label (str);
			vlp->buttons = g_slist_append (vlp->buttons, wid);
			g_object_set_data (G_OBJECT (wid), "row_ref", GINT_TO_POINTER (rows));
			g_object_set_data (G_OBJECT (wid), "col_ref", GINT_TO_POINTER (cols));
			g_free (str);
			g_signal_connect (G_OBJECT (wid), "toggled",
					  G_CALLBACK (view2_check_button_toggled), matrix);
			gtk_table_attach_defaults (GTK_TABLE (table), wid, i, i+1, j, j+1);
			gtk_widget_show (wid);
			view2_update_toggle_button_from_model (matrix, GTK_TOGGLE_BUTTON (wid));
			i++;
			if (i == matrix->priv->gtk_table_view_nb_cols) {
				j++;
				i = 0;
			}
		}
	}
}

static void
view2_model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, ViewData *vd)
{
	GnomeDbMatrix *matrix = vd->matrix;
	GSList *list;
	gint iter_row_ref, button_row_ref;
	/* find the toggle buttons which refer to the row identified by iter */
	
	gtk_tree_model_get (matrix->priv->model, iter, COLUMN_ROW_NUM, &iter_row_ref, -1);
	list = VIEW2_LIST_PRIVATE (vd->private)->buttons;
	while (list) {
		button_row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (list->data), "row_ref"));
		if (button_row_ref == iter_row_ref)
			view2_update_toggle_button_from_model (matrix, GTK_TOGGLE_BUTTON (list->data));
		list = g_slist_next (list);
	}
}

static void
view2_check_button_toggled (GtkToggleButton *button, GnomeDbMatrix *matrix)
{
	GtkTreePath *path;
	gint row_ref, col_ref;
	gboolean assoc_enabled;

	/* we don't want the toggle button to change its status now */
	g_signal_handlers_block_by_func (button, view2_check_button_toggled, matrix);
	gtk_toggle_button_set_active (button, !gtk_toggle_button_get_active (button));
	
	/* create a GtkTreePath and get AssocChunck and AssocData */
	row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "row_ref"));
	col_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "col_ref"));
	
	path = gtk_tree_path_new_from_indices (row_ref, -1);
	assoc_enabled = allviews_data_cell_toggled (path, row_ref, col_ref, matrix);
	gtk_tree_path_free (path);

	/* finaly set the correct button state */
	gtk_toggle_button_set_active (button, assoc_enabled);
	g_signal_handlers_unblock_by_func (button, view2_check_button_toggled, matrix);
}

static void 
view2_update_toggle_button_from_model (GnomeDbMatrix *matrix, GtkToggleButton *button)
{
	GtkTreePath *path;
	gint row_ref, col_ref;
	AssocChunck *ac;
	AssocData *ad = NULL;
	GtkTreeIter iter;
	GSList *list;
	gboolean set = FALSE;
	gboolean changed = FALSE;
	gboolean valid = TRUE;

	/* create a GtkTreePath and get AssocChunck and AssocData */
	row_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "row_ref"));
	col_ref = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "col_ref"));
	
	path = gtk_tree_path_new_from_indices (row_ref, -1);
	gtk_tree_model_get_iter (matrix->priv->model, &iter, path);
	gtk_tree_path_free (path);
	
	gtk_tree_model_get (matrix->priv->model, &iter, COLUMN_ASSOC_CHUNCK, &ac, -1);
	list = ac->assoc_data_list;
	while (list && !ad) {
		if (ASSOC_DATA (list->data)->cols_row == col_ref)
			ad = ASSOC_DATA (list->data);
		list = g_slist_next (list);
	}
	
	/* state computations */
	if (ad) {
		set = assoc_data_is_enabled (ad);
		changed = assoc_data_has_been_changed (ad);
		valid = assoc_data_is_valid (ad);
	}

	g_signal_handlers_block_by_func (G_OBJECT (button),
					 G_CALLBACK (view2_check_button_toggled), matrix);
	gtk_toggle_button_set_active (button, set);
	/* FIXME: use the changed and valid attributes for visual feedback */
	g_signal_handlers_unblock_by_func (G_OBJECT (button),
					   G_CALLBACK (view2_check_button_toggled), matrix);
}

static void
view2_clean (GnomeDbMatrix *matrix, ViewData *vd)
{
	View2ListPrivate *vlp = VIEW2_LIST_PRIVATE (vd->private);
	if (!vlp)
		return;

	g_signal_handler_disconnect (G_OBJECT (matrix->priv->model), vlp->signal_id);
	g_slist_free (vlp->buttons);
	g_free (vlp);
	vd->private = NULL;
}







/**
 * gnome_db_matrix_set_view_type
 * @matrix: a #GnomeDbMatrix widget
 * @type: a #GnomeDbMatrixType value
 *
 * Select which way the @matrix widget must present its interface
 */
void
gnome_db_matrix_set_view_type (GnomeDbMatrix *matrix, GnomeDbMatrixType type)
{
	g_return_if_fail (matrix && IS_GNOME_DB_MATRIX (matrix));
	g_return_if_fail (matrix->priv);

	if (matrix->priv->view_mode == type)
		return;
	matrix->priv->view_mode = type;

	/* set the correct page to display */
	if (matrix->priv->rs_rows && matrix->priv->rs_cols && matrix->priv->rs_contents) {
		if (matrix->priv->view_pages [type] == 0) {
			ViewData *vd = matrix->priv->view_data [type];
			GtkWidget *wid = (vd->init_view) (matrix, vd);
			(vd->update_view) (matrix, vd);
			gtk_widget_show (wid);
			matrix->priv->view_pages [type] = 
				gtk_notebook_append_page (GTK_NOTEBOOK (matrix->priv->notebook), wid, NULL);
		}
		gtk_notebook_set_current_page (GTK_NOTEBOOK (matrix->priv->notebook), 
					       matrix->priv->view_pages [type]);
	}
}

/*
 *
 * Modification buttons (Commit changes, Reset matrix)
 *
 */

static void action_commit_cb (GtkAction *action, GnomeDbMatrix *matrix);
static void action_reset_cb (GtkAction *action, GnomeDbMatrix *matrix);

static GtkActionEntry ui_actions[] = {
	{ "WorkWidgetCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "WorkWidgetReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)}
};

static const gchar *ui_actions_info =
"<ui>"
"  <toolbar name='ToolBar'>"
"    <toolitem action='WorkWidgetCommit'/>"
"    <toolitem action='WorkWidgetReset'/>"
"  </toolbar>"
"</ui>";


static GtkWidget *
modif_buttons_make (GnomeDbMatrix *matrix)
{
	GtkActionGroup *actions;
	GtkUIManager *ui;

	actions = gtk_action_group_new ("Actions");
	matrix->priv->actions_group = actions;

	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), matrix);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, actions, 0);
	gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);
	matrix->priv->uimanager = ui;
	matrix->priv->modif_all = gtk_ui_manager_get_widget (ui, "/ToolBar");

	return matrix->priv->modif_all;
}

static void
action_commit_cb (GtkAction *action, GnomeDbMatrix *matrix)
{
	model_commit_changes (matrix);
}

static void
action_reset_cb (GtkAction *action, GnomeDbMatrix *matrix)
{
	model_update_complete (matrix, FALSE);
}


static void
modif_buttons_update (GnomeDbMatrix *matrix)
{
	gboolean changed;
	GtkAction *action;

	changed = gnome_db_matrix_has_been_changed (GNOME_DB_DATA_WIDGET (matrix));
	action = gtk_ui_manager_get_action (matrix->priv->uimanager, "/ToolBar/WorkWidgetCommit");
	g_object_set (G_OBJECT (action), "sensitive", changed, NULL);
	action = gtk_ui_manager_get_action (matrix->priv->uimanager, "/ToolBar/WorkWidgetReset");
	g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);
}

/*
 * writes the changes made to the model back to the database
 */
static gboolean do_run_query (GnomeDbMatrix *matrix, GnomeDbQuery *query);
static void
model_commit_changes (GnomeDbMatrix *matrix)
{
	GSList *list;
	list = matrix->priv->assoc_chuncks;
	while (list) {
		AssocChunck *ac = ASSOC_CHUNCK (list->data);
		GSList *adlist = ac->assoc_data_list;
		GSList *newadlist = g_slist_copy (ac->assoc_data_list);
		
		while (adlist) {
			GnomeDbQuery *query = NULL;
			AssocData *ad = ASSOC_DATA (adlist->data);
			
			if (assoc_data_is_valid (ad)) {
				if (ad->contents_row < 0) {
					if (ad->req_ins_del)
						query = matrix->priv->query_insert;
				}
				else {
					if (ad->req_ins_del)
						query = matrix->priv->query_delete;
					else {
						if (assoc_data_has_been_changed (ad))
							query = matrix->priv->query_update;
					}
				}
				
				if (query) {
					set_params_from_assoc_data (matrix, ac->rows_row, ad->cols_row, ad);
					if (do_run_query (matrix, query)) {
						/* remove the AssocData */
						newadlist = g_slist_remove (newadlist, ad);
						assoc_data_free (ad);
					}
				}
			}
			
			adlist = g_slist_next (adlist);
		}
		
		g_slist_free (ac->assoc_data_list);
		ac->assoc_data_list = newadlist;
		
		list = g_slist_next (list);
	}
	model_update_complete (matrix, TRUE);
}

static gboolean
do_run_query (GnomeDbMatrix *matrix, GnomeDbQuery *query)
{
	return utility_query_execute_modif (query, matrix->priv->work_context,
					    matrix->priv->mode & GNOME_DB_ACTION_ASK_CONFIRM_INSERT,
					    matrix->priv->mode & GNOME_DB_ACTION_ASK_CONFIRM_UPDATE,
					    matrix->priv->mode & GNOME_DB_ACTION_ASK_CONFIRM_DELETE,
					    GTK_WIDGET (matrix), NULL, NULL);
}



/*
 * GnomeDbDataWidget interface implementation
 */
static void
gnome_db_matrix_run (GnomeDbDataWidget *iface, guint mode)
{
	GnomeDbMatrix *matrix;

	g_return_if_fail (iface && IS_GNOME_DB_MATRIX (iface));
	matrix = GNOME_DB_MATRIX (iface);
	g_return_if_fail (matrix->priv);
	g_return_if_fail (matrix->priv->query_select_contents);
	
	/* REM : we don't check for missing parameters: the user must do it himself
	 * using the gnome_db_data_widget_get_exec_context() function and gnome_db_data_set_is_valid() */

	/*
	 * Signals connecting
	 */
	if (matrix->priv->args_context)
		g_signal_connect (G_OBJECT (matrix->priv->args_context), "changed",
				  G_CALLBACK (arg_param_changed_cb), matrix);
	matrix->priv->has_run = TRUE;

	/*
	 * Actual start
	 */
	if (mode)
		matrix->priv->mode = mode;
	arg_param_changed_cb (NULL, matrix);
}


static void
gnome_db_matrix_set_mode (GnomeDbDataWidget *iface, guint mode)
{
	GnomeDbMatrix *matrix;

	g_return_if_fail (iface && IS_GNOME_DB_MATRIX (iface));
	matrix = GNOME_DB_MATRIX (iface);
	g_return_if_fail (matrix->priv);

	matrix->priv->mode = mode;

	/* updating the various possible actions */
 	modif_buttons_update (matrix);
}

void
gnome_db_matrix_set_entry_editable (GnomeDbDataWidget *iface, GnomeDbQfield *field, gboolean editable)
{
	g_return_if_fail (iface && IS_GNOME_DB_MATRIX (iface));
	g_warning ("The gnome_db_data_widget_entry_set_editable() method is not available for this widget class!");
}


static void
gnome_db_matrix_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions)
{
	GnomeDbMatrix *matrix;

	g_return_if_fail (iface && IS_GNOME_DB_MATRIX (iface));
	matrix = GNOME_DB_MATRIX (iface);
	g_return_if_fail (matrix->priv);

	if (show_actions)
		gtk_widget_show (matrix->priv->modif_all);
	else
		gtk_widget_hide (matrix->priv->modif_all);
}

static GnomeDbParameter *
gnome_db_matrix_get_param_for_field (GnomeDbDataWidget *iface, GnomeDbQfield *field, const gchar *field_name, gboolean in_exec_context)
{
	GnomeDbMatrix *matrix;
	GnomeDbParameter *param = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_MATRIX (iface), NULL);
	matrix = GNOME_DB_MATRIX (iface);
	g_return_val_if_fail (matrix->priv, NULL);
	g_return_val_if_fail (field || (field_name && *field_name), NULL);
	
	if (!in_exec_context)
		return NULL;

	if (field) {
		/* use the 'field' argument */
		gpointer q_sel_field;
		g_return_val_if_fail (field && IS_GNOME_DB_QFIELD (field), NULL);
			
		q_sel_field = g_hash_table_lookup (matrix->priv->replacements, field);
		g_return_val_if_fail (q_sel_field, NULL);
		
		param = gnome_db_data_set_find_parameter_for_field (matrix->priv->args_context, GNOME_DB_QFIELD (q_sel_field));
		if (!param)
			param = gnome_db_data_set_find_parameter_for_field (matrix->priv->args_context, field);
	}
	else {
		/* use the 'field_name' argument */
		GnomeDbQfield *f;

		f = (GnomeDbQfield*) gnome_db_entity_get_field_by_name (GNOME_DB_ENTITY (matrix->priv->query_select_rows), field_name);
		if (!f)
			f = (GnomeDbQfield *) gnome_db_entity_get_field_by_name (GNOME_DB_ENTITY (matrix->priv->query_select_cols), field_name);

		if (f) 
			param = gnome_db_data_set_find_parameter_for_field (matrix->priv->args_context, f);
	}

	return param;
}

static gboolean
gnome_db_matrix_has_been_changed (GnomeDbDataWidget *iface)
{
	GnomeDbMatrix *matrix;
	GSList *list;
	gboolean changed = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_MATRIX (iface), FALSE);
	matrix = GNOME_DB_MATRIX (iface);
	g_return_val_if_fail (matrix->priv, FALSE);

	list = matrix->priv->assoc_chuncks;
	while (list && !changed) {
		GSList *assoc_list = ASSOC_CHUNCK (list->data)->assoc_data_list;
		
		while (assoc_list && !changed) {
			changed = assoc_data_has_been_changed (ASSOC_DATA (assoc_list->data));
			assoc_list = g_slist_next (assoc_list);
		}

		list = g_slist_next (list);
	}

	return changed;
}

static GnomeDbDataSet *
gnome_db_matrix_get_exec_context (GnomeDbDataWidget *iface)
{
	GnomeDbMatrix *matrix;

	g_return_val_if_fail (iface && IS_GNOME_DB_MATRIX (iface), NULL);
	matrix = GNOME_DB_MATRIX (iface);
	g_return_val_if_fail (matrix->priv, NULL);
	
	return matrix->priv->args_context;
}

static GtkActionGroup *
gnome_db_matrix_get_actions_group (GnomeDbDataWidget *iface)
{
	GnomeDbMatrix *matrix;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_MATRIX (iface), NULL);
	matrix = GNOME_DB_MATRIX (iface);
	g_return_val_if_fail (matrix->priv, NULL);

	return matrix->priv->actions_group;
}


/*
 * Functions to work on AssocData structures
 */

static AssocData *
assoc_data_create (GnomeDbMatrix *matrix, AssocChunck *ac, gint cols_row)
{
	AssocData *ad;
	GList *values;
	
	g_return_val_if_fail (ac, NULL);

	values = build_values_list (matrix->priv->rs_cols, cols_row, matrix->priv->cols_pk_no);
	/* create a new AssocData */
	ad = g_new0 (AssocData, 1);
	ad->ac = ac;
	ad->cols_row = cols_row;
	ad->pk_value = gda_value_new_list (values);
	g_list_free (values);
	ad->req_ins_del = TRUE;
	ad->contents_row = -1;
	ad->data = NULL; 
	ad->data_orig = NULL;
	ac->assoc_data_list = g_slist_append (ac->assoc_data_list, ad);
	
	return ad;
}

static void
assoc_data_free (AssocData *ad)
{
	if (ad->data)
		gda_value_free (ad->data);
	if (ad->data_orig)
		gda_value_free (ad->data_orig);
	if (ad->pk_value)
		gda_value_free (ad->pk_value);
	g_free (ad);
}

static gboolean
assoc_data_is_enabled (AssocData *ad)
{
	if (((ad->contents_row >= 0) && !ad->req_ins_del) ||
	    ((ad->contents_row < 0) && ad->req_ins_del))
		return TRUE;
	else
		return FALSE;
}

static void
assoc_data_enable (AssocData *ad)
{
	if (assoc_data_is_enabled (ad))
		return;

	g_return_if_fail (ad->contents_row >= 0);

	/* set not to be deleted */
	ad->req_ins_del = FALSE;
}

/*
 * Returns: NULL if @ad has been removed, or @ad otherwise.
 */
static AssocData *
assoc_data_disable (AssocData *ad)
{
	if (! assoc_data_is_enabled (ad))
		return ad;

	if (ad->contents_row >= 0) 
		/* set to be deleted */
		ad->req_ins_del = TRUE;
	else {
		/* remove from AssocChunck */
		AssocChunck *ac = ad->ac;
		ac->assoc_data_list = g_slist_remove (ac->assoc_data_list, ad);
		assoc_data_free (ad);
		ad = NULL;
	}

	return ad;
}

static gboolean
assoc_data_has_been_changed (AssocData *ad)
{
	gboolean changed = FALSE;

	if (ad->req_ins_del)
		changed = TRUE;

	if (! changed) {
		if (ad->data_orig)
			changed = gda_value_compare (ad->data, ad->data_orig);
	}

	return changed;
}

static gboolean
assoc_data_is_valid (AssocData *ad)
{
	return ad->data_valid;
}
