/* gnome-db-data-set.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-data-set.h"
#include "gnome-db-query.h"
#include "marshal.h"
#include <string.h>
#include "gnome-db-data-model.h"
#include "gnome-db-result-set.h"
#include "gnome-db-parameter.h"
#include "gnome-db-field.h"
#include "gnome-db-referer.h"
#include "gnome-db-entity.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-qfield.h"
#include "gnome-db-server.h"
#include "gnome-db-model-wrapper.h"

/* 
 * Main static functions 
 */
static void gnome_db_data_set_class_init (GnomeDbDataSetClass * class);
static void gnome_db_data_set_init (GnomeDbDataSet * srv);
static void gnome_db_data_set_dispose (GObject   * object);
static void gnome_db_data_set_finalize (GObject   * object);

static void gnome_db_data_set_set_property (GObject              *object,
					    guint                 param_id,
					    const GValue         *value,
					    GParamSpec           *pspec);
static void gnome_db_data_set_get_property (GObject              *object,
					    guint                 param_id,
					    GValue               *value,
					    GParamSpec           *pspec);

static void gnome_db_data_set_free_node (GnomeDbDataSet *dataset, GnomeDbDataSetNode *node);
static void nullified_param_cb (GnomeDbParameter *param, GnomeDbDataSet *dataset);
static void changed_param_cb (GnomeDbParameter *param, GnomeDbDataSet *dataset);

#ifdef debug
static void gnome_db_data_set_dump                     (GnomeDbDataSet *dataset, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	PARAM_CHANGED,
	LAST_SIGNAL
};

static gint gnome_db_data_set_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbDataSetPrivate
{
	GHashTable *param_default_values;      /* key = param, value = new GdaValue */
	GHashTable *param_default_aliases;     /* key = param, value = alias parameter */
	GHashTable *aliases_default_param;     /* key = alias parameter, value = param */

	GHashTable *param_repl;                /* key = parameter, value = parameter used instead because of similarity */
};



/* module error */
GQuark gnome_db_data_set_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_data_set_error");
	return quark;
}


GType
gnome_db_data_set_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDataSetClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_data_set_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDataSet),
			0,
			(GInstanceInitFunc) gnome_db_data_set_init
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbDataSet", &info, 0);
	}
	return type;
}


static void
gnome_db_data_set_class_init (GnomeDbDataSetClass *class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_data_set_signals[PARAM_CHANGED] =
		g_signal_new ("param_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbDataSetClass, param_changed),
			      NULL, NULL,
			      marshal_VOID__OBJECT, G_TYPE_NONE, 1,
			      G_TYPE_OBJECT);

	class->param_changed = NULL;

	object_class->dispose = gnome_db_data_set_dispose;
	object_class->finalize = gnome_db_data_set_finalize;

	/* Properties */
	object_class->set_property = gnome_db_data_set_set_property;
	object_class->get_property = gnome_db_data_set_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_data_set_dump;
#endif
}

static void
gnome_db_data_set_init (GnomeDbDataSet *gnome_db_data_set)
{
	gnome_db_data_set->priv = g_new0 (GnomeDbDataSetPrivate, 1);
	gnome_db_data_set->parameters = NULL;
	gnome_db_data_set->nodes = NULL;
	gnome_db_data_set->hints = g_hash_table_new (NULL, NULL);

	gnome_db_data_set->priv->param_default_values = g_hash_table_new_full (NULL, NULL, 
									       NULL, (GDestroyNotify) gda_value_free);
	gnome_db_data_set->priv->param_default_aliases = g_hash_table_new (NULL, NULL);
	gnome_db_data_set->priv->aliases_default_param = g_hash_table_new (NULL, NULL);

	gnome_db_data_set->priv->param_repl = g_hash_table_new (NULL, NULL);
}

static void compute_dataset_nodes (GnomeDbDataSet *dataset);
static void gnome_db_data_set_real_add_param (GnomeDbDataSet *dataset, GnomeDbParameter *param);

/**
 * gnome_db_data_set_new
 * @dict: a #GnomeDbDict object
 * @params: a list of #GnomeDbParameter objects
 *
 * Creates a new #GnomeDbDataSet object, and populates it with the list given as argument.
 * The list can then be freed as it gets copied. All the parameters in @params are referenced counted
 * and modified, so they should not be used anymore afterwards, and the @params list gets copied
 * (so it should be freed by the caller).
 *
 * Returns: a new #GnomeDbDataSet object
 */
GObject *
gnome_db_data_set_new (GnomeDbDict *dict, GSList *params)
{
	GObject *obj;
	GnomeDbDataSet *gnome_db_data_set;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_DATA_SET_TYPE, "dict", ASSERT_DICT (dict), NULL);
        gnome_db_data_set = GNOME_DB_DATA_SET (obj);
        gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_data_set), 0);

	/* add the parameters */
	while (params) {
		gnome_db_data_set_real_add_param (gnome_db_data_set, GNOME_DB_PARAMETER (params->data));
		params = g_slist_next (params);
	}
	compute_dataset_nodes (gnome_db_data_set);

	return obj;
}

typedef struct {
	GHashTable *lookup_hash;
	GHashTable *dest_hash;
} ForeachStruct;

static void
copy_hash_foreach (gpointer key, gpointer value, ForeachStruct *fhs)
{
	if (fhs->lookup_hash)
		g_hash_table_insert (fhs->dest_hash, g_hash_table_lookup (fhs->lookup_hash, key), value);
	else
		g_hash_table_insert (fhs->dest_hash, key, value);
}

/**
 * gnome_db_data_set_new_copy
 * @orig: a #GnomeDbDataSet to make a copy of
 * @replacements: a place where to store the parameters and queries replacements, or %NULL
 * 
 * Copy constructor. @replacements is used and modified only if @copy_params is TRUE.
 *
 * Returns: a the new copy of @orig
 */
GObject *
gnome_db_data_set_new_copy (GnomeDbDataSet *orig, GHashTable *replacements)
{
	GObject   *obj;
	GnomeDbDataSet *gnome_db_data_set;
	GnomeDbDict *dict;
	GHashTable *repl;
	GSList *list;
	GnomeDbParameter *param;
	guint hint;

	g_return_val_if_fail (orig && IS_GNOME_DB_DATA_SET (orig), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	obj = g_object_new (GNOME_DB_DATA_SET_TYPE, "dict", dict, NULL);
	gnome_db_data_set = GNOME_DB_DATA_SET (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_data_set), 0);


	if (replacements)
		repl = replacements;
	else
		repl = g_hash_table_new (NULL, NULL);
	
	/* copy parameters and hints */
	if (orig->parameters) {
		list = orig->parameters;
		while (list) {
			/* param copy */
			param = GNOME_DB_PARAMETER (gnome_db_parameter_new_copy (GNOME_DB_PARAMETER (list->data)));
			g_hash_table_insert (repl, list->data, param);
			gnome_db_data_set->parameters = g_slist_append (gnome_db_data_set->parameters, param);
			
			gnome_db_base_connect_nullify (param,
						       G_CALLBACK (nullified_param_cb), gnome_db_data_set);
			g_signal_connect (G_OBJECT (param), "changed",
					  G_CALLBACK (changed_param_cb), gnome_db_data_set);
			
			/* hint copy */
			hint = GPOINTER_TO_UINT (g_hash_table_lookup (orig->hints, list->data));
			if (hint)
				g_hash_table_insert (gnome_db_data_set->hints, param, GUINT_TO_POINTER (hint));
			
			list = g_slist_next (list);
		}
	}
	
	/* finish nodes copy */
	list = orig->nodes;
	while (list) {
		GnomeDbDataSetNode *node, *orig;
		
		orig = GNOME_DB_DATA_SET_NODE (list->data);
		node = g_new0 (GnomeDbDataSetNode, 1);
		
		node->param = g_hash_table_lookup (repl, orig->param);
		
		if (orig->params) {
			GSList *params = orig->params;
			while (params) {
				node->params = g_slist_append (node->params, 
							       g_hash_table_lookup (repl, params->data));
				params = g_slist_next (params);
			}
		}
		
		if (orig->pos_for_param) {
			ForeachStruct fhs;
			
			fhs.lookup_hash = repl;
			node->pos_for_param = g_hash_table_new (NULL, NULL);
			fhs.dest_hash = node->pos_for_param;
			g_hash_table_foreach (orig->pos_for_param, 
					      (GHFunc) copy_hash_foreach, &fhs);
		}

		if (orig->data_for_param) 
			node->data_for_param = GNOME_DB_DATA_MODEL (gnome_db_data_model_copy (orig->data_for_param, 
											      repl));
		
		gnome_db_data_set->nodes = g_slist_append (gnome_db_data_set->nodes, node);
		g_hash_table_insert (repl, orig, node);
		
		list = g_slist_next (list);
	}
	
	/* make the replacements be taken into account */
	list = gnome_db_data_set->parameters;
	while (list) {
		gnome_db_referer_replace_refs (GNOME_DB_REFERER (list->data), repl);
		list = g_slist_next (list);
	}
	
	if (!replacements)
		g_hash_table_destroy (repl);

#ifdef debug_NO
	g_print ("############ DataSet copy ORIG:\n");
	gnome_db_data_set_dump (orig, 3);
	g_print ("############ DataSet copy copy:\n");
	gnome_db_data_set_dump (gnome_db_data_set, 3);
	g_print ("############ DataSet copy end.\n");
#endif

	return obj;
}

/**
 * gnome_db_data_set_new_from_spec
 * @xml_spec: a string
 * @error: a place to store the error, or %NULL
 *
 * Creates a new #GnomeDbDataSet object from the @xml_spec
 * specifications
 *
 * Returns: a new object, or %NULL if an error occured
 */
GObject *
gnome_db_data_set_new_from_spec (GnomeDbDict *dict, const gchar *xml_spec, GError **error)
{
	GObject *obj = NULL;
	GSList *params = NULL, *sources = NULL;
	GSList *list;
	
	xmlDocPtr doc;
	xmlNodePtr root, cur;
	GnomeDbServer *srv;
	gboolean allok = TRUE;
	gchar *str;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	srv = gnome_db_dict_get_server (ASSERT_DICT (dict));

	/* string parsing */
	doc = xmlParseMemory (xml_spec, strlen (xml_spec));
	if (!doc)
		return NULL;
	root = xmlDocGetRootElement (doc);
	if (strcmp (root->name, "data-set-spec") != 0){
		g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
			     _("Spec's root node != 'data-set-spec': '%s'"), root->name);
		return NULL;
	}

	/* Making the list of parameters */
	root = root->xmlChildrenNode;
	while (xmlNodeIsText (root)) 
		root = root->next; 

	if (strcmp (root->name, "parameters") != 0){
		g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
			     _("Missing node <parameters>: '%s'"), root->name);
		return NULL;
	}

	for (cur = root->xmlChildrenNode; (cur != NULL) && allok; cur = cur->next) {
		if (xmlNodeIsText (cur)) 
			continue;

		if (!strcmp (cur->name, "parameter")) {
			GnomeDbParameter *param = NULL;
			GnomeDbServerDataType *dtype = NULL;
			gchar *str;
			gboolean dtype_created = FALSE;
			
			/* find data type and create GnomeDbParameter */
			str = xmlGetProp(cur, "dbmstype");
			if (str) {
				dtype = gnome_db_server_get_data_type_by_name (srv, str);
				g_free (str);
			}
			if (!dtype) {
				str = xmlGetProp (cur, "gdatype");
				if (str) {
					GdaValueType gtype;

					gtype = gda_type_from_string (str);
					if (gtype != GDA_VALUE_TYPE_UNKNOWN) {
						GnomeDbServerInfo *info;
						gint i;
						
						info = gnome_db_server_get_server_info (srv);
						if (info) {
							for (i = 0; (i < info->value_nb_tests_types) && (!dtype); i ++) {
								if (info->value_test_gda_types [i] == gtype)
									dtype = gnome_db_server_get_data_type_by_name (srv, 
											        info->value_test_data_types [i]);
							}
						}

						if (!dtype) {
							/* create a GnomeDbServerDataType for that 'gda-type' */
							dtype = GNOME_DB_SERVER_DATA_TYPE (gnome_db_server_data_type_new (srv));
							gnome_db_server_data_type_set_sqlname (dtype, str);
							gnome_db_server_data_type_set_gda_type (dtype, gtype);
							gnome_db_server_declare_custom_data_type (srv, dtype);
							dtype_created = TRUE;
						}
					}
					g_free (str);					
				}
			}

			if (!dtype) {
				str = xmlGetProp(cur, "name");
				
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
					     _("Can't find a data type for parameter '%s'"), str ? str : "unnamed");
				g_free (str);

				allok = FALSE;
				continue;
			}

			param = GNOME_DB_PARAMETER (gnome_db_parameter_new (dict, dtype));
			if (dtype_created)
				g_object_unref (dtype);
			params = g_slist_append (params, param);

			/* set parameter's attributes */
			str = xmlGetProp(cur, "id");
			if (str) {
				g_object_set (G_OBJECT (param), "string_id", str, NULL);
				g_free (str);
			}
			str = xmlGetProp(cur, "name");
			if (str) {
				gnome_db_base_set_name (GNOME_DB_BASE (param), str);
				g_free (str);
			}
			str = xmlGetProp(cur, "descr");
			if (str) {
				gnome_db_base_set_description (GNOME_DB_BASE (param), str);
				g_free (str);
			}
			str = xmlGetProp(cur, "null-ok");
			if (str) {
				gnome_db_parameter_set_not_null (param, (*str == 'T') || (*str == 't') ? FALSE : TRUE);
				g_free (str);
			}
			else
				gnome_db_parameter_set_not_null (param, FALSE);
			str = xmlGetProp(cur, "plugin");
			if (str) {
				g_object_set (G_OBJECT (param), "handler_plugin", str, NULL);
				g_free (str);
			}

			str = xmlGetProp(cur, "source");
			if (str) 
				g_object_set_data (G_OBJECT (param), "source", str);
		}
	}

	/* Parameters' sources, not mandatory */
	root = root->next;
	while (xmlNodeIsText (root)) 
		root = root->next; 
	if (allok && root && !strcmp (root->name, "sources")){
		for (cur = root->xmlChildrenNode; (cur != NULL) && allok; cur = cur->next) {
			if (xmlNodeIsText (cur)) 
				continue;

			if (!strcmp (cur->name, "data-array")) {
				GdaDataModel *model;

				str = xmlGetProp(cur, "name");
				model = gda_data_model_array_new_from_xml_node (cur, error);
				if (model) {
					GnomeDbModelWrapper *wrapper;

					wrapper = gnome_db_model_wrapper_new (dict, model);
					if (str)
						gnome_db_base_set_name (GNOME_DB_BASE (wrapper), str);
					g_object_unref (model);
					sources = g_slist_prepend (sources, wrapper);
				}
				else 
					allok = FALSE; /* error has already been set */
				g_free (str);
			}
		}
	}
	
	/* affecting parameters' sources fields to each parameter */
	list = params;
	while (list && allok) {
		str = g_object_get_data (G_OBJECT (list->data), "source");
		if (str) {
			gchar *ptr1, *ptr2=NULL, *tok;
			
			ptr1 = strtok_r (str, ":", &tok);
			if (ptr1)
				ptr2 = strtok_r (NULL, ":", &tok);
			
			if (ptr1 && ptr2) {
				GSList *tmp = sources;
				GnomeDbModelWrapper *wrapper = NULL;
				while (tmp && !wrapper) {
					if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (tmp->data)), ptr1))
						wrapper = GNOME_DB_MODEL_WRAPPER (tmp->data);
					tmp = g_slist_next (tmp);
				}

				if (wrapper) {
					gint fno;

					fno = atoi (ptr2);
					if ((fno <= 0) ||
					    (fno > gda_data_model_get_n_columns (GDA_DATA_MODEL (wrapper)))) {
						g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
							     _("Field number %d not found in source named '%s'"), fno, ptr1); 
						allok = FALSE;	
					}
					else {
						GnomeDbField *field;

						field = gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (wrapper), fno - 1);
						if (!gnome_db_parameter_set_source_field (GNOME_DB_PARAMETER (list->data),
											  field, error))
							allok = FALSE;
						else {
							/* rename the wrapper with the current param's name */
							g_object_set_data (G_OBJECT (wrapper), "newname", 
									   gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
							g_object_set_data (G_OBJECT (wrapper), "newdescr", 
									   gnome_db_base_get_description (GNOME_DB_BASE (list->data)));
						}
					}
				}
				else {
					g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
						     _("Can't find parameter source named '%s'"), ptr1); 
					allok = FALSE;
				}
			}
			else {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_XML_SPEC_ERROR,
					     _("Parameter source specification should be \"<source name>:<field pos>\": '%s'"), 
					     str);
				allok = FALSE;
			}

			g_object_set_data (G_OBJECT (list->data), "source", NULL);
			g_free (str);
		}
		list = g_slist_next (list);
	}

	/* setting prepared new names from sources (wrappers) */
	list = sources;
	while (list) {
		str = g_object_get_data (G_OBJECT (list->data), "newname");
		if (str) {
			gnome_db_base_set_name (GNOME_DB_BASE (list->data), str);
			g_object_set_data (G_OBJECT (list->data), "newname", NULL);
		}
		str = g_object_get_data (G_OBJECT (list->data), "newdescr");
		if (str) {
			gnome_db_base_set_description (GNOME_DB_BASE (list->data), str);
			g_object_set_data (G_OBJECT (list->data), "newdescr", NULL);
		}
		list = g_slist_next (list);
	}

	/* parameters' values, constraints: TODO */
	
	/* GnomeDbDataSet creation */
	if (allok) 
		obj = gnome_db_data_set_new (dict, params);

	g_slist_foreach (params, (GFunc) g_object_unref, NULL);
	g_slist_free (params);
	g_slist_foreach (sources, (GFunc) g_object_unref, NULL);
	g_slist_free (sources);

	xmlFreeDoc(doc);
	return obj;
}

/**
 * gnome_db_data_set_get_spec
 * @dataset: a #GnomeDbDataSet object
 *
 * Get the specification as an XML string. See the gnome_db_data_set_new_from_spec()
 * form more information about the XML specification string format.
 *
 * Returns: a new string
 */
gchar *
gnome_db_data_set_get_spec (GnomeDbDataSet *dataset)
{
	xmlDocPtr doc;
	xmlNodePtr root;
	xmlChar *xmlbuff;
	int buffersize;
	GSList *list;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), NULL);

	doc = xmlNewDoc ("1.0");
	g_return_val_if_fail (doc, NULL);
	root = xmlNewDocNode (doc, NULL, "data-set-spec", NULL);
	xmlDocSetRootElement (doc, root);

	/* parameters list */
	list = dataset->parameters;
	while (list) {
		xmlNodePtr node;
		GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
		GnomeDbServerDataType *dtype;
		gchar *str;

		node = xmlNewTextChild (root, NULL, "parameter", NULL);
		g_object_get (G_OBJECT (param), "string_id", &str, NULL);
		if (str) {
			xmlSetProp (node, "id", str);
			g_free (str);
		}
		str = gnome_db_base_get_name (GNOME_DB_BASE (param));
		if (str)
			xmlSetProp (node, "name", str);		
		str = gnome_db_base_get_description (GNOME_DB_BASE (param));
		if (str)
			xmlSetProp (node, "descr", str);		

		dtype = gnome_db_parameter_get_data_type (param);
		if (dtype) {
			xmlSetProp (node, "dbmstype", gnome_db_server_data_type_get_sqlname (dtype));
			xmlSetProp (node, "gdatype", gda_type_to_string (gnome_db_server_data_type_get_gda_type (dtype)));
		}

		xmlSetProp (node, "null-ok", gnome_db_parameter_get_not_null (param) ? "FALSE" : "TRUE");
		g_object_get (G_OBJECT (param), "handler_plugin", &str, NULL);
		if (str) {
			xmlSetProp (node, "plugin", str);
			g_free (str);
		}
		
		list = g_slist_next (list);
	}

	/* parameters' values, sources, constraints: TODO */

	xmlDocDumpFormatMemory(doc, &xmlbuff, &buffersize, 1);
	g_print ((gchar *) xmlbuff);
	
	xmlFreeDoc(doc);
	return (gchar *) xmlbuff;
}


static void
nullified_param_cb (GnomeDbParameter *param, GnomeDbDataSet *dataset)
{
	g_assert (g_slist_find (dataset->parameters, param));
	dataset->parameters = g_slist_remove (dataset->parameters, param);
	g_signal_handlers_disconnect_by_func (G_OBJECT (param),
					      G_CALLBACK (nullified_param_cb), dataset);
	g_signal_handlers_disconnect_by_func (G_OBJECT (param),
					      G_CALLBACK (changed_param_cb), dataset);

	if (dataset->nodes) {
		GnomeDbDataSetNode *node;
		gboolean finished = FALSE;
		GSList *list = dataset->nodes;

		while (list && !finished) {
			node = GNOME_DB_DATA_SET_NODE (list->data);
			if (node->params && g_slist_find (node->params, param)) {
				node->params = g_slist_remove (node->params, param);
				if (!node->params) {
					g_object_unref (node->data_for_param);
					dataset->nodes = g_slist_remove (dataset->nodes, node);
					g_free (node);
				}
				finished = TRUE;
			}
			list = g_slist_next (list);
		}
	}
	g_object_unref (G_OBJECT (param));
}

static void
changed_param_cb (GnomeDbParameter *param, GnomeDbDataSet *dataset)
{
	/* signal the parameter change */
	gnome_db_base_changed (GNOME_DB_BASE (dataset));
#ifdef debug_signal
	g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (dataset), gnome_db_data_set_signals[PARAM_CHANGED], 0, param);
#ifdef debug_signal
	g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
}

static void
gnome_db_data_set_dispose (GObject *object)
{
	GnomeDbDataSet *gnome_db_data_set;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATA_SET (object));

	gnome_db_data_set = GNOME_DB_DATA_SET (object);
	if (gnome_db_data_set->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
	}

	if (gnome_db_data_set->hints) {
		g_hash_table_destroy (gnome_db_data_set->hints);
		gnome_db_data_set->hints = NULL;
	}
	
	/* free the nodes if there are some */
	while (gnome_db_data_set->nodes)
		gnome_db_data_set_free_node (gnome_db_data_set, GNOME_DB_DATA_SET_NODE (gnome_db_data_set->nodes->data));

	/* free the parameters list */
	while (gnome_db_data_set->parameters)
		nullified_param_cb (GNOME_DB_PARAMETER (gnome_db_data_set->parameters->data), gnome_db_data_set);


	/* parent class */
	parent_class->dispose (object);
}

static void foreach_finalize_alias (GnomeDbParameter *param, GnomeDbParameter *alias, GnomeDbDataSet *dataset);
static void
gnome_db_data_set_finalize (GObject *object)
{
	GnomeDbDataSet *gnome_db_data_set;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATA_SET (object));

	gnome_db_data_set = GNOME_DB_DATA_SET (object);
	if (gnome_db_data_set->priv) {
		g_hash_table_destroy (gnome_db_data_set->priv->param_default_values);
		g_hash_table_foreach (gnome_db_data_set->priv->param_default_aliases,
				      (GHFunc) foreach_finalize_alias, gnome_db_data_set);
		g_hash_table_destroy (gnome_db_data_set->priv->param_default_aliases);
		g_hash_table_destroy (gnome_db_data_set->priv->aliases_default_param);
		g_hash_table_destroy (gnome_db_data_set->priv->param_repl);

		g_free (gnome_db_data_set->priv);
		gnome_db_data_set->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_data_set_set_property (GObject              *object,
				guint                 param_id,
				const GValue         *value,
				GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbDataSet *gnome_db_data_set;

	gnome_db_data_set = GNOME_DB_DATA_SET (object);
	if (gnome_db_data_set->priv) {
		switch (param_id) {
		case PROP:
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_data_set_get_property (GObject              *object,
				guint                 param_id,
				GValue               *value,
				GParamSpec           *pspec)
{
	GnomeDbDataSet *gnome_db_data_set;
	gnome_db_data_set = GNOME_DB_DATA_SET (object);
	
	if (gnome_db_data_set->priv) {
		switch (param_id) {
		case PROP:
			break;
		}	
	}
}

/*
 * Resets and computes dataset->nodes, and if some nodes already exist, they are previously discarded
 */
static void 
compute_dataset_nodes (GnomeDbDataSet *dataset)
{
	gboolean err = FALSE;
	GSList *list;
	GHashTable *hash;       /* key = a data_for_param referenced by a param, value = corresponding node */
	GHashTable *node_repls; /* key = a node, value = replacements hash created for that node */
	GHashTable *new_rs;     /* key = query used as a source, value = GnomeDbResultSet created for it */
	gboolean firstinit;

	hash = g_hash_table_new (NULL, NULL);
	node_repls = g_hash_table_new (NULL, NULL);
	new_rs = g_hash_table_new (NULL, NULL);

	firstinit = dataset->nodes ? FALSE : TRUE;

	/*
	 * Creation of the GnomeDbDataSetNode nodes, for the parameters which don't already have one
	 */
	list = dataset->parameters;
	while (list && !err) {
		GnomeDbField *ref_field;
		
		if (gnome_db_data_set_find_node_for_param (dataset, GNOME_DB_PARAMETER (list->data))) {
			/* this param already is in a GnomeDbDataSetNode */
			list = g_slist_next (list);
			continue;
		}
		
		ref_field = gnome_db_parameter_get_source_field (GNOME_DB_PARAMETER (list->data));
		if (ref_field) {
			/* 
			 * the parameter has a value constrained by a data_for_param 
			 */
			GnomeDbEntity *ent;
			GHashTable *repl = NULL;
			GnomeDbQuery *query = NULL;

			if (!firstinit)
				/* the solution would be to memorize the 'new_rs' hash table and re-use it for any subsequent
				 * call to this function */
				g_warning ("The GnomeDbDataSet::%s() function does not yet know how to compute\n"
					   "new GnomeDbDataSetNode structures when adding parameters which have a source field:\n"
					   "The layout of the new GnomeDbDataSetNode list may not be the correct one!", 
					   __FUNCTION__);

			ent = gnome_db_field_get_entity (ref_field);

			/* if 'ent' is a query, then we create a new GnomeDbResultSet to replace the GnomeDbQuery,
			 * and also prepare for the replacements
			 * to translate fom 'ent' to the new GnomeDbResultSet (which is also an entity) */
			if (IS_GNOME_DB_QUERY (ent)) {
				GnomeDbDataSetNode *node;

				node = g_hash_table_lookup (hash, ent);
				if (node)
					ent = GNOME_DB_ENTITY (node->data_for_param);
				else {
					/* transform the query into a GnomeDbDataModel */
					GnomeDbResultSet *rs;
					
					rs = g_hash_table_lookup (new_rs, ent);
					if (!rs) {
						rs = GNOME_DB_RESULT_SET (gnome_db_result_set_new (GNOME_DB_QUERY (ent), 
												   NULL));
						gnome_db_base_set_name (GNOME_DB_BASE (rs),
									gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
						gnome_db_base_set_description (GNOME_DB_BASE (rs),
									gnome_db_base_get_description (GNOME_DB_BASE (list->data)));
						g_hash_table_insert (new_rs, ent, rs);
						g_object_get (G_OBJECT (rs), "init_query_replacements", &repl, NULL);
					}
					query = GNOME_DB_QUERY (ent);
					ent = GNOME_DB_ENTITY (rs);
				}
			}

			if (!ent || !IS_GNOME_DB_DATA_MODEL (ent))
				err = TRUE;
			else {
				GnomeDbDataSetNode *node = g_hash_table_lookup (hash, ent);
				if (!node) {
					GnomeDbDataSetNode *node;

					node = g_new0 (GnomeDbDataSetNode, 1);
					node->data_for_param = GNOME_DB_DATA_MODEL (ent);
					node->params = g_slist_append (NULL, list->data);

					g_hash_table_insert (hash, ent, node);
					if (query) {
						g_hash_table_insert (hash, query, node);
						g_hash_table_insert (node_repls, node, repl);
					}
					else
						g_object_ref (G_OBJECT (ent));

					dataset->nodes = g_slist_append (dataset->nodes, node);
				}
				else 
					node->params = g_slist_append (node->params, list->data);
			}
		}
		else {
			/* 
			 * the parameter is free fill 
			 */
			GnomeDbDataSetNode *node;
			
			node = g_new0 (GnomeDbDataSetNode, 1);
			node->param = list->data;
			dataset->nodes = g_slist_append (dataset->nodes, node);
		}

		list = g_slist_next (list);
	}



	/* 
	 * destination and source fields replacements for each parameter
	 */
	list = dataset->parameters;
	while (list) {
		GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
		GSList *tmplist;
		GnomeDbField *ref_field;

		GnomeDbEntity *ent;
		gpointer rs;
		
		/* destination fields */
		tmplist = gnome_db_parameter_get_dest_fields (param);
		while (tmplist) {
			ent = gnome_db_field_get_entity (GNOME_DB_FIELD (tmplist->data));
			rs = g_hash_table_lookup (new_rs, ent);
			if (rs) {
				GHashTable *repl;
				
				g_object_get (G_OBJECT (rs), "init_query_replacements", &repl, NULL);
				gnome_db_parameter_replace_ref (param, repl);
			}
			tmplist = g_slist_next (tmplist);
		}

		/* source fields */
		ref_field = gnome_db_parameter_get_source_field (param);
		if (ref_field) {
			ent = gnome_db_field_get_entity (ref_field);
			rs = g_hash_table_lookup (new_rs, ent);
			if (rs) {
				GHashTable *repl;
				
				g_object_get (G_OBJECT (rs), "init_query_replacements", &repl, NULL);
				gnome_db_parameter_replace_ref (param, repl);
			}
		}

		list = g_slist_next (list);
	}

	/* 
	 * creating params_pos_in_query in each node where node->params is not NULL 
	 */
	list = dataset->nodes;
	while (list) {
		GnomeDbDataSetNode *node = GNOME_DB_DATA_SET_NODE (list->data);
		if (node->params) {
			GSList *params;
			GHashTable *hash_pos;
			gint pos;
			GnomeDbField *field;
			
			hash_pos = g_hash_table_new (NULL, NULL);
			node->pos_for_param = hash_pos;
			
			params = node->params;
			while (params) {
				field = gnome_db_parameter_get_source_field (GNOME_DB_PARAMETER (params->data));
				pos = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (node->data_for_param), field);
				if (pos < 0) {
					err = TRUE;
					g_warning ("Can't find position of field %p into data model %p for parameter %p\n",
						   field, node->data_for_param, params->data);
				}
				
				g_hash_table_insert (hash_pos, params->data, GINT_TO_POINTER (pos));
				params = g_slist_next (params);
			}
		}
		list = g_slist_next (list);
	}

	/* memory de-allocations */
	g_hash_table_destroy (hash);
	g_hash_table_destroy (node_repls);
	g_hash_table_destroy (new_rs);

#ifdef debug_NO
	gnome_db_data_set_dump (dataset, 0);
#endif
}

/**
 * gnome_db_data_set_add_param
 * @dataset: a #GnomeDbDataSet object
 * @param: a #GnomeDbParameter object
 *
 * Adds @param to the list of parameters managed within @dataset.
 * WARNING: the dataset may decide not to use the @param parameter, but to
 * modify another parameter already present within the dataset. The publicly available
 * lists from the @dataset object may also be changed in the process.
 */
void
gnome_db_data_set_add_param (GnomeDbDataSet *dataset, GnomeDbParameter *param)
{
	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));

	gnome_db_data_set_real_add_param (dataset, param);
	compute_dataset_nodes (dataset);
}

static void
gnome_db_data_set_real_add_param (GnomeDbDataSet *dataset, GnomeDbParameter *param)
{
	GSList *params;
	GnomeDbParameter *similar = NULL;
	GSList *param_dest_fields;

	if (g_slist_find (dataset->parameters, param))
		return;

	/* 
	 * try to find a similar parameter in the dataset->parameters:
	 * a parameter B is similar to a parameter A if the list of fields which use the value
	 * of A and the list of fields which use the value of B have at least one element in common.
	 */
	param_dest_fields = gnome_db_parameter_get_dest_fields (param);
	params = dataset->parameters;
	while (params && !similar) {
		GSList *list1;
		list1 = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (params->data));

		while (list1 && !similar) {
			if (g_slist_find (param_dest_fields, list1->data))
				similar = GNOME_DB_PARAMETER (params->data);
			list1 = g_slist_next (list1);
		}
		params = g_slist_next (params);
	}
	
	if (similar) {
#ifdef debug_NO
		g_print ("Param %p and %p are similar, keeping %p only\n", similar, param, similar);
#endif
		g_hash_table_insert (dataset->priv->param_repl, param, similar);
		/* don't use @param, but instead use similar and update the list of fields
		   similar is for */
		while (param_dest_fields) {
			gnome_db_parameter_add_dest_field (similar, GNOME_DB_FIELD (param_dest_fields->data));
			param_dest_fields = g_slist_next (param_dest_fields);
		}
	}
	else {
		/* really add @param to the dataset */
		dataset->parameters = g_slist_append (dataset->parameters, param);
		gnome_db_base_connect_nullify (param,
					       G_CALLBACK (nullified_param_cb), dataset);
		g_signal_connect (G_OBJECT (param), "changed",
				  G_CALLBACK (changed_param_cb), dataset);

		g_object_ref (G_OBJECT (param));
		gnome_db_parameter_replace_ref (param, dataset->priv->param_repl);
	}
}

/**
 * gnome_db_data_set_merge_dataset_params
 * @dataset: a #GnomeDbDataSet object
 * @dataset_to_merge: a #GnomeDbDataSet object
 *
 * Add to @dataset all the parameters of @dataset_to_merge.
 */
void
gnome_db_data_set_merge_dataset_params (GnomeDbDataSet *dataset, GnomeDbDataSet *dataset_to_merge)
{
	GSList *params = dataset_to_merge->parameters;
	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));
	g_return_if_fail (dataset_to_merge && IS_GNOME_DB_DATA_SET (dataset_to_merge));

	while (params) {
		gnome_db_data_set_real_add_param (dataset, GNOME_DB_PARAMETER (params->data));
		params = g_slist_next (params);
	}
	compute_dataset_nodes (dataset);
}

/**
 * gnome_db_dataset_is_coherent
 * @dataset: a #GnomeDbDataSet object
 * @error:
 *
 * Checks that @dataset has a coherent structure
 *
 * Returns: TRUE if @dataset is coherent
 */
gboolean
gnome_db_data_set_is_coherent (GnomeDbDataSet *dataset, GError **error)
{
	GSList *list;
	gint current_pos = 0;
	/* To test:
	   - For each parameter, check that the value provider is a field of the same type
	   - For each parameter, check that the value provider belongs to a GnomeDbDataModel
	   - For each parameter, check that all the dependancies parameter are also in the dataset
	   - the nodes settings are correct
	*/
	list = dataset->parameters;
	while (list) {
		GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
		GnomeDbField *value_prov;
		GnomeDbDataSetNode *node;
		GSList *depend;
		
		node = gnome_db_data_set_find_node_for_param (dataset, param);
		if (!node) {
			g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_NO_NODE_ERROR,
				     _("Value provider for parameter '%s' is not listed in any dataset node "
				       "(internal error)"),
				     gnome_db_base_get_name (GNOME_DB_BASE (param)));
			return FALSE;
		}

		/*
		 * testing the value providers
		 */
		value_prov = gnome_db_parameter_get_source_field (param);
		if (value_prov) {
			GnomeDbDataModel *model;
			
			/* data type of value_prov */
			if (gnome_db_parameter_get_data_type (param) != gnome_db_field_get_data_type (value_prov)) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_VALUE_PROV_DATA_TYPE_ERROR,
					     _("Data types for parameter '%s' and value provider differ "
					       "(respectively %s and %s)"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)),
					     gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_parameter_get_data_type (param))),
					     gnome_db_base_get_name (GNOME_DB_BASE (gnome_db_field_get_data_type (value_prov))));
				return FALSE;
			}

			/* type of query to which value_prov belongs */
			model = GNOME_DB_DATA_MODEL (gnome_db_field_get_entity (value_prov));
			if (node->param ||
			    (node->data_for_param != model) ||
			    ! node->pos_for_param ||
			    (GPOINTER_TO_INT (g_hash_table_lookup (node->pos_for_param, param)) < 0)) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_NODE_OUTDATED_ERROR,
					     _("Parameter '%s' is value-constrained by an entity not in the data set"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)));
				return FALSE;
			}

			/* value_prov is visible */
			if (!gnome_db_entity_has_field (GNOME_DB_ENTITY (node->data_for_param), value_prov)) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_VALUE_PROV_INVISIBLE_ERROR,
					     _("Value provider for parameter '%s' is not visible in the model"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)));
				return FALSE;
			}

		}
		else {
			/* No value provider, test the node structure */
			if (node->data_for_param ||
			    node->params) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_NODE_OUTDATED_ERROR,
					     _("Parameter '%s' has changed since insertion in the dataset"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)));
				return FALSE;
			}
		}

		/*
		 * testing the parameter dependencies which must be within the dataset and before
		 * param in the parameters list
		 */
		depend = gnome_db_parameter_get_dependencies (param);
		while (depend) {
			gint i = g_slist_index (dataset->parameters, depend->data);

			/* dependency param in dataset */
			if (i < 0) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_DEPENDENCY_NOT_IN_DATASET_ERROR,
					     _("Parameter '%s' has a dependency parameter ('%s') which is "
					       "not handled in the dataset"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)),
					     gnome_db_base_get_name (GNOME_DB_BASE (depend->data)));
				return FALSE;
			}
			
			/* dependency param before tested param in the parameters list */
			if (i >= current_pos) {
				g_set_error (error, GNOME_DB_DATA_SET_ERROR, GNOME_DB_DATA_SET_DEPENDENCY_POSITION_ERROR,
					     _("Parameter '%s' has a dependency parameter ('%s') which is "
					       "after itself the dataset"),
					     gnome_db_base_get_name (GNOME_DB_BASE (param)),
					     gnome_db_base_get_name (GNOME_DB_BASE (depend->data)));
				return FALSE;
			}

			depend = g_slist_next (depend);
		}

		list = g_slist_next (list);
		current_pos++;
	}
	
	return TRUE;
}


/**
 * gnome_db_data_set_free_node
 * @dataset: a #GnomeDbDataSet object
 * @node: a #GnomeDbDataSetNode structure
 *
 * Removes @node from @dataset and de-allocates any memory in the process.
 */
static void
gnome_db_data_set_free_node (GnomeDbDataSet *dataset, GnomeDbDataSetNode *node)
{
	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));
	g_return_if_fail (node);
	g_return_if_fail (dataset->priv);
	g_return_if_fail (g_slist_find (dataset->nodes, node));

	if (node->param)
		node->param = NULL;
	if (node->data_for_param)
		g_object_unref (G_OBJECT (node->data_for_param));
	if (node->params)
		g_slist_free (node->params);
	if (node->pos_for_param)
		g_hash_table_destroy (node->pos_for_param);

	dataset->nodes = g_slist_remove (dataset->nodes, node);
	g_free (node);
}

/**
 * gnome_db_data_set_is_valid
 * @dataset: a #GnomeDbDataSet object
 *
 * Tells if all the dataset's parameters have valid data
 *
 * Returns: TRUE if the dataset is valid
 */
gboolean
gnome_db_data_set_is_valid (GnomeDbDataSet *dataset)
{
	GSList *params;
	gboolean retval = TRUE;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), FALSE);
	g_return_val_if_fail (dataset->priv, FALSE);

	params = dataset->parameters;
	while (params && retval) {
		if (!gnome_db_parameter_is_valid (GNOME_DB_PARAMETER (params->data))) 
			retval = FALSE;
		
#ifdef debug_NO
		g_print ("== PARAM %p: valid= %d, value=%s\n", params->data, gnome_db_parameter_is_valid (GNOME_DB_PARAMETER (params->data)),
			 gnome_db_parameter_get_value (GNOME_DB_PARAMETER (params->data)) ?
			 gda_value_stringify (gnome_db_parameter_get_value (GNOME_DB_PARAMETER (params->data))) : "Null");
#endif
		params = g_slist_next (params);
	}

	return retval;
}

/**
 * gnome_db_data_set_needs_user_input
 * @dataset: a #GnomeDbDataSet object
 *
 * Tells if the dataset needs the user to provide some values for some parameters.
 * This is the case when either the dataset is invalid, or the dataset is valid and
 * contains some #GnomeDbParameter objects which are valid but are defined to require 
 * a user input.
 *
 * Note: this function may return TRUE even though @dataset is valid.
 *
 * Returns: TRUE if a user input is required for the dataset
 */
gboolean
gnome_db_data_set_needs_user_input (GnomeDbDataSet *dataset)
{
	GSList *params;
	gboolean retval = FALSE;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), FALSE);
	g_return_val_if_fail (dataset->priv, FALSE);

	if (!gnome_db_data_set_is_valid (dataset))
		return TRUE;

	params = dataset->parameters;
	while (params && !retval) {
		if (gnome_db_parameter_requires_user_input (GNOME_DB_PARAMETER (params->data)))
			retval = TRUE;
		
		params = g_slist_next (params);
	}

	return retval;
}

/**
 * gnome_db_data_set_find_parameter
 * @dataset: a #GnomeDbDataSet object
 * @param_name: the name of the requested parameter
 *
 * Finds a #GnomeDbParameter using its name
 *
 * Returns: a #GnomeDbParameter or %NULL
 */
GnomeDbParameter *
gnome_db_data_set_find_parameter (GnomeDbDataSet *dataset, const gchar *param_name)
{
	GnomeDbParameter *param = NULL;
	GSList *list;
	gchar *string_id;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), NULL);
	g_return_val_if_fail (dataset->priv, NULL);

	list = dataset->parameters;
	while (list && !param) {
		if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), param_name))
			param = GNOME_DB_PARAMETER (list->data);
		if (!param) {
			g_object_get (G_OBJECT (list->data), "string_id", &string_id, NULL);
			if (!strcmp (string_id, param_name))
				param = GNOME_DB_PARAMETER (list->data);
		}
		list = g_slist_next (list);
	}

	return param;
}

/**
 * gnome_db_data_set_find_parameter_for_field
 * @dataset: a #GnomeDbDataSet object
 * @for_field: a #GnomeDbField object
 *
 * Finds a #GnomeDbParameter which is to be used by @for_field
 *
 * Returns: a #GnomeDbParameter or %NULL
 */
GnomeDbParameter *
gnome_db_data_set_find_parameter_for_field (GnomeDbDataSet *dataset, GnomeDbField *for_field)
{
	GnomeDbParameter *param = NULL;
	GSList *list;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), NULL);
	g_return_val_if_fail (dataset->priv, NULL);

	list = dataset->parameters;
	while (list && !param) {
		GSList *for_fields = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
		if (for_fields &&  g_slist_find (for_fields, for_field))
			param = GNOME_DB_PARAMETER (list->data);
		list = g_slist_next (list);
	}

	return param;
}

/**
 * gnome_db_data_set_find_node_for_param
 * @dataset: a #GnomeDbDataSet object
 * @param: a #GnomeDbParameter object
 *
 * Finds a #GnomeDbDataSetNode which is for @param
 *
 * Returns: a #GnomeDbDataSetNode or %NULL
 */
GnomeDbDataSetNode *
gnome_db_data_set_find_node_for_param (GnomeDbDataSet *dataset, GnomeDbParameter *param)
{
	GnomeDbDataSetNode *retval = NULL, *node;
	GSList *list;

	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), NULL);
	g_return_val_if_fail (dataset->priv, NULL);
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (g_slist_find (dataset->parameters, param), NULL);

	list = dataset->nodes;
	while (list && !retval) {
		node = GNOME_DB_DATA_SET_NODE (list->data);

		if (node->param) {
			if (node->param == param)
				retval = node;
		}
		else {
			if (g_slist_find (node->params, param))
				retval = node;
		}

		list = g_slist_next (list);
	}

	return retval;
}


/**
 * gnome_db_data_set_set_param_default_value
 * @dataset: a #GnomeDbDataSet object
 * @param: a #GnomeDbParameter object, managed by @dataset
 * @value: a #GdaValue, of the same type as @param, or %NULL
 *
 * Stores @value in @dataset to make it possible for @dataset's users to find a default value
 * for @param when one is required, instead of %NULL.
 *
 * @dataset only provides a storage functionnality, the way the value obtained with 
 * gnome_db_data_set_get_param_default_value() is used is up to @dataset's user.
 */
void
gnome_db_data_set_set_param_default_value (GnomeDbDataSet *dataset, GnomeDbParameter *param, const GdaValue *value)
{
	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (g_slist_find (dataset->parameters, param));

	if (value && ! gda_value_is_null (value)) {
		g_return_if_fail (gda_value_get_type (value) ==
				  gnome_db_server_data_type_get_gda_type (gnome_db_parameter_get_data_type (param)));
		g_hash_table_insert (dataset->priv->param_default_values, param, gda_value_copy (value));
	}
	else
		g_hash_table_remove (dataset->priv->param_default_values, param);
}

static void nullified_alias_cb (GnomeDbParameter *alias, GnomeDbDataSet *dataset);

/*
 * gnome_db_data_set_set_param_default_alias
 * @dataset: a #GnomeDbDataSet object
 * @param: a #GnomeDbParameter object, managed by @dataset
 * @alias: a #GnomeDbParameter object, of the same type as @param, or %NULL
 *
 * Stores a reference to @alias in @dataset to make it possible for @dataset's users to find a default value
 * for @param when one is required, instead of %NULL.
 *
 * @dataset only provides a storage functionnality, the way the value obtained with 
 * gnome_db_data_set_get_param_default_value() is used is up to @dataset's user.
 */
void
gnome_db_data_set_set_param_default_alias (GnomeDbDataSet *dataset, GnomeDbParameter *param, GnomeDbParameter *alias)
{
	GnomeDbParameter *oldalias;

	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (g_slist_find (dataset->parameters, param));

	/* remove the old alias if there was one */
	oldalias = g_hash_table_lookup (dataset->priv->param_default_aliases, param);
	if (oldalias)
		nullified_alias_cb (oldalias, dataset);

	/* take care of new alias if there is one */
	if (alias) {
		g_return_if_fail (alias != param);
		g_return_if_fail (alias && IS_GNOME_DB_PARAMETER (alias));
		g_return_if_fail (gnome_db_parameter_get_data_type (param) == 
				  gnome_db_parameter_get_data_type (alias));
		g_hash_table_insert (dataset->priv->param_default_aliases, param, alias);
		g_hash_table_insert (dataset->priv->aliases_default_param, alias, param);
		gnome_db_base_connect_nullify (alias, G_CALLBACK (nullified_alias_cb), dataset);
		g_object_ref (G_OBJECT (alias));
	}
}

static void
nullified_alias_cb (GnomeDbParameter *alias, GnomeDbDataSet *dataset)
{
	GnomeDbParameter *param;

	g_signal_handlers_disconnect_by_func (G_OBJECT (alias),
					      G_CALLBACK (nullified_alias_cb), dataset);
	param = g_hash_table_lookup (dataset->priv->aliases_default_param, alias);
	g_hash_table_remove (dataset->priv->param_default_aliases, param);
	g_hash_table_remove (dataset->priv->aliases_default_param, alias);
	g_object_unref (G_OBJECT (alias));
}

static void
foreach_finalize_alias (GnomeDbParameter *param, GnomeDbParameter *alias, GnomeDbDataSet *dataset)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (alias),
					      G_CALLBACK (nullified_alias_cb), dataset);
	g_object_unref (G_OBJECT (alias));
}

/**
 * gnome_db_data_set_get_param_default_value
 * @dataset: a #GnomeDbDataSet object
 * @param: a #GnomeDbParameter object, managed by @dataset
 *
 * Retreives a prefered default value to be used by @dataset's users when necessary.
 * The usage of such values is decided by @dataset's users.
 * 
 * @dataset only offers to store the value
 * using gnome_db_data_set_set_param_default_value() or to store a #GnomeDbParameter reference from which to get
 * a value using gnome_db_data_set_set_param_default_alias().
 *
 * Returns: a #GdaValue, or %NULL.
 */
const GdaValue *
gnome_db_data_set_get_param_default_value  (GnomeDbDataSet *dataset, GnomeDbParameter *param)
{
	const GdaValue *value;
	g_return_val_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset), NULL);
	g_return_val_if_fail (dataset->priv, NULL);
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);

	value = g_hash_table_lookup (dataset->priv->param_default_values, param);
	if (value)
		return value;
	else {
		GnomeDbParameter *alias;
		
		alias = g_hash_table_lookup (dataset->priv->param_default_aliases, param);
		if (alias && gnome_db_parameter_is_valid (alias))
			return gnome_db_parameter_get_value (alias);
	}

	return NULL;
}

#ifdef debug
static void
gnome_db_data_set_dump (GnomeDbDataSet *dataset, guint offset)
{
	gchar *str;
        guint i;
	
	g_return_if_fail (dataset && IS_GNOME_DB_DATA_SET (dataset));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (dataset->priv) {
		GError *error = NULL;
		GSList *list;
                g_print ("%s" D_COL_H1 "GnomeDbDataSet" D_COL_NOR " %p (id=%d)",
                         str, dataset, gnome_db_base_get_id (GNOME_DB_BASE (dataset)));
		if (! gnome_db_data_set_is_coherent (dataset, &error)) {
			g_print (" " D_COL_ERR "not coherent: " D_COL_NOR "%s", error->message);
			g_error_free (error);
		}
		g_print ("\n");

		g_print ("%s     ***** Parameters:\n", str);
		list = dataset->parameters;
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}

		g_print ("\n%s     ***** Nodes:\n", str);
		list = dataset->nodes;
		while (list) {
			GnomeDbDataSetNode *node = GNOME_DB_DATA_SET_NODE (list->data);
			if (node->param) 
				g_print ("%s     * " D_COL_H1 "GnomeDbDataSetNode %p" D_COL_NOR " for param %p\n", 
					 str, node, node->param);
			else {
				GSList *params = node->params;
				g_print ("%s     * " D_COL_H1 "GnomeDbDataSetNode %p" D_COL_NOR "\n", str, node);
				gnome_db_base_dump (GNOME_DB_BASE (node->data_for_param), offset+10);
				while (params) {
					g_print ("%s       -> param %p (%s)\n", str, params->data, 
						 gnome_db_base_get_name (GNOME_DB_BASE (params->data)));
					params = g_slist_next (params);
				}
			}
			list = g_slist_next (list);
		}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, dataset);
}
#endif

