/* gnome-db-table-field.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-table-field.h"
#include "gnome-db-table.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "marshal.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-constraint.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_table_field_class_init (GnomeDbTableFieldClass * class);
static void gnome_db_table_field_init (GnomeDbTableField * srv);
static void gnome_db_table_field_dispose (GObject   * object);
static void gnome_db_table_field_finalize (GObject   * object);

static void gnome_db_table_field_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_table_field_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_table_field_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_table_field_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_table_field_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_table_field_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void                   gnome_db_table_field_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_table_field_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_table_field_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_table_field_renderer_init      (GnomeDbRendererIface *iface);
static GdaXqlItem     *gnome_db_table_field_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar          *gnome_db_table_field_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_table_field_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

#ifdef debug
static void            gnome_db_table_field_dump            (GnomeDbTableField *field, guint offset);
#endif

/* When the DbTable or GnomeDbServerDataType is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbTableField *field);

/* Function to get a GnomeDbDataHandler from a GnomeDbTableField */
static GnomeDbDataHandler *gnome_db_table_field_handler_func (GnomeDbServer *srv, GObject *obj);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_table_field_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_DB_TABLE
};


/* private structure */
struct _GnomeDbTableFieldPrivate
{
	GnomeDbServerDataType *data_type;
	GnomeDbTable        *table;
	gint              length;     /* -1 if not applicable */
	gint              scale;      /* 0 if not applicable */
	GdaValue         *default_val;/* NULL if no default value */
};
#define FIELD_SRV(field) (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (field))))


/* module error */
GQuark gnome_db_table_field_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_table_field_error");
	return quark;
}


guint
gnome_db_table_field_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbTableFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_table_field_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbTableField),
			0,
			(GInstanceInitFunc) gnome_db_table_field_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_table_field_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_table_field_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_table_field_renderer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbTableField", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_FIELD_TYPE, &field_info);
		g_type_add_interface_static (type, GNOME_DB_RENDERER_TYPE, &renderer_info);
	}
	return type;
}

static void 
gnome_db_table_field_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_table_field_get_xml_id;
	iface->save_to_xml = gnome_db_table_field_save_to_xml;
	iface->load_from_xml = gnome_db_table_field_load_from_xml;
}

static void
gnome_db_table_field_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_table_field_get_entity;
	iface->get_data_type = gnome_db_table_field_get_data_type;
}

static void
gnome_db_table_field_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_table_field_render_as_xql;
	iface->render_as_sql = gnome_db_table_field_render_as_sql;
	iface->render_as_str = gnome_db_table_field_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_table_field_class_init (GnomeDbTableFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_table_field_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbTableFieldClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_table_field_dispose;
	object_class->finalize = gnome_db_table_field_finalize;

	/* Properties */
	object_class->set_property = gnome_db_table_field_set_property;
	object_class->get_property = gnome_db_table_field_get_property;
	g_object_class_install_property (object_class, PROP_DB_TABLE,
					 g_param_spec_pointer ("db_table", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_table_field_dump;
#endif
}


static void
gnome_db_table_field_init (GnomeDbTableField * gnome_db_table_field)
{
	gnome_db_table_field->priv = g_new0 (GnomeDbTableFieldPrivate, 1);
	gnome_db_table_field->priv->table = NULL;
	gnome_db_table_field->priv->data_type = NULL;
	gnome_db_table_field->priv->length = -1;
	gnome_db_table_field->priv->scale = 0;
}


/**
 * gnome_db_table_field_new
 * @dict: a #GnomeDbDict object
 * @type:  a #GnomeDbServerDataType object (the field's type)
 *
 * Creates a new GnomeDbTableField object
 *
 * Returns: the new object
 */
GObject*
gnome_db_table_field_new (GnomeDbDict *dict, GnomeDbServerDataType *type)
{
	GObject   *obj;
	GnomeDbTableField *gnome_db_table_field;

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
	if (type)
		g_return_val_if_fail (IS_GNOME_DB_SERVER_DATA_TYPE (type), NULL);

	obj = g_object_new (GNOME_DB_TABLE_FIELD_TYPE, "dict", dict, NULL);
	gnome_db_table_field = GNOME_DB_TABLE_FIELD (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_table_field), 0);

	if (type)
		gnome_db_table_field_set_data_type (gnome_db_table_field, type);

	/* Extra init */
	gnome_db_server_set_object_func_handler (gnome_db_dict_get_server (dict), gnome_db_table_field_handler_func);
	
	return obj;
}

static GnomeDbDataHandler *
gnome_db_table_field_handler_func (GnomeDbServer *srv, GObject *obj)
{
	if (IS_GNOME_DB_TABLE_FIELD (obj)) 
		return gnome_db_server_get_handler_by_gda (srv, 
						     gnome_db_server_data_type_get_gda_type (GNOME_DB_TABLE_FIELD (obj)->priv->data_type));
	else
		return NULL;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbTableField *field)
{
	gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static void
gnome_db_table_field_dispose (GObject *object)
{
	GnomeDbTableField *gnome_db_table_field;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TABLE_FIELD (object));

	gnome_db_table_field = GNOME_DB_TABLE_FIELD (object);
	if (gnome_db_table_field->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_table_field->priv->table) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_table_field->priv->table),
							      G_CALLBACK (nullified_object_cb), gnome_db_table_field);
			gnome_db_table_field->priv->table = NULL;
		}
		if (gnome_db_table_field->priv->data_type) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_table_field->priv->data_type),
							      G_CALLBACK (nullified_object_cb), gnome_db_table_field);
			gnome_db_table_field->priv->data_type = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_table_field_finalize (GObject   * object)
{
	GnomeDbTableField *gnome_db_table_field;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TABLE_FIELD (object));

	gnome_db_table_field = GNOME_DB_TABLE_FIELD (object);
	if (gnome_db_table_field->priv) {

		g_free (gnome_db_table_field->priv);
		gnome_db_table_field->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_table_field_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbTableField *gnome_db_table_field;

	gnome_db_table_field = GNOME_DB_TABLE_FIELD (object);
	if (gnome_db_table_field->priv) {
		switch (param_id) {
		case PROP_DB_TABLE:
			if (gnome_db_table_field->priv->table) {
				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_table_field->priv->table),
								      G_CALLBACK (nullified_object_cb), gnome_db_table_field);
				gnome_db_table_field->priv->table = NULL;
			}

			ptr = g_value_get_pointer (value);
			if (ptr && IS_GNOME_DB_TABLE (ptr)) {
				gnome_db_table_field->priv->table = GNOME_DB_TABLE (ptr);
				gnome_db_base_connect_nullify (ptr,
							 G_CALLBACK (nullified_object_cb), gnome_db_table_field);
			}
			break;
		}
	}
}

static void
gnome_db_table_field_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbTableField *gnome_db_table_field;
	gnome_db_table_field = GNOME_DB_TABLE_FIELD (object);
	
	if (gnome_db_table_field->priv) {
		switch (param_id) {
		case PROP_DB_TABLE:
			g_value_set_pointer (value, gnome_db_table_field->priv->table);
			break;
		}	
	}
}


/**
 * gnome_db_table_field_get_length
 * @field: a #GnomeDbTableField  object
 *
 * Set the length of a field.
 *
 */
void
gnome_db_table_field_set_length (GnomeDbTableField *field, gint length)
{
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (field->priv);
	if (length <= 0)
		field->priv->length = -1;
	else
		field->priv->length = length;

	/* signal the modification */
	gnome_db_base_changed (GNOME_DB_BASE (field));
}

/**
 * gnome_db_table_field_get_length
 * @field: a #GnomeDbTableField  object
 *
 * Get the length of a field.
 *
 * Returns: the size of the corresponding data type has a fixed size, or -1
 */
gint
gnome_db_table_field_get_length (GnomeDbTableField *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), -1);
	g_return_val_if_fail (field->priv, -1);

	return field->priv->length;
}

/**
 * gnome_db_table_field_get_scale
 * @field: a #GnomeDbTableField  object
 *
 * Set the scale of a field.
 *
 */
void
gnome_db_table_field_set_scale (GnomeDbTableField *field, gint scale)
{
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (field->priv);
	if (scale <= 0)
		field->priv->scale = 0;
	else
		field->priv->scale = scale;

	/* signal the modification */
	gnome_db_base_changed (GNOME_DB_BASE (field));
}

/**
 * gnome_db_table_field_get_scale
 * @field: a #GnomeDbTableField  object
 *
 * Get the scale of a field.
 *
 * Returns: the size of the corresponding data type has a fixed size, or -1
 */
gint
gnome_db_table_field_get_scale (GnomeDbTableField *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), -1);
	g_return_val_if_fail (field->priv, -1);

	return field->priv->scale;
}

/**
 * gnome_db_table_field_get_constraints
 * @field: a #GnomeDbTableField  object
 *
 * Get all the constraints which impact the given field. Constraints are of several type:
 * NOT NULL, primary key, foreign key, check constrains
 *
 * Returns: a new list of #GnomeDbConstraint objects
 */
GSList *
gnome_db_table_field_get_constraints (GnomeDbTableField *field)
{
	GSList *retval = NULL;
	GSList *list, *table_cons;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), NULL);
	g_return_val_if_fail (field->priv, NULL);
	g_return_val_if_fail (field->priv->table, NULL);

	table_cons = gnome_db_table_get_constraints (field->priv->table);
	list = table_cons;
	while (list) {
		if (gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field))
			retval = g_slist_append (retval, list->data);
		list = g_slist_next (list);
	}
	g_slist_free (table_cons);
	
	return retval;
}

/**
* gnome_db_table_field_set_data_type
* @field: a #GnomeDbTableField  object
* @type: a #GnomeDbServerDataType object
*
* Sets the data type of the field
*/
void
gnome_db_table_field_set_data_type (GnomeDbTableField *field, GnomeDbServerDataType *type)
{
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (field->priv);
	g_return_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type));

	if (field->priv->data_type)
		g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->data_type),
						      G_CALLBACK (nullified_object_cb), field);
	field->priv->data_type = type;
	gnome_db_base_connect_nullify (type, 
				 G_CALLBACK (nullified_object_cb), field);

	/* signal the modification */
	gnome_db_base_changed (GNOME_DB_BASE (field));
}


/**
 * gnome_db_table_field_set_default_value
 * @field: a #GnomeDbTableField object
 * @value: a #GdaValue value or NULL
 *
 * Sets (or replace) the default value for the field. WARNING: the default value's data type can be
 * different from the field's data type (this is the case for example if the default value is a 
 * function like Postgres's default value for the SERIAL data type).
 */
void
gnome_db_table_field_set_default_value (GnomeDbTableField *field, const GdaValue *value)
{
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (field->priv);

	if (field->priv->default_val) {
		gda_value_free (field->priv->default_val);
		field->priv->default_val = NULL;
	}

	if (value)
		field->priv->default_val = gda_value_copy (value);

	/* signal the modification */
	gnome_db_base_changed (GNOME_DB_BASE (field));
}

/**
 * gnome_db_table_field_get_default_value
 * @field: a #GnomeDbTableField object
 * 
 * Get the default value for the field if ne exists
 *
 * Returns: the default value
 */
const GdaValue *
gnome_db_table_field_get_default_value (GnomeDbTableField *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	return field->priv->default_val;
}

gboolean
gnome_db_table_field_is_null_allowed (GnomeDbTableField *field)
{
	gboolean retval = TRUE;
	GSList *list, *table_cons;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);
	g_return_val_if_fail (field->priv->table, FALSE);

	table_cons = gnome_db_table_get_constraints (field->priv->table);
	list = table_cons;
	while (list && retval) {
		if ((gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data))==CONSTRAINT_NOT_NULL) &&
		    gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field))
			retval = FALSE;
		list = g_slist_next (list);
	}
	g_slist_free (table_cons);
	
	return retval;
}

gboolean
gnome_db_table_field_is_pkey_part (GnomeDbTableField *field)
{
	gboolean retval = FALSE;
	GSList *constraints, *list;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);
	g_return_val_if_fail (field->priv->table, FALSE);

	constraints = gnome_db_table_get_constraints (field->priv->table);
	list = constraints;
	while (list && !retval) {
		if ((gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == CONSTRAINT_PRIMARY_KEY) &&
		    gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field))
			retval = TRUE;
		list = g_slist_next (list);
	}
	g_slist_free (constraints);

	return retval;
}

gboolean
gnome_db_table_field_is_pkey_alone (GnomeDbTableField *field)
{
	gboolean retval = FALSE;
	GSList *constraints, *list;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);
	g_return_val_if_fail (field->priv->table, FALSE);

	constraints = gnome_db_table_get_constraints (field->priv->table);
	list = constraints;
	while (list && !retval) {
		if ((gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == CONSTRAINT_PRIMARY_KEY) &&
		    gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field)) {
			GSList *fields = gnome_db_constraint_pkey_get_fields (GNOME_DB_CONSTRAINT (list->data));
			retval = g_slist_length (fields) == 1 ? TRUE : FALSE;
			g_slist_free (fields);
		}
		list = g_slist_next (list);
	}
	g_slist_free (constraints);

	return retval;
}

gboolean
gnome_db_table_field_is_fkey_part (GnomeDbTableField *field)
{
	gboolean retval = FALSE;
	GSList *constraints, *list;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);
	g_return_val_if_fail (field->priv->table, FALSE);

	constraints = gnome_db_table_get_constraints (field->priv->table);
	list = constraints;
	while (list && !retval) {
		if ((gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == CONSTRAINT_FOREIGN_KEY) &&
		    gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field))
			retval = TRUE;
		list = g_slist_next (list);
	}
	g_slist_free (constraints);

	return retval;
}

gboolean
gnome_db_table_field_is_fkey_alone (GnomeDbTableField *field)
{
	gboolean retval = FALSE;
	GSList *constraints, *list;

	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);
	g_return_val_if_fail (field->priv->table, FALSE);

	constraints = gnome_db_table_get_constraints (field->priv->table);
	list = constraints;
	while (list && !retval) {
		if ((gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == CONSTRAINT_FOREIGN_KEY) &&
		    gnome_db_constraint_uses_field (GNOME_DB_CONSTRAINT (list->data), field)) {
			GSList *fields = gnome_db_constraint_fkey_get_fields (GNOME_DB_CONSTRAINT (list->data));
			GSList *list2;
			retval = g_slist_length (fields) == 1 ? TRUE : FALSE;

			list2 = fields;
			while (list2) {
				g_free (list2->data);
				list2 = g_slist_next (list2);
			}
			g_slist_free (fields);
		}
		list = g_slist_next (list);
	}
	g_slist_free (constraints);

	return retval;	
}

#ifdef debug
static void
gnome_db_table_field_dump (GnomeDbTableField *field, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (field->priv)
                g_print ("%s" D_COL_H1 "GnomeDbTableField" D_COL_NOR " %s (%p)\n",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (field)), field);
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, field);
}
#endif


/* 
 * GnomeDbField interface implementation
 */
static GnomeDbEntity *
gnome_db_table_field_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_TABLE_FIELD (iface)->priv->table);
}

static GnomeDbServerDataType *
gnome_db_table_field_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);

	return GNOME_DB_TABLE_FIELD (iface)->priv->data_type;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_table_field_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *t_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);

	t_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_TABLE_FIELD (iface)->priv->table));
	xml_id = g_strdup_printf ("%s:FI%s", t_xml_id, gnome_db_base_get_name (GNOME_DB_BASE (iface)));
	g_free (t_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_table_field_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbTableField *field;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);

	field = GNOME_DB_TABLE_FIELD (iface);

	node = xmlNewNode (NULL, "GNOME_DB_FIELD");
	
	str = gnome_db_table_field_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (field)));
	if (gnome_db_base_get_owner (GNOME_DB_BASE (field)))
		xmlSetProp (node, "owner", gnome_db_base_get_owner (GNOME_DB_BASE (field)));
	xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (field)));
	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (field->priv->data_type));
	xmlSetProp (node, "type", str);
	g_free (str);
	str = g_strdup_printf ("%d", field->priv->length);
	xmlSetProp (node, "length", str);
	g_free (str);
	str = g_strdup_printf ("%d", field->priv->scale);
	xmlSetProp (node, "scale", str);
	g_free (str);

	if (field->priv->default_val) {
		GnomeDbDataHandler *dh;
		GdaValueType vtype;
		
		vtype = gda_value_get_type (field->priv->default_val);
		xmlSetProp (node, "default_gda_type", gda_type_to_string (vtype));

		dh = gnome_db_server_get_handler_by_gda (FIELD_SRV (field), vtype);
		str = gnome_db_data_handler_get_str_from_value (dh, field->priv->default_val);
		xmlSetProp (node, "default", str);
		g_free (str);
	}

	if (gnome_db_server_object_has_handler (FIELD_SRV (field), G_OBJECT (field))) {
		GnomeDbDataHandler *dh;
		dh = gnome_db_server_get_object_handler (FIELD_SRV (field), G_OBJECT (field));
		xmlSetProp (node, "plugin", gnome_db_data_handler_get_plugin_name (dh));
	}

	return node;
}

static gboolean
gnome_db_table_field_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbTableField *field;
	gchar *prop;
	gboolean name = FALSE, type = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	field = GNOME_DB_TABLE_FIELD (iface);
	if (strcmp (node->name, "GNOME_DB_FIELD")) {
		g_set_error (error,
			     GNOME_DB_TABLE_FIELD_ERROR,
			     GNOME_DB_TABLE_FIELD_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_FIELD>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		name = TRUE;
		gnome_db_base_set_name (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "owner");
	if (prop) {
		gnome_db_base_set_owner (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		GnomeDbServerDataType *dt = gnome_db_server_get_data_type_by_xml_id (FIELD_SRV (field), prop);
		if (dt) {
			gnome_db_table_field_set_data_type (field, dt);
			type = TRUE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "length");
	if (prop) {
		field->priv->length = atoi (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "scale");
	if (prop) {
		field->priv->scale = atoi (prop);
		g_free (prop);
	}
	
	prop = xmlGetProp (node, "plugin");
	if (prop) {
		GnomeDbDataHandler *dh;
		dh = gnome_db_server_get_handler_by_name (FIELD_SRV (field), prop);
		if (dh) 
			gnome_db_server_set_object_handler (FIELD_SRV (field), G_OBJECT (field), dh);
		g_free (prop);
	}

	prop = xmlGetProp (node, "default");
	if (prop) {
		gchar *str2;

		str2 = xmlGetProp (node, "default_gda_type");
		if (str2) {
			GdaValueType vtype;
			GnomeDbDataHandler *dh;
			GdaValue *value;
			
			vtype = gda_type_from_string (str2);			
			dh = gnome_db_server_get_handler_by_gda (FIELD_SRV (field), vtype);
			value = gnome_db_data_handler_get_value_from_str (dh, prop, vtype);
			gnome_db_table_field_set_default_value (field, value);
			gda_value_free (value);

			g_free (str2);
		}
		g_free (prop);
	}



	if (name && type)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_TABLE_FIELD_ERROR,
			     GNOME_DB_TABLE_FIELD_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_FIELD>"));
		return FALSE;
	}
}


/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_table_field_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	GdaXqlItem *node = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return node;
}

static gchar *
gnome_db_table_field_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return str;
}

static gchar *
gnome_db_table_field_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE_FIELD (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return str;
}
