/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include "ccd-selector-group-priv.h"

typedef struct {
	GSList *selectors;
} ccd_selector_set_t;

/**
 * ccd_selector_group_t:
 * @sets:		
 * @min_specificity_e:	
 * @dangling_selectors:
 * 
 * Represents a set of associated styling information. 
 **/
struct ccd_selector_group_ {
	GTree		*sets;
	unsigned int	 n_selectors;
	unsigned int	 min_specificity_e;
	GSList		*dangling_selectors;
};

static int
compare (size_t	 key1,
	 size_t	 key2, 
	 void	*data)
{
	return key1 - key2;
}

static void
free_set (ccd_selector_set_t *set)
{
	GSList		*iter;
	ccd_selector_t	*selector;

	g_assert (set);

	iter = set->selectors;
	while (iter) {
		selector = (ccd_selector_t *) iter->data;
		iter = g_slist_remove (iter, selector);
		ccd_selector_free (selector);
	}

	g_free (set);
}

/**
 * ccd_selector_group_new:
 *
 * Create an empty selector group.
 *
 * A newly allocated #ccd_selector_group_t.
 **/
ccd_selector_group_t *
ccd_selector_group_new (void)
{
	ccd_selector_group_t *self;

	self = g_new0 (ccd_selector_group_t, 1);
	self->sets = g_tree_new_full ((GCompareDataFunc) compare, NULL, NULL, 
					(GDestroyNotify) free_set);
	self->n_selectors = 0;
	self->min_specificity_e = CCD_SELECTOR_MAX_SPECIFICITY;

	return self;
}

/**
 * ccd_selector_group_free:
 * @self: a #ccd_selector_group_t.
 * 
 * Free the selector group and all associated resources.
 **/
void
ccd_selector_group_free (ccd_selector_group_t *self)
{
	g_assert (self);

	g_tree_destroy (self->sets);
	g_free (self);
}

/*
 * The attribute `min_specificity_e' is decremented before each recursive 
 * invocation, so base styles are cascaded root-first.
 */
static void
query_base_r (ccd_selector_group_t	*self,
	      ccd_node_t const		*node)
{
	ccd_node_class_t const	*node_class;
	ccd_selector_t		*selector;
	ccd_node_t		*base;
	char const		*type_name;
	unsigned int		 specificity_e;

	g_assert (self->min_specificity_e >= 0);

	node_class = ccd_node_get_class ();

	specificity_e = self->min_specificity_e;
	base = node_class->get_base_style (node);
	if (base) {
		/* recurse */
		g_assert (self->min_specificity_e > 0);
		self->min_specificity_e--;
		query_base_r (self, base);
		node_class->release (base);
	}

	/* create dangling base type selector and remember for later fixup */
	type_name = node_class->get_type (node);
	selector = ccd_base_type_selector_new (type_name, specificity_e);
	self->dangling_selectors = g_slist_prepend (self->dangling_selectors, selector);
}

/*
 * Takes ownership of the selector.
 */
void
ccd_selector_group_add_selector (ccd_selector_group_t	*self, 
				 ccd_selector_t		*selector)
{
	ccd_selector_set_t	*set;
	size_t			 specificity;

	g_return_if_fail (self && selector);

	/* insert or update the selector group */
	specificity = ccd_selector_get_specificity (selector);
	set = g_tree_lookup (self->sets, GSIZE_TO_POINTER (specificity));
	if (!set) {
		set = g_new0 (ccd_selector_set_t, 1);
		g_tree_insert (self->sets, GSIZE_TO_POINTER (specificity), set);
	}
	set->selectors = g_slist_prepend (set->selectors, selector);
	self->n_selectors++;
}

static unsigned int
calculate_min_specificity_e (ccd_selector_group_t	*group,
			     unsigned int		 n_specificities)
{
	unsigned int specificity_e;

	/* The tree is walked in order, so we remember how many 
	 * specificities `e' will be required to insert the merged selectors at 
	 * the right place. `- 1' because "group->min_specificity_e" already has
	 * the next free value. */
	g_assert (((signed) group->min_specificity_e - (signed) n_specificities - 1) >= 0);
	specificity_e = group->min_specificity_e - n_specificities - 1;

	group->min_specificity_e -= n_specificities;
	g_assert (group->min_specificity_e >= 0);

	return specificity_e;
}

typedef struct {
	ccd_selector_group_t	*self;
	bool			 as_base;
	unsigned int		 specificity_e;
} traverse_merge_info_t;

static bool
traverse_merge (size_t			 	 specificity,
		ccd_selector_set_t const	*set,
		traverse_merge_info_t		*info)
{
	GSList const		*iter;
	ccd_selector_t const	*selector;
	ccd_selector_t		*new_selector;

	g_assert (info->self && set);

	iter = set->selectors;
	while (iter) {
		selector = (ccd_selector_t const *) iter->data;
		if (info->as_base) {
			new_selector = ccd_selector_copy_as_base (selector, info->specificity_e);
		} else {
			new_selector = ccd_selector_copy (selector);
		}
		ccd_selector_group_add_selector (info->self, new_selector);
		iter = iter->next;
	}

	info->specificity_e++;

	return false;
}

void
ccd_selector_group_merge (ccd_selector_group_t		*self,
			  ccd_selector_group_t const	*group)
{
	traverse_merge_info_t info;

	g_assert (self && group);

	info.self = self;
	info.as_base = false;
	info.specificity_e = 0;
	g_tree_foreach (group->sets, (GTraverseFunc) traverse_merge, &info);
}

void
ccd_selector_group_merge_base (ccd_selector_group_t		*self,
			       ccd_selector_group_t const	*group)
{
	traverse_merge_info_t info;

	g_assert (self && group);

	info.self = self;
	info.as_base = true;
	info.specificity_e = calculate_min_specificity_e (self, 
				self->n_selectors);

	g_tree_foreach (group->sets, (GTraverseFunc) traverse_merge, &info);
}

GSList const *
ccd_selector_group_get_dangling_selectors (ccd_selector_group_t const *self)
{
	g_assert (self);

	return self->dangling_selectors;
}

void
ccd_selector_group_clear_dangling_selectors (ccd_selector_group_t *self)
{
	ccd_selector_t	*selector;
	GSList		*iter;

	g_return_if_fail (self && self->dangling_selectors);

	iter = self->dangling_selectors;
	while (iter) {
		selector = (ccd_selector_t *) iter->data;
		iter = g_slist_remove (iter, selector);
		ccd_selector_free (selector);
	}

	self->dangling_selectors = NULL;
}

typedef struct {
	ccd_node_t const	*node;
	ccd_selector_group_t	*result_group;
	bool			 as_base;
	unsigned int		 specificity_e;
	bool			 ret;
} traverse_query_info_t;

static bool
traverse_query (size_t			 specificity,
		ccd_selector_set_t	*set,
		traverse_query_info_t	*info)
{
	ccd_selector_t const	*selector;
	ccd_selector_t		*new_selector;
	GSList const		*iter;
	bool			 ret;

	iter = set->selectors;
	while (iter) {
		selector = (ccd_selector_t const *) iter->data;
		ret = ccd_selector_query_apply (selector, info->node, NULL);
		if (ret) {
			if (info->as_base) {
				new_selector = ccd_selector_copy_as_base (selector, info->specificity_e);
				info->specificity_e++;
			} else {
				new_selector = ccd_selector_copy (selector);
			}
			ccd_selector_group_add_selector (info->result_group, new_selector);
			info->ret = true;
		}
		iter = iter->next;
	}

	return false;
}

bool
ccd_selector_group_query_collect (ccd_selector_group_t const	*self, 
				  ccd_node_t const		*node, 
				  ccd_selector_group_t		*result_group,
				  bool				 as_base)
{
	traverse_query_info_t info;

	g_assert (self && self->sets && node && result_group);

	info.node = node;
	info.result_group = result_group;
	info.as_base = as_base;
	if (as_base) {
		info.specificity_e = calculate_min_specificity_e (result_group,
					self->n_selectors);
	}
	info.ret = false;

	g_tree_foreach (self->sets, (GTraverseFunc) traverse_query, &info);

	return info.ret;
}

typedef struct {
	ccd_node_t const	*node;
	ccd_style_t		*style;
	bool			 ret;
} traverse_match_info_t;

static bool
traverse_match (size_t			 specificity,
		ccd_selector_set_t	*set,
		traverse_match_info_t	*info)
{
	GSList const *iter;

	iter = set->selectors;
	while (iter) {
		info->ret |= ccd_selector_query_apply ((ccd_selector_t const *) iter->data, 
						 info->node, info->style);
		iter = iter->next;
	}

	return false;
}

bool
ccd_selector_group_query_apply (ccd_selector_group_t const	*self,
				ccd_node_t const		*node,
				ccd_style_t			*style)
{
	traverse_match_info_t info;

	g_assert (self && self->sets && node && style);

	info.node = node;
	info.style = style;
	info.ret = false;

	g_tree_foreach (self->sets, (GTraverseFunc) traverse_match, &info);

	return info.ret;
}

static bool
traverse_apply (size_t			 specificity,
		ccd_selector_set_t	*set,
		ccd_style_t		*style)
{
	GSList const *iter;

	iter = set->selectors;
	while (iter) {
		ccd_selector_apply ((ccd_selector_t const *) iter->data, style);
		iter = iter->next;
	}

	return false;
}

/**
 * ccd_selector_group_apply:
 * @self:	a #ccd_selector_group_t.
 * @style:	a #ccd_style_t.
 *
 * Apply the styling information held by #self to #style.
 **/
void
ccd_selector_group_apply (ccd_selector_group_t const	*self, 
			  ccd_style_t			*style)
{
	g_assert (self && self->sets && style);

	g_tree_foreach (self->sets, (GTraverseFunc) traverse_apply, style);
}

static bool
traverse_dump (size_t			 specificity,
	       ccd_selector_set_t	*set,
	       void			*data)
{
	GSList const *iter;

	iter = set->selectors;
	while (iter) {
		ccd_selector_dump ((ccd_selector_t const *) iter->data);
		iter = iter->next;
	}

	return false;
}

/**
 * ccd_selector_group_dump:
 * @self:	a ccd_selector_group_t.
 *
 * Print informations about the internal state of this object.
 **/
void
ccd_selector_group_dump (ccd_selector_group_t const *self)
{
	GSList const *iter;

	g_return_if_fail (self);

	g_tree_foreach (self->sets, (GTraverseFunc) traverse_dump, NULL);

	iter = self->dangling_selectors;
	while (iter) {
		printf ("(dangling) ");
		ccd_selector_dump ((ccd_selector_t const *) iter->data);
		iter = iter->next;
	}
}

