/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <stdbool.h>
#include <string.h>
#include "ccd-function.h"
#include "ccd-image.h"

#if CCD_WITH_RSVG
#include <librsvg/rsvg.h>
#include <librsvg/rsvg-cairo.h>
#endif

void
ccd_image_discard (ccd_image_t *self)
{
	if (self->uri) {
		g_free (self->uri);
		self->uri = NULL;
	}

	if (self->surface) {
		cairo_surface_destroy (self->surface);
		self->surface = NULL;
	}
}

static char *
parse_uri (char const	 *uri,
	   char		**id)
{	
	char const	*suffix;
	char const	*hash;
	char		*basename;

	g_return_val_if_fail (uri, NULL);

	suffix = strrchr (uri, '.');
	hash = strrchr (uri, '#');

	basename = NULL;
	if (suffix && hash && hash > suffix && hash[1] != '\0') {
		basename = g_strndup (uri, hash - uri);
		*id = g_strdup (hash + 1);
	} else {
		basename = g_strdup (uri);
		*id = NULL;
	}

	return basename;
}

#if CCD_WITH_RSVG

static bool
load_svg (ccd_image_t	*self,
	  char const	*uri,
	  char const	*id)
{
	RsvgHandle		*handle;
	GError			*error;
	RsvgDimensionData	 dimensions;
	cairo_t			*cr;
	cairo_status_t		 status;

	error = NULL;
	handle = rsvg_handle_new_from_file (uri, &error);
	if (error) {
		g_warning (error->message);
		g_warning ("Could not load `%s'", uri);
		g_error_free (error);
		return false;
	}

	rsvg_handle_get_dimensions (handle, &dimensions);
	self->surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 
						    dimensions.width,
						    dimensions.height);

	status = cairo_surface_status(self->surface);
	if (status != CAIRO_STATUS_SUCCESS) {
		g_warning (cairo_status_to_string (status));
		cairo_surface_destroy (self->surface);
		self->surface = NULL;
		return false;
	}

	cr = cairo_create (self->surface);
	rsvg_handle_render_cairo_sub (handle, cr, id);
	cairo_destroy (cr), cr = NULL;

	g_object_unref (G_OBJECT (handle)), handle = NULL;

	return true;
}

#endif

static bool
load_image (ccd_image_t *self)
{
	char	*basename;
	char	*id;
	bool	 matched;

	basename = parse_uri (self->uri, &id);
	g_return_val_if_fail (basename, NULL);

	matched = false;
#if CCD_WITH_RSVG
	if (!matched &&
	    g_str_has_suffix (basename, ".svg")) {
		matched = true;
		load_svg (self, basename, id);
	}
#endif

	if (!matched) {
		self->surface = cairo_image_surface_create_from_png (basename);

	}

	g_free (basename);
	g_free (id);

	return (bool) self->surface;
}

ccd_property_spec_t
ccd_image_parse (ccd_image_t	 *self,
		 CRTerm const	**value)
{
	switch ((*value)->type) {
	case TERM_IDENT:
		return ccd_property_parse_spec (value);
	case TERM_URI:
		self->uri = ccd_function_invoke ("url", *value);
		*value = (*value)->next;
		return load_image (self) ? 
			CCD_PROPERTY_SPEC_SET :
			CCD_PROPERTY_SPEC_UNSET;
	default:
		return CCD_PROPERTY_SPEC_UNSET;
	}
}

void
ccd_image_dump (ccd_image_t const *self)
{
	printf (self->uri);
}

