/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * om-bluetooth: code to establish a Bluetooth connection for OBEX FTP
 * Copyright (C) 2007  James Henstridge
 * Copyright (C) 2007  Bastien Nocera
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdlib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>
#include <bluetooth/rfcomm.h>

#include "om-bluetooth.h"
#include "om-utils.h"

#define d(x) x


static int
connect_rfcomm (bdaddr_t *src, bdaddr_t *dst, uint8_t channel)
{
	struct sockaddr_rc addr;
	int fd = -1;

	fd = socket (AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if (fd < 0)
		goto err;

	memset (&addr, 0, sizeof(addr));
	addr.rc_family  = AF_BLUETOOTH;

	bacpy (&addr.rc_bdaddr, src);
	addr.rc_channel = 0;

	if (bind (fd, (struct sockaddr *) &addr, sizeof (addr)) < 0) {
		d(g_printerr ("connect_rfcomm: bind: %s",
			      g_strerror (errno)));
		goto err;
	}

	addr.rc_family  = AF_BLUETOOTH;
	bacpy(&addr.rc_bdaddr, dst);
	addr.rc_channel = channel;

	if (connect(fd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		d(g_printerr ("connect_rfcomm: connect: %s",
			      g_strerror (errno)));
		goto err;
	}

	return fd;
 err:
	if (fd >= 0)
		close (fd);
	return -1;
}

/* From Johan Hedberg:
 *
 * some Nokia Symbian phones have two OBEX FTP services: one
 * identified with the normal UUID and another with a Nokia specific
 * 128 bit UUID. The service found behind the normal identifier is
 * very limited in features on these phones while the other one
 * supports full OBEX FTP (don't ask me why).
 */
static const uint8_t nokia_ftp_uuid[16] =
	{0x00, 0x00, 0x50, 0x05, 0x00, 0x00, 0x10, 0x00,
	 0x80, 0x00, 0x00, 0x02, 0xee, 0x00, 0x00, 0x01};

/* Determine a score for a particular SDP record.  Higher numbers are
 * better.  A 0 means service is not usable.
 *
 * Currently this gives a score of "2" for Nokia's proprietary OBEX
 * file transfer service, and a "1" for the standard OBEX file
 * transfer service.
 */
static int
score_sdp_record(sdp_record_t *rec)
{
	sdp_list_t *classes = NULL, *tmp;
	int score = 0;

	if (sdp_get_service_classes (rec, &classes) != 0)
		goto end;

	for (tmp = classes; tmp != NULL; tmp = tmp->next) {
		uuid_t *uuid = tmp->data;

		if (uuid->type == SDP_UUID128 &&
		    memcmp(&uuid->value.uuid128, nokia_ftp_uuid,
			   sizeof(nokia_ftp_uuid)) == 0) {
			score = 2;
			break;
		} else if (uuid->type == SDP_UUID16 &&
			   uuid->value.uuid16 == OBEX_FILETRANS_SVCLASS_ID) {
			score = 1;
			break;
		}
	}

 end:
	sdp_list_free(classes, free);
	return score;
}

static int
get_rfcomm_channel (sdp_record_t *rec)
{
	int channel = -1;
	sdp_list_t *protos = NULL;

	if (sdp_get_access_protos (rec, &protos) != 0)
		goto end;

	channel = sdp_get_proto_port (protos, RFCOMM_UUID);
 end:
	sdp_list_foreach(protos, (sdp_list_func_t)sdp_list_free, 0);
	sdp_list_free(protos, 0);
	return channel;
}

/* Determine whether the given device supports OBEX FTP, and if so
 * what the RFCOMM channel number for the service is.
 */
static int
find_obexftp_channel (bdaddr_t *adapter, bdaddr_t *device)
{
	sdp_session_t *sdp = NULL;
	sdp_list_t *search = NULL, *attrs = NULL, *recs = NULL, *tmp;
	uuid_t browse_uuid, rfcomm_uuid, obex_uuid;
	uint16_t proto_desc_list = SDP_ATTR_PROTO_DESC_LIST;
	uint16_t svclass_id_list = SDP_ATTR_SVCLASS_ID_LIST;
	int best_score = 0;
	int result = -1;

	sdp = sdp_connect (adapter, device, SDP_RETRY_IF_BUSY);
	if (!sdp) {
		d(g_printerr("find_obexftp_channel: "
			     "Can't connect to SDP service"));
		goto end;
	}

	/* Normally, we'd just do a search for OBEX_FILETRANS_SVCLASS_ID,
	 * but we also want to check multiple services.  So instead, we
	 * browse all rfcomm obex services.
	 */
	sdp_uuid16_create(&browse_uuid, PUBLIC_BROWSE_GROUP);
	sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
	sdp_uuid16_create(&obex_uuid, OBEX_UUID);
	search = sdp_list_append (NULL, &browse_uuid);
	search = sdp_list_append (search, &rfcomm_uuid);
	search = sdp_list_append (search, &obex_uuid);

	attrs = sdp_list_append (NULL, &proto_desc_list);
	attrs = sdp_list_append (attrs, &svclass_id_list);

	if (sdp_service_search_attr_req (sdp, search,
					 SDP_ATTR_REQ_INDIVIDUAL, attrs,
					 &recs)) {
		d(g_printerr("find_obexftp_channel: "
			     "Could not browse services"));
		goto end;
	}
	for (tmp = recs; tmp != NULL; tmp = tmp->next) {
		sdp_record_t *rec = tmp->data;
		int channel, score = score_sdp_record (rec);

		/* If this service is better than what we've
		 * previously seen, try and get the channel number.
		 */
		if (score > best_score) {
			channel = get_rfcomm_channel (rec);
			if (channel > 0) {
				best_score = score;
				result = channel;
			}
		}
	}
	if (best_score == 0)
		d(g_printerr("find_obexftp_channel: "
			     "no suitable channels found"));
 end:
	sdp_list_free (recs, (sdp_free_func_t)sdp_record_free);
	sdp_list_free (search, NULL);
	sdp_list_free (attrs, NULL);
	sdp_close(sdp);

	return result;
}

int
om_bluetooth_connect (const gchar *bda, GnomeVFSResult *result)
{
	bdaddr_t src, dst;
	int channel, fd = -1;

	if (!om_utils_check_bda (bda)) {
		*result = GNOME_VFS_ERROR_INVALID_URI;
		goto end;
	}

	*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;

	bacpy (&src, BDADDR_ANY);
	str2ba(bda, &dst);

	channel = find_obexftp_channel(&src, &dst);
	if (channel < 0)
		goto end;

	fd = connect_rfcomm (&src, &dst, channel);

	if (fd >= 0)
		*result = GNOME_VFS_OK;
 end:
	return fd;
}

void
om_bluetooth_disconnect (int fd)
{
	close (fd);
}
