
/*
 * $Header: /cvs/gnome/gperfmeter/src/perfmeter.c,v 1.45 2004/11/15 17:12:49 richb Exp $
 *
 * Copyright (c) 1990-2004 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 */

#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdarg.h>
#include <stdlib.h>
#include <locale.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "callbacks.h"
#include "access/factory.h"
#include "perfmeter.h"
#include "perfmeter.xbm"

#ifdef SOLARIS
#include <sys/systeminfo.h>
#else
#include <sys/utsname.h>
#endif /*SOLARIS*/

#include <X11/keysym.h>
#include <X11/cursorfont.h>

enum {
    M_CPU = 0, M_LOAD,     M_DISK,  M_PAGE, M_CNTX, 
    M_SWAP,    M_INTR,     M_PKTS,  M_COLL, M_ERRS, 
    M_EXIT,    M_BAR,      M_STRIP, M_BOTH, M_MBAR, 
    M_SETUP,   M_CONTENTS, M_ABOUT
};

static void show_icons(gboolean);
static void menuitem_icon_visibility_notify(GConfClient *, guint,
                                            GConfEntry *, gpointer);

static void gperfmeter_set_atk_relation(GtkWidget *, 
                                        GtkWidget *, AtkRelationType);
static void create_MainWin(void);
static void create_PopupMenu(void);
static void create_pixmaps(void);
static GtkWidget *get_menubar_widget(const char *);
static GtkWidget *get_popup_menu_widget(const char *);
static void menu_handler(gpointer, int, GtkWidget *);
static gboolean DismissPerfmeterCB(GtkWidget *, GdkEvent *, gpointer);
static gboolean MainWinKeyHandler(GtkWidget *, GdkEventKey *, gpointer);
static gboolean WindowHandler(GtkWidget *, GdkEventProperty *, gpointer);
static gboolean CanvasExpose(GtkWidget *, GdkEventExpose *, gpointer);
static gboolean CanvasButtonHandler(GtkWidget *, GdkEventButton *, gpointer);
static gboolean CanvasResize(GtkWidget *, GdkEventConfigure *, gpointer);
static gboolean WindowConfigure(GtkWidget *, GdkEventConfigure *, gpointer);
static void AdjustMenuBar(void);
static void ShowMeter(GtkWidget *, int);
static void HideLiveActionBar(void);
static void HideStripChart(void);
static void PmShowHelp(void);
static void ShowBothChart(void);
static void style_set_callback(GtkWidget *);
static void ToggleMBWidget(gboolean);
static void ValidateStripBarPopup(void);

static gint gperfmeter_accessible_get_n_children(AtkObject *);
static AtkObject *gperfmeter_accessible_ref_child(AtkObject *, gint);

static gboolean check_flag = FALSE;


/* Array for holding the AtkObject structures for the children */
AtkObject *pango_accessible[PmMAX_METERS];

GConfValue *GetGConfResource(char *, char *, char *);

void PutResource(char *, char *, char *, ResType, gpointer);
void ValidateStripBarPopup();


/* Generate the menus and popup menus using a GtkItemFactory */
 
static GtkItemFactoryEntry main_menu[] = {
    { N_("/_File"),                     NULL,         NULL,         0,          "<Branch>"     },
    { N_("/File/_Quit"),                "<control>Q", menu_handler, M_EXIT,     "<StockItem>", GTK_STOCK_QUIT },
    { N_("/_Edit"),                     NULL,         NULL,         0,          "<Branch>" }, 
    { N_("/Edit/_Preferences"),         NULL,         menu_handler, M_SETUP,    "<StockItem>", GTK_STOCK_PREFERENCES }, 
    { N_("/_View"),                     NULL,         NULL,         0,          "<Branch>"     },
    { N_("/View/_Metrics"),             NULL,         NULL,         0,          "<Branch>" }, 
    { N_("/View/Metrics/_CPU"),         "<control>1", menu_handler, M_CPU,      "<ToggleItem>" }, 
    { N_("/View/Metrics/_Load"),        "<control>2", menu_handler, M_LOAD,     "<ToggleItem>" }, 
    { N_("/View/Metrics/_Disk"),        "<control>3", menu_handler, M_DISK,     "<ToggleItem>" }, 
    { N_("/View/Metrics/sep1"),         NULL,         NULL,         0,          "<Separator>"  }, 
    { N_("/View/Metrics/_Page"),        "<control>4", menu_handler, M_PAGE,     "<ToggleItem>" }, 
    { N_("/View/Metrics/C_ontext"),     "<control>5", menu_handler, M_CNTX,     "<ToggleItem>" }, 
    { N_("/View/Metrics/_Job Swaps"),   "<control>6", menu_handler, M_SWAP,     "<ToggleItem>" }, 
    { N_("/View/Metrics/sep2"),         NULL,         NULL,         0,          "<Separator>"  }, 
    { N_("/View/Metrics/_Interrupts"),  "<control>7", menu_handler, M_INTR,     "<ToggleItem>" }, 
    { N_("/View/Metrics/Pac_kets"),     "<control>8", menu_handler, M_PKTS,     "<ToggleItem>" }, 
    { N_("/View/Metrics/Collisio_ns"),  "<control>9", menu_handler, M_COLL,     "<ToggleItem>" }, 
    { N_("/View/Metrics/_Errors"),      "<control>0", menu_handler, M_ERRS,     "<ToggleItem>" }, 
    { N_("/View/_Bar Chart"),           NULL,         menu_handler, M_BAR,      "<RadioItem>" },
    { N_("/View/_Strip Chart"),         NULL,         menu_handler, M_STRIP,    "/View/Bar Chart" },
    { N_("/View/B_oth Charts"),         NULL,         menu_handler, M_BOTH,     "/View/Bar Chart" },
    { N_("/View/sep1"),                 NULL,         NULL,         0,          "<Separator>"  },
    { N_("/View/Men_u Bar"),            "<control>M", menu_handler, M_MBAR,     "<ToggleItem>" },
    { N_("/_Help"),                     NULL,         NULL,         0,          "<Branch>" },
    { N_("/Help/_Contents"),            "F1",         menu_handler, M_CONTENTS, "<StockItem>", GTK_STOCK_HELP  },
    { N_("/Help/_About"),               NULL,         menu_handler, M_ABOUT,    "<StockItem>", GNOME_STOCK_ABOUT },

};

static GtkItemFactoryEntry popup_menu[] = {
    { N_("/_CPU"),             NULL, menu_handler, M_CPU,   "<ToggleItem>" },
    { N_("/_Load"),            NULL, menu_handler, M_LOAD,  "<ToggleItem>" },
    { N_("/_Disk"),            NULL, menu_handler, M_DISK,  "<ToggleItem>" },
    { N_("/sep1"),             NULL, NULL,         0,       "<Separator>"  },
    { N_("/Pa_ge"),            NULL, menu_handler, M_PAGE,  "<ToggleItem>" },
    { N_("/Con_text"),         NULL, menu_handler, M_CNTX,  "<ToggleItem>" },
    { N_("/_Job Swaps"),       NULL, menu_handler, M_SWAP,  "<ToggleItem>" },
    { N_("/sep2"),             NULL, NULL,         0,       "<Separator>"  },
    { N_("/_Interrupts"),      NULL, menu_handler, M_INTR,  "<ToggleItem>" },
    { N_("/Pac_kets"),         NULL, menu_handler, M_PKTS,  "<ToggleItem>" },
    { N_("/Collisio_ns"),      NULL, menu_handler, M_COLL,  "<ToggleItem>" },
    { N_("/_Errors"),          NULL, menu_handler, M_ERRS,  "<ToggleItem>" },
    { N_("/sep3"),             NULL, NULL,         0,       "<Separator>"  },
    { N_("/Show B_othChart"),  NULL, menu_handler, M_BOTH,  "<RadioItem>" },
    { N_("/Show _BarChart"),   NULL, menu_handler, M_BAR,   "/Show BothChart" },
    { N_("/Show _StripChart"), NULL, menu_handler, M_STRIP, "/Show BothChart" },
    { N_("/Show _MenuBar"),    NULL, menu_handler, M_MBAR,  "<ToggleItem>" },
    { N_("/_Preferences"),     NULL, menu_handler, M_SETUP, NULL },
    { N_("/sep4"),             NULL, NULL,         0,       "<Separator>" },
};


PmUIStruct *UI;
PmStruct *PM;
static PangoLayout *layout;
static int doingShowMeter = FALSE;

static guint notify;
static GConfClient *client = NULL;

static void Update(void);
static void CalculateRC(void);
static void DrawGraph(int, gboolean);
static void DrawDead(void);
static void Initialize(void);
static void SetColorMode(void);
static void PostInitialize(void);
static void CreateInterface(void);
static void PmShutdown(void);
static void ResetPopupMenu(void);
static void PmSizeGraphArea(int, int);
static void PmActivate(gboolean);

/* LAB = Live Action Bar */
#define PmSEPARATOR_THICKNESS 2
#define PmLAB_SEPARATOR       5
#define PmLAB_WIDTH           16
#define PmHALF_LAB_WIDTH      8      /* Half the size LAB_WIDTH */

/* Inter-client communications */
static GdkAtom XA_WM_STATE;


int 
main(int argc, char **argv)
{
    int i, pargc;
    char **pargv;

    SETLOCALE(LC_ALL, "");
    BINDTEXTDOMAIN(GETTEXT_PACKAGE, GNOMELOCALEDIR);
    TEXTDOMAIN(GETTEXT_PACKAGE);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    BIND_TEXTDOMAIN_CODESET(GETTEXT_PACKAGE, "UTF-8");
#endif

    /* Copy the command arguments before the call to 
     * gnome_init_with_popt_table() in ParseCmdOptions().
     */
    pargc = argc;
    pargv = (char **) calloc(argc+1, sizeof(char *));
    for (i = 0; i < argc; i++) {
        pargv[i] = argv[i];
    }

    /* Initialize the perfmeter data structures */
    Initialize();

    /* Save the original command line arguments */
    UI->Argc = argc;
    UI->Argv = argv;

    /* Initialize and load the application resources, plus parse cmd line. */
    UI->initializing = TRUE;
    PmLoadResources(pargc, pargv);
    g_free(pargv);
    XA_WM_STATE = gdk_atom_intern("WM_STATE", FALSE);

    PmLock();

    /* Set the color mode */
    SetColorMode();

    /* Create the main window */
    create_MainWin();

    create_pixmaps();

    /* Calculate numbers of rows and columns */
    CalculateRC();

    /* Set the icon for the main window. */
    gdk_window_set_icon(UI->MainWin->window, NULL, UI->IconBM, UI->IconBM);
    gdk_window_set_icon_name(UI->MainWin->window, "gperfmeter");

    /* Create the canvas popup menu. */
    create_PopupMenu();
 
    /* Set the mininimum canvas size */
    PmSetMinSize();

    PmShow(TRUE);

    CreateInterface();

    /* Start to monitor */
    if (!PmMonitor()) {
        GtkWidget *dialog;

        dialog = gtk_message_dialog_new(NULL,
                     GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR,
                     GTK_BUTTONS_CLOSE,
                     (" Please enable rstatd. "));

        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return(1);
    }
    
    PmUnLock();
    ValidateStripBarMenu();
    PostInitialize();
    UI->initializing = FALSE;
    setup_factory();
    gtk_main();

    PmSignalHandler(0);
    return(0);
}


static void
create_pixmaps(void)
{
    /* Create the dead pixmap. */
    UI->DeadPM = gdk_pixmap_create_from_xpm(UI->MainWin->window,
                                            NULL, NULL, 
                                            GPERF_XPMDIR "/dead.xpm");
    PM->DeadWp = 64;
    PM->DeadHp = 64;

    /* Create the sick pixmap. */
    UI->SickPM = gdk_pixmap_create_from_xpm(UI->MainWin->window,
                                            NULL, NULL, 
                                            GPERF_XPMDIR "/sick.xpm");
    PM->SickWp = 64;
    PM->SickHp = 64;

    /* Create the application icon bitmap. */
    UI->IconBM = gdk_bitmap_create_from_data(UI->MainWin->window,
                                             (const gchar *) perfmeter_xbm_bits,
                                             perfmeter_xbm_width,
                                             perfmeter_xbm_height);
}


/*ARGSUSED*/
static char *
item_factory_translate_func(const char *path, gpointer func_data)
{
    return(_(path));
}


static void
show_icons(gboolean use_image)
{
    GtkWidget *quit_image, *about_image, *pref_image, *help_image;

    quit_image = gtk_image_menu_item_get_image(
                   GTK_IMAGE_MENU_ITEM(get_menubar_widget("/File/Quit")));
    pref_image = gtk_image_menu_item_get_image
                 (GTK_IMAGE_MENU_ITEM(get_menubar_widget("/Edit/Preferences")));
    help_image = gtk_image_menu_item_get_image
                 (GTK_IMAGE_MENU_ITEM(get_menubar_widget("/Help/Contents")));
    about_image = gtk_image_menu_item_get_image
                 (GTK_IMAGE_MENU_ITEM(get_menubar_widget("/Help/About")));

    if (use_image) {
        gtk_widget_show(quit_image);
        gtk_widget_show(pref_image);
        gtk_widget_show(help_image);
        gtk_widget_show(about_image);
    } else {
        gtk_widget_hide(quit_image);
        gtk_widget_hide(pref_image);
        gtk_widget_hide(help_image);
        gtk_widget_hide(about_image);
    }
}


/*ARGSUSED*/
static void
menuitem_icon_visibility_notify(GConfClient *client, guint cnxn_id,
                                GConfEntry *entry, gpointer user_data)
{
    show_icons(gconf_value_get_bool(entry->value));
}


static void
SetBackingStoreInfo(int meter)
{
    XSetWindowAttributes swattr;
    XWindowAttributes wattr;
    GdkWindow *cw = gnome_perfmeter_get_chart_win(
                        GNOME_PERFMETER(UI->GMeters[meter]));

    /* Turn on backing store on the chart window */
    swattr.backing_store = WhenMapped;

    XChangeWindowAttributes(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(cw),
                            CWBackingStore, &swattr);

    /* Get the current canvas window attributes */
    XGetWindowAttributes(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(cw), &wattr);

    if (wattr.backing_store == NotUseful) {
        /* Turn off backingstore flag */
        UI->BackingStore = FALSE;
#ifdef DEBUG
        PRINTF("Backing Store is OFF\n");
#endif
    }
}


static void
MakeNewMeter(int n)
{
    GtkWidget *chart;
    char buf[40];

    UI->GMeters[n] = gnome_perfmeter_new();
    gnome_perfmeter_set_maxval(GNOME_PERFMETER(UI->GMeters[n]),
                               PM->Meters[n].TRName);
    SPRINTF(buf, "%1d", PM->Meters[n].CurMax);
    if (n == PM_CPU) {
        SPRINTF(buf, "%1d%%", PM->Meters[n].CurMax);
    }
    gnome_perfmeter_set_meter_type(GNOME_PERFMETER(UI->GMeters[n]), buf);
    gtk_widget_hide_all(UI->GMeters[n]);
    gtk_widget_ref(UI->GMeters[n]);

    chart = gnome_perfmeter_get_chart(GNOME_PERFMETER(UI->GMeters[n]));
    g_signal_connect(G_OBJECT(chart), "expose_event",
                     GTK_SIGNAL_FUNC(CanvasExpose), (gpointer) n);
    g_signal_connect(G_OBJECT(chart), "button_press_event",
                     GTK_SIGNAL_FUNC(CanvasButtonHandler), (gpointer) n);
    g_signal_connect(G_OBJECT(chart), "configure_event",
                     GTK_SIGNAL_FUNC(CanvasResize), (gpointer) n);
}


static void
AdjustMeterDisplay(int meter, int addMeter)
{
    GtkWidget *box;
    GtkWidget *GMeter = UI->GMeters[meter];

    if (PM->Orientation == PM_VERTICAL) {
        box = UI->MeterVBox;
        gnome_perfmeter_show_vsep(GNOME_PERFMETER(GMeter), (meter > 0));
    } else {
        box = UI->MeterHBox;
        gnome_perfmeter_show_hsep(GNOME_PERFMETER(GMeter), (meter > 0));
    }

    if (addMeter == TRUE) {
        gtk_box_pack_start(GTK_BOX(box), GMeter, TRUE, TRUE, 0);
        gtk_widget_show_all(GMeter);
    } else {
        gtk_container_remove(GTK_CONTAINER(box), GMeter);
    }
}


void
ToggleDirection(int newDir)
{
    int i, meter;

    if (PM->Orientation != newDir) {
        for (i = 0; i < PM->NMeters; i++) {
	    meter = PM->DisplayList[i];
            AdjustMeterDisplay(meter, FALSE);
        }
        gtk_widget_hide(newDir == PM_VERTICAL ? UI->MeterHBox : UI->MeterVBox);

        PM->Orientation = newDir;

        for (i = 0; i < PM->NMeters; i++) {
	    meter = PM->DisplayList[i];
            AdjustMeterDisplay(meter, TRUE);
        }
        gtk_widget_show(newDir == PM_VERTICAL ? UI->MeterVBox : UI->MeterHBox);
    }
}


static void
create_MainWin(void)
{
    AtkObject *access_object;
    AtkObjectClass *access_object_class;
    GConfValue *xgptr, *ygptr, *wgptr, *hgptr;
    gboolean use_image;
    GError *error;
    int i, x, y, width, height;
    int count = sizeof(main_menu) / sizeof(main_menu[0]);

    UI->MainWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    UI->accel = gtk_accel_group_new();
    g_signal_connect(G_OBJECT(UI->MainWin), "key_press_event",
                     GTK_SIGNAL_FUNC(MainWinKeyHandler), NULL);
    g_signal_connect(G_OBJECT(UI->MainWin), "delete_event",
                     GTK_SIGNAL_FUNC(DismissPerfmeterCB), NULL);
    g_signal_connect(G_OBJECT(UI->MainWin), "property_notify_event",
                     GTK_SIGNAL_FUNC(WindowHandler), NULL);

    /* This gets your window automatically resizing down on hiding of its
     * parts.
     */
    gtk_window_set_resizable(GTK_WINDOW(UI->MainWin), TRUE);
    gtk_window_set_default_size(GTK_WINDOW(UI->MainWin), 250, -1);

    /* Get the initial position of the gperfmeter window. */
    if ((xgptr = GetGConfResource(NULL, NULL, "WindowXPos"))) {
        x = gconf_value_get_int(xgptr);
    }

    if ((ygptr = GetGConfResource(NULL, NULL, "WindowYPos"))) {
        y = gconf_value_get_int(ygptr);
    }
    if (xgptr && ygptr) {
        gtk_window_move(GTK_WINDOW(UI->MainWin), x, y);
    }

    /* Get the initial width/height of the gperfmeter window. */
    if ((wgptr = GetGConfResource(NULL, NULL, "WindowWidth"))) {
	width = gconf_value_get_int(wgptr);
    }

    if ((hgptr = GetGConfResource(NULL, NULL, "WindowHeight"))) {
	height = gconf_value_get_int(hgptr);
    }
    if (wgptr && hgptr) {
        gtk_window_set_default_size(GTK_WINDOW(UI->MainWin), width, height);
    }

    set_title();

    /* Main vbox */
    UI->VBox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(UI->VBox);
    gtk_container_add(GTK_CONTAINER(UI->MainWin), UI->VBox);

    /* Make menubar from itemfactory */
    UI->fact = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", UI->accel);
    gtk_item_factory_set_translate_func(UI->fact, item_factory_translate_func,
                                        NULL, NULL);
    gtk_item_factory_create_items(UI->fact, count, main_menu, NULL);
    UI->MenuBar = gtk_item_factory_get_widget(UI->fact, "<main>");

    client = gconf_client_get_default();
    error = NULL;
    use_image = gconf_client_get_bool(client,
                    "/desktop/gnome/interface/menus_have_icons",&error);
    if (error) {
        g_printerr(_("There was an error loading config value for whether to use image in menus. (%s)\n"), error->message);
        g_error_free(error);
    } else {
       show_icons(use_image);
    }

    notify = gconf_client_notify_add(client, 
                 "/desktop/gnome/interface/menus_have_icons",
                 menuitem_icon_visibility_notify, NULL, NULL, &error);
    if (error) {
        g_printerr(_("There was an error subscribing to notification of menu icon visibilty changes. (%s)\n"), error->message);
        g_error_free(error);
    }

    gtk_widget_show(UI->MenuBar);
    gtk_box_pack_start(GTK_BOX(UI->VBox), UI->MenuBar, FALSE, FALSE, 0);

    UI->MeterHBox = gtk_hbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(UI->VBox), UI->MeterHBox, TRUE, TRUE, 0);
    if (PM->Orientation == PM_HORIZONTAL) {
        gtk_widget_show(UI->MeterHBox);
    }

    UI->MeterVBox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(UI->VBox), UI->MeterVBox, TRUE, TRUE, 0);
    if (PM->Orientation == PM_VERTICAL) {
        gtk_widget_show(UI->MeterVBox);
    }

    for (i = 0; i < PmMAX_METERS; i++) {
        MakeNewMeter(i);
    }

    for (i = 0; i < PM->NMeters; i++) {
	int meter = PM->DisplayList[i];

        AdjustMeterDisplay(meter, TRUE);
    }

    access_object = gtk_widget_get_accessible(UI->VBox);

    atk_object_set_name(access_object, _("Container VBox"));

    access_object_class = ATK_OBJECT_GET_CLASS(access_object);
    access_object_class->get_n_children = gperfmeter_accessible_get_n_children;
    access_object_class->ref_child = gperfmeter_accessible_ref_child;

    g_signal_connect(G_OBJECT(UI->VBox), "style_set",
                     G_CALLBACK(style_set_callback), UI->VBox);
    g_signal_connect(G_OBJECT(UI->MainWin), "configure_event",
                     G_CALLBACK(WindowConfigure), NULL);

    gtk_widget_realize(UI->MainWin);

    gperfmeter_set_atk_relation(get_menubar_widget("/View/Menu Bar"), 
                                UI->MenuBar, ATK_RELATION_CONTROLLER_FOR);
    gperfmeter_set_atk_relation(UI->MenuBar, 
                                get_menubar_widget("/View/Menu Bar"), 
                                ATK_RELATION_CONTROLLED_BY);
    /* This will listen for our menubar keys and stuff */
    gtk_window_add_accel_group(GTK_WINDOW(UI->MainWin), UI->accel);
}


static void
create_PopupMenu()
{
    int count = sizeof(popup_menu) / sizeof(popup_menu[0]);

    UI->pfact = gtk_item_factory_new(GTK_TYPE_MENU, "<popup>", UI->accel);
    gtk_item_factory_set_translate_func(UI->pfact, item_factory_translate_func,
                                        NULL, NULL);

    gtk_item_factory_create_items(UI->pfact, count, popup_menu, NULL);
    UI->PopupMenu = gtk_item_factory_get_widget(UI->pfact, "<popup>");
    gtk_container_set_border_width(GTK_CONTAINER(UI->PopupMenu), 1);
}


/* Helper function to get the menubar item from its path. */

GtkWidget *
get_menubar_widget(const char *path)
{
    return(gtk_item_factory_get_item(UI->fact, path));
}


/* Helper function to get the popup menu item from its path. */

GtkWidget *
get_popup_menu_widget(const char *path)
{
    return(gtk_item_factory_get_item(UI->pfact, path));
}


/* Callback to handle delete events for the "About Perfmeter" popup window. */
 
/*ARGSUSED*/
gboolean
DismissAbout(GtkWidget *widget, GdkEvent  *event, gpointer  user_data)
{
    UI->Aboutdialog = NULL;
 
    return(FALSE);
}


static void
AboutPerf(void)
{
    static GtkWidget *about = NULL;
    gchar *authors[] = {
                   "Rich Burridge <rich.burridge@sun.com>",
                   NULL
    };
    gchar *documenters[] = {
                   NULL
    };

    /* Translator credits */
    gchar *translator_credits = _("translator_credits");

    PmBusy(TRUE);
    if (about != NULL)
    {
        gdk_window_show(about->window);
        gdk_window_raise(about->window);
        PmBusy(FALSE);
        return;
    }
    about = gnome_about_new(_("The Performance Meter"), VERSION,
                            "(C) 1998 the Free Software Foundation",
                            _("Monitor disk, memory, CPU and network usage"),
                            (const char **) authors,
                            (const char **) documenters,
                            strcmp(translator_credits, "translator_credits") !=0 ? translator_credits : NULL, NULL);
    g_signal_connect(G_OBJECT(about), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &about);
    gtk_widget_show(about);

    PmBusy(FALSE);
}


/* Handle all the various menu items. */

/*ARGSUSED*/
void
menu_handler(gpointer data, int action, GtkWidget *item)
{
    /* We don't want to do stupid things while we are setting defaults. */
    if (UI->initializing) {
        return;
    }

    switch (action) {
        case M_CPU:
        case M_LOAD:
        case M_DISK:
        case M_PAGE:
        case M_CNTX:
        case M_SWAP:
        case M_INTR:
        case M_PKTS:
        case M_COLL:
        case M_ERRS:
            ShowMeter(item, action);
            break;

        case M_EXIT:
            PmShutdown();
            break;

        case M_BAR:
            if (check_flag) {
                return;
            }
            HideLiveActionBar();
            break;

        case M_STRIP:
            if (check_flag) {
                return;
            }
            HideStripChart();
            break;

        case M_BOTH:
            if (check_flag) {
                return;
            }
            ShowBothChart();
            break;


        case M_MBAR:
            /* This is the *ONLY* place where you should call AdjustMenuBar
             * because everything else simply toggles the checkbutton in the
             * menu, causing it to come here and do its job.
             */
            AdjustMenuBar();
            break;

        case M_SETUP:
            PmShowProps();
            break;


        case M_CONTENTS:
            PmShowHelp();
            break;

        case M_ABOUT:
            AboutPerf();
            break;
    }

    PmSaveResources();    
}


/* Initialize the application data structures . */

static void 
Initialize()
{
#ifndef SOLARIS
    struct utsname utsbuf;
#endif /*SOLARIS*/
    char buf[1000];
    int count;

    /* Perfmeter resource structure */
    PM = (PmStruct *) calloc(1, sizeof(PmStruct));

    /* Set the application class name */
    PM->AppClass = "Gperfmeter";
    PM->AppName  = "gnome-perfmeter";

    /* Perfmeter meter data structures */
    PM->Meters = (PmMeterStruct *) calloc(PmMAX_METERS,
        sizeof(PmMeterStruct));

    /* Perfmeter host data structure */
    PMD = (PmDataStruct *) calloc(1, sizeof(PmDataStruct));

    /* Interface data structure */
    UI = (PmUIStruct *) calloc(1, sizeof(PmUIStruct));

    /* Interface meter data structures */
    UI->Meters = (PmUIMeterStruct *) calloc(PmMAX_METERS,
                                            sizeof(PmUIMeterStruct));

    /* Size & reset the meter data cache */
    PmResizeMeterData(500);
    PmResetMeterData();

    /* Get the name of the current host */
    buf[0] = '\0';
#ifdef SOLARIS
    count = sysinfo(SI_HOSTNAME, buf, 1000);
#else
    uname(&utsbuf);
    STRCPY(buf, utsbuf.nodename);
    count = strlen(buf);
#endif /*SOLARIS*/

    PM->LocalMachine = (char *) calloc(count + 1, sizeof(char));
    STRCPY(PM->LocalMachine, buf);
    PM->ParentPID = getpid();
}


static void 
SetColorMode()
{
    GdkVisual *visual;

    visual = gdk_visual_get_system();
    if ((visual->depth > 1) &&
        ((visual->type == GDK_VISUAL_PSEUDO_COLOR) ||
         (visual->type == GDK_VISUAL_STATIC_COLOR) ||
         (visual->type == GDK_VISUAL_TRUE_COLOR) ||
         (visual->type == GDK_VISUAL_DIRECT_COLOR))) {
        UI->UseColor = TRUE;
    } else {
        /* staticgray and grayscale */
        UI->UseColor = FALSE;
    }

#ifdef DEBUG
    PRINTF("Use Color = %d\n", UI->UseColor);
#endif
}


static void 
PostInitialize()
{
    GnomePerfmeter *GMeter = GNOME_PERFMETER(UI->GMeters[PM->DisplayList[0]]);
    int i, state;
    gint width, height;

    /* Set the signal handlers */
    PmSetSignalHandlers();

    /* Set the backing store flag */
    state = DoesBackingStore(UI->RScreen);

    if (state == WhenMapped || state == Always) {
        UI->BackingStore = TRUE;
#ifdef DEBUG
        PRINTF("Backing Store is ON\n");
#endif
    }
    else {
        UI->BackingStore = FALSE;
#ifdef DEBUG
        PRINTF("Backing Store is OFF\n");
#endif
    }

    if (UI->BackingStore) {
        for (i = 0; i < PM->NMeters; i++) {
    	    int meter = PM->DisplayList[i];

            SetBackingStoreInfo(meter);
        }
    }

    UI->WaitCursor = gdk_cursor_new(XC_watch);

    /* Set the meter colors */
    for (i = 0; i < PM->NMeters; i++) {
        PmSetMeterColor(PM->DisplayList[i]);
    }

    /* Set the graph type */
    PmSetGraphType(PM->GraphType);

    /* Calculate the graph size based on the canvas size */
    width = gnome_perfmeter_get_chart_width(GMeter);
    height = gnome_perfmeter_get_chart_height(GMeter);

    if (width < 10 || height < 10) {
        /* We assume no canvas size was specified */
        if (PM->ShowMenubar == TRUE) {
            int mwidth = UI->MenuBar->allocation.width;

            /* So that menubar doesn't appear squished */
            PM->AreaWp = mwidth;
        } else {
            PM->AreaWp = 64;
        }
        PM->AreaHp = 64;
#if DEBUG
        PRINTF("What no canvas size specified!\n");
#endif
    } else {
        PmSizeGraphArea(width, height);
    }

    ResetPopupMenu();
    PmSaveCommand();
}


/* Create user interface. Ie. do all the things that Glade can't. */

static void 
CreateInterface()
{
    GnomePerfmeter *GMeter;
    GdkWindow *cw;

    /* There will always be at least one meter in the display list. */
    GMeter = GNOME_PERFMETER(UI->GMeters[PM->DisplayList[0]]);
    cw = gnome_perfmeter_get_chart_win(GMeter);

    UI->ViewMeters[M_CPU]   = get_menubar_widget("/View/Metrics/CPU");
    UI->ViewMeters[M_LOAD]  = get_menubar_widget("/View/Metrics/Load");
    UI->ViewMeters[M_DISK]  = get_menubar_widget("/View/Metrics/Disk");
    UI->ViewMeters[M_PAGE]  = get_menubar_widget("/View/Metrics/Page");
    UI->ViewMeters[M_CNTX]  = get_menubar_widget("/View/Metrics/Context");
    UI->ViewMeters[M_SWAP]  = get_menubar_widget("/View/Metrics/Job Swaps");
    UI->ViewMeters[M_INTR]  = get_menubar_widget("/View/Metrics/Interrupts");
    UI->ViewMeters[M_PKTS]  = get_menubar_widget("/View/Metrics/Packets");
    UI->ViewMeters[M_COLL]  = get_menubar_widget("/View/Metrics/Collisions");
    UI->ViewMeters[M_ERRS]  = get_menubar_widget("/View/Metrics/Errors");

/* Set data values for the ten meter menu entries in the canvas popup menu. */

    UI->PopupMeters[M_CPU]  = get_popup_menu_widget("/CPU");
    UI->PopupMeters[M_LOAD] = get_popup_menu_widget("/Load");
    UI->PopupMeters[M_DISK] = get_popup_menu_widget("/Disk");
    UI->PopupMeters[M_PAGE] = get_popup_menu_widget("/Page");
    UI->PopupMeters[M_CNTX] = get_popup_menu_widget("/Context");
    UI->PopupMeters[M_SWAP] = get_popup_menu_widget("/Job Swaps");
    UI->PopupMeters[M_INTR] = get_popup_menu_widget("/Interrupts");
    UI->PopupMeters[M_PKTS] = get_popup_menu_widget("/Packets");
    UI->PopupMeters[M_COLL] = get_popup_menu_widget("/Collisions");
    UI->PopupMeters[M_ERRS] = get_popup_menu_widget("/Errors");

/* This doesn't do the callback so we have to call ToggleMBWidget() to 
 * hide/show it.
 */

    ToggleMenuBar(PM->ShowMenubar);    
    ToggleMBWidget(PM->ShowMenubar);

    UI->FgPixel = GMeter->chart->style->fg[GTK_STATE_NORMAL];
    UI->BgPixel = GMeter->chart->style->bg[GTK_STATE_NORMAL];

    /* Sets Label and Canvas Color. */
    PM->CanvasColor = UI->BgPixel;
    PM->LabelColor =  UI->FgPixel;

    UI->CanvasGC = gdk_gc_new(cw);
    gdk_gc_set_exposures(UI->CanvasGC, TRUE);

    UI->FillGC = gdk_gc_new(cw);
    gdk_gc_set_fill(UI->FillGC, GDK_SOLID);

    PmSetColors();
    gperfmeter_set_atk_relation(get_popup_menu_widget("/Show MenuBar"), 
                                UI->MenuBar, ATK_RELATION_CONTROLLER_FOR);
    gperfmeter_set_atk_relation(UI->MenuBar, 
                                get_popup_menu_widget("/Show MenuBar"), 
                                ATK_RELATION_CONTROLLED_BY);

    UI->UICreated = TRUE;
}


/*ARGSUSED*/
static gboolean
CanvasExpose(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{
    DrawGraph((int) user_data, TRUE);

    return(FALSE);
}


/* Handle configure events on the meter drawing canvas. */

/*ARGSUSED*/
static gboolean
CanvasResize(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data)
{
    gint width  = widget->allocation.width;
    gint height = widget->allocation.height;

    if (UI->UICreated == FALSE) {
        return(TRUE);
    }

    PmLock();

#ifdef DEBUG
    PRINTF("Resize width = %d, height = %d\n", width, height);
#endif

    PmSetMinSize();
    PmSizeGraphArea(width, height);
    PmRefresh(TRUE);
    DrawGraph((int) user_data, TRUE);
    PmUnLock();

    return(TRUE);
}


/*ARGSUSED*/
static gboolean
WindowConfigure(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data)
{
    gint x, y;
    gint width	= widget->allocation.width;
    gint height = widget->allocation.height;

    gtk_window_get_position(GTK_WINDOW(widget), &x, &y);

    /* Save current x/y position of the gperfmeter window. */
    PutResource(NULL, NULL, "WindowXPos", RES_INT, GINT_TO_POINTER(x));
    PutResource(NULL, NULL, "WindowYPos", RES_INT, GINT_TO_POINTER(y));

    /* Save current width/height of the gperfmeter window. */
    PutResource(NULL, NULL, "WindowWidth", RES_INT, GINT_TO_POINTER(width));
    PutResource(NULL, NULL, "WindowHeight", RES_INT, GINT_TO_POINTER(height));

    return(FALSE);
}


/* Handle other events on the meter canvas. */

/*ARGSUSED*/
static gboolean
CanvasButtonHandler(GtkWidget *widget, GdkEventButton *event, 
                    gpointer user_data)
{
    if (event->button == 3) {
        gtk_menu_popup(GTK_MENU(UI->PopupMenu),
                       NULL, NULL, NULL, NULL, event->button, event->time);
    } else if (event->button == 1) {
        if (!PM->inFrontPanel) {
            if (!GTK_WIDGET_VISIBLE(UI->MenuBar)) {

                /* Not in front panel and menubar is hidden. Show menubar. */
                PM->ShowMenubar = TRUE;
                PmSetMinSize();
                ToggleMenuBar(PM->ShowMenubar);
                gtk_widget_map(UI->MenuBar);
                PmResetPropsDecor();
            }
        }
    }

    return(FALSE);
}


/* Show the specified meter. */

static void
ShowMeter(GtkWidget *item, int meter)
{
    gboolean set = GTK_CHECK_MENU_ITEM(item)->active;

    if (doingShowMeter == FALSE) {
        doingShowMeter = TRUE;

#ifdef DEBUG
        PRINTF("ShowMeter: Set meter %d to %s\n", meter, set ? "TRUE": "FALSE");
#endif

        AdjustMeterDisplay(meter, set);
        if (set) {
            SetBackingStoreInfo(meter);
            PmSetMeterColor(meter);
        }

        PmResetMeter(meter, set);
        PmPropsSetDisplayList();
        ResetPopupMenu();

        PmRedisplay();
        doingShowMeter = FALSE;
    }
}


static void
HideLiveActionBar(void)
{
    PM->ShowLiveActionBar = TRUE;
    PM->ShowStripChart = FALSE;
    PM->StripChartActivity = TRUE;
    PmSetMinSize();
    PmRedisplay();
    PmResetPropsDecor();
    PM->StripChartActivity = FALSE;
    ValidateStripBarMenu();
}


/* Hide strip chart. */

static void
HideStripChart(void)
{
    PM->ShowLiveActionBar = FALSE;
    PM->ShowStripChart = TRUE;
    PM->StripChartActivity = TRUE;
    PmSetMinSize();
    PmRedisplay();
    PmResetPropsDecor();
    PM->StripChartActivity = FALSE;
    ValidateStripBarMenu();
}

static void
ShowBothChart(void)
{
    PM->ShowLiveActionBar = TRUE;
    PM->ShowStripChart = TRUE;
    PM->StripChartActivity = TRUE;
    PmSetMinSize();
    PmRedisplay();
    PmResetPropsDecor();
    PM->StripChartActivity = FALSE;
    ValidateStripBarMenu();
}


/* This function needs a better name - but what this does is handles toggling
 * the menubar status. This is *the* function you want to call when you want
 * to do anything with the menubar. This will toggle the menu item which in
 * turn will generate a "toggled" callback which will go to
 * menu_handler(), M_MBAR, which will call ToggleMenuBar() which will call
 * AdjustMenuBar(), and this is what you want. Perhaps AdjustMenuBar can be
 * done away with and this function can take its name or something. Or code
 * from ToggleMenuBar can live in AdjustMenuBar and this function renamed to
 * ToggleMenuBar().
 */

void
ToggleMenuBar(int visible)
{
    GtkWidget *menu_item,*popup_item;

    /* Toggle the MenuBar in MenuBar menu */
    menu_item = get_menubar_widget("/View/Menu Bar");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), visible);

    /* Toggle the MenuBar in PopUp menu */
    popup_item = get_popup_menu_widget("/Show MenuBar");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(popup_item), visible);
}


static void
ToggleMBWidget(gboolean makeVisible)
{
    if (makeVisible) {
        gtk_widget_show(UI->MenuBar);
    } else {
        gtk_widget_hide(UI->MenuBar);
    }
}


static void
AdjustMenuBar()
{
    if (GTK_WIDGET_VISIBLE(UI->MenuBar)) {
        PM->ShowMenubar = FALSE;
        PmSetMinSize();
        ToggleMenuBar(FALSE);    
        ToggleMBWidget(FALSE);
    } else {
        PM->ShowMenubar = TRUE;
        PmSetMinSize();
        ToggleMenuBar(TRUE);    
        ToggleMBWidget(TRUE);
    }
    PmResetPropsDecor();
}


/*ARGSUSED*/
static gboolean
DismissPerfmeterCB(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    g_object_unref(G_OBJECT(layout));
    PmShutdown();

    return(FALSE);
}


static void
PmShowHelp()
{
    GError *error = NULL;

    gnome_help_display("gnome-perfmeter", NULL, &error);
    if (error) {
        g_warning("help error: %s\n", error->message);
        g_error_free(error);
        error = NULL;
    }
}


/* Shutdown the application. */

static void 
PmShutdown()
{
    gtk_main_quit();
}


typedef struct _PropWMState {
    CARD32    state;
    BITS32    icon;
} PropWMState;

#define PROP_WM_STATE_ELEMENTS 2


/*ARGSUSED*/
static gboolean 
WindowHandler(GtkWidget *widget, GdkEventProperty *event, gpointer user_data)
{
    static gboolean setupShowing = FALSE;

    if (event->atom == XA_WM_STATE) {
        PropWMState *property = NULL;
        Atom actual_type;
        int actual_format, ret_val;
        unsigned long nitems, leftover;

#ifdef DEBUG
        PRINTF("PropertyNotify WM_STATE Atom = %d, State = %d\n",
               event->atom, event->state);
#endif

        /* Get the property data */
        ret_val = XGetWindowProperty(GDK_DISPLAY(),
                                     GDK_WINDOW_XWINDOW(widget->window),
                                     gdk_x11_atom_to_xatom (XA_WM_STATE), 0L,
                                     PROP_WM_STATE_ELEMENTS, FALSE,
                                     gdk_x11_atom_to_xatom (XA_WM_STATE),
                                     &actual_type, &actual_format,
                                     &nitems, &leftover, 
                                     (unsigned char **) &property);
 
        if (!((ret_val == Success) &&
            (actual_type == gdk_x11_atom_to_xatom (XA_WM_STATE)) &&
            (nitems == PROP_WM_STATE_ELEMENTS))) {

            /* The property could not be retrieved or is not
             * correctly set up. Therefore, we will assume
             * there is no window manager running and do nothing.
             */
            if (property) {
                XFree((char *) property);
 
                return(FALSE);
            }
        } 

        /* Set the new display state */
        switch (property->state) {
            case NormalState:
                PM->Showing = TRUE;
                PmActivate(TRUE);
                PmRedisplay();

                /* Popup prop sheet if it was showing earlier */
                if (setupShowing) {
                    PmShowProps();
                }
                break;

            case WithdrawnState:
            break;

            case IconicState:
                PM->Showing = FALSE;
                if (PM->CollectWhenIconized == FALSE) {
                    PmActivate(FALSE);
                }
                setupShowing = PmHideProps();
                break;
        }
 
        /* Free the property */
        XFree((char *) property);
    }

    return(TRUE);
}


/* Reset the toggles in the view menu and the canvas popup menus to reflect 
 * the correct meter state.
 */

static void 
ResetPopupMenu()
{
    int i, j, match;

    if (!UI->PopupMenu) {
        return;
    }

#ifdef DEBUG
    PRINTF("ResetPopupMenu: ");
#endif
    for (i = 0; i < PmMAX_METERS; i++) {
        match = 0;
        for (j = 0; j < PM->NMeters; j++) {
            if (i == PM->DisplayList[j]) {
                match = 1;
                break;
            }
        }
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(UI->PopupMeters[i]),
                                       match);

        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(UI->ViewMeters[i]),
                                       match);

#ifdef DEBUG
        PRINTF("%d ", match);
#endif
    }
#ifdef DEBUG
    PRINTF("\n");
#endif

    gdk_flush();
}


/* Checks the three "Show" items on the option menu depending upon whether
 * these options are set.
 */

extern void 
ValidateStripBarMenu()
{
    GtkWidget *radio_widget;

    /* Check_flag stops the recursive call */  
    check_flag = TRUE;
    
    if (PM->ShowLiveActionBar && PM->ShowStripChart) {
        radio_widget=get_menubar_widget("/View/Both Charts");
    } else if (PM->ShowLiveActionBar) {
        radio_widget=get_menubar_widget("/View/Bar Chart");
    } else {
        radio_widget=get_menubar_widget("/View/Strip Chart");
    }

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(radio_widget), TRUE);

    /* Validate the popup menu also */
    ValidateStripBarPopup(); 
    check_flag = FALSE;    
}


static void 
ValidateStripBarPopup()
{
    GtkWidget *radio_widget;

    if (PM->ShowLiveActionBar && PM->ShowStripChart) {
        radio_widget=get_popup_menu_widget("/Show BothChart");
    } else if (PM->ShowLiveActionBar) {
        radio_widget=get_popup_menu_widget("/Show BarChart");
    } else {
        radio_widget=get_popup_menu_widget("/Show StripChart");
    }

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(radio_widget), TRUE);
}


/* Timer callback to get the next host datapoint and redisplay the meters. */

/*ARGSUSED*/
static gint
UpdateTimer(gpointer data)
{
    if (UI->TimerId == 0) {
        return(TRUE);
    }

    /* Update the meter data */
    if (!PM->IsDead) {
        if (PmUpdateData()) {
            PmScaleData();
        }

        Update();

        if (PM->Log) {
            PMLOGDATA();
        }
    }

    return(TRUE);
}


/* Interrupt timer callback to reset the meters. */

/*ARGSUSED*/
extern gint 
InterruptTimer(gpointer data)
{
    PmActivate(FALSE);
    PmActivate(TRUE);

    return(TRUE);
}


/* Set an interrupt timer to cause a full reset and redisplay. */

void 
PmSetRedisplayInterrupt()
{
    gtk_timeout_add(0, InterruptTimer, NULL);
}


/* Update the meter with the new data point. */

static void
Update(void)
{
    int i, meter;

    if (PM->LockCount > 0 || !PM->Showing) {
        return;
    }

    gdk_flush();

    if (PM->IsDead || PM->IsSick) {
        DrawDead();
	return;
    }

    for (i = 0; i < PM->NMeters; i++) {
        meter = PM->DisplayList[i];
        DrawGraph(meter, FALSE);
    }
}


static void
DrawDead()
{
    int i, x, y;

    if (PM->IsDead || PM->IsSick) {
        PmRefresh(FALSE);

        for (i = 0; i < PM->NMeters; i++) {
            int meter = PM->DisplayList[i];
            GnomePerfmeter *gp = GNOME_PERFMETER(UI->GMeters[meter]);
            GdkWindow *cw = gnome_perfmeter_get_chart_win(gp);
            int width = gnome_perfmeter_get_chart_width(gp);
            int height = gnome_perfmeter_get_chart_height(gp);

            if (PM->IsDead) {
                x = (width - PM->DeadWp) / 2;
                y = (height - PM->DeadHp) / 2;
                PmAssignMax(x, 0);
                PmAssignMax(y, 0);

                gdk_draw_drawable(cw, UI->CanvasGC, UI->DeadPM,
                                  0, 0, x, y, PM->DeadWp, PM->DeadHp);
            } else {
                x = (width - PM->SickWp) / 2;
                y = (height - PM->SickHp) / 2;
                PmAssignMax(x, 0);
                PmAssignMax(y, 0);

                gdk_draw_drawable(cw, UI->CanvasGC, UI->SickPM,
                                  0, 0, x, y, PM->SickWp, PM->SickHp);
            }
        }
    }
}


/* Calculate the rows and columns. */

static void 
CalculateRC()
{
    double ratio;

    if (PM->Orientation == PM_HORIZONTAL) {
        if (PM->NMeters <= PM->WrapCount) {
            PM->Cols = PM->NMeters;
            PM->Rows = 1;
        } else {
            PM->Cols = PM->WrapCount;
            ratio = (double) PM->NMeters / PM->Cols;
            PM->Rows = (int) ceil(ratio);
        }
    } else {                                       /* PM_VERTICAL */
        if (PM->NMeters <= PM->WrapCount) {
            PM->Rows = PM->NMeters;
            PM->Cols = 1;
        } else {
            PM->Rows = PM->WrapCount;
            ratio = (double) PM->NMeters / PM->Rows;
            PM->Cols = (int) ceil(ratio);
        }
    }
}


/* Set the graph type data. */

void 
PmSetGraphType(PmGraphType GraphType)
{
    PM->GraphType = GraphType;            /* LINE or SOLID */
}


/* Calculate the minimum canvas size. */

void
PmSetMinSize()
{
    GdkWindowHints geometry_mask;
    GdkGeometry geometry;
    int chartw = 0, charth = 0, i, minh, minw, neww, newh;
    GnomePerfmeter *GMeter;
    int mbarw = UI->MenuBar->requisition.width;	  /* Width of the menubar. */
    int mbarh = UI->MenuBar->requisition.height;  /* Height of the menubar. */

    CalculateRC();

    for (i = 0; i < PM->NMeters; i++) {
        GMeter = GNOME_PERFMETER(UI->GMeters[PM->DisplayList[i]]);

        neww = gnome_perfmeter_get_min_width(GMeter);
        chartw = PmMax(chartw, neww);

        newh = gnome_perfmeter_get_min_height(GMeter);
        charth = PmMax(charth, newh);
    }

    if (PM->Orientation == PM_HORIZONTAL) {
	minw = chartw * PM->NMeters;
	minh = charth;

	if (PM->ShowMenubar == TRUE) {
            minw = PmMax(minw, mbarw);
            minh = minh + mbarh;
	}
    } else {
	minw = chartw;
	minh = charth * PM->NMeters;

	if (PM->ShowMenubar == TRUE) {
            minw = PmMax(minw, mbarw);
            minh = minh + mbarh;
	}
    }

    geometry_mask = GDK_HINT_MIN_SIZE;
    geometry.min_width = minw;
    geometry.min_height = minh;
    gtk_window_set_geometry_hints(GTK_WINDOW(UI->MainWin),
                                  UI->MainWin, &geometry, geometry_mask);

#ifdef DEBUG
    PRINTF("SetMinSize %d,%d\n", minw, minh);
#endif
}


/* Size the graph area. */

static void
PmSizeGraphArea(int Wp, int Hp)
{
    PM->AreaWp = Wp;
    PM->MeterHp = PM->AreaHp = Hp;

    if (PM->ShowLiveActionBar) {
        PM->MeterWp = PM->AreaWp - 2 * PmLAB_SEPARATOR - PmLAB_WIDTH;
    } else {
        PM->MeterWp = PM->AreaWp;
    }

    if (PM->ShowStripChart == FALSE) {
        /* Don't show meter */
        PM->LabStart = (int) (PM->AreaWp / 2) - PmHALF_LAB_WIDTH;
        PM->MeterWp = 0;
    }

#ifdef DEBUG
    PRINTF("SizeGraphArea %d,%d\n", PM->AreaWp, PM->AreaHp);
#endif

    PmResizeMeterData(PM->MeterWp);
}

/* Scale the meter data to the visible area. */

static int 
ScaleValue(PmMeterStruct *m, int num)
{
    int i, p, sp;

    /* get the data point */
    if (num > PM->CData) {
        i = PM->NData - (num - PM->CData);
    } else {
        i = PM->CData - num;
    }

    p = m->Data[i];

    if (p > m->CurMax * FSCALE) {
        sp = 0;
    } else {
        sp = PM->MeterHp - 
            (int) (((double) (p * PM->MeterHp)) / 
            ((double) (FSCALE * m->CurMax)));
    }

#ifdef DEBUGDATA
    PRINTF("Raw Data = %d, Scaled Data = %d\n", p, sp);
#endif

    return(sp);
}


/* Show or hide the specified meter. */

void 
PmResetMeter(int meter, gboolean show)
{
    int i, j;

    if (PM->Meters[meter].Showing == show) {
        return;
    }

    if (show == TRUE) {
        PM->Meters[meter].Showing = TRUE;
        PM->Meters[meter].Redisplay = TRUE;

        /* add to the display list */
        for (i = 0; i < PmMAX_METERS; i++) {
            if (PM->DisplayList[i] < 0) {
                PM->DisplayList[i] = meter;
                PM->NMeters++;
                break;
            }    
        }    
    } else {
        PM->Meters[meter].Showing = FALSE;

        /* Remove from the display list */
        for (i = 0; i < PmMAX_METERS; i++) {
            if (PM->DisplayList[i] == meter) {
                /* move the rest to the left */
                for (j = i + 1; j < PmMAX_METERS; j++, i++) {
                    PM->DisplayList[i] = PM->DisplayList[j];
                }
                PM->DisplayList[PmMAX_METERS-1] = -1;
                PM->NMeters--;
                break;
            }
        }
    }

    if (PM->NMeters == 0) {
        /* Always show at least one meter */
        PmResetMeter(PM_CPU, TRUE);
        return;
    } else if (PM->LockCount == 0) {
        PmSetMinSize();
        PmRedisplay();
    }
}


void
PmRedisplay()
{
    int i;

    if (PM->LockCount > 0 || !PM->Showing) {
        return;
    }

    if (PM->IsDead || PM->IsSick) {
        DrawDead();
        return;
    }

    for (i = 0; i < PM->NMeters; i++) {
        int meter = PM->DisplayList[i];

	PM->Meters[meter].Redisplay = TRUE;
        DrawGraph(meter, TRUE);
        PM->Meters[meter].Redisplay = FALSE;
    }
}


/* Show a warning message dialog popup. */

void 
PmWarning(int code, char *arg)
{
    char *charset, *msg, *tmp_msg;
    GtkWidget *dialog;

/* If this is the child process, just return as no graphics operations
 * should be attempted from a fork()'ed process.
 */

   if (PM->ParentPID != getpid()) {
       return;
   }

    switch (code) {
        case PmWRN_HOSTNOTFOUND:
            msg = _("Host \"%s\" is unknown\n");
            break;

        case PmWRN_HELPSELECT:
            msg = _("%s: Select a component within the application.");
            break;

        case PmWRN_HOSTCONNECT:
            if (PM->IsRemote) {
                msg = _("Unable to connect to the remote host:\n");
            } else {
                msg = _("Unable to connect to the local host:\n");
            }
            break;

        default:
            msg = _("%s: Unknown message code.\n");
    }

    if (!UI->MainWin) {
        charset = g_locale_from_utf8(msg, -1, NULL, NULL, NULL);
        FPRINTF(stderr, charset, arg);
        g_free(charset);
        return;
    }

    tmp_msg = (char *) malloc(strlen(msg) + strlen(arg) + 1);
    SPRINTF(tmp_msg, msg, arg);

    dialog = gtk_message_dialog_new(NULL, 
                             GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                             GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE,
                             tmp_msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog); 
    free(tmp_msg);
}


/* Show an error message. */

void 
PmError(int code, char *arg)
{
    char *charset, *msg, *tmp_msg;
    GtkWidget *dialog;

    switch (code) {
        case PmERR_OPENDISPLAY:
            msg = _("%s: Error opening X Display %s\n");
            break;

        case PmERR_LABELFONT:
            msg = _("%s: Invalid label font specified.\n");
            break;

        case PmERR_HELPSELECT:
            msg = _("%s: Help Selection Error, Operation terminated.\n");
            break;

        case PmERR_SAVERESFILE:
            msg = _("%s: Error saving resource file.\n");
            break;

        case PmERR_OPENRESFILE:
            msg = _("%s: Error opening resource file.\n");
            break;

        default:
            msg = _("%s: Unknown message code.\n");
    }

    if (!UI->MainWin) {
        charset = g_locale_from_utf8(msg, -1, NULL, NULL, NULL);
        FPRINTF(stderr, charset, PM->AppClass, arg);
        g_free(charset);
        return;
    }

    if (arg) {
        tmp_msg = (char *) malloc(strlen(msg) + strlen(PM->AppClass) +
                                                strlen(arg) + 1);
    } else {
        tmp_msg = (char *) malloc(strlen(msg) + strlen(arg) + 1);
    }

    SPRINTF(tmp_msg, msg, PM->AppClass, arg);
    dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                                    tmp_msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog); 

    free(tmp_msg);
}


static void
DrawGraph(int meter, gboolean redraw)
{
    PmUIMeterStruct *M;
    int i, end;
    int x1 = 0, y1, x2 = 0, y2, yc, base;
    GdkWindow *cw;

    GnomePerfmeter *gp = GNOME_PERFMETER(UI->GMeters[meter]);
    int AreaWp, AreaHp, MeterWp, MeterHp, LabStart;

    cw = gnome_perfmeter_get_chart_win(gp);

    AreaWp = gnome_perfmeter_get_chart_width(gp)-1;
    MeterHp = AreaHp = gnome_perfmeter_get_chart_height(gp)-1;

    if (PM->ShowLiveActionBar) {
	MeterWp = AreaWp - 2 * PmLAB_SEPARATOR - PmLAB_WIDTH;
    } else {
	MeterWp = AreaWp;
    }

    if (PM->ShowStripChart == FALSE) {
	/* Don't show meter */
	LabStart = (int) (AreaWp / 2) - PmHALF_LAB_WIDTH;
	MeterWp = 0;
    }

#ifdef DEBUGDRAW
    PRINTF("Draw Graph %d\n", meter);
#endif

    M = &(UI->Meters[meter]);
    if (M->DrawGC == NULL || M->LimitGC == NULL) {
        PmSetMeterColor(meter);
    }

    base = MeterHp;                       /* base line of graph */
    end = PmMin(PM->NData, MeterWp - 1);  /* number of points to draw */

    if (redraw || PM->Meters[meter].Redisplay) {
        /* Clear the entire meter */
        redraw = TRUE;

        /* Redraw the background */
        /* Live action bar should also get cleared */
        gdk_draw_rectangle(cw, UI->FillGC, TRUE, 0, 0, AreaWp+1, AreaHp+1);

#ifdef DEBUGDRAW
    PRINTF("Clear meter %d\n", meter);
#endif
    } else {
        if (UI->BackingStore) {
            /* Move the existing points to the left */
            gdk_draw_drawable(cw, UI->FillGC, cw, 1, 0, 0, 0, 
                              MeterWp+1, MeterHp+1);

            gdk_draw_rectangle(cw, UI->FillGC, TRUE, MeterWp, 0, 1, MeterHp+1);

            /* Clear live action bar */
            gdk_draw_rectangle(cw, UI->FillGC, TRUE,
                               MeterWp+PmLAB_SEPARATOR, 0,
                               PmLAB_WIDTH+1, MeterHp+1);

            /* Draw the last two points */
            end = PmMin(end, 2);

#ifdef DEBUGDRAW
            PRINTF("Move graph %d\n", meter);
#endif
        } else {
            /* Move the existing points to the left */
            gdk_draw_drawable(cw, UI->FillGC, cw, 1, 0, 0, 0, 
                              MeterWp+1, MeterHp+1);

            gdk_draw_rectangle(cw, UI->FillGC, TRUE, MeterWp, 0, 
                               1, MeterHp+1);

/* XXX:richb - need to correctly handle when there is no backing store. */
            /* PRINTF("DrawGraph: NO BACKING STORE ... problems!\n"); */

            /* redraw all of the points to fill in any blank areas
             * generated from the CopyArea
             */

#ifdef DEBUGDRAW
            PRINTF("Move graph %d\n", meter);
#endif
        }
    }

    /* Scale the ceiling point */
    if (PM->Meters[meter].Limit < PM->Meters[meter].CurMax) {
        yc = base - (int) (((double)
            PM->Meters[meter].Limit / (double) PM->Meters[meter].CurMax) *
            MeterHp);
    } else {
        /* Set the ceiling position out of bounds */
        yc = 0;
    }

#ifdef DEBUGDRAW
    PRINTF("Graph %d Base = %d, Ceiling = %d, Num Points = %d\n",
       meter, base, yc, end);
#endif

    if (PM->ShowStripChart == TRUE) {       /* Draw stripchart */
        if (PM->GraphType == PM_LINE && PM->NData > 1) {
            /* Calculate the first point (must have at least two data points) */
            x1 = MeterWp - 1;
            y1 = ScaleValue(&PM->Meters[meter], 0);

            x2 = y2 = 0;

            for (i = 1; i < end; i++, x1 = x2, y1 = y2) {
                x2 = x1 - 1;
                y2 = ScaleValue(&PM->Meters[meter], i);

                if (y1 <= yc && y2 <= yc) {
                    /* Both points are above the line */
                    gdk_draw_line(cw, M->LimitGC, x1, y1, x2, y2);
#ifdef DEBUGDRAW
                    PRINTF("Graph BOTH ABOVE (%d,%d),(%d,%d)\n",
                           x1, y1, x2, y2);
#endif
                } else if (y1 <= yc) {
                    /* New point is above limit */
                    gdk_draw_line(cw, M->LimitGC, x1, y1, x2, yc);
                    gdk_draw_line(cw, M->DrawGC, x2, yc, x2, y2);

#ifdef DEBUGDRAW
                    PRINTF("Graph NEW  ABOVE (%d,%d),(%d,%d),(%d,%d)\n",
                           x1, y1, x2, yc, x2, y2);
#endif
                } else if (y2 <= yc) {
                    /* Previous point is above Limit */
                    gdk_draw_line(cw, M->DrawGC, x1, y1, x2, yc);
                    gdk_draw_line(cw, M->LimitGC, x2, yc, x2, y2);

#ifdef DEBUGDRAW
                    PRINTF("Graph PREV ABOVE (%d,%d),(%d,%d),(%d,%d)\n",
                           x1, y1, x2, yc, x2, y2);
#endif
                } else {
                    /* None are above the limit */
                    gdk_draw_line(cw, M->DrawGC, x1, y1, x2, y2);

#ifdef DEBUGDRAW
                    PRINTF("Graph NORMAL (%d,%d), (%d,%d)\n", x1, y1, x2, y2);
#endif
                }
            }
        } else if (PM->GraphType == PM_SOLID) /* SOLID */ {
            for (i = 0, x1 = MeterWp-1, y1 = base; i < end; i++, x1--) {
                x2 = x1;
                y2 = ScaleValue(&PM->Meters[meter], i);

                if (y2 <= yc) {
                    gdk_draw_line(cw, M->DrawGC, x1, y1, x2, yc);
                    gdk_draw_line(cw, M->LimitGC, x2, yc, x2, y2);
#ifdef DEBUGDRAW
                    PRINTF("Graph LIMIT (%d,%d), (%d,%d), (%d,%d)\n",
                           x1, y1, x2, yc, x2, y2);
#endif
                } else {
                    gdk_draw_line(cw, M->DrawGC, x1, y1, x2, y2);

#ifdef DEBUGDRAW
                    PRINTF("Graph NORMAL (%d,%d), (%d,%d)\n", x1, y1, x2, y2);
#endif
                }
            }
        }
    }

    /* Fill out live action bar */
    if (PM->ShowLiveActionBar) {
        if (PM->ShowStripChart == TRUE) {
            x1 = MeterWp + PmLAB_SEPARATOR + PmLAB_WIDTH - 1;
        } else {
            x1 = LabStart +  PmLAB_WIDTH - 1;
        }
        y1 = base;
        x2 = x1;
        y2 = ScaleValue(&PM->Meters[meter], 0);

        for (i = 0; i < PmLAB_WIDTH; i++, x1--) {
            x2 = x1;

            if (y2 <= yc) {
                gdk_draw_line(cw, M->DrawGC, x1, y1, x2, yc);
                gdk_draw_line(cw, M->LimitGC, x1, yc, x2, y2);
            } else {
                gdk_draw_line(cw, M->DrawGC, x1, y1, x2, y2);
            }

            /* clear above the bar */
            gdk_draw_line(cw, UI->FillGC, x1, y2, x2, y1 - PM->MeterHp);
        }
    }

    /* Draw the limit line */
    if ((PM->ShowLimit) &&
        (PM->Meters[meter].Limit < PM->Meters[meter].CurMax)) {
        if (PM->ShowStripChart == TRUE) {
            x1 = 0;
            x2 = MeterWp - 1;
            y1 = y2 = yc;
            gdk_draw_line(cw, M->LimitGC, x1, y1, x2, y2);
        }

        if ((PM->ShowStripChart) && (PM->ShowLiveActionBar))  {
            x1 = MeterWp + PmLAB_SEPARATOR - 1;
            x2 = MeterWp + PmLAB_SEPARATOR + PmLAB_WIDTH - 1;
        }
        if (!PM->ShowStripChart) {
            x1 = LabStart;
            x2 = x1 + PmLAB_WIDTH - 1;
        }
        y1 = y2 = yc;
        gdk_draw_line(cw, M->LimitGC, x1, y1, x2, y2);
    }
}


/* Set the new display list and update the visible meters. */

gboolean 
PmSetDisplayList(int *DL)
{
    int i, meter;
    gboolean update = FALSE;

    PmLock();

    if (DL) {
        for (i = 0; i < PmMAX_METERS; i++) {
            if (PM->DisplayList[i] != DL[i]) {
                /* Something has changed */
                update = TRUE;
            }
        }
    } else {
        update = TRUE;
    }
  
    if (update) {
        /* Initialize the visible meter counter */
        PM->NMeters = 0;
      
        /* Turn all meters off */
        for (i = 0; i < PmMAX_METERS; i++) {
            PM->Meters[i].Redisplay = FALSE;
            PM->Meters[i].Showing = FALSE;
        }

        if (DL) {
            /* Reset the visible meters & counter */
            for (i = 0; i < PmMAX_METERS; i++) {
                meter = PM->DisplayList[i] = DL[i];
          
                if (meter >= 0) {
                    PM->Meters[meter].Redisplay = TRUE;
                    PM->Meters[meter].Showing = TRUE;
                    PM->NMeters++;
                }
            }
        } else {
            for (i = 0; i < PmMAX_METERS; i++) {
                PM->DisplayList[i] = -1;
            }
        }
    }

    /* If no meters are visible, make the CPU meter visible */
    if (PM->NMeters < 1) {
        PmResetMeter(PM_CPU, TRUE);
    }

#ifdef DEBUG
    PRINTF("SetDisplayList ");
    for (i = 0; i < PmMAX_METERS; i++) {
        PRINTF("%d ", PM->DisplayList[i]);
    }
    PRINTF("\n");
#endif

    ResetPopupMenu();
    PmUnLock();

    return(update);
}


/* Increment the lock count. */

void 
PmLock()
{
    PM->LockCount++;
#ifdef DEBUG
    PRINTF("LOCK Count = %d\n", PM->LockCount);
#endif
}


/* Decrement the lock count. */

void 
PmUnLock()
{
    if (PM->LockCount > 0) {
        if (PM->LockCount == 1 && UI->initializing == FALSE) {
            gdk_flush();
        }
        PM->LockCount--;
    }
#ifdef DEBUG
    PRINTF("UNLOCK Count = %d\n", PM->LockCount);
#endif
}


void
PmSetMeterColor(int meter)
{
    GnomePerfmeter *GMeter = GNOME_PERFMETER(UI->GMeters[meter]);
    GdkWindow *cw = gnome_perfmeter_get_chart_win(GMeter);
    GdkGC *DrawGC, *LimitGC;
    GdkGCValues values;
    GdkGCValuesMask mask;
    GdkColor MeterColor;
    GdkColor LimitColor;
    GdkColor black;

    if (cw != NULL) {
        gdk_color_parse("black", &black);

        /* Save the existing ones */
        DrawGC  = UI->Meters[meter].DrawGC;
        LimitGC = UI->Meters[meter].LimitGC;

        /* Set the colors */
        if (UI->UseColor) {
            MeterColor = PM->Meters[meter].MeterColor;
            LimitColor = PM->Meters[meter].LimitColor;
            gdk_colormap_alloc_color(UI->ColorMap, &MeterColor, FALSE, TRUE);
            gdk_colormap_alloc_color(UI->ColorMap, &LimitColor, FALSE, TRUE);
        } else {
            MeterColor = LimitColor = black;
            gdk_colormap_alloc_color(UI->ColorMap, &MeterColor, FALSE, TRUE);
            gdk_colormap_alloc_color(UI->ColorMap, &LimitColor, FALSE, TRUE);
        }

        mask = GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | GDK_GC_LINE_WIDTH |
               GDK_GC_LINE_STYLE | GDK_GC_CAP_STYLE | GDK_GC_JOIN_STYLE;
        values.line_width = 0;
        values.line_style = LineSolid;
        values.cap_style = CapButt;
        values.join_style = JoinMiter;

        values.foreground = MeterColor;
        values.background = PM->CanvasColor;
        UI->Meters[meter].DrawGC = gdk_gc_new_with_values(cw, &values, mask);
        values.foreground = LimitColor;
        values.background = PM->CanvasColor;
        UI->Meters[meter].LimitGC = gdk_gc_new_with_values(cw, &values, mask);

        /* Release the old GC's */
        if (DrawGC) {
            g_object_unref(DrawGC);
        }
        if (LimitGC) {
            g_object_unref(LimitGC);
        }
    }
}


void
PmSetColors()
{
    int i;

    if (UI->UseColor) {
#if DEBUG
        PRINTF("PM->CanvasColor=%d\n", PM->CanvasColor);
#endif

        for (i = 0; i < PmMAX_METERS; i++) {
            gnome_perfmeter_set_chart_bg_color(GNOME_PERFMETER(UI->GMeters[i]), 
                                               PM->CanvasColor);
        }

        gdk_gc_set_foreground(UI->CanvasGC, &PM->CanvasColor);
        gdk_gc_set_background(UI->CanvasGC, &PM->CanvasColor);

        gdk_gc_set_foreground(UI->FillGC, &PM->CanvasColor);
    }
}


void
PmRefresh(gboolean redisplay)
{
    int i;

    for (i = 0; i < PM->NMeters; i++) {
        int meter = PM->DisplayList[i];

        gnome_perfmeter_clear_chart(GNOME_PERFMETER(UI->GMeters[meter]),
                                    redisplay);
    }
}


/* Activate the monitoring. */

static void 
PmActivate(gboolean active)
{
    guint32 interval = 0;

    if (PM->Active == active) {
        return;
    }

    if (PM->Active) {
        /* Stop collecting */
        if (UI->TimerId) {
            gtk_timeout_remove(UI->TimerId);
            UI->TimerId = 0;
        }

        /* Reset the collect data */
        PmResetMeterData();

#ifdef DEBUG
        PRINTF("Stop Collecting\n");
#endif
    } else {
        /* Start collecting */
        if (!PM->IsDead) {
            PmRefresh(TRUE);

            /* Update the meter data */
            if (PmUpdateData()) {
                PmScaleData();
            }

            /* Redisplay the meters */
            Update();

            if (PM->Log) {
                PMLOGDATA();
            }
        }

        /* Set a new timer */
        interval = PM->SampleInterval * PM->SampleModifier * 1000;
        UI->TimerId = gtk_timeout_add(interval, UpdateTimer, NULL);

#ifdef DEBUG
        PRINTF("Start Collecting\n");
#endif
    }
    PM->Active = active;
}


static void
SetSampleTime(int value)
{
    PM->SampleInterval = value; 
    PM->SampleModifier = 1; 
    PmAssignMax(PM->SampleInterval, 1); 
    PmRedisplay(); 
    PmResetProps(); 
}


/* Handle the window keyboard accelerators:
 *
 *              a   - show all meters
 *              n   - toggle monitoring mode (local/remote)
 *              s   - toggle graph style
 *              ^   - toggle showing the limit line
 *              t   - toggle writing to the log file
 *              y   - toggle orientation (horizontal/vertical)
 *              r   - toggle showing strip chart
 *              1-9 - set sampletime to a range from 1-9 seconds
 *              +   - increment the sample time by 1 second
 *              -   - decrement the sample time by 1 second
 */

/*ARGSUSED*/
static gboolean
MainWinKeyHandler(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
    int i, dl[PmMAX_METERS];

    if ((event->state) && (event->keyval != GDK_F10) ||
        (!event->state) && (event->keyval == GDK_F10)) {
        return(FALSE);
    }

    switch (event->keyval) {
        case GDK_a:                                    /* Show all meters. */ 
            for (i = 0; i < PmMAX_METERS; i++) {
                dl[i] = i;
            }
            PmLock();
            PMSETDISPLAYLIST(dl);
            PmSetMinSize();
            PmUnLock();
            PmRedisplay();
            break;


        case GDK_n:              /* Toggle monitoring mode (local/remote). */ 
            if (PM->IsRemote) {
                if (PM->AutoLoadByHost) {
                    PmShow(FALSE);
                }

                PM->IsRemote = FALSE;
                PMMONITOR();
 
                if (PM->AutoLoadByHost) {
                    PmReset(NULL);
                    PmShow(TRUE);
                }
            } else {
                if (PM->AutoLoadByHost) {
                    PmShow(FALSE);
                }
             
                PM->IsRemote = TRUE;
                PMMONITOR();
             
                if (PM->AutoLoadByHost) {
                    PmReset(NULL);
                    PmShow(TRUE);
                }
            }    
            PmResetProps(); 
            break;

        case GDK_s:                    /* Toggle graph style (line/solid). */
#if DEBUG
            PRINTF("GraphType=%s\n",
                   (PM->GraphType == PM_LINE) ? "LINE" : "SOLID");
#endif
            if (PM->ShowStripChart == TRUE) {
                if ((PM->GraphType == PM_LINE) ||
                    (PM->GraphType == PM_SOLID)) {
                    if (PM->GraphType == PM_LINE) {
                        PmSetGraphType(PM_SOLID);
                    } else {
                        PmSetGraphType(PM_LINE);
                    }
                    PmSetMinSize();
                    PmRedisplay();
                }
            }
            PmResetProps();
            break;

        case GDK_acute:                  /* Toggle showing the limit line. */
            PM->ShowLimit = !PM->ShowLimit;
            PmResetProps();
            break;

        case GDK_t:                     /* Toggle writing to the log file. */
            if (PM->Log) {
                PmLogClose();
            } else {
                PM->Log = PmLogOpen();
            }
            PmResetProps(); 
            break; 

        case GDK_y:                                 /* Toggle orientation. */
            if (PM->Orientation == PM_HORIZONTAL) {
                PM->Orientation = PM_VERTICAL;
            } else {
                PM->Orientation = PM_HORIZONTAL;
            }
            PmSetMinSize();
            PmRedisplay();
            PmResetProps();
            break;
 
        case GDK_r:                         /* Toggle showing strip chart. */
            PM->ShowStripChart = (PM->ShowStripChart) ? FALSE : TRUE;
            PM->StripChartActivity = TRUE;
            PmSetMinSize();
            PmRedisplay();
            PM->StripChartActivity = FALSE;
            PmResetPropsDecor();
            break;

        case GDK_1:                        /* Set sampletime to 1 seconds. */  
            SetSampleTime(1);
            break;

        case GDK_2:                        /* Set sampletime to 2 seconds. */  
            SetSampleTime(2);
            break;

        case GDK_3:                        /* Set sampletime to 3 seconds. */  
            SetSampleTime(3);
            break;

        case GDK_4:                        /* Set sampletime to 4 seconds. */  
            SetSampleTime(4);
            break;

        case GDK_5:                        /* Set sampletime to 5 seconds. */  
            SetSampleTime(5);
            break;

        case GDK_6:                        /* Set sampletime to 6 seconds. */  
            SetSampleTime(6);
            break;

        case GDK_7:                        /* Set sampletime to 7 seconds. */  
            SetSampleTime(7);
            break;

        case GDK_8:                        /* Set sampletime to 8 seconds. */  
            SetSampleTime(8);
            break;

        case GDK_9:                        /* Set sampletime to 9 seconds. */
            SetSampleTime(9);
            break;

        case GDK_minus:          /* Increment the sample time by 1 second. */
            SetSampleTime(PM->SampleInterval++);
            break;

        case GDK_plus:           /* Decrement the sample time by 1 second. */
            SetSampleTime(PM->SampleInterval--);
            break;

        case GDK_F10:           /* Popup Menu */
            if (event->state & GDK_SHIFT_MASK) {
                gtk_menu_popup(GTK_MENU(UI->PopupMenu),
                               NULL, NULL, NULL, NULL, 0, event->time);
            }
            break;

        default:
            /* We didn't handle it - let something else work it out. */
            return(FALSE);
    }

    return(TRUE);
}


static void
style_set_callback(GtkWidget *widget)
{
    GnomePerfmeter *GMeter = GNOME_PERFMETER(UI->GMeters[PM->DisplayList[0]]);
    GtkWidget *chart = gnome_perfmeter_get_chart(GMeter);

    if (GTK_WIDGET_REALIZED(widget)){
        create_MainWin();
        PM->CanvasColor = chart->style->bg[GTK_STATE_NORMAL];
        PmSetColors();
        PmRedisplay();
    }
}


/* Set the remote host name (or "localhost") in the titlebar of the window. */

void
set_title()
{
    char title[MAXPATHLEN + PmMAX_LINE];

    SPRINTF(title, "%s - %s", (PM->IsRemote ? PM->Hostname : _("localhost")),
            _("Performance Meter"));
    gtk_window_set_title(GTK_WINDOW(UI->MainWin), title);
}


/* Start a new monitor. */

gboolean 
PmMonitor()
{
    /* Deactivate the monitor */
    PmActivate(FALSE);

    /* Connect to the new host */
    if (PmConnectToHost()) {      
        /* Activate the monitor */
        PmActivate(TRUE);

        set_title();
        if (PM->IsRemote) {
            PmAddHostToList(PM->Hostname);
        }
     
        PmRedisplay();
        return(TRUE);
    }

    return(FALSE);
}


/* Show/Hide the meters. */

void 
PmShow(gboolean show)
{
    if (show) {
        gdk_flush();
        gtk_widget_show(UI->MainWin);
        PM->Showing = TRUE;
    } else {
        gtk_widget_hide(UI->MainWin);
        PM->Showing = FALSE;
    }
}


/* Reset the application to its last saved state. */

void 
PmReset(char *filename)
{
    int i;

    if (PmResetResources(filename)) {
        /* General colors */
        PmSetColors();

        /* Meter colors */
        for (i = 0; i < PM->NMeters; i++) {
            PmSetMeterColor(PM->DisplayList[i]);
        }

        /* Graph type */
        PmSetGraphType(PM->GraphType);
        PmSetMinSize();

        PmSetMinSize();
        PmLogClose();
        PmResetProps();
        PmRedisplay();
    }
}


/* Set the busy cursor on the main & props windows. */

void 
PmBusy(gboolean busy)
{
    if (busy) {
        UI->Busy++;
        gdk_window_set_cursor(UI->MainWin->window, UI->WaitCursor);
    } else if (UI->Busy > 0) {
        UI->Busy--;

        if (UI->Busy == 0) {
            gdk_window_set_cursor(UI->MainWin->window, NULL);
        }
    }

    PmPropsBusy(busy);
}


static void
gperfmeter_set_atk_relation(GtkWidget *obj1, GtkWidget *obj2,
                            AtkRelationType rel_type)
{
    AtkObject *atk_obj1, *atk_obj2;
    AtkRelationSet *relation_set;
    AtkObject *targets[1];
    AtkRelation *relation;

    atk_obj1 = gtk_widget_get_accessible(obj1);
    atk_obj2 = gtk_widget_get_accessible(obj2);

    if (!(GTK_IS_ACCESSIBLE(atk_obj1)) || !(GTK_IS_ACCESSIBLE(atk_obj2))) {
        return;
    }

    relation_set = atk_object_ref_relation_set(atk_obj1);
    targets[0] = atk_obj2;

    relation = atk_relation_new(targets, 1, rel_type);
    atk_relation_set_add(relation_set, relation);

    g_object_unref(G_OBJECT(relation));
}


/*ARGSUSED*/
static gint 
gperfmeter_accessible_get_n_children(AtkObject *obj)
{
    return(PmMAX_METERS);
}


static
AtkObject *
gperfmeter_accessible_ref_child(AtkObject *obj, gint i)
{
    AtkRegistry *default_registry;
    AtkObjectFactory *factory;
    GtkWidget *widget;
    gint j;

    static gint first_time = 1;

    widget = GTK_ACCESSIBLE(obj)->widget;
    g_return_val_if_fail(widget != NULL, NULL);
    g_return_val_if_fail(i < PmMAX_METERS, NULL);

    if (first_time) {
        default_registry = atk_get_default_registry();
        factory = atk_registry_get_factory(default_registry, 
                                           PANGO_TYPE_LAYOUT);

        for (j = 0; j < PmMAX_METERS; j++) {
            pango_accessible[j] = atk_object_factory_create_accessible(factory,
                                                             G_OBJECT(layout));
            atk_object_set_parent(pango_accessible[j], obj);
            atk_object_set_name(pango_accessible[j], _(PM->Meters[j].RName));
        }
        first_time--;
    }

    return(pango_accessible[i]);
}
