/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gst/gst.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <cstuff.h>


#define DVB_TYPE_EPG_SCANNER (dvb_epg_scanner_get_type ())
#define DVB_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScanner))
#define DVB_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))
#define DVB_IS_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_SCANNER))
#define DVB_IS_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EPG_SCANNER))
#define DVB_EPG_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))

typedef struct _DVBEPGScanner DVBEPGScanner;
typedef struct _DVBEPGScannerClass DVBEPGScannerClass;
typedef struct _DVBEPGScannerPrivate DVBEPGScannerPrivate;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_SETTINGS (dvb_settings_get_type ())
#define DVB_SETTINGS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SETTINGS, DVBSettings))
#define DVB_SETTINGS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SETTINGS, DVBSettingsClass))
#define DVB_IS_SETTINGS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SETTINGS))
#define DVB_IS_SETTINGS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SETTINGS))
#define DVB_SETTINGS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SETTINGS, DVBSettingsClass))

typedef struct _DVBSettings DVBSettings;
typedef struct _DVBSettingsClass DVBSettingsClass;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_TYPE_SCHEDULE (dvb_schedule_get_type ())
#define DVB_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCHEDULE, DVBSchedule))
#define DVB_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCHEDULE, DVBScheduleClass))
#define DVB_IS_SCHEDULE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCHEDULE))
#define DVB_IS_SCHEDULE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCHEDULE))
#define DVB_SCHEDULE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCHEDULE, DVBScheduleClass))

typedef struct _DVBSchedule DVBSchedule;
typedef struct _DVBScheduleClass DVBScheduleClass;

#define DVB_TYPE_EVENT (dvb_event_get_type ())
#define DVB_EVENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT, DVBEvent))
#define DVB_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT, DVBEventClass))
#define DVB_IS_EVENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT))
#define DVB_IS_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT))
#define DVB_EVENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT, DVBEventClass))

typedef struct _DVBEvent DVBEvent;
typedef struct _DVBEventClass DVBEventClass;
typedef struct _DVBEventPrivate DVBEventPrivate;

#define DVB_EVENT_TYPE_AUDIO_COMPONENT (dvb_event_audio_component_get_type ())
#define DVB_EVENT_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponent))
#define DVB_EVENT_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))
#define DVB_EVENT_IS_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_IS_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_AUDIO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))

typedef struct _DVBEventAudioComponent DVBEventAudioComponent;
typedef struct _DVBEventAudioComponentClass DVBEventAudioComponentClass;

#define DVB_EVENT_TYPE_VIDEO_COMPONENT (dvb_event_video_component_get_type ())
#define DVB_EVENT_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponent))
#define DVB_EVENT_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))
#define DVB_EVENT_IS_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_IS_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_VIDEO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))

typedef struct _DVBEventVideoComponent DVBEventVideoComponent;
typedef struct _DVBEventVideoComponentClass DVBEventVideoComponentClass;

#define DVB_EVENT_TYPE_TELETEXT_COMPONENT (dvb_event_teletext_component_get_type ())
#define DVB_EVENT_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponent))
#define DVB_EVENT_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))
#define DVB_EVENT_IS_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_IS_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_TELETEXT_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))

typedef struct _DVBEventTeletextComponent DVBEventTeletextComponent;
typedef struct _DVBEventTeletextComponentClass DVBEventTeletextComponentClass;
typedef struct _DVBEventAudioComponentPrivate DVBEventAudioComponentPrivate;
typedef struct _DVBEventVideoComponentPrivate DVBEventVideoComponentPrivate;
typedef struct _DVBEventTeletextComponentPrivate DVBEventTeletextComponentPrivate;

struct _DVBEPGScanner {
	GObject parent_instance;
	DVBEPGScannerPrivate * priv;
};

struct _DVBEPGScannerClass {
	GObjectClass parent_class;
};

struct _DVBEPGScannerPrivate {
	DVBDeviceGroup* _DeviceGroup;
	GstElement* pipeline;
	GStaticRecMutex __lock_pipeline;
	GQueue* channels;
	GSource* scan_source;
	GSource* queue_source;
	gint stop_counter;
	GMainContext* context;
	GMainLoop* loop;
	GThread* worker_thread;
	guint bus_watch_id;
};

/**
     * Represents an EPG event (i.e. a show with all its information)
     */
struct _DVBEvent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventPrivate * priv;
	guint id;
	guint year;
	guint month;
	guint hour;
	guint day;
	guint minute;
	guint second;
	guint duration;
	guint running_status;
	gboolean free_ca_mode;
	char* name;
	char* description;
	char* extended_description;
	GSList* audio_components;
	GSList* video_components;
	GSList* teletext_components;
};

struct _DVBEventClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEvent *self);
};

struct _DVBEventAudioComponent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventAudioComponentPrivate * priv;
	char* type;
};

struct _DVBEventAudioComponentClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEventAudioComponent *self);
};

struct _DVBEventVideoComponent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventVideoComponentPrivate * priv;
	gboolean high_definition;
	char* aspect_ratio;
	gint frequency;
};

struct _DVBEventVideoComponentClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEventVideoComponent *self);
};

struct _DVBEventTeletextComponent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventTeletextComponentPrivate * priv;
	char* type;
};

struct _DVBEventTeletextComponentClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEventTeletextComponent *self);
};


static gint dvb_epg_scanner_CHECK_EIT_INTERVAL;
static gint dvb_epg_scanner_CHECK_EIT_INTERVAL = 0;
static gpointer dvb_epg_scanner_parent_class = NULL;

GType dvb_epg_scanner_get_type (void);
GType dvb_device_group_get_type (void);
GType dvb_channel_get_type (void);
#define DVB_EPG_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_EPG_SCANNER, DVBEPGScannerPrivate))
enum  {
	DVB_EPG_SCANNER_DUMMY_PROPERTY,
	DVB_EPG_SCANNER_DEVICE_GROUP
};
#define DVB_EPG_SCANNER_WAIT_FOR_EIT_DURATION 10
#define DVB_EPG_SCANNER_PIPELINE_TEMPLATE "dvbsrc name=dvbsrc adapter=%u frontend=%u pids=0:16:17:18 stats-reporting-interval=0 ! mpegtsparse ! fakesink silent=true"
void dvb_epg_scanner_set_DeviceGroup (DVBEPGScanner* self, DVBDeviceGroup* value);
GType dvb_settings_get_type (void);
DVBSettings* dvb_factory_get_settings (void);
gint dvb_settings_get_integer (DVBSettings* self, const char* group_name, const char* key, GError** error);
#define DVB_SETTINGS_EPG_SECTION "epg"
#define DVB_SETTINGS_SCAN_INTERVAL "scan_interval"
DVBEPGScanner* dvb_epg_scanner_new (DVBDeviceGroup* device);
DVBEPGScanner* dvb_epg_scanner_construct (GType object_type, DVBDeviceGroup* device);
DVBDeviceGroup* dvb_epg_scanner_get_DeviceGroup (DVBEPGScanner* self);
guint dvb_device_group_get_Id (DVBDeviceGroup* self);
static void dvb_epg_scanner_remove_timeouts (DVBEPGScanner* self);
static void dvb_epg_scanner_reset (DVBEPGScanner* self);
void dvb_epg_scanner_stop (DVBEPGScanner* self);
static void* dvb_epg_scanner_worker (DVBEPGScanner* self);
static void* _dvb_epg_scanner_worker_gthread_func (gpointer self);
GType dvb_channel_list_get_type (void);
DVBChannelList* dvb_device_group_get_Channels (DVBDeviceGroup* self);
guint dvb_channel_get_Frequency (DVBChannel* self);
GType dvb_device_get_type (void);
DVBDevice* dvb_device_group_get_next_free_device (DVBDeviceGroup* self);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
static gboolean dvb_epg_scanner_bus_watch_func (DVBEPGScanner* self, GstBus* bus, GstMessage* message);
static gboolean _dvb_epg_scanner_bus_watch_func_gst_bus_func (GstBus* bus, GstMessage* message, gpointer self);
static gboolean dvb_epg_scanner_scan_new_frequency (DVBEPGScanner* self);
static gboolean _dvb_epg_scanner_scan_new_frequency_gsource_func (gpointer self);
gboolean dvb_epg_scanner_start (DVBEPGScanner* self);
static gboolean _dvb_epg_scanner_start_gsource_func (gpointer self);
GType dvb_schedule_get_type (void);
DVBSchedule* dvb_channel_get_Schedule (DVBChannel* self);
void dvb_schedule_remove_expired_events (DVBSchedule* self);
void dvb_channel_setup_dvb_source (DVBChannel* self, GstElement* source);
void dvb_epg_scanner_on_eit_structure (DVBEPGScanner* self, const GstStructure* structure);
static guint dvb_epg_scanner_get_uint_val (const GstStructure* structure, const char* name);
DVBChannel* dvb_channel_list_get_channel (DVBChannelList* self, guint sid);
DVBEvent* dvb_event_new (void);
DVBEvent* dvb_event_construct (GType object_type);
gpointer dvb_event_ref (gpointer instance);
void dvb_event_unref (gpointer instance);
GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_value_set_event (GValue* value, gpointer v_object);
gpointer dvb_value_get_event (const GValue* value);
GType dvb_event_get_type (void);
gpointer dvb_event_audio_component_ref (gpointer instance);
void dvb_event_audio_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_audio_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_audio_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_audio_component (const GValue* value);
GType dvb_event_audio_component_get_type (void);
gpointer dvb_event_video_component_ref (gpointer instance);
void dvb_event_video_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_video_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_video_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_video_component (const GValue* value);
GType dvb_event_video_component_get_type (void);
gpointer dvb_event_teletext_component_ref (gpointer instance);
void dvb_event_teletext_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_teletext_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_teletext_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_teletext_component (const GValue* value);
GType dvb_event_teletext_component_get_type (void);
DVBEventAudioComponent* dvb_event_audio_component_new (void);
DVBEventAudioComponent* dvb_event_audio_component_construct (GType object_type);
DVBEventVideoComponent* dvb_event_video_component_new (void);
DVBEventVideoComponent* dvb_event_video_component_construct (GType object_type);
DVBEventTeletextComponent* dvb_event_teletext_component_new (void);
DVBEventTeletextComponent* dvb_event_teletext_component_construct (GType object_type);
void dvb_schedule_add (DVBSchedule* self, DVBEvent* event);
static GObject * dvb_epg_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void dvb_epg_scanner_finalize (GObject* obj);
static void dvb_epg_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_epg_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
         * @device: The device where EPG should be collected from
         */
DVBEPGScanner* dvb_epg_scanner_construct (GType object_type, DVBDeviceGroup* device) {
	GError * _inner_error_;
	DVBEPGScanner * self;
	g_return_val_if_fail (device != NULL, NULL);
	_inner_error_ = NULL;
	self = g_object_newv (object_type, 0, NULL);
	dvb_epg_scanner_set_DeviceGroup (self, device);
	if (dvb_epg_scanner_CHECK_EIT_INTERVAL == (-1)) {
		DVBSettings* _tmp0_;
		DVBSettings* settings;
		_tmp0_ = NULL;
		settings = (_tmp0_ = dvb_factory_get_settings (), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
		{
			gint _tmp1_;
			_tmp1_ = dvb_settings_get_integer (settings, DVB_SETTINGS_EPG_SECTION, DVB_SETTINGS_SCAN_INTERVAL, &_inner_error_);
			if (_inner_error_ != NULL) {
				if (_inner_error_->domain == G_KEY_FILE_ERROR) {
					goto __catch3_g_key_file_error;
				}
				goto __finally3;
			}
			dvb_epg_scanner_CHECK_EIT_INTERVAL = _tmp1_ * 60;
		}
		goto __finally3;
		__catch3_g_key_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("EPGScanner.vala:66: %s", e->message);
				dvb_epg_scanner_CHECK_EIT_INTERVAL = 15 * 60;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally3:
		if (_inner_error_ != NULL) {
			(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
		(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
	}
	return self;
}


DVBEPGScanner* dvb_epg_scanner_new (DVBDeviceGroup* device) {
	return dvb_epg_scanner_construct (DVB_TYPE_EPG_SCANNER, device);
}


/**
         * Stop collecting EPG data
         */
void dvb_epg_scanner_stop (DVBEPGScanner* self) {
	g_return_if_fail (self != NULL);
	g_debug ("EPGScanner.vala:76: Stopping EPG scan for group %u (%d)", dvb_device_group_get_Id (self->priv->_DeviceGroup), self->priv->stop_counter);
	if (self->priv->stop_counter == 0) {
		dvb_epg_scanner_remove_timeouts (self);
		dvb_epg_scanner_reset (self);
	}
	self->priv->stop_counter = self->priv->stop_counter + 1;
}


static void dvb_epg_scanner_remove_timeouts (DVBEPGScanner* self) {
	GMainLoop* _tmp2_;
	g_return_if_fail (self != NULL);
	if (self->priv->scan_source != NULL) {
		GSource* _tmp0_;
		g_source_destroy (self->priv->scan_source);
		_tmp0_ = NULL;
		self->priv->scan_source = (_tmp0_ = NULL, (self->priv->scan_source == NULL) ? NULL : (self->priv->scan_source = (g_source_unref (self->priv->scan_source), NULL)), _tmp0_);
	}
	if (self->priv->queue_source != NULL) {
		GSource* _tmp1_;
		g_source_destroy (self->priv->queue_source);
		_tmp1_ = NULL;
		self->priv->queue_source = (_tmp1_ = NULL, (self->priv->queue_source == NULL) ? NULL : (self->priv->queue_source = (g_source_unref (self->priv->queue_source), NULL)), _tmp1_);
	}
	g_main_loop_quit (self->priv->loop);
	_tmp2_ = NULL;
	self->priv->loop = (_tmp2_ = NULL, (self->priv->loop == NULL) ? NULL : (self->priv->loop = (g_main_loop_unref (self->priv->loop), NULL)), _tmp2_);
	g_thread_join (self->priv->worker_thread);
	self->priv->worker_thread = NULL;
}


/* Main Thread */
static void* dvb_epg_scanner_worker (DVBEPGScanner* self) {
	void* result;
	g_return_val_if_fail (self != NULL, NULL);
	g_main_loop_run (self->priv->loop);
	result = NULL;
	return result;
}


static void dvb_epg_scanner_reset (DVBEPGScanner* self) {
	DVBChannel* c;
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		if (self->priv->pipeline != NULL) {
			GSource* _tmp0_;
			GSource* bus_watch_source;
			GstElement* _tmp1_;
			_tmp0_ = NULL;
			bus_watch_source = (_tmp0_ = g_main_context_find_source_by_id (self->priv->context, self->priv->bus_watch_id), (_tmp0_ == NULL) ? NULL : g_source_ref (_tmp0_));
			if (bus_watch_source != NULL) {
				g_source_destroy (bus_watch_source);
				self->priv->bus_watch_id = (guint) 0;
			}
			gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
			gst_element_get_state (self->priv->pipeline, NULL, NULL, (GstClockTime) (-1));
			_tmp1_ = NULL;
			self->priv->pipeline = (_tmp1_ = NULL, (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp1_);
			(bus_watch_source == NULL) ? NULL : (bus_watch_source = (g_source_unref (bus_watch_source), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	/* clear doesn't unref for us so we do this instead*/
	c = NULL;
	while (TRUE) {
		DVBChannel* _tmp2_;
		_tmp2_ = NULL;
		if (!((c = (_tmp2_ = (DVBChannel*) g_queue_pop_head (self->priv->channels), (c == NULL) ? NULL : (c = (g_object_unref (c), NULL)), _tmp2_)) != NULL)) {
			break;
		}
	}
	/* Vala unref's Channel instances for us*/
	g_queue_clear (self->priv->channels);
	(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
}


static void* _dvb_epg_scanner_worker_gthread_func (gpointer self) {
	return dvb_epg_scanner_worker (self);
}


static gboolean _dvb_epg_scanner_bus_watch_func_gst_bus_func (GstBus* bus, GstMessage* message, gpointer self) {
	return dvb_epg_scanner_bus_watch_func (self, bus, message);
}


static gboolean _dvb_epg_scanner_scan_new_frequency_gsource_func (gpointer self) {
	return dvb_epg_scanner_scan_new_frequency (self);
}


/**
         * Start collection EPG data for all channels
         */
gboolean dvb_epg_scanner_start (DVBEPGScanner* self) {
	gboolean result;
	GError * _inner_error_;
	GMainLoop* _tmp0_;
	GThread* _tmp1_;
	GeeHashSet* unique_frequencies;
	DVBDevice* device;
	GSource* _tmp7_;
	g_return_val_if_fail (self != NULL, FALSE);
	_inner_error_ = NULL;
	g_debug ("EPGScanner.vala:134: Starting EPG scan for group %u (%d)", dvb_device_group_get_Id (self->priv->_DeviceGroup), self->priv->stop_counter);
	_tmp0_ = NULL;
	self->priv->loop = (_tmp0_ = g_main_loop_new (self->priv->context, FALSE), (self->priv->loop == NULL) ? NULL : (self->priv->loop = (g_main_loop_unref (self->priv->loop), NULL)), _tmp0_);
	_tmp1_ = g_thread_create (_dvb_epg_scanner_worker_gthread_func, self, TRUE, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	self->priv->worker_thread = _tmp1_;
	self->priv->stop_counter = self->priv->stop_counter - 1;
	if (self->priv->stop_counter > 0) {
		result = FALSE;
		return result;
	}
	self->priv->stop_counter = 0;
	/* TODO scan all channels?*/
	unique_frequencies = gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal);
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) dvb_device_group_get_Channels (self->priv->_DeviceGroup));
		while (TRUE) {
			DVBChannel* c;
			guint freq;
			if (!gee_iterator_next (_c_it)) {
				break;
			}
			c = (DVBChannel*) gee_iterator_get (_c_it);
			freq = dvb_channel_get_Frequency (c);
			if (!gee_abstract_collection_contains ((GeeAbstractCollection*) unique_frequencies, GUINT_TO_POINTER (freq))) {
				DVBChannel* _tmp2_;
				gee_abstract_collection_contains ((GeeAbstractCollection*) unique_frequencies, GUINT_TO_POINTER (freq));
				_tmp2_ = NULL;
				g_queue_push_tail (self->priv->channels, (_tmp2_ = c, (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_)));
			}
			(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
	}
	device = dvb_device_group_get_next_free_device (self->priv->_DeviceGroup);
	if (device == NULL) {
		result = FALSE;
		(unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL));
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		return result;
	}
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		GstBus* bus;
		{
			char* _tmp3_;
			GstElement* _tmp4_;
			GstElement* _tmp5_;
			GstElement* _tmp6_;
			_tmp3_ = NULL;
			_tmp4_ = NULL;
			_tmp5_ = (_tmp4_ = gst_parse_launch (_tmp3_ = g_strdup_printf (DVB_EPG_SCANNER_PIPELINE_TEMPLATE, dvb_device_get_Adapter (device), dvb_device_get_Frontend (device)), &_inner_error_), _tmp3_ = (g_free (_tmp3_), NULL), _tmp4_);
			if (_inner_error_ != NULL) {
				goto __catch4_g_error;
				goto __finally4;
			}
			_tmp6_ = NULL;
			self->priv->pipeline = (_tmp6_ = _tmp5_, (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp6_);
		}
		goto __finally4;
		__catch4_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_error ("EPGScanner.vala:161: Could not create pipeline: %s", e->message);
				result = FALSE;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				(unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL));
				(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
				return result;
			}
		}
		__finally4:
		if (_inner_error_ != NULL) {
			(unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL));
			(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return FALSE;
		}
		bus = gst_element_get_bus (self->priv->pipeline);
		self->priv->bus_watch_id = gst_bus_add_watch_context (bus, _dvb_epg_scanner_bus_watch_func_gst_bus_func, self, self->priv->context);
		(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	_tmp7_ = NULL;
	self->priv->scan_source = (_tmp7_ = g_timeout_source_new_seconds ((guint) DVB_EPG_SCANNER_WAIT_FOR_EIT_DURATION), (self->priv->scan_source == NULL) ? NULL : (self->priv->scan_source = (g_source_unref (self->priv->scan_source), NULL)), _tmp7_);
	g_source_set_callback (self->priv->scan_source, _dvb_epg_scanner_scan_new_frequency_gsource_func, g_object_ref (self), g_object_unref);
	g_source_attach (self->priv->scan_source, self->priv->context);
	result = FALSE;
	(unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL));
	(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
	return result;
}


static gboolean _dvb_epg_scanner_start_gsource_func (gpointer self) {
	return dvb_epg_scanner_start (self);
}


/**
         * Scan the next frequency for EPG data
         */
static gboolean dvb_epg_scanner_scan_new_frequency (DVBEPGScanner* self) {
	gboolean result;
	DVBChannel* channel;
	g_return_val_if_fail (self != NULL, FALSE);
	if (g_queue_is_empty (self->priv->channels)) {
		GSource* _tmp0_;
		g_debug ("EPGScanner.vala:182: Finished EPG scan for group %u", dvb_device_group_get_Id (self->priv->_DeviceGroup));
		dvb_epg_scanner_reset (self);
		/* Time the next iteration*/
		_tmp0_ = NULL;
		self->priv->queue_source = (_tmp0_ = g_timeout_source_new_seconds ((guint) dvb_epg_scanner_CHECK_EIT_INTERVAL), (self->priv->queue_source == NULL) ? NULL : (self->priv->queue_source = (g_source_unref (self->priv->queue_source), NULL)), _tmp0_);
		g_source_set_callback (self->priv->queue_source, _dvb_epg_scanner_start_gsource_func, g_object_ref (self), g_object_unref);
		g_source_attach (self->priv->queue_source, self->priv->context);
		result = FALSE;
		return result;
	}
	channel = (DVBChannel*) g_queue_pop_head (self->priv->channels);
	dvb_schedule_remove_expired_events (dvb_channel_get_Schedule (channel));
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		GstElement* dvbsrc;
		gst_element_set_state (self->priv->pipeline, GST_STATE_READY);
		dvbsrc = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), "dvbsrc");
		dvb_channel_setup_dvb_source (channel, dvbsrc);
		gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
		(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	result = TRUE;
	(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	return result;
}


static gboolean dvb_epg_scanner_bus_watch_func (DVBEPGScanner* self, GstBus* bus, GstMessage* message) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (bus != NULL, FALSE);
	g_return_val_if_fail (message != NULL, FALSE);
	switch (message->type) {
		case GST_MESSAGE_ELEMENT:
		{
			if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-read-failure") == 0) {
				g_critical ("EPGScanner.vala:212: Could not read from DVB device");
				dvb_epg_scanner_stop (self);
			} else {
				if (_vala_strcmp0 (gst_structure_get_name (message->structure), "eit") == 0) {
					dvb_epg_scanner_on_eit_structure (self, message->structure);
				}
			}
			break;
		}
		case GST_MESSAGE_ERROR:
		{
			GError* gerror;
			char* debug;
			char* _tmp3_;
			char* _tmp2_;
			GError* _tmp1_;
			GError* _tmp0_;
			gerror = NULL;
			debug = NULL;
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			(gst_message_parse_error (message, &_tmp0_, &_tmp2_), gerror = (_tmp1_ = _tmp0_, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp1_));
			debug = (_tmp3_ = _tmp2_, debug = (g_free (debug), NULL), _tmp3_);
			g_critical ("EPGScanner.vala:223: %s %s", gerror->message, debug);
			dvb_epg_scanner_stop (self);
			result = FALSE;
			(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
			debug = (g_free (debug), NULL);
			return result;
		}
		default:
		{
			break;
		}
	}
	result = TRUE;
	return result;
}


void dvb_epg_scanner_on_eit_structure (DVBEPGScanner* self, const GstStructure* structure) {
	GValue events;
	guint size;
	GValue val = {0};
	const GstStructure* event;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	events = *gst_structure_get_value (structure, "events");
	if (!G_VALUE_HOLDS (&events, gst_value_list_get_type ())) {
		return;
	}
	size = gst_value_list_get_size (&events);
	event = NULL;
	{
		guint i;
		/* Iterate over events*/
		i = (guint) 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				guint sid;
				DVBChannel* channel;
				guint event_id;
				DVBEvent* event_class;
				const char* _tmp1_;
				char* name;
				gboolean _tmp2_;
				const char* _tmp5_;
				char* desc;
				gboolean _tmp6_;
				const char* _tmp9_;
				char* ext_desc;
				gboolean _tmp10_;
				gboolean free_ca;
				GValue components;
				guint components_len;
				GValue comp_val = {0};
				const GstStructure* component;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < size)) {
					break;
				}
				val = *gst_value_list_get_value (&events, i);
				event = gst_value_get_structure (&val);
				sid = dvb_epg_scanner_get_uint_val (structure, "service-id");
				channel = dvb_channel_list_get_channel (dvb_device_group_get_Channels (self->priv->_DeviceGroup), sid);
				if (channel == NULL) {
					g_warning ("EPGScanner.vala:250: Could not find channel %u for this device", sid);
					(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
					return;
				}
				event_id = dvb_epg_scanner_get_uint_val (event, "event-id");
				event_class = dvb_event_new ();
				event_class->id = event_id;
				event_class->year = dvb_epg_scanner_get_uint_val (event, "year");
				event_class->month = dvb_epg_scanner_get_uint_val (event, "month");
				event_class->day = dvb_epg_scanner_get_uint_val (event, "day");
				event_class->hour = dvb_epg_scanner_get_uint_val (event, "hour");
				event_class->minute = dvb_epg_scanner_get_uint_val (event, "minute");
				event_class->second = dvb_epg_scanner_get_uint_val (event, "second");
				event_class->duration = dvb_epg_scanner_get_uint_val (event, "duration");
				event_class->running_status = dvb_epg_scanner_get_uint_val (event, "running-status");
				_tmp1_ = NULL;
				name = (_tmp1_ = gst_structure_get_string (event, "name"), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
				_tmp2_ = FALSE;
				if (name != NULL) {
					_tmp2_ = g_utf8_validate (name, -1, NULL);
				} else {
					_tmp2_ = FALSE;
				}
				if (_tmp2_) {
					char* _tmp4_;
					const char* _tmp3_;
					_tmp4_ = NULL;
					_tmp3_ = NULL;
					event_class->name = (_tmp4_ = (_tmp3_ = name, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), event_class->name = (g_free (event_class->name), NULL), _tmp4_);
				}
				_tmp5_ = NULL;
				desc = (_tmp5_ = gst_structure_get_string (event, "description"), (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
				_tmp6_ = FALSE;
				if (desc != NULL) {
					_tmp6_ = g_utf8_validate (desc, -1, NULL);
				} else {
					_tmp6_ = FALSE;
				}
				if (_tmp6_) {
					char* _tmp8_;
					const char* _tmp7_;
					_tmp8_ = NULL;
					_tmp7_ = NULL;
					event_class->description = (_tmp8_ = (_tmp7_ = desc, (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_)), event_class->description = (g_free (event_class->description), NULL), _tmp8_);
				}
				_tmp9_ = NULL;
				ext_desc = (_tmp9_ = gst_structure_get_string (event, "extended-text"), (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
				_tmp10_ = FALSE;
				if (ext_desc != NULL) {
					_tmp10_ = g_utf8_validate (ext_desc, -1, NULL);
				} else {
					_tmp10_ = FALSE;
				}
				if (_tmp10_) {
					char* _tmp12_;
					const char* _tmp11_;
					_tmp12_ = NULL;
					_tmp11_ = NULL;
					event_class->extended_description = (_tmp12_ = (_tmp11_ = ext_desc, (_tmp11_ == NULL) ? NULL : g_strdup (_tmp11_)), event_class->extended_description = (g_free (event_class->extended_description), NULL), _tmp12_);
				}
				free_ca = FALSE;
				gst_structure_get_boolean (event, "free-ca-mode", &free_ca);
				event_class->free_ca_mode = free_ca;
				components = *gst_structure_get_value (event, "components");
				components_len = gst_value_list_get_size (&components);
				component = NULL;
				{
					guint j;
					j = (guint) 0;
					{
						gboolean _tmp13_;
						_tmp13_ = TRUE;
						while (TRUE) {
							if (!_tmp13_) {
								j++;
							}
							_tmp13_ = FALSE;
							if (!(j < components_len)) {
								break;
							}
							comp_val = *gst_value_list_get_value (&components, j);
							component = gst_value_get_structure (&comp_val);
							if (_vala_strcmp0 (gst_structure_get_name (component), "audio") == 0) {
								DVBEventAudioComponent* audio;
								char* _tmp15_;
								const char* _tmp14_;
								DVBEventAudioComponent* _tmp16_;
								audio = dvb_event_audio_component_new ();
								_tmp15_ = NULL;
								_tmp14_ = NULL;
								audio->type = (_tmp15_ = (_tmp14_ = gst_structure_get_string (component, "type"), (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_)), audio->type = (g_free (audio->type), NULL), _tmp15_);
								_tmp16_ = NULL;
								event_class->audio_components = g_slist_append (event_class->audio_components, (_tmp16_ = audio, (_tmp16_ == NULL) ? NULL : dvb_event_audio_component_ref (_tmp16_)));
								(audio == NULL) ? NULL : (audio = (dvb_event_audio_component_unref (audio), NULL));
							} else {
								if (_vala_strcmp0 (gst_structure_get_name (component), "video") == 0) {
									DVBEventVideoComponent* video;
									gboolean highdef;
									char* _tmp18_;
									const char* _tmp17_;
									gint freq;
									DVBEventVideoComponent* _tmp19_;
									video = dvb_event_video_component_new ();
									highdef = FALSE;
									gst_structure_get_boolean (component, "high-definition", &highdef);
									video->high_definition = highdef;
									_tmp18_ = NULL;
									_tmp17_ = NULL;
									video->aspect_ratio = (_tmp18_ = (_tmp17_ = gst_structure_get_string (component, "high-definition"), (_tmp17_ == NULL) ? NULL : g_strdup (_tmp17_)), video->aspect_ratio = (g_free (video->aspect_ratio), NULL), _tmp18_);
									freq = 0;
									gst_structure_get_int (component, "frequency", &freq);
									video->frequency = freq;
									_tmp19_ = NULL;
									event_class->video_components = g_slist_append (event_class->video_components, (_tmp19_ = video, (_tmp19_ == NULL) ? NULL : dvb_event_video_component_ref (_tmp19_)));
									(video == NULL) ? NULL : (video = (dvb_event_video_component_unref (video), NULL));
								} else {
									if (_vala_strcmp0 (gst_structure_get_name (component), "teletext") == 0) {
										DVBEventTeletextComponent* teletext;
										char* _tmp21_;
										const char* _tmp20_;
										DVBEventTeletextComponent* _tmp22_;
										teletext = dvb_event_teletext_component_new ();
										_tmp21_ = NULL;
										_tmp20_ = NULL;
										teletext->type = (_tmp21_ = (_tmp20_ = gst_structure_get_string (component, "type"), (_tmp20_ == NULL) ? NULL : g_strdup (_tmp20_)), teletext->type = (g_free (teletext->type), NULL), _tmp21_);
										_tmp22_ = NULL;
										event_class->teletext_components = g_slist_append (event_class->teletext_components, (_tmp22_ = teletext, (_tmp22_ == NULL) ? NULL : dvb_event_teletext_component_ref (_tmp22_)));
										(teletext == NULL) ? NULL : (teletext = (dvb_event_teletext_component_unref (teletext), NULL));
									}
								}
							}
						}
					}
				}
				/*debug ("Adding new event: %s", event_class.to_string ());*/
				dvb_schedule_add (dvb_channel_get_Schedule (channel), event_class);
				(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
				(event_class == NULL) ? NULL : (event_class = (dvb_event_unref (event_class), NULL));
				name = (g_free (name), NULL);
				desc = (g_free (desc), NULL);
				ext_desc = (g_free (ext_desc), NULL);
			}
		}
	}
}


static guint dvb_epg_scanner_get_uint_val (const GstStructure* structure, const char* name) {
	guint result;
	guint val;
	g_return_val_if_fail (structure != NULL, 0U);
	g_return_val_if_fail (name != NULL, 0U);
	val = 0U;
	gst_structure_get_uint (structure, name, &val);
	result = val;
	return result;
}


DVBDeviceGroup* dvb_epg_scanner_get_DeviceGroup (DVBEPGScanner* self) {
	DVBDeviceGroup* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_DeviceGroup;
	return result;
}


void dvb_epg_scanner_set_DeviceGroup (DVBEPGScanner* self, DVBDeviceGroup* value) {
	g_return_if_fail (self != NULL);
	self->priv->_DeviceGroup = value;
	g_object_notify ((GObject *) self, "DeviceGroup");
}


static GObject * dvb_epg_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBEPGScannerClass * klass;
	GObjectClass * parent_class;
	DVBEPGScanner * self;
	klass = DVB_EPG_SCANNER_CLASS (g_type_class_peek (DVB_TYPE_EPG_SCANNER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_EPG_SCANNER (obj);
	{
		GQueue* _tmp0_;
		GMainContext* _tmp1_;
		_tmp0_ = NULL;
		self->priv->channels = (_tmp0_ = g_queue_new (), (self->priv->channels == NULL) ? NULL : (self->priv->channels = (g_queue_free (self->priv->channels), NULL)), _tmp0_);
		self->priv->stop_counter = 0;
		_tmp1_ = NULL;
		self->priv->context = (_tmp1_ = g_main_context_new (), (self->priv->context == NULL) ? NULL : (self->priv->context = (g_main_context_unref (self->priv->context), NULL)), _tmp1_);
	}
	return obj;
}


static void dvb_epg_scanner_class_init (DVBEPGScannerClass * klass) {
	dvb_epg_scanner_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBEPGScannerPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_epg_scanner_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_epg_scanner_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_epg_scanner_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_epg_scanner_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_EPG_SCANNER_DEVICE_GROUP, g_param_spec_object ("DeviceGroup", "DeviceGroup", "DeviceGroup", DVB_TYPE_DEVICE_GROUP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	dvb_epg_scanner_CHECK_EIT_INTERVAL = -1;
}


static void dvb_epg_scanner_instance_init (DVBEPGScanner * self) {
	self->priv = DVB_EPG_SCANNER_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_pipeline);
}


static void dvb_epg_scanner_finalize (GObject* obj) {
	DVBEPGScanner * self;
	self = DVB_EPG_SCANNER (obj);
	g_static_rec_mutex_free (&self->priv->__lock_pipeline);
	(self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL));
	(self->priv->channels == NULL) ? NULL : (self->priv->channels = (g_queue_free (self->priv->channels), NULL));
	(self->priv->scan_source == NULL) ? NULL : (self->priv->scan_source = (g_source_unref (self->priv->scan_source), NULL));
	(self->priv->queue_source == NULL) ? NULL : (self->priv->queue_source = (g_source_unref (self->priv->queue_source), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (g_main_context_unref (self->priv->context), NULL));
	(self->priv->loop == NULL) ? NULL : (self->priv->loop = (g_main_loop_unref (self->priv->loop), NULL));
	G_OBJECT_CLASS (dvb_epg_scanner_parent_class)->finalize (obj);
}


GType dvb_epg_scanner_get_type (void) {
	static GType dvb_epg_scanner_type_id = 0;
	if (dvb_epg_scanner_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBEPGScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_epg_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEPGScanner), 0, (GInstanceInitFunc) dvb_epg_scanner_instance_init, NULL };
		dvb_epg_scanner_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBEPGScanner", &g_define_type_info, 0);
	}
	return dvb_epg_scanner_type_id;
}


static void dvb_epg_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBEPGScanner * self;
	gpointer boxed;
	self = DVB_EPG_SCANNER (object);
	switch (property_id) {
		case DVB_EPG_SCANNER_DEVICE_GROUP:
		g_value_set_object (value, dvb_epg_scanner_get_DeviceGroup (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_epg_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBEPGScanner * self;
	self = DVB_EPG_SCANNER (object);
	switch (property_id) {
		case DVB_EPG_SCANNER_DEVICE_GROUP:
		dvb_epg_scanner_set_DeviceGroup (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




