/* gbf-am-build.c
 *
 * Copyright (C) 2000  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Dave Camp
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <regex.h>
#include "gbf-i18n.h"
#include "gbf-am-build.h"

/* FIXME: mark strings for translation and this file to POTFILES.in */

typedef struct {
	GbfAmProject *project;
	GbfBuildType type;
	int id;
	int num_channels;
	GList *callbacks;

	/* Regex structures. */
	struct re_pattern_buffer dir_buf;
	struct re_pattern_buffer warn_buf;
	struct re_pattern_buffer err_buf;
	struct re_registers reg;

	/* Build info. */
	char *build_dir;
} BuildInfo;

static void
build_msg (BuildInfo      *info,
	   GbfBuildMessage type,
	   gpointer        msg)
{
	GList *l;

	for (l = info->callbacks; l; l = l->next) {
		GbfAmBuildCallback *cb = l->data;
		(* cb->callback) (GBF_PROJECT (info->project),
				  type, msg,
				  cb->user_data);
	}
}

static void
parse_output (BuildInfo  *info,
	      const char *line)
{
	/* Check for directory changes. */
	if (re_search (&info->dir_buf, line, strlen (line), 0,
		       strlen (line), &info->reg) != -1) {
		if (info->reg.num_regs >= 2) {
			if (info->build_dir) {
				g_free (info->build_dir);
				info->build_dir = NULL;
			}
			info->build_dir = g_strndup (line + info->reg.start[1],
						     info->reg.end[1] - info->reg.start[1]);
		}
	}

	/* Check for warnings & errors. */
	if (re_search (&info->warn_buf, line, strlen (line), 0,
		       strlen (line), &info->reg) != -1) {
		GbfBuildWarning *warn;
		char *text;

		/* Create new warning. */
		warn = g_new0 (GbfBuildWarning, 1);
		text = g_strndup (line + info->reg.start[1],
				  info->reg.end[1] - info->reg.start[1]);
		warn->filename = g_strconcat (info->build_dir, "/", text, NULL);
		g_free (text);
		text = g_strndup (line + info->reg.start[2],
				  info->reg.end[2] - info->reg.start[2]);
		warn->line = atoi (text);
		g_free (text);
		warn->warning = g_strndup (line + info->reg.start[3],
					   info->reg.end[3] - info->reg.start[3]);
		warn->output = g_strdup (line);

		build_msg (info, GBF_BUILD_WARNING, warn);
	} else if (re_search (&info->err_buf, line, strlen (line), 0,
			      strlen (line), &info->reg) != -1) {
		GbfBuildError *err;
		char *text;

		/* Create new error. */
		err = g_new0 (GbfBuildError, 1);
		text = g_strndup (line + info->reg.start[1],
				  info->reg.end[1] - info->reg.start[1]);
		err->filename = g_strconcat (info->build_dir, "/", text, NULL);
		g_free (text);
		text = g_strndup (line + info->reg.start[2],
				  info->reg.end[2] - info->reg.start[2]);
		err->line = atoi (text);
		g_free (text);
		err->error = g_strndup (line + info->reg.start[3],
					info->reg.end[3] - info->reg.start[3]);
		err->output = g_strdup (line);

		build_msg (info, GBF_BUILD_ERROR, err);
	} else {
		build_msg (info, GBF_BUILD_OUTPUT, (gpointer)line);
	}
}

static gboolean
build_output_cb (GIOChannel  *chan,
		 GIOCondition cond,
		 gpointer     user_data)
{
	BuildInfo *info = user_data;
	char *line;
	gsize len;
	gsize term;
	GError *err = NULL;
	GIOStatus status;

	status = g_io_channel_read_line (chan, &line, &len, &term, &err);
	if (status != G_IO_STATUS_NORMAL || line == NULL || err != NULL) {
		info->num_channels--;
		if (info->num_channels == 0) {
			build_msg (info, GBF_BUILD_END, "Build ended");

			g_signal_emit_by_name (G_OBJECT (info->project),
					       "build_stop", TRUE);

			/* Cleanup. */
			if (info->build_dir)
				g_free (info->build_dir);
			if (info->dir_buf.fastmap)
				g_free (info->dir_buf.fastmap);
			if (info->warn_buf.fastmap)
				g_free (info->warn_buf.fastmap);
			if (info->err_buf.fastmap)
				g_free (info->err_buf.fastmap);
			g_free (info);
		}

		return FALSE;
	}

	parse_output (info, line);
	g_free (line);

	return TRUE;
}

static gboolean
compile_pattern (struct re_pattern_buffer *buf,
		 const char               *pattern)
{
	memset (buf, 0, sizeof (struct re_pattern_buffer));
	buf->translate = NULL;
	buf->fastmap = g_malloc (256);
	buf->allocated = 0;
	buf->buffer = NULL;
	buf->can_be_null = 0;
	buf->no_sub = 0;

	if (!re_compile_pattern (pattern, strlen (pattern), buf)) {
		if (re_compile_fastmap (buf) != 0) {
			g_warning ("IMPORTANT REGEX FAILED TO CREASTE FASTMAP\n");
			g_free (buf->fastmap);
			buf->fastmap = NULL;
		}
	} else {
		g_warning ("IMPORTANT REGEX FAILED TO COMPILE\n");
		return FALSE;
	}

	return TRUE;
}

int
gbf_build_run (GbfAmProject    *project,
	       GbfBuildType     type,
	       const char      *project_dir,
	       GList           *callbacks)
{
	static const char *dir_regex = "Entering directory `([^']+)'";
	static const char *warn_regex = "^([^:]+):([0-9]+): warning: (.+)$";
	static const char *err_regex = "^([^:]+):([0-9]+): (.+)$";
	static const char *prepare_argv[] = { "./autogen.sh", "--prefix=/gnome", NULL };
	static const char *configure_argv[] = { "./configure", "--prefix=/gnome", NULL };
	static const char *clean_argv[] = {"/usr/bin/make", "clean", NULL };
	static const char *all_argv[] = { "/usr/bin/make", "all", NULL };
	static const char *install_argv[] = {"/usr/bin/make", "install", NULL };
	static int buildid = 0;
	const char **argv = NULL;
	BuildInfo *info;
	char *tmp, *msg;
	int output, err, pid;
	GIOChannel *out_channel, *err_channel;
	reg_syntax_t old_options;

	switch (type) {
	case GBF_BUILD_PREPARE:
		argv = prepare_argv;
		break;
	case GBF_BUILD_CONFIGURE:
		argv = configure_argv;
		break;
	case GBF_BUILD_CLEAN:
		argv = clean_argv;
		break;
	case GBF_BUILD_ALL:
		argv = all_argv;
		break;
	case GBF_BUILD_CURRENT:
		g_warning ("no build for current");
		break;
	case GBF_BUILD_INSTALL:
		argv = install_argv;
		break;
	}

	if (!g_spawn_async_with_pipes (project_dir,
				       (char**)argv, NULL,
				       0,
				       NULL, NULL,
				       &pid, 
				       NULL, &output, &err,
				       NULL)) {
		g_warning ("Couldn't spawn %s\n", argv[0]);
		return -1;
	}

	out_channel = g_io_channel_unix_new (output);
	err_channel = g_io_channel_unix_new (err);

	info = g_new0 (BuildInfo, 1);
	info->project = project;
	info->type = type;
	info->id = ++buildid;
	info->num_channels = 2;
	info->callbacks = callbacks;
	info->build_dir = NULL;

	/* Intialize regexs. */
	old_options = re_syntax_options;
	re_syntax_options = RE_SYNTAX_EGREP;

	if (!compile_pattern (&info->dir_buf, dir_regex) ||
	    !compile_pattern (&info->warn_buf, warn_regex) ||
	    !compile_pattern (&info->err_buf, err_regex)) {
		g_warning ("failed to compile regexs necessary for build output parsing");
		return -1;
	}

	re_syntax_options = old_options;

	g_signal_emit_by_name (G_OBJECT (project), "build_start");

	tmp = g_strjoinv (" ", (char **) argv);
	msg = g_strconcat (tmp, "\n", NULL);
	g_free (tmp);
	build_msg (info, GBF_BUILD_START, msg);
	g_free (msg);

	g_io_add_watch (out_channel, G_IO_IN | G_IO_ERR | G_IO_HUP, 
			build_output_cb, 
			info);
	g_io_channel_unref (out_channel);
	g_io_add_watch (err_channel, G_IO_IN | G_IO_ERR | G_IO_HUP, 
			build_output_cb, 
			info);
	g_io_channel_unref (err_channel);

	return info->id;
}

void
gbf_build_cancel (int build_id)
{
}
