/* value.cc
 *
 * Copyright (C) 2000-2002 GConfmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gconfmm/value.h"
#include "gconfmm/schema.h"
#include "gconfmm/list_conversion.h"

namespace Gnome
{

namespace Conf
{

Value::Value(GConfValueType type = GCONF_VALUE_INVALID)
{
  m_pValue = gconf_value_new(type);
}

Value::Value(const GConfValue* pValue)
{
  m_pValue = 0;

  if(pValue)
  {
    m_pValue = gconf_value_copy(const_cast<GConfValue*>(pValue)); //Should be gconf_value_copy(const)
  }
}

Value::Value(const Value& src)
{
  m_pValue = 0;

  operator=(src);
}

Value::~Value()
{
  clear();
}

Value& Value::operator=(const Value& src)
{
  if(&src != this)
  {
    clear();

    if(src.gobj())
      m_pValue = gconf_value_copy(const_cast<GConfValue*>(src.gobj()));
  }

  return *this;
}

void Value::clear()
{
  if(m_pValue)
  {
    gconf_value_free(m_pValue);
    m_pValue = 0;
  }
}

const GConfValue* Value::gobj() const
{
  return m_pValue;
}

GConfValue* Value::gobj()
{
  return m_pValue;
}

void Value::set(gint val)
{
  gconf_value_set_int(gobj(), val);
}

void Value::set(const Glib::ustring& val)
{
  gconf_value_set_string(gobj(), val.c_str());
}

void Value::set(gdouble val)
{
  gconf_value_set_float(gobj(), val);
}

void Value::set(bool val)
{
  gconf_value_set_bool(gobj(), (bool)val);
}

void Value::set(const Schema& sc)
{
  gconf_value_set_schema(gobj(), const_cast<GConfSchema*>(sc.gobj()));
}

void Value::set_car(const Value& car)
{
  gconf_value_set_car(gobj(), const_cast<GConfValue*>(car.gobj()));
}

void Value::set_cdr(const Value& cdr)
{
  gconf_value_set_cdr(gobj(), const_cast<GConfValue*>(cdr.gobj()));
}

GConfValueType Value::get_type() const
{
  return gobj()->type;
}

Glib::ustring Value::get_string() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_value_get_string(gobj());
  if(pchVal)
    strResult = pchVal;

  return strResult;
}


gint Value::get_int() const
{
  return gconf_value_get_int(gobj());
}

gdouble Value::get_float() const
{
  return gconf_value_get_float(gobj());
}

bool Value::get_bool() const
{
  return (bool)gconf_value_get_bool(gobj());
}

Value Value::get_car() const
{
  const GConfValue* pValue = gconf_value_get_car(gobj());
  if(pValue)
  {
    return Value(pValue);
  }
  else
  {
    return Value();
  }
}

Value Value::get_cdr() const
{
  const GConfValue* pValue = gconf_value_get_cdr(gobj());
  if(pValue)
  {
    return Value(pValue);
  }
  else
  {
    return Value();
  }
}

const void Value::get_schema(Schema& schema) const
{
  const GConfSchema* pSchema = gconf_value_get_schema(gobj());
  if(pSchema)
  {
    schema = Schema(pSchema);
  }
  else
  {
    schema = Schema();
  }

}

Glib::ustring Value::to_string() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_value_to_string(const_cast<GConfValue*>(gobj()));
  if(pchVal)
    strResult = pchVal;

  return strResult;
}

Value::type_list_float Value::get_list_of_floats() const
{
  return ListHelpers::container_from_gslist_of_values_of_floats(
    gconf_value_get_list(gobj()) );
}

Value::type_list_int Value::get_list_of_ints() const
{
  return ListHelpers::container_from_gslist_of_values_of_ints(
    gconf_value_get_list(gobj()) );
}


Value::type_list_bool Value::get_list_of_bools() const
{
  return ListHelpers::container_from_gslist_of_values_of_bools(
    gconf_value_get_list(gobj()) );
}

void Value::get_list_of_schemas(type_list_schema& list) const
{
  list = ListHelpers::container_from_gslist_of_values_of_schemas(
    gconf_value_get_list(gobj()) );
}


Glib::SArray Value::get_list_of_strings() const
{
  return ListHelpers::container_from_gslist_of_values_of_strings(
    gconf_value_get_list(gobj()) );
}

void Value::set(const Glib::SArray& list)
{
  GSList* pList = ListHelpers::gslist_of_values_from_container(list);

  gconf_value_set_list_type(gobj(), GCONF_VALUE_STRING);
  gconf_value_set_list(gobj(), pList);

  ListHelpers::gslist_of_values_destroy(pList);
}

void Value::set(const type_list_int& list)
{
  GSList* pList = ListHelpers::gslist_of_values_from_container(list);

  gconf_value_set_list_type(gobj(), GCONF_VALUE_INT);
  gconf_value_set_list(gobj(), pList);

  ListHelpers::gslist_of_values_destroy(pList);
}

void Value::set(const type_list_bool& list)
{
  GSList* pList = ListHelpers::gslist_of_values_from_container(list);

  gconf_value_set_list_type(gobj(), GCONF_VALUE_BOOL);
  gconf_value_set_list(gobj(), pList);

  ListHelpers::gslist_of_values_destroy(pList);
}

void Value::set(const type_list_float& list)
{
  GSList* pList = ListHelpers::gslist_of_values_from_container(list);

  gconf_value_set_list_type(gobj(), GCONF_VALUE_FLOAT);
  gconf_value_set_list(gobj(), pList);

  ListHelpers::gslist_of_values_destroy(pList);
}

void Value::set(const type_list_schema& list)
{
  GSList* pList = ListHelpers::gslist_of_values_from_container(list);

  gconf_value_set_list_type(gobj(), GCONF_VALUE_SCHEMA);
  gconf_value_set_list(gobj(), pList);

  ListHelpers::gslist_of_values_destroy(pList);
}


} /* namespace Conf */
} /* namespace Gnome */
