#!/usr/bin/env python
from distutils.core import setup
from distutils.command.install_data import install_data
from distutils.command.install_lib import install_lib
from distutils.dep_util import newer
from distutils.log import info
from fnmatch import fnmatch
import os
import sys
import shutil

from gazpacho import application

def listfiles(*dirs):
    dir, pattern = os.path.split(os.path.join(*dirs))
    return [os.path.join(dir, filename)
            for filename in os.listdir(os.path.abspath(dir))
                if filename[0] != '.' and fnmatch(filename, pattern)]


class InstallData(install_data):
    def run(self):
        # we don't usually have the tools to compile po files in win32
        if sys.platform != 'win32':
            self.data_files.extend(self._compile_po_files())
        
        install_data.run(self)

    def _compile_po_files(self):
        data_files = []
        for po in listfiles('po', '*.po'):
            lang = os.path.basename(po[:-3])
            mo = os.path.join('locale', lang,
                              'LC_MESSAGES', 'gazpacho.mo')

            if not os.path.exists(mo) or newer(po, mo):
                directory = os.path.dirname(mo)
                if not os.path.exists(directory):
                    info("creating %s" % directory)
                    os.makedirs(directory)
                cmd = 'msgfmt -o %s %s' % (mo, po)
                info('compiling %s -> %s' % (po, mo))
                if os.system(cmd) != 0:
                    raise SystemExit("Error while running msgfmt")
            dest = os.path.dirname(os.path.join('share', mo))
            data_files.append((dest, [mo]))
            
        return data_files

class InstallLib(install_lib):

    def install(self):
        installed_template = '''# Generated by setup.py do not modify
import os
prefix = r"%(prefix)s"
pixmaps_dir = os.path.join(prefix, 'share', 'gazpacho', 'pixmaps')
languages_dir = os.path.join(prefix, 'share', 'locale')
catalogs_dir = os.path.join(prefix, 'share', 'gazpacho', 'catalogs')
resources_dir = os.path.join(prefix, 'share', 'gazpacho', 'resources')
glade_dir = os.path.join(prefix, 'share', 'gazpacho', 'glade')
widgets_dir = os.path.join(prefix, 'lib', 'gazpacho', 'widgets')
docs_dir = os.path.join(prefix, 'share', 'doc', 'gazpacho')
'''
        filename = os.path.join(self.build_dir, 'gazpacho', 
                                '__installed__.py')
        install = self.distribution.get_command_obj('install')
	# Overkill: Don't generate the file if we have the same prefix
        if (os.path.exists(filename) and not newer(__file__, filename)):
	    d = {}
 	    try:
                # This may have security implications, but do we care?
                execfile(filename, {}, d)		
	    except Exception, e:
                raise SystemExit('Error: %s' % e)
	    if d.get('prefix') == install.prefix:
                return install_lib.install(self)

        self.mkpath(os.path.dirname(filename))
        fp = open(filename, 'w')
        fp.write(installed_template % dict(prefix=install.prefix))
        fp.close()
	return install_lib.install(self) + [filename]

data_files = [
    ('share/gazpacho/catalogs', listfiles('catalogs', 'gtk+.xml')),
    ('share/gazpacho/pixmaps', listfiles('pixmaps', '*.png')),
    ('share/doc/gazpacho', ('AUTHORS', 'ChangeLog', 'CONTRIBUTORS',
                            'COPYING', 'README', 'NEWS')),
    ('share/doc/gazpacho/examples', listfiles('examples', '*')),
    ('share/gazpacho/resources/gtk+', listfiles('resources', 'gtk+', '*.png')),
    ('share/gazpacho/glade', listfiles('glade', '*.glade')),
    ('lib/gazpacho/widgets', listfiles('gazpacho', 'widgets', 'gtk+.py'))
    ]

# In Win32 we can't call our script 'gazpacho' since all the files must have
# an extension if we just want to double click on them to run the program.
# We can't neither call it gazpacho.py because we have already a package called
# gazpacho in site-packages and doing so would make us run recursive imports
if sys.platform == 'win32':
    src = os.path.join('bin', 'gazpacho')
    shutil.copy2(src, os.path.join('bin', 'launch-gazpacho.py'))
    shutil.copy2(src, os.path.join('bin', 'launch-gazpacho.pyw'))
    scripts = ['bin/launch-gazpacho.py',
               'bin/launch-gazpacho.pyw']
else:
    scripts = ['bin/gazpacho']
    data_files.append(('share/applications',
                       ['gazpacho.desktop']))

setup(name='gazpacho',
      version=application.__version__,
      description='GTK+ GUI Designer',
      author='SICEm S.L.',
      author_email='lgs@sicem.biz',
      url='http://gruppy.sicem.biz',
      license='LGPL',
      packages=['gazpacho', 'gazpacho.loader'],
      scripts=scripts,
      data_files=data_files,
      cmdclass={'install_data': InstallData,
                'install_lib': InstallLib})

