/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Patch by Alan Carriou <cariou_alan@yahoo.fr> (C) 2004-2005
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>

 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.

 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.

 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef H_LIB_ACPI
#  define H_LIB_ACPI

/************************************************************************
 * Macros
 ************************************************************************/

#  define ACPIDEV "/proc/acpi/info"

  /* use ? */
#  ifndef ACPI_32_BIT_SUPPORT
#    define ACPI_32_BIT_SUPPORT      0x0002
#  endif

/*
 * "safe" free macro.
 * "data" is evaluated twice, so don't put in any fancy thing like "*t++",
 * because it just won't work.
 */
#  define MYFREE(data)                                                    \
        do {                                                              \
            free(data);                                                   \
            (data) = NULL;                                                \
        } while (0)

/************************************************************************
 * Type definitions
 ************************************************************************/

/*
 * The ACPI Specification (version 2.0c) tells that the naming
 * conventions (a.k.a. "ACPI Namespace") imposes 32-bits-long names. So a
 * char[4] is, theoretically, enough to remember any device name. But many
 * DSDTs are not 100% spec-compliant, so I keep some margin using a "char *"
 * as long a I do not know how the kernel deals with it.
 */

/************************
 * AC adapter infos
 ************************/

typedef enum {
    On_line,
    Off_line,
    Other                       /* "Unknown" would conflict with bat_status definition */
} ac_power;

/* structure to hold information about ONE ac adapter
 *
 * Managing several acad's can be a real need. The "ACPI Specification 2.0c"
 * (section 11.4) gives the example of a laptop plugged in a docking station,
 * each one having its own ACad, which means that 2 AC adapters are listed in
 * the global acpi namespace.
 *
 * I do not have such hardware, so I do not know yet how it managed by any
 * OS implementation ; neither do I know about the possibility of an ACad to
 * be declared as "on-line" when another is "off-line". I haven't read the 
 * whole 518 pages of the acpi spec, and i do not think i'll have the time for 
 * this, so I would apreciate any help !
 *
 * For now, we scan each acad at update time, to know which ones are Online/
 * Offline/"Unknown". 
 *
 * As far as the GUI is concerned, the icon "ac power on-line" is displayed
 * when the status of any AC adapter (see the structure ACPI_Infos) is set to
 * "On_line".
 */
typedef struct s_AC_adapter AC_adapter;

struct s_AC_adapter {
    AC_adapter *next;
    char *name;                 /* device name given by OS acpi interface */
    char *statefile;            /* name of the state file */
    ac_power status;
};

/************************
 * Battery infos
 ************************/
typedef enum {
    Charging,
    Discharging,
    Unknown
} bat_status;

typedef struct s_Battery Battery;

struct s_Battery {
    Battery *next;
    char *name;
    char *statefile;            /* name of the state file (device present state) */
    char *infofile;             /* name of the info file (device characteristics) */
    int plugged;                /* 1 when a battery is plugged in the slot, else 0 */
    double capacity;            /* "last full capacity" */
    int percentage;             /* load percentage = remain/capacity * 100 */
    bat_status status;          /* charging / discharging / unknown */
};

/************************
 * Thermal zone infos
 ************************/

/*
 * structure to hold information about ONE thermal sensor ( = "Thermal Zone"
 * in ACPI naming standard).
 *
 * temp is set to 0 when unconsitent data is read, and can be higher than 99C.
 */
typedef struct s_Thermal_Zone Thermal_Zone;

/* as used in the linux source drivers/acpi/thermal.c (at least for 2.6.8.1-12mdk) */
typedef unsigned long int temperature;

struct s_Thermal_Zone {
    Thermal_Zone *next;
    char *name;
    char *tempfile;             /* name of the 'temperature' file */
    temperature temp;           /* the temperature read by the sensor, in celcius degrees */
};

/************************
 * Global infos
 ************************/

/* structure to hold every acpi information */
typedef struct s_AcpiInfos AcpiInfos;

struct s_AcpiInfos {
    Battery *bat;               /* chained list of battery slots */
    Battery *bat_first;         /* the currently monitored battery */

    Thermal_Zone *thermal;      /* the same... */
    Thermal_Zone *thermal_first;

    AC_adapter *ac;             /* chained list of AC adapters */
    ac_power AC_power;          /* are we on line ? */

    /* Temperature (in celcius degree) over which alarm is raised. */
    temperature thermal_alarm_level;

    /* Percentage of available battery power under which alarm is raised */
    int bat_alarm_level;
};

/* 
 * This union allows to do some generic work (mainly in acpi_detect()).
 * It needs that every struct defines "next" as its first field.
 */
typedef union {
    void *next;
    Battery bat;
    Thermal_Zone tz;
    AC_adapter acad;
} acpi_device;

/************************************************************************
 * Prototypes
 ************************************************************************/

/*
 * Tells if the system has an ACPI support.
 *
 * Returns 0 if ACPI is present, 1 if not, and 2 if ACPI is enabled but
 * we cannot read the ACPI data.
 */
int acpi_exists(void);

/*
 * Detects every ACPI device defined in the directories of, for instance,
 * /proc/acpi/battery. Note that the directory name can end with a
 * trailing '/', although acpi_detect works the same with or without.
 */
void *acpi_detect(char *, void *(*)(struct dirent *));

/*          
 * Reads the informations (temperature, battery status...) from /proc/acpi. It
 * is called within update().
 *
 * Does nothing if the AcpiInfos is NULL. Does nothing for a device type where
 * no device has been intialized (i.e. does not try to read battery status if
 * we know there is no battery...).
 */
void acpi_update_status(AcpiInfos *);

/*
 * Initiates all ACPI informations.
 *
 * If there is no battery, or no AC adapter, or no thermal zone, then it
 * tries to detect the most devices.
 * If the AcpiInfos* is NULL, then it does nothing.
 */
void acpi_detect_devices(AcpiInfos *);

/*
 * Used to build some file names as "/proc/acpi/battery/C11C/state" from
 * "/proc/acpi" + "/battery/"+"C11C"+"/state".
 * 
 * Returns NULL in case of error.
 */
char *strcat4(char *, char *, char *, char *);

/*
 * Function called when exit() is raised. Frees every malloc()ed memory
 * inside the cur_acpi_infos.
 */
void acpi_cleanup(void);

/* Returns a non-null value if the alarm should be raised */
int acpi_check_alarm(AcpiInfos *);

/*
 * Read each line of the named file, looking for a line beginning
 * with the searched string.
 *
 * If such a line is found, the function copies the rest of the line, skipping leading
 * spaces and tabulations, and the final '\n', in the given char buffer and returns.
 * If the rest of the line is longer than the buffer length given, only the
 * (length - 1) first characters are copied, and the last character is set to '\0'.
 * The function then returns "dest".
 *
 * If such a line is not found, or any error occurs (file does not exist, unreadable,
 * bad parameters...), the function returns NULL and does not modify the buffer.
 *
 * The function opens and closes the file by itself.
 */
char *parse_file(char * /* file_name */ , char * /* searched */ ,
                 char * /* dest */ , size_t);

#endif                          /* H_LIB_ACPI */
