"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const _1 = require(".");
const chai_1 = require("chai");
const paths = require("path");
const mkdirp = require("mkdirp");
const os = require("os");
const common_spec_1 = require("./common.spec");
const rimraf = require("rimraf");
/* eslint-disable no-unused-expressions */
chai_1.use(require('chai-string'));
describe('clangd-contexts-config', function () {
    let configFile;
    this.beforeAll(common_spec_1.createTmpDir);
    this.beforeEach(function () {
        configFile = common_spec_1.tmpFile('.clangd-contexts', `{
"workspaceName": "CDT.cloud Clangd Contexts Test",
"projects": [
    {
        "path": "app",
        "contextDirs": "nested"
    },
    {
        "path": "lib",
        "contextDirs": "flat"
    }
]
}`);
    });
    describe('getClangdContextsConfig', function () {
        it('load a simple config file', function () {
            const simpleConfig = _1.getClangdContextsConfig(configFile);
            chai_1.expect(simpleConfig === null || simpleConfig === void 0 ? void 0 : simpleConfig.path).to.equal(paths.dirname(configFile));
            chai_1.expect(simpleConfig === null || simpleConfig === void 0 ? void 0 : simpleConfig.workspaceName).to.equal('CDT.cloud Clangd Contexts Test');
            chai_1.expect(simpleConfig === null || simpleConfig === void 0 ? void 0 : simpleConfig.projects).to.have.nested.property('[0].contextDirs', 'nested');
            chai_1.expect(simpleConfig === null || simpleConfig === void 0 ? void 0 : simpleConfig.projects).to.have.nested.property('[1].path', 'lib');
        });
        it('search up the parent directories', function () {
            const cwd = paths.resolve(configFile, '..', 'some', 'nested', 'dir');
            mkdirp.sync(cwd);
            const simpleConfig = _1.getClangdContextsConfig(cwd);
            chai_1.expect(simpleConfig === null || simpleConfig === void 0 ? void 0 : simpleConfig.path).to.equal(paths.dirname(configFile));
        });
        it('load a non-existent config file', function () {
            const simpleConfig = _1.getClangdContextsConfig(os.tmpdir());
            chai_1.expect(simpleConfig).to.be.undefined;
        });
    });
    describe('ClangdContextsConfig', function () {
        this.afterEach(function () {
            rimraf.sync(paths.resolve(configFile, '..', 'app'));
            rimraf.sync(paths.resolve(configFile, '..', 'lib'));
        });
        it('getClangdProjects', function () {
            var _a;
            const projects = (_a = _1.getClangdContextsConfig(configFile)) === null || _a === void 0 ? void 0 : _a.getClangdProjects();
            const absoluteDirs = ['app', 'lib'].map(dir => paths.resolve(configFile, '..', dir));
            chai_1.expect(projects).to.have.members(absoluteDirs);
        });
        it('getClangdContexts (no contexts)', function () {
            var _a;
            const contexts = (_a = _1.getClangdContextsConfig(configFile)) === null || _a === void 0 ? void 0 : _a.getClangdContexts();
            chai_1.expect(contexts).to.be.empty;
        });
        it('getClangdContexts', function () {
            var _a;
            // These are nested
            common_spec_1.tmpFile('app/Debug/x86-64/compile_commands.json', '');
            common_spec_1.tmpFile('app/Release/x86-64/compile_commands.json', '');
            // And these are flat
            common_spec_1.tmpFile('lib/Debug_x86-64/compile_commands.json', '');
            common_spec_1.tmpFile('lib/Release_x86-64/compile_commands.json', '');
            const contexts = (_a = _1.getClangdContextsConfig(configFile)) === null || _a === void 0 ? void 0 : _a.getClangdContexts();
            chai_1.expect(contexts).to.have.members(['Debug/x86-64', 'Release/x86-64', 'Debug_x86-64', 'Release_x86-64']);
        });
        it('getClangdContexts (uniqueness)', function () {
            var _a;
            common_spec_1.tmpFile('app/Debug_x86-64/compile_commands.json', '');
            common_spec_1.tmpFile('app/Release_x86-64/compile_commands.json', '');
            common_spec_1.tmpFile('lib/Debug_x86-64/compile_commands.json', '');
            common_spec_1.tmpFile('lib/Release_x86-64/compile_commands.json', '');
            const contexts = (_a = _1.getClangdContextsConfig(configFile)) === null || _a === void 0 ? void 0 : _a.getClangdContexts();
            chai_1.expect(contexts).to.have.members(['Debug_x86-64', 'Release_x86-64']);
        });
    });
});
//# sourceMappingURL=clangd-contexts-config.spec.js.map