"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExternalUriOpenerPriority = exports.TokenizationRegistry = exports.InlayHintKind = exports.CommentMode = exports.CommentThreadCollapsibleState = exports.Command = exports.FoldingRangeKind = exports.SymbolKinds = exports.isLocationLink = exports.DocumentHighlightKind = exports.SignatureHelpTriggerKind = exports.InlineCompletionTriggerKind = exports.CompletionItemKinds = exports.EncodedTokenizationResult = exports.TokenizationResult = exports.Token = exports.TokenMetadata = void 0;
const uri_1 = require("../../base/common/uri");
const range_1 = require("./core/range");
const tokenizationRegistry_1 = require("./tokenizationRegistry");
const codicons_1 = require("../../base/common/codicons");
/**
 * @internal
 */
class TokenMetadata {
    static getLanguageId(metadata) {
        return (metadata & 255 /* LANGUAGEID_MASK */) >>> 0 /* LANGUAGEID_OFFSET */;
    }
    static getTokenType(metadata) {
        return (metadata & 768 /* TOKEN_TYPE_MASK */) >>> 8 /* TOKEN_TYPE_OFFSET */;
    }
    static getFontStyle(metadata) {
        return (metadata & 15360 /* FONT_STYLE_MASK */) >>> 10 /* FONT_STYLE_OFFSET */;
    }
    static getForeground(metadata) {
        return (metadata & 8372224 /* FOREGROUND_MASK */) >>> 14 /* FOREGROUND_OFFSET */;
    }
    static getBackground(metadata) {
        return (metadata & 4286578688 /* BACKGROUND_MASK */) >>> 23 /* BACKGROUND_OFFSET */;
    }
    static getClassNameFromMetadata(metadata) {
        const foreground = this.getForeground(metadata);
        let className = 'mtk' + foreground;
        const fontStyle = this.getFontStyle(metadata);
        if (fontStyle & 1 /* Italic */) {
            className += ' mtki';
        }
        if (fontStyle & 2 /* Bold */) {
            className += ' mtkb';
        }
        if (fontStyle & 4 /* Underline */) {
            className += ' mtku';
        }
        if (fontStyle & 8 /* Strikethrough */) {
            className += ' mtks';
        }
        return className;
    }
    static getInlineStyleFromMetadata(metadata, colorMap) {
        const foreground = this.getForeground(metadata);
        const fontStyle = this.getFontStyle(metadata);
        let result = `color: ${colorMap[foreground]};`;
        if (fontStyle & 1 /* Italic */) {
            result += 'font-style: italic;';
        }
        if (fontStyle & 2 /* Bold */) {
            result += 'font-weight: bold;';
        }
        let textDecoration = '';
        if (fontStyle & 4 /* Underline */) {
            textDecoration += ' underline';
        }
        if (fontStyle & 8 /* Strikethrough */) {
            textDecoration += ' line-through';
        }
        if (textDecoration) {
            result += `text-decoration:${textDecoration};`;
        }
        return result;
    }
    static getPresentationFromMetadata(metadata) {
        const foreground = this.getForeground(metadata);
        const fontStyle = this.getFontStyle(metadata);
        return {
            foreground: foreground,
            italic: Boolean(fontStyle & 1 /* Italic */),
            bold: Boolean(fontStyle & 2 /* Bold */),
            underline: Boolean(fontStyle & 4 /* Underline */),
            strikethrough: Boolean(fontStyle & 8 /* Strikethrough */),
        };
    }
}
exports.TokenMetadata = TokenMetadata;
class Token {
    constructor(offset, type, language) {
        this._tokenBrand = undefined;
        this.offset = offset;
        this.type = type;
        this.language = language;
    }
    toString() {
        return '(' + this.offset + ', ' + this.type + ')';
    }
}
exports.Token = Token;
/**
 * @internal
 */
class TokenizationResult {
    constructor(tokens, endState) {
        this._tokenizationResultBrand = undefined;
        this.tokens = tokens;
        this.endState = endState;
    }
}
exports.TokenizationResult = TokenizationResult;
/**
 * @internal
 */
class EncodedTokenizationResult {
    constructor(tokens, endState) {
        this._encodedTokenizationResultBrand = undefined;
        this.tokens = tokens;
        this.endState = endState;
    }
}
exports.EncodedTokenizationResult = EncodedTokenizationResult;
/**
 * @internal
 */
var CompletionItemKinds;
(function (CompletionItemKinds) {
    const byKind = new Map();
    byKind.set(0 /* Method */, codicons_1.Codicon.symbolMethod);
    byKind.set(1 /* Function */, codicons_1.Codicon.symbolFunction);
    byKind.set(2 /* Constructor */, codicons_1.Codicon.symbolConstructor);
    byKind.set(3 /* Field */, codicons_1.Codicon.symbolField);
    byKind.set(4 /* Variable */, codicons_1.Codicon.symbolVariable);
    byKind.set(5 /* Class */, codicons_1.Codicon.symbolClass);
    byKind.set(6 /* Struct */, codicons_1.Codicon.symbolStruct);
    byKind.set(7 /* Interface */, codicons_1.Codicon.symbolInterface);
    byKind.set(8 /* Module */, codicons_1.Codicon.symbolModule);
    byKind.set(9 /* Property */, codicons_1.Codicon.symbolProperty);
    byKind.set(10 /* Event */, codicons_1.Codicon.symbolEvent);
    byKind.set(11 /* Operator */, codicons_1.Codicon.symbolOperator);
    byKind.set(12 /* Unit */, codicons_1.Codicon.symbolUnit);
    byKind.set(13 /* Value */, codicons_1.Codicon.symbolValue);
    byKind.set(15 /* Enum */, codicons_1.Codicon.symbolEnum);
    byKind.set(14 /* Constant */, codicons_1.Codicon.symbolConstant);
    byKind.set(15 /* Enum */, codicons_1.Codicon.symbolEnum);
    byKind.set(16 /* EnumMember */, codicons_1.Codicon.symbolEnumMember);
    byKind.set(17 /* Keyword */, codicons_1.Codicon.symbolKeyword);
    byKind.set(27 /* Snippet */, codicons_1.Codicon.symbolSnippet);
    byKind.set(18 /* Text */, codicons_1.Codicon.symbolText);
    byKind.set(19 /* Color */, codicons_1.Codicon.symbolColor);
    byKind.set(20 /* File */, codicons_1.Codicon.symbolFile);
    byKind.set(21 /* Reference */, codicons_1.Codicon.symbolReference);
    byKind.set(22 /* Customcolor */, codicons_1.Codicon.symbolCustomColor);
    byKind.set(23 /* Folder */, codicons_1.Codicon.symbolFolder);
    byKind.set(24 /* TypeParameter */, codicons_1.Codicon.symbolTypeParameter);
    byKind.set(25 /* User */, codicons_1.Codicon.account);
    byKind.set(26 /* Issue */, codicons_1.Codicon.issues);
    /**
     * @internal
     */
    function toIcon(kind) {
        let codicon = byKind.get(kind);
        if (!codicon) {
            console.info('No codicon found for CompletionItemKind ' + kind);
            codicon = codicons_1.Codicon.symbolProperty;
        }
        return codicon;
    }
    CompletionItemKinds.toIcon = toIcon;
    const data = new Map();
    data.set('method', 0 /* Method */);
    data.set('function', 1 /* Function */);
    data.set('constructor', 2 /* Constructor */);
    data.set('field', 3 /* Field */);
    data.set('variable', 4 /* Variable */);
    data.set('class', 5 /* Class */);
    data.set('struct', 6 /* Struct */);
    data.set('interface', 7 /* Interface */);
    data.set('module', 8 /* Module */);
    data.set('property', 9 /* Property */);
    data.set('event', 10 /* Event */);
    data.set('operator', 11 /* Operator */);
    data.set('unit', 12 /* Unit */);
    data.set('value', 13 /* Value */);
    data.set('constant', 14 /* Constant */);
    data.set('enum', 15 /* Enum */);
    data.set('enum-member', 16 /* EnumMember */);
    data.set('enumMember', 16 /* EnumMember */);
    data.set('keyword', 17 /* Keyword */);
    data.set('snippet', 27 /* Snippet */);
    data.set('text', 18 /* Text */);
    data.set('color', 19 /* Color */);
    data.set('file', 20 /* File */);
    data.set('reference', 21 /* Reference */);
    data.set('customcolor', 22 /* Customcolor */);
    data.set('folder', 23 /* Folder */);
    data.set('type-parameter', 24 /* TypeParameter */);
    data.set('typeParameter', 24 /* TypeParameter */);
    data.set('account', 25 /* User */);
    data.set('issue', 26 /* Issue */);
    /**
     * @internal
     */
    function fromString(value, strict) {
        let res = data.get(value);
        if (typeof res === 'undefined' && !strict) {
            res = 9 /* Property */;
        }
        return res;
    }
    CompletionItemKinds.fromString = fromString;
})(CompletionItemKinds = exports.CompletionItemKinds || (exports.CompletionItemKinds = {}));
/**
 * How an {@link InlineCompletionsProvider inline completion provider} was triggered.
 */
var InlineCompletionTriggerKind;
(function (InlineCompletionTriggerKind) {
    /**
     * Completion was triggered automatically while editing.
     * It is sufficient to return a single completion item in this case.
     */
    InlineCompletionTriggerKind[InlineCompletionTriggerKind["Automatic"] = 0] = "Automatic";
    /**
     * Completion was triggered explicitly by a user gesture.
     * Return multiple completion items to enable cycling through them.
     */
    InlineCompletionTriggerKind[InlineCompletionTriggerKind["Explicit"] = 1] = "Explicit";
})(InlineCompletionTriggerKind = exports.InlineCompletionTriggerKind || (exports.InlineCompletionTriggerKind = {}));
var SignatureHelpTriggerKind;
(function (SignatureHelpTriggerKind) {
    SignatureHelpTriggerKind[SignatureHelpTriggerKind["Invoke"] = 1] = "Invoke";
    SignatureHelpTriggerKind[SignatureHelpTriggerKind["TriggerCharacter"] = 2] = "TriggerCharacter";
    SignatureHelpTriggerKind[SignatureHelpTriggerKind["ContentChange"] = 3] = "ContentChange";
})(SignatureHelpTriggerKind = exports.SignatureHelpTriggerKind || (exports.SignatureHelpTriggerKind = {}));
/**
 * A document highlight kind.
 */
var DocumentHighlightKind;
(function (DocumentHighlightKind) {
    /**
     * A textual occurrence.
     */
    DocumentHighlightKind[DocumentHighlightKind["Text"] = 0] = "Text";
    /**
     * Read-access of a symbol, like reading a variable.
     */
    DocumentHighlightKind[DocumentHighlightKind["Read"] = 1] = "Read";
    /**
     * Write-access of a symbol, like writing to a variable.
     */
    DocumentHighlightKind[DocumentHighlightKind["Write"] = 2] = "Write";
})(DocumentHighlightKind = exports.DocumentHighlightKind || (exports.DocumentHighlightKind = {}));
/**
 * @internal
 */
function isLocationLink(thing) {
    return thing
        && uri_1.URI.isUri(thing.uri)
        && range_1.Range.isIRange(thing.range)
        && (range_1.Range.isIRange(thing.originSelectionRange) || range_1.Range.isIRange(thing.targetSelectionRange));
}
exports.isLocationLink = isLocationLink;
/**
 * @internal
 */
var SymbolKinds;
(function (SymbolKinds) {
    const byKind = new Map();
    byKind.set(0 /* File */, codicons_1.Codicon.symbolFile);
    byKind.set(1 /* Module */, codicons_1.Codicon.symbolModule);
    byKind.set(2 /* Namespace */, codicons_1.Codicon.symbolNamespace);
    byKind.set(3 /* Package */, codicons_1.Codicon.symbolPackage);
    byKind.set(4 /* Class */, codicons_1.Codicon.symbolClass);
    byKind.set(5 /* Method */, codicons_1.Codicon.symbolMethod);
    byKind.set(6 /* Property */, codicons_1.Codicon.symbolProperty);
    byKind.set(7 /* Field */, codicons_1.Codicon.symbolField);
    byKind.set(8 /* Constructor */, codicons_1.Codicon.symbolConstructor);
    byKind.set(9 /* Enum */, codicons_1.Codicon.symbolEnum);
    byKind.set(10 /* Interface */, codicons_1.Codicon.symbolInterface);
    byKind.set(11 /* Function */, codicons_1.Codicon.symbolFunction);
    byKind.set(12 /* Variable */, codicons_1.Codicon.symbolVariable);
    byKind.set(13 /* Constant */, codicons_1.Codicon.symbolConstant);
    byKind.set(14 /* String */, codicons_1.Codicon.symbolString);
    byKind.set(15 /* Number */, codicons_1.Codicon.symbolNumber);
    byKind.set(16 /* Boolean */, codicons_1.Codicon.symbolBoolean);
    byKind.set(17 /* Array */, codicons_1.Codicon.symbolArray);
    byKind.set(18 /* Object */, codicons_1.Codicon.symbolObject);
    byKind.set(19 /* Key */, codicons_1.Codicon.symbolKey);
    byKind.set(20 /* Null */, codicons_1.Codicon.symbolNull);
    byKind.set(21 /* EnumMember */, codicons_1.Codicon.symbolEnumMember);
    byKind.set(22 /* Struct */, codicons_1.Codicon.symbolStruct);
    byKind.set(23 /* Event */, codicons_1.Codicon.symbolEvent);
    byKind.set(24 /* Operator */, codicons_1.Codicon.symbolOperator);
    byKind.set(25 /* TypeParameter */, codicons_1.Codicon.symbolTypeParameter);
    /**
     * @internal
     */
    function toIcon(kind) {
        let icon = byKind.get(kind);
        if (!icon) {
            console.info('No codicon found for SymbolKind ' + kind);
            icon = codicons_1.Codicon.symbolProperty;
        }
        return icon;
    }
    SymbolKinds.toIcon = toIcon;
})(SymbolKinds = exports.SymbolKinds || (exports.SymbolKinds = {}));
class FoldingRangeKind {
    /**
     * Creates a new {@link FoldingRangeKind}.
     *
     * @param value of the kind.
     */
    constructor(value) {
        this.value = value;
    }
}
exports.FoldingRangeKind = FoldingRangeKind;
/**
 * Kind for folding range representing a comment. The value of the kind is 'comment'.
 */
FoldingRangeKind.Comment = new FoldingRangeKind('comment');
/**
 * Kind for folding range representing a import. The value of the kind is 'imports'.
 */
FoldingRangeKind.Imports = new FoldingRangeKind('imports');
/**
 * Kind for folding range representing regions (for example marked by `#region`, `#endregion`).
 * The value of the kind is 'region'.
 */
FoldingRangeKind.Region = new FoldingRangeKind('region');
/**
 * @internal
 */
var Command;
(function (Command) {
    /**
     * @internal
     */
    function is(obj) {
        if (!obj || typeof obj !== 'object') {
            return false;
        }
        return typeof obj.id === 'string' &&
            typeof obj.title === 'string';
    }
    Command.is = is;
})(Command = exports.Command || (exports.Command = {}));
/**
 * @internal
 */
var CommentThreadCollapsibleState;
(function (CommentThreadCollapsibleState) {
    /**
     * Determines an item is collapsed
     */
    CommentThreadCollapsibleState[CommentThreadCollapsibleState["Collapsed"] = 0] = "Collapsed";
    /**
     * Determines an item is expanded
     */
    CommentThreadCollapsibleState[CommentThreadCollapsibleState["Expanded"] = 1] = "Expanded";
})(CommentThreadCollapsibleState = exports.CommentThreadCollapsibleState || (exports.CommentThreadCollapsibleState = {}));
/**
 * @internal
 */
var CommentMode;
(function (CommentMode) {
    CommentMode[CommentMode["Editing"] = 0] = "Editing";
    CommentMode[CommentMode["Preview"] = 1] = "Preview";
})(CommentMode = exports.CommentMode || (exports.CommentMode = {}));
var InlayHintKind;
(function (InlayHintKind) {
    InlayHintKind[InlayHintKind["Type"] = 1] = "Type";
    InlayHintKind[InlayHintKind["Parameter"] = 2] = "Parameter";
})(InlayHintKind = exports.InlayHintKind || (exports.InlayHintKind = {}));
/**
 * @internal
 */
exports.TokenizationRegistry = new tokenizationRegistry_1.TokenizationRegistry();
/**
 * @internal
 */
var ExternalUriOpenerPriority;
(function (ExternalUriOpenerPriority) {
    ExternalUriOpenerPriority[ExternalUriOpenerPriority["None"] = 0] = "None";
    ExternalUriOpenerPriority[ExternalUriOpenerPriority["Option"] = 1] = "Option";
    ExternalUriOpenerPriority[ExternalUriOpenerPriority["Default"] = 2] = "Default";
    ExternalUriOpenerPriority[ExternalUriOpenerPriority["Preferred"] = 3] = "Preferred";
})(ExternalUriOpenerPriority = exports.ExternalUriOpenerPriority || (exports.ExternalUriOpenerPriority = {}));
//# sourceMappingURL=languages.js.map