# Commands covered:  clock
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1995-1998 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: clock.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

set env(LC_TIME) POSIX

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

test clock-1.1 {clock tests} {
    list [catch {clock} msg] $msg
} {1 {wrong # args: should be "clock option ?arg ...?"}}
test clock-1.2 {clock tests} {
    list [catch {clock foo} msg] $msg
} {1 {bad option "foo": must be clicks, format, scan, or seconds}}

# clock clicks
test clock-2.1 {clock clicks tests} {
    expr [clock clicks]+1
    concat {}
} {}
test clock-2.2 {clock clicks tests} {
    set start [clock clicks]
    after 10
    set end [clock clicks]
    expr "$end > $start"
} {1}
test clock-2.3 {clock clicks tests} {
    list [catch {clock clicks foo} msg] $msg
} {1 {bad switch "foo": must be -milliseconds}}
test clock-2.4 {clock clicks tests} {
    expr [clock clicks -milliseconds]+1
    concat {}
} {}
test clock-2.5 {clock clicks tests, millisecond timing test} {
    set start [clock clicks -milli]
    after 10
    set end [clock clicks -milli]
    # 60 msecs seems to be the max time slice under Windows 95/98
    expr {($end > $start) && (($end - $start) <= 60)}
} {1}
test clock-2.6 {clock clicks, milli with too much abbreviation} {
    list [catch { clock clicks {} } msg] $msg
} {1 {bad switch "": must be -milliseconds}}
test clock-2.7 {clock clicks, milli with too much abbreviation} {
    list [catch { clock clicks - } msg] $msg
} {1 {bad switch "-": must be -milliseconds}}

# clock format
test clock-3.1 {clock format tests} {unixOnly} {
    set clockval 657687766
    clock format $clockval -format {%a %b %d %I:%M:%S %p %Y} -gmt true
} {Sun Nov 04 03:02:46 AM 1990}
test clock-3.2 {clock format tests} {
    # TCL_USE_TIMEZONE_VAR

    catch {set oldtz $env(TZ)}
    set env(TZ) PST
    set x {}
    append x [clock format 863800000 -format %Z -gmt 1]
    append x [set env(TZ)]
    catch {unset env(TZ); set env(TZ) $oldtz}
    set x
} {GMTPST}
test clock-3.3 {clock format tests} {
    # tzset() under Borland doesn't seem to set up tzname[] for local 
    # timezone, which caused "clock format" to think that %Z was an invalid
    # string.  Don't care about answer, just that test runs w/o error.

    clock format 863800000 -format %Z
    set x {}
} {}
test clock-3.4 {clock format tests} {
    # tzset() under Borland doesn't seem to set up tzname[] for gmt timezone.
    # tzset() under MSVC has the following weird observed behavior:
    #	 First time we call "clock format [clock seconds] -format %Z -gmt 1"
    #	 we get "GMT", but on all subsequent calls we get the current time 
    #	 zone string, even though env(TZ) is GMT and the variable _timezone 
    #    is 0.

    set x {}
    append x [clock format 863800000 -format %Z -gmt 1]
    append x [clock format 863800000 -format %Z -gmt 1]
} {GMTGMT}
test clock-3.5 {clock format tests} {
    list [catch {clock format} msg] $msg
} {1 {wrong # args: should be "clock format clockval ?-format string? ?-gmt boolean?"}}
test clock-3.6 {clock format tests} {
    list [catch {clock format foo} msg] $msg
} {1 {expected integer but got "foo"}}
test clock-3.7 {clock format tests} {unixOrPc} {
    set clockval 657687766
    clock format $clockval -format "%a %b %d %I:%M:%S %p %Y" -gmt true
} "Sun Nov 04 03:02:46 AM 1990"
test clock-3.8 {clock format tests} {
    list [catch {clock format a b c d e g} msg] $msg
} {1 {wrong # args: should be "clock format clockval ?-format string? ?-gmt boolean?"}}
test clock-3.9 {clock format tests} {unixOrPc nonPortable} {
    set clockval -1
    clock format $clockval -format "%a %b %d %I:%M:%S %p %Y" -gmt true
} "Wed Dec 31 11:59:59 PM 1969"
test clock-3.10 {clock format tests} {
    list [catch {clock format 123 -bad arg} msg] $msg
} {1 {bad switch "-bad": must be -format or -gmt}}
test clock-3.11 {clock format tests} {
    clock format 123 -format "x"
} x
test clock-3.12 {clock format tests} {
    clock format 123 -format ""
} ""
test clock-3.13 {clock format with non-ASCII character in the format string} {
    set oldenc [encoding system] 
    encoding system iso8859-1
    set res [clock format 0 -format \u00c4]
    encoding system $oldenc
    unset oldenc
    set res
} "\u00c4"

# Bug 942078

test clock-3.14 {change of time zone} -setup {
    catch { unset oldTZ }
    if { [info exists env(TZ)] } {
	set oldTZ $env(TZ)
    }
} -body {
    set env(TZ) PST8PDT
    set s [clock format 0 -format %H%M]
    set env(TZ) GMT0
    append s -[clock format 0 -format %H%M]
} -cleanup {
    if { [info exists oldTZ] } {
	set env(TZ) $oldTZ
	unset oldTZ
    } else {
	unset env(TZ)
    }
} -result {1600-0000}
    

# clock scan
test clock-4.1 {clock scan tests} {
    list [catch {clock scan} msg] $msg
} {1 {wrong # args: should be "clock scan dateString ?-base clockValue? ?-gmt boolean?"}}
test clock-4.2 {clock scan tests} {
    list [catch {clock scan "bad-string"} msg] $msg
} {1 {unable to convert date-time string "bad-string"}}
test clock-4.3 {clock scan tests} {
    clock format [clock scan "14 Feb 92" -gmt true] \
      -format {%m/%d/%y %I:%M:%S %p} -gmt true
} {02/14/92 12:00:00 AM}
test clock-4.4 {clock scan tests} {
    clock format [clock scan "Feb 14, 1992 12:20 PM" -gmt true] \
      -format {%m/%d/%y %I:%M:%S %p} -gmt true
} {02/14/92 12:20:00 PM}
test clock-4.5 {clock scan tests} {
    clock format \
      [clock scan "Feb 14, 1992 12:20 PM" -base 319363200 -gmt true] \
      -format {%m/%d/%y %I:%M:%S %p} -gmt true
} {02/14/92 12:20:00 PM}
test clock-4.6 {clock scan tests} {
    set time [clock scan "Oct 23,1992 15:00"]
    clock format $time -format {%b %d,%Y %H:%M}
} {Oct 23,1992 15:00}
test clock-4.7 {clock scan tests} {
    set time [clock scan "Oct 23,1992 15:00 GMT"]
    clock format $time -format {%b %d,%Y %H:%M GMT} -gmt true
} {Oct 23,1992 15:00 GMT}
test clock-4.8 {clock scan tests} {
    set time [clock scan "Oct 23,1992 15:00" -gmt true]
    clock format $time -format {%b %d,%Y %H:%M GMT} -gmt true
} {Oct 23,1992 15:00 GMT}
test clock-4.9 {clock scan tests} {
    list [catch {clock scan "Jan 12" -bad arg} msg] $msg
} {1 {bad switch "-bad": must be -base or -gmt}}
# The following two two tests test the two year date policy
test clock-4.10 {clock scan tests} {
    set time [clock scan "1/1/71" -gmt true]
    clock format $time -format {%b %d,%Y %H:%M GMT} -gmt true
} {Jan 01,1971 00:00 GMT}
test clock-4.11 {clock scan tests} {
    set time [clock scan "1/1/37" -gmt true]
    clock format $time -format {%b %d,%Y %H:%M GMT} -gmt true
} {Jan 01,2037 00:00 GMT}

test clock-4.12 {clock scan, relative times} {
    set time [clock scan "Oct 23, 1992 -1 day"]
    clock format $time -format {%b %d, %Y}
} "Oct 22, 1992"
test clock-4.13 {clock scan, ISO 8601 base date format} {
    set time [clock scan "19921023"]
    clock format $time -format {%b %d, %Y}
} "Oct 23, 1992"
test clock-4.14 {clock scan, ISO 8601 expanded date format} {
    set time [clock scan "1992-10-23"]
    clock format $time -format {%b %d, %Y}
} "Oct 23, 1992"
test clock-4.15 {clock scan, DD-Mon-YYYY format} {
    set time [clock scan "23-Oct-1992"]
    clock format $time -format {%b %d, %Y}
} "Oct 23, 1992"
test clock-4.16 {clock scan, ISO 8601 point in time format} {
    set time [clock scan "19921023T235959"]
    clock format $time -format {%b %d, %Y %H:%M:%S}
} "Oct 23, 1992 23:59:59"
test clock-4.17 {clock scan, ISO 8601 point in time format} {
    set time [clock scan "19921023 235959"]
    clock format $time -format {%b %d, %Y %H:%M:%S}
} "Oct 23, 1992 23:59:59"
test clock-4.18 {clock scan, ISO 8601 point in time format} {
    set time [clock scan "19921023T000000"]
    clock format $time -format {%b %d, %Y %H:%M:%S}
} "Oct 23, 1992 00:00:00"

# CLOCK SCAN REAL TESTS
# We use 5am PST, 31-12-1999 as the base for these scans because irrespective
# of your local timezone it should always give us times on December 31, 1999
set 5amPST 946645200
test clock-4.19 {clock scan, number meridian} {
    set t1 [clock scan "5 am" -base $5amPST -gmt true]
    set t2 [clock scan "5 pm" -base $5amPST -gmt true]
    set t3 [clock scan "5 a.m." -base $5amPST -gmt true]
    set t4 [clock scan "5 p.m." -base $5amPST -gmt true]
    list \
	    [clock format $t1 -format {%b %d, %Y %H:%M:%S} -gmt true] \
	    [clock format $t2 -format {%b %d, %Y %H:%M:%S} -gmt true] \
	    [clock format $t3 -format {%b %d, %Y %H:%M:%S} -gmt true] \
	    [clock format $t4 -format {%b %d, %Y %H:%M:%S} -gmt true]
} [list "Dec 31, 1999 05:00:00" "Dec 31, 1999 17:00:00" \
	"Dec 31, 1999 05:00:00" "Dec 31, 1999 17:00:00"]
test clock-4.20 {clock scan, number:number meridian} {
    clock format [clock scan "5:30 pm" -base $5amPST -gmt true] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 17:30:00"
test clock-4.21 {clock scan, number:number-timezone} {
    clock format [clock scan "00:00-0800" -gmt true -base $5amPST] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 08:00:00"
test clock-4.22 {clock scan, number:number:number o_merid} {
    clock format [clock scan "8:00:00" -gmt true -base $5amPST] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 08:00:00"
test clock-4.23 {clock scan, number:number:number o_merid} {
    clock format [clock scan "8:00:00 am" -gmt true -base $5amPST] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 08:00:00"
test clock-4.24 {clock scan, number:number:number o_merid} {
    clock format [clock scan "8:00:00 pm" -gmt true -base $5amPST] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 20:00:00"
test clock-4.25 {clock scan, number:number:number-timezone} {
    clock format [clock scan "00:00:30-0800" -gmt true -base $5amPST] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Dec 31, 1999 08:00:30"
test clock-4.26 {clock scan, DST for days} {
    clock scan "tomorrow" -base [clock scan "19991031 00:00:00"]
} [clock scan "19991101 00:00:00"]
test clock-4.27 {clock scan, DST for days} {
    clock scan "yesterday" -base [clock scan "19991101 00:00:00"]
} [clock scan "19991031 00:00:00"]
test clock-4.28 {clock scan, day} knownBug {
    clock format [clock scan "Monday" -gmt true -base 946627200] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Jan 03, 2000 00:00:00"
test clock-4.29 {clock scan, number/number} {
    clock format [clock scan "1/1" -gmt true -base 946627200] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Jan 01, 1999 00:00:00"
test clock-4.30 {clock scan, number/number} {
    clock format [clock scan "1/1/1999" -gmt true -base 946627200] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Jan 01, 1999 00:00:00"
test clock-4.31 {clock scan, number/number} {
    clock format [clock scan "19990101" -gmt true -base 946627200] \
	    -format {%b %d, %Y %H:%M:%S} -gmt true
} "Jan 01, 1999 00:00:00"
test clock-4.32 {clock scan, relative minutes} {
    clock scan "now + 1 minute" -base 946627200
} 946627260
test clock-4.33 {clock scan, relative minutes} {
    clock scan "now +1 minute" -base 946627200
} 946627260
test clock-4.34 {clock scan, relative minutes} {
    clock scan "now 1 minute" -base 946627200
} 946627260
test clock-4.35 {clock scan, relative minutes} {
    clock scan "now - 1 minute" -base 946627200
} 946627140
test clock-4.36 {clock scan, relative minutes} {
    clock scan "now -1 minute" -base 946627200
} 946627140
test clock-4.37 {clock scan, day of week} {
    clock format [clock scan "wednesday" -base [clock scan 20000112]] \
	    -format {%b %d, %Y}
} "Jan 12, 2000"
test clock-4.38 {clock scan, next day of week} {
    clock format [clock scan "next wednesday" -base [clock scan 20000112]] \
	    -format {%b %d, %Y}
} "Jan 19, 2000"
test clock-4.39 {clock scan, day of week} {
    clock format [clock scan "thursday" -base [clock scan 20000112]] \
	    -format {%b %d, %Y}
} "Jan 13, 2000"
test clock-4.40 {clock scan, next day of week} {
    clock format [clock scan "next thursday" -base [clock scan 20000112]] \
	    -format {%b %d, %Y}
} "Jan 20, 2000"

# weekday specification and base.
test clock-4.41 {2nd monday in november} {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set nov8th [clock scan 11/8/$i]
      set monday [clock scan monday -base $nov8th]
      lappend res [clock format $monday -format %Y-%m-%d]
    }
    set res
} {1991-11-11 1992-11-09 1993-11-08 1994-11-14 1995-11-13 1996-11-11}
test clock-4.42 {2nd monday in november (2nd try)} {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set nov1th [clock scan 11/1/$i]
      set monday [clock scan "2 monday" -base $nov1th]
      lappend res [clock format $monday -format %Y-%m-%d]
    }
    set res
} {1991-11-11 1992-11-09 1993-11-08 1994-11-14 1995-11-13 1996-11-11}
test clock-4.43 {last monday in november} {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set dec1th [clock scan 12/1/$i]
      set monday [clock scan "monday 1 week ago" -base $dec1th]
      lappend res [clock format $monday -format %Y-%m-%d]
    }
    set res
} {1991-11-25 1992-11-30 1993-11-29 1994-11-28 1995-11-27 1996-11-25}

test clock-4.44 {2nd monday in november} knownBug {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set nov8th [clock scan 11/8/$i -gmt 1]
      set monday [clock scan monday -base $nov8th -gmt 1]
      lappend res [clock format $monday -format %Y-%m-%d -gmt 1]
    }
    set res
} {1991-11-11 1992-11-09 1993-11-08 1994-11-14 1995-11-13 1996-11-11}
test clock-4.45 {2nd monday in november (2nd try)} knownBug {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set nov1th [clock scan 11/1/$i -gmt 1]
      set monday [clock scan "2 monday" -base $nov1th -gmt 1]
      lappend res [clock format $monday -format %Y-%m-%d -gmt 1]
    }
    set res
} {1991-11-11 1992-11-09 1993-11-08 1994-11-14 1995-11-13 1996-11-11}
test clock-4.46 {last monday in november} knownBug {
    set res {}
    foreach i {91 92 93 94 95 96} {
      set dec1th [clock scan 12/1/$i -gmt 1]
      set monday [clock scan "monday 1 week ago" -base $dec1th -gmt 1]
      lappend res [clock format $monday -format %Y-%m-%d -gmt 1]
    }
    set res
} {1991-11-25 1992-11-30 1993-11-29 1994-11-28 1995-11-27 1996-11-25}
test clock-4.47 {ago with multiple relative units} {
    set base [clock scan "12/31/1999 00:00:00"]
    set res [clock scan "2 days 2 hours ago" -base $base]
    expr {$base - $res}
} 180000

# clock seconds
test clock-5.1 {clock seconds tests} {
    expr [clock seconds]+1
    concat {}
} {}
test clock-5.2 {clock seconds tests} {
    list [catch {clock seconds foo} msg] $msg
} {1 {wrong # args: should be "clock seconds"}}
test clock-5.3 {clock seconds tests} {
    set start [clock seconds]
    after 2000
    set end [clock seconds]
    expr "$end > $start"
} {1}

# The following dates check certain roll over dates
set day [expr 24 * 60 * 60]
test clock-6.1 {clock roll over dates} {
    set time [clock scan "12/31/1998" -gmt true]
    clock format [expr $time + $day] -format {%b %d,%Y %H:%M GMT} -gmt true
} {Jan 01,1999 00:00 GMT}
test clock-6.2 {clock roll over dates} {
    set time [clock scan "12/31/1999" -gmt true]
    clock format [expr $time + $day] -format {%b %d,%Y %H:%M GMT} -gmt true
} {Jan 01,2000 00:00 GMT}
test clock-6.3 {clock roll over dates} {
    set time [clock scan "2/28/2000" -gmt true]
    clock format [expr $time + $day] -format {%b %d,%Y %H:%M GMT} -gmt true
} {Feb 29,2000 00:00 GMT}
test clock-6.4 {clock roll over dates} {
    set time [clock scan "2/29/2000" -gmt true]
    clock format [expr $time + $day] -format {%b %d,%Y %H:%M GMT} -gmt true
} {Mar 01,2000 00:00 GMT}
test clock-6.5 {clock roll over dates} {
    set time [clock scan "January 1, 2000" -gmt true]
    clock format $time -format %A -gmt true
} {Saturday}
test clock-6.6 {clock roll over dates} {
    set time [clock scan "January 1, 2000" -gmt true]
    clock format $time -format %j -gmt true
} {001}
test clock-6.7 {clock roll over dates} {
    set time [clock scan "February 29, 2000" -gmt true]
    clock format $time -format %A -gmt true
} {Tuesday}
test clock-6.8 {clock roll over dates} {
    set time [clock scan "February 29, 2000" -gmt true]
    clock format $time -format %j -gmt true
} {060}
test clock-6.9 {clock roll over dates} {
    set time [clock scan "March 1, 2000" -gmt true]
    clock format $time -format %A -gmt true
} {Wednesday}
test clock-6.10 {clock roll over dates} {
    set time [clock scan "March 1, 2000" -gmt true]
    clock format $time -format %j -gmt true
} {061}
test clock-6.11 {clock roll over dates} {
    set time [clock scan "March 1, 2001" -gmt true]
    clock format $time -format %j -gmt true
} {060}

test clock-7.1 {clock scan next monthname} {
    clock format [clock scan "next june" -base [clock scan "june 1, 2000"]] \
	    -format %m.%Y
} "06.2001"
test clock-7.2 {clock scan next monthname} {
    clock format [clock scan "next july" -base [clock scan "june 1, 2000"]] \
	    -format %m.%Y
} "07.2000"
test clock-7.3 {clock scan next monthname} {
    clock format [clock scan "next may" -base [clock scan "june 1, 2000"]] \
	    -format %m.%Y
} "05.2001"

# We use 5am PST, 31-12-1999 as the base for these scans because irrespective
# of your local timezone it should always give us times on December 31
set 5amPST 946645200
test clock-8.1 {clock scan midnight/gmt range bug 413397} {
    set fmt "%m/%d"
    list [clock format [clock scan year -base $5amPST -gmt 0] -format $fmt] \
	    [clock format [clock scan year -base $5amPST -gmt 1] -format $fmt]
} {12/31 12/31}

::tcltest::testConstraint needPST [expr {
    [regexp {^(Pacific.*|P[DS]T)$} [clock format 1 -format %Z]]
    && ([clock format 1 -format %s] != "%s")
}]

test clock-9.1 {%s gmt testing} {needPST} {

    # Note that this test will fail if the strftime on the underlying
    # system doesn't support the %s format group.  Systems that are known
    # to have trouble include the native C libraries on AIX and HP-UX

    # We need PST to guarantee the difference value below, and %s isn't
    # valid on all OSes (like Solaris).
    set s 100000
    set a [clock format $s -format %s -gmt 0]
    set b [clock format $s -format %s -gmt 1]
    # This should be the offset in seconds between current locale and GMT.
    # This didn't seem to be correctly on Windows until the fix for
    # Bug #559376, which fiddled with env(TZ) when -gmt 1 was used.
    # It's hard-coded to check P[SD]T now. (8 hours)
    set c [expr {$b-$a}]
} {28800}

::tcltest::testConstraint percentG \
    [expr { ![catch { clock format 0 -format %G -gmt true } y1970]
	    && $y1970 eq {1970} }]

test clock-10.0 {Can strftime do %G?} {
    clock format 0 -format %G -gmt true
} 1970
test clock-10.1 {ISO week-based calendar 2000-W52-1} {percentG} {
    clock format 977702400 -format {%a %A %g %G %u %V %w} -gmt true; # 2000-12-25
} {Mon Monday 00 2000 1 52 1}
test clock-10.2 {ISO week-based calendar 2000-W52-7} {percentG} {
    clock format 978220800 -format {%a %A %g %G %u %V %w} -gmt true; # 2000-12-31
} {Sun Sunday 00 2000 7 52 0}
test clock-10.3 {ISO week-based calendar 2001-W01-1} {percentG} {
    clock format 978307200 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-1-1
} {Mon Monday 01 2001 1 01 1}
test clock-10.4 {ISO week-based calendar 2001-W01-7} {percentG} {
    clock format 978825600 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-1-7
} {Sun Sunday 01 2001 7 01 0}
test clock-10.5 {ISO week-based calendar 2001-W02-1} {percentG} {
    clock format 978912000 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-1-8
} {Mon Monday 01 2001 1 02 1}
test clock-10.6 {ISO week-based calendar 2001-W52-1} {percentG} {
    clock format 1009152000 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-12-24
} {Mon Monday 01 2001 1 52 1}
test clock-10.7 {ISO week-based calendar 2001-W52-7} {percentG} {
    clock format 1009670400 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-12-30
} {Sun Sunday 01 2001 7 52 0}
test clock-10.8 {ISO week-based calendar 2002-W01-1} {percentG} {
    clock format 1009756800 -format {%a %A %g %G %u %V %w} -gmt true; # 2001-12-31
} {Mon Monday 02 2002 1 01 1}
test clock-10.9 {ISO week-based calendar 2002-W01-2} {percentG} {
    clock format 1009843200 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-1-1
} {Tue Tuesday 02 2002 2 01 2}
test clock-10.10 {ISO week-based calendar 2002-W01-7} {percentG} {
    clock format 1010275200 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-1-6
} {Sun Sunday 02 2002 7 01 0}
test clock-10.11 {ISO week-based calendar 2002-W02-1} {percentG} {
    clock format 1010361600 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-1-7
} {Mon Monday 02 2002 1 02 1}
test clock-10.12 {ISO week-based calendar 2002-W52-1} {percentG} {
    clock format 1040601600 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-12-23
} {Mon Monday 02 2002 1 52 1}
test clock-10.13 {ISO week-based calendar 2002-W52-7} {percentG} {
    clock format 1041120000 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-12-29
} {Sun Sunday 02 2002 7 52 0}
test clock-10.14 {ISO week-based calendar 2003-W01-1} {percentG} {
    clock format 1041206400 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-12-30
} {Mon Monday 03 2003 1 01 1}
test clock-10.15 {ISO week-based calendar 2003-W01-2} {percentG} {
    clock format 1041292800 -format {%a %A %g %G %u %V %w} -gmt true; # 2002-12-31
} {Tue Tuesday 03 2003 2 01 2}
test clock-10.16 {ISO week-based calendar 2003-W01-3} {percentG} {
    clock format 1041379200 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-1-1
} {Wed Wednesday 03 2003 3 01 3}
test clock-10.17 {ISO week-based calendar 2003-W01-7} {percentG} {
    clock format 1041724800 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-1-5
} {Sun Sunday 03 2003 7 01 0}
test clock-10.18 {ISO week-based calendar 2003-W02-1} {percentG} {
    clock format 1041811200 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-1-6
} {Mon Monday 03 2003 1 02 1}
test clock-10.19 {ISO week-based calendar 2003-W52-1} {percentG} {
    clock format 1072051200 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-12-22
} {Mon Monday 03 2003 1 52 1}
test clock-10.20 {ISO week-based calendar 2003-W52-7} {percentG} {
    clock format 1072569600 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-12-28
} {Sun Sunday 03 2003 7 52 0}
test clock-10.21 {ISO week-based calendar 2004-W01-1} {percentG} {
    clock format 1072656000 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-12-29
} {Mon Monday 04 2004 1 01 1}
test clock-10.22 {ISO week-based calendar 2004-W01-3} {percentG} {
    clock format 1072828800 -format {%a %A %g %G %u %V %w} -gmt true; # 2003-12-31
} {Wed Wednesday 04 2004 3 01 3}
test clock-10.23 {ISO week-based calendar 2004-W01-4} {percentG} {
    clock format 1072915200 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-1-1
} {Thu Thursday 04 2004 4 01 4}
test clock-10.24 {ISO week-based calendar 2004-W01-7} {percentG} {
    clock format 1073174400 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-1-4
} {Sun Sunday 04 2004 7 01 0}
test clock-10.25 {ISO week-based calendar 2004-W02-1} {percentG} {
    clock format 1073260800 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-1-5
} {Mon Monday 04 2004 1 02 1}
test clock-10.26 {ISO week-based calendar 2004-W52-1} {percentG} {
    clock format 1103500800 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-12-20
} {Mon Monday 04 2004 1 52 1}
test clock-10.27 {ISO week-based calendar 2004-W52-7} {percentG} {
    clock format 1104019200 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-12-26
} {Sun Sunday 04 2004 7 52 0}
test clock-10.28 {ISO week-based calendar 2004-W53-1} {percentG} {
    clock format 1104105600 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-12-27
} {Mon Monday 04 2004 1 53 1}
test clock-10.29 {ISO week-based calendar 2004-W53-5} {percentG} {
    clock format 1104451200 -format {%a %A %g %G %u %V %w} -gmt true; # 2004-12-31
} {Fri Friday 04 2004 5 53 5}
test clock-10.30 {ISO week-based calendar 2004-W53-6} {percentG} {
    clock format 1104537600 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-1-1
} {Sat Saturday 04 2004 6 53 6}
test clock-10.31 {ISO week-based calendar 2004-W53-7} {percentG} {
    clock format 1104624000 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-1-2
} {Sun Sunday 04 2004 7 53 0}
test clock-10.32 {ISO week-based calendar 2005-W01-1} {percentG} {
    clock format 1104710400 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-1-3
} {Mon Monday 05 2005 1 01 1}
test clock-10.33 {ISO week-based calendar 2005-W01-7} {percentG} {
    clock format 1105228800 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-1-9
} {Sun Sunday 05 2005 7 01 0}
test clock-10.34 {ISO week-based calendar 2005-W02-1} {percentG} {
    clock format 1105315200 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-1-10
} {Mon Monday 05 2005 1 02 1}
test clock-10.35 {ISO week-based calendar 2005-W52-1} {percentG} {
    clock format 1135555200 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-12-26
} {Mon Monday 05 2005 1 52 1}
test clock-10.36 {ISO week-based calendar 2005-W52-6} {percentG} {
    clock format 1135987200 -format {%a %A %g %G %u %V %w} -gmt true; # 2005-12-31
} {Sat Saturday 05 2005 6 52 6}
test clock-10.37 {ISO week-based calendar 2005-W52-7} {percentG} {
    clock format 1136073600 -format {%a %A %g %G %u %V %w} -gmt true; # 2006-1-1
} {Sun Sunday 05 2005 7 52 0}
test clock-10.38 {ISO week-based calendar 2006-W01-1} {percentG} {
    clock format 1136160000 -format {%a %A %g %G %u %V %w} -gmt true; # 2006-1-2
} {Mon Monday 06 2006 1 01 1}
test clock-10.39 {ISO week-based calendar 2006-W01-7} {percentG} {
    clock format 1136678400 -format {%a %A %g %G %u %V %w} -gmt true; # 2006-1-8
} {Sun Sunday 06 2006 7 01 0}
test clock-10.40 {ISO week-based calendar 2006-W02-1} {percentG} {
    clock format 1136764800 -format {%a %A %g %G %u %V %w} -gmt true; # 2006-1-9
} {Mon Monday 06 2006 1 02 1}
test clock-10.41 {ISO week-based calendar 2009-W52-1} {percentG} {
    clock format 1261353600 -format {%a %A %g %G %u %V %w} -gmt true; # 2009-12-21
} {Mon Monday 09 2009 1 52 1}
test clock-10.42 {ISO week-based calendar 2009-W52-7} {percentG} {
    clock format 1261872000 -format {%a %A %g %G %u %V %w} -gmt true; # 2009-12-27
} {Sun Sunday 09 2009 7 52 0}
test clock-10.43 {ISO week-based calendar 2009-W53-1} {percentG} {
    clock format 1261958400 -format {%a %A %g %G %u %V %w} -gmt true; # 2009-12-28
} {Mon Monday 09 2009 1 53 1}
test clock-10.44 {ISO week-based calendar 2009-W53-4} {percentG} {
    clock format 1262217600 -format {%a %A %g %G %u %V %w} -gmt true; # 2009-12-31
} {Thu Thursday 09 2009 4 53 4}
test clock-10.45 {ISO week-based calendar 2009-W53-5} {percentG} {
    clock format 1262304000 -format {%a %A %g %G %u %V %w} -gmt true; # 2010-1-1
} {Fri Friday 09 2009 5 53 5}
test clock-10.46 {ISO week-based calendar 2009-W53-7} {percentG} {
    clock format 1262476800 -format {%a %A %g %G %u %V %w} -gmt true; # 2010-1-3
} {Sun Sunday 09 2009 7 53 0}
test clock-10.47 {ISO week-based calendar 2010-W01-1} {percentG} {
    clock format 1262563200 -format {%a %A %g %G %u %V %w} -gmt true; # 2010-1-4
} {Mon Monday 10 2010 1 01 1}
test clock-10.48 {ISO week-based calendar 2010-W01-7} {percentG} {
    clock format 1263081600 -format {%a %A %g %G %u %V %w} -gmt true; # 2010-1-10
} {Sun Sunday 10 2010 7 01 0}
test clock-10.49 {ISO week-based calendar 2010-W02-1} {percentG} {
    clock format 1263168000 -format {%a %A %g %G %u %V %w} -gmt true; # 2010-1-11
} {Mon Monday 10 2010 1 02 1}

test clock-41.1 {regression test - format group %k when hour is 0 } {
    clock format 0 -format %k -gmt true
} { 0}

# cleanup
::tcltest::cleanupTests
return
