/******************************************************************************
 * Copyright (c) 2003, 2006 IBM Corporation and others.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    IBM Corporation - initial API and implementation 
 ****************************************************************************/

package org.eclipse.gmf.runtime.diagram.ui.handles;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.ImageFigure;
import org.eclipse.draw2d.Label;
import org.eclipse.draw2d.Locator;
import org.eclipse.draw2d.PositionConstants;
import org.eclipse.draw2d.StackLayout;
import org.eclipse.draw2d.TreeSearch;
import org.eclipse.gef.DragTracker;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.handles.AbstractHandle;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.internal.l10n.DiagramUIPluginImages;
import org.eclipse.gmf.runtime.diagram.ui.internal.tools.ConnectionHandleTool;
import org.eclipse.gmf.runtime.diagram.ui.l10n.SharedImages;
import org.eclipse.swt.graphics.Image;

/**
 * This is the handle figure used to represent a connection handle.
 * 
 * @author cmahoney
 */
public class ConnectionHandle extends AbstractHandle {

	/**
	 * An enumeration of connection directions.
	 * OUTGOING = source to target
	 * INCOMING = target to source
	 */
	public static final class HandleDirection {
		private HandleDirection() {
		    // empty
		}

		/** source to target */
		public static final HandleDirection OUTGOING = new HandleDirection();

		/** target to source */
		public static final HandleDirection INCOMING = new HandleDirection();
	}

	/** the error icon that can be superimposed on the connection handle image */
	private static final ImageFigure ERROR_IMAGE = new ImageFigure(SharedImages
		.get(SharedImages.IMG_ERROR));

	static {
		ERROR_IMAGE.setSize(SharedImages.get(SharedImages.IMG_ERROR)
			.getBounds().width, SharedImages.get(SharedImages.IMG_ERROR)
			.getBounds().height);
	}

	/** direction that the relationship is to be created */
	private HandleDirection handleDirection;

	/**
	 * Creates a new <code>ConnectionHandle</code>.
	 * @param ownerEditPart the editpart for which the handle belongs
	 * @param relationshipDirection direction that the relationship is to be created
	 * @param tooltip the tooltip
	 */
	public ConnectionHandle(
		IGraphicalEditPart ownerEditPart,
		HandleDirection relationshipDirection,
		String tooltip) {

		setOwner(ownerEditPart);
		setRelationshipDirection(relationshipDirection);
		setToolTip(new Label(tooltip));

		// A stack layout is used so that the error icon can be overlayed on top.
		setLayoutManager(new StackLayout());
	}
	
	/**
	 * @see org.eclipse.gef.handles.AbstractHandle#createDragTracker()
	 */
	protected DragTracker createDragTracker() {
		return new ConnectionHandleTool(this);
	}

	/**
	 * @see org.eclipse.draw2d.IFigure#findFigureAt(int, int, org.eclipse.draw2d.TreeSearch)
	 */
	public IFigure findFigureAt(int x, int y, TreeSearch search) {
		// return the ConnectionHandle and not the children figures
		if (containsPoint(x, y)) {
			return this;
		}
		return super.findFigureAt(x, y, search);
	}

	/**
	 * Make public.
	 * @see org.eclipse.gef.handles.AbstractHandle#setLocator(org.eclipse.draw2d.Locator)
	 */
	public void setLocator(Locator locator) {
		super.setLocator(locator);
	}

	/**
	 * Make public.
	 * @see org.eclipse.gef.handles.AbstractHandle#getOwner()
	 */
	public GraphicalEditPart getOwner() {
		return super.getOwner();
	}

	/**
	 * Sets the direction that the relationship is to be created.
	 * @param direction the <code>HandleDirection</code> that the relationship is to be created
	 */
	protected void setRelationshipDirection(HandleDirection direction) {
		handleDirection = direction;
	}

	/**
	 * Is this for incoming relationships?
	 * @return true if this is for incoming relationships, false otherwise
	 */
	public boolean isIncoming() {
		return handleDirection == HandleDirection.INCOMING;
	}

	/**
	 * Superimposes an error icon on this connection handle.
	 */
	public void addErrorIcon() {
		add(ERROR_IMAGE);
	}

	/**
	 * Removes the error icon if it is being displayed.
	 */
	public void removeErrorIcon() {
		if (getChildren().contains(ERROR_IMAGE)) {
			remove(ERROR_IMAGE);
		}
	}

	/**
	 * Updates the images used for the handles, based on the side they will
	 * appear on.  Sets the location of the handles using the locator.
	 * @see org.eclipse.draw2d.IFigure#validate()
	 */
	public void validate() {
		if (isValid())
			return;

		removeAll();
		int side = ((ConnectionHandleLocator) getLocator())
			.getBorderSide();
		Image image = getImage(side);

		ImageFigure imageFigure = new ImageFigure(image);
		imageFigure.setSize(image.getBounds().width, image.getBounds().height);
		add(imageFigure);

		setSize(imageFigure.getSize().getUnioned(ERROR_IMAGE.getSize()));

		super.validate();
	}
    
    /**
     * Gets the image to be used for the connection handle given the side of the
     * shape where the connection handle will appear. A call to
     * <code>isIncoming()</code> will reveal the direction of the handle.
     * 
     * @param side
     *            the side of the shape where the connection handle will appear,
     *            a value in PositionConstants
     * @return the image to be used for the connection handle
     */
    protected Image getImage(int side) {
        if (side == PositionConstants.WEST) {
            return isIncoming() ? DiagramUIPluginImages
                .get(DiagramUIPluginImages.IMG_HANDLE_INCOMING_WEST)
                : DiagramUIPluginImages
                    .get(DiagramUIPluginImages.IMG_HANDLE_OUTGOING_WEST);
        } else if (side == PositionConstants.EAST) {
            return isIncoming() ? DiagramUIPluginImages
                .get(DiagramUIPluginImages.IMG_HANDLE_INCOMING_EAST)
                : DiagramUIPluginImages
                    .get(DiagramUIPluginImages.IMG_HANDLE_OUTGOING_EAST);
        } else if (side == PositionConstants.SOUTH) {
            return isIncoming() ? DiagramUIPluginImages
                .get(DiagramUIPluginImages.IMG_HANDLE_INCOMING_SOUTH)
                : DiagramUIPluginImages
                    .get(DiagramUIPluginImages.IMG_HANDLE_OUTGOING_SOUTH);
        } else {
            return isIncoming() ? DiagramUIPluginImages
                .get(DiagramUIPluginImages.IMG_HANDLE_INCOMING_NORTH)
                : DiagramUIPluginImages
                    .get(DiagramUIPluginImages.IMG_HANDLE_OUTGOING_NORTH);
        }
    }

}
