/**
 ********************************************************************************
 * Copyright (c) 2021 Robert Bosch GmbH and others.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.validations.sim.software;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.app4mc.amalthea.model.AbstractMemoryElement;
import org.eclipse.app4mc.amalthea.model.ActivityGraphItem;
import org.eclipse.app4mc.amalthea.model.AmaltheaIndex;
import org.eclipse.app4mc.amalthea.model.HwAccessElement;
import org.eclipse.app4mc.amalthea.model.Memory;
import org.eclipse.app4mc.amalthea.model.ProcessingUnit;
import org.eclipse.app4mc.amalthea.model.RunnableCall;
import org.eclipse.app4mc.amalthea.model.Scheduler;
import org.eclipse.app4mc.amalthea.model.SchedulerAllocation;
import org.eclipse.app4mc.amalthea.model.util.DeploymentUtil;
import org.eclipse.app4mc.validation.annotation.Validation;
import org.eclipse.app4mc.validation.core.ValidationDiagnostic;
import org.eclipse.emf.common.util.EList;

/**
 * Checks feasibility of a memory access (e.g. label, modelabel or channel)
 * 
 * <ul>
 * <li>Checks if a memory access is feasible for simulation. The test requires some model
 * meandering: 
 * - find the memory the label is mapped to (memory mapping) 
 * - find processing units that the task calling the host runnable may be
 *   assigned to (task & scheduler mapping) 
 * - check if all processing units have access to the memory, 
 * if yes -> success, else -> fail
 * </li> 
 * </ul>
 */

@Validation(id = "Sim-Software-LabelAccessFeasibility", checks = {
		"Checks if a label access can be performed from certain runnable, as a hosting processing unit must be able to access the label's memory location." })

class SimSoftwareMemoryAccessFeasibility{

	public static boolean validate(List<ValidationDiagnostic> results, ActivityGraphItem activityGraphItem,
			final AbstractMemoryElement abstractMemoryElement) {
		//store all memories the label is mapped to (1 entry expected)
		final Set<Memory> memories = DeploymentUtil.getMapping(abstractMemoryElement);
		if (memories.isEmpty() || memories.size() > 1) {
			/*
			 * No memory location was found for this label. As this check is part of another validation we will
			 * return at this point without adding an issue. This validation ma be re-run when the mapping is 
			 * complete.
			 */
			return true;
		}
		
		// store all processing units
		// label accesses might be contained in activity graphs within runnables or
		// processes (tasks or ISRs)
		HashSet<ProcessingUnit> processingUnits = new HashSet<ProcessingUnit>();
		if (activityGraphItem.getContainingRunnable() != null) {
			EList<RunnableCall> runnableCalls = activityGraphItem.getContainingRunnable().getRunnableCalls();
			for (RunnableCall runnableCall : runnableCalls) {
				if (runnableCall.getContainingProcess() != null) {
					processingUnits.addAll(
							DeploymentUtil.getAssignedCoreForProcess(runnableCall.getContainingProcess()));
				}
			}
		} else if (activityGraphItem.getContainingProcess() != null) {
			processingUnits.addAll(
					DeploymentUtil.getAssignedCoreForProcess(activityGraphItem.getContainingProcess()));
		} else if (activityGraphItem.eContainingFeature() instanceof Scheduler) {
			Scheduler scheduler = (Scheduler) activityGraphItem.eContainingFeature();
			for (SchedulerAllocation schedulerAllocation : AmaltheaIndex.getReferringObjects(scheduler,
					SchedulerAllocation.class)) {
				processingUnits.addAll(schedulerAllocation.getResponsibility());
			}
		}					
		
		if (processingUnits.isEmpty()) {
			/*
			 * 	No executing was found for the (task/ISR) contexts in which this activity item's runnable may be launched.
			 *	This is not per se a problem within this validation since the mapping of tasks to schedulers, and schedulers to cores
			 * 	is handled in other validations. However it prevents this validation from checking, if a valid hw access element has been defined.
			 *	Therefore we return at this point without adding issues, only when the mappings are complete, this needs to be evaluated as 
			 * 	a task/scheduler combination that is not fully mapped will not be brought to execution. When the mapping is complete this validation 
			 * 	may be re-run to completion.
			 *
			*/
			return true;
		}

		
		for (ProcessingUnit processingUnit : processingUnits) {
			for (Memory memory : memories) {
				Set<HwAccessElement> hwAccessElements = 
						AmaltheaIndex.getReferringObjects(memory,HwAccessElement.class);
				boolean foundAccessElement = false;
				for (HwAccessElement hwAccessElement : hwAccessElements) {
					if (hwAccessElement.getSource() == processingUnit) {
						foundAccessElement = true;
						break;
					}
				}
				if (!foundAccessElement) { // unable to locate and access element that links Processing unit and memory
					return false;
				}
			}
		}
		return true;
	}

}
