# frozen_string_literal: true

require "singleton"

module GitlabQuality
  module TestTooling
    module TestMetricsExporter
      class Config
        include Singleton

        # GCS client configuration object used to push metrics as json file to gcs bucket
        #
        class GCS
          def initialize(project_id:, credentials:, bucket_name:, metrics_file_name:, dry_run: false)
            @project_id = project_id
            @credentials = credentials
            @bucket_name = bucket_name
            @metrics_file_name = metrics_file_name
            @dry_run = dry_run
          end

          attr_reader :project_id, :credentials, :bucket_name, :metrics_file_name, :dry_run
        end

        # ClickHouse client configuration object
        #
        class ClickHouse
          def initialize(url:, database:, table_name:, username:, password:)
            @url = url
            @database = database
            @table_name = table_name
            @username = username
            @password = password
          end

          attr_reader :url, :database, :table_name, :username, :password
        end

        class << self
          def configuration
            Config.instance
          end

          def configure
            yield(configuration)
          end
        end

        attr_reader :gcs_config, :clickhouse_config, :initial_run
        attr_accessor :run_type
        attr_writer :extra_rspec_metadata_keys,
          :skip_record_proc,
          :test_retried_proc,
          :custom_metrics_proc,
          :spec_file_path_prefix,
          :logger

        # rubocop:disable Style/TrivialAccessors -- allows documenting that setting config enables the export as well as document input class type

        # Enable metrics export to gcs bucket by setting configuration object
        #
        # @param config [Config::GCS]
        # @return [GCS]
        def gcs_config=(config)
          @gcs_config = config
        end

        # Enable metrics export to clickhouse by setting configuration object
        #
        # @param config [Config::ClickHouse]
        # @return [ClickHouse]
        def clickhouse_config=(config)
          @clickhouse_config = config
        end

        # Additional columns to be created in the table if initial_run setup is used
        # Columns should be defined in the format used for ALTER TABLE query, example;
        #   [
        #     "feature_category LowCardinality(String) DEFAULT ''",
        #     "level LowCardinality(String) DEFAULT ''"
        #   ]
        #
        # @param columns [Array]
        # @return [Array]
        def extra_metadata_columns=(columns)
          @extra_metadata_columns = columns
        end

        # rubocop:enable Style/TrivialAccessors

        # Marks execution as initial run and performs setup tasks before running tests, like creating database in ClickHouse
        #
        # @return [Boolean]
        def initial_run!
          @initial_run = true
        end

        # Additional metadata columns used during initial table creation
        #
        # @return [Array]
        def extra_metadata_columns
          @extra_metadata_columns ||= []
        end

        # Extra rspec metadata keys to include in exported metrics
        #
        # @return [Array<Symbol>]
        def extra_rspec_metadata_keys
          @extra_rspec_metadata_keys ||= []
        end

        # Extra path prefix for constructing full file path within mono-repository setups
        #
        # @return [String]
        def spec_file_path_prefix
          @spec_file_path_prefix ||= ""
        end

        # A lambda that determines whether to skip recording a test result
        #
        # This is useful when you would want to skip initial failure when retrying specs is set up in a separate process
        # and you want to avoid duplicate records
        #
        # @return [Proc]
        def skip_record_proc
          @skip_record_proc ||= ->(_example) { false }
        end

        # A lambda that determines whether a test was retried or not
        #
        # @return [Proc]
        def test_retried_proc
          @test_retried_proc ||= ->(_example) { false }
        end

        # A lambda that return hash with additional custom metrics
        #
        # @return [Proc]
        def custom_metrics_proc
          @custom_metrics_proc ||= ->(_example) { {} }
        end

        # Logger instance
        #
        # @return [Logger]
        def logger
          @logger ||= Logger.new($stdout, level: Logger::INFO)
        end
      end
    end
  end
end
