# frozen_string_literal: true

require 'time'

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      class Evaluation
        include Concerns::IssueConcern
        include Concerns::WorkItemConcern

        BASE_LABELS_FOR_SEARCH = ['feature::addition'].freeze
        FEATURE_READINESS_TRACKING_LABEL = 'tracking feature readiness'

        def initialize(token:, project: nil, group: nil, limit_to_minutes: nil, epic_iid: nil, search_labels: [], dry_run: false)
          @token = token
          @project = "#{group}/#{project}"
          @group = group
          @limit_to_minutes = limit_to_minutes
          @epic_iid = epic_iid
          @search_labels = search_labels
          @dry_run = dry_run
          @analyzed_epics = []
        end

        def invoke!
          created_after = utc_time_minus_mins(limit_to_minutes)

          epics = fetch_epics(created_after)

          epics.compact.each do |epic|
            @analyzed_epics << process_epic(epic)
          rescue StandardError => e
            puts "ERROR processing epic #{epic[:epic_web_url]} due to: #{e}"
          end

          report_epics(analyzed_epics)
        end

        private

        attr_reader :token, :project, :dry_run, :group, :limit_to_minutes, :epic_iid, :search_labels, :analyzed_epics

        def work_items_client
          @work_items_client ||= (dry_run ? GitlabClient::WorkItemsDryClient : GitlabClient::WorkItemsClient).new(token: token, project: project, group: group)
        end

        def fetch_epics(created_after)
          return [fetch_work_item(epic_iid, work_items_client, [:hierarchy, :labels])] if epic_iid

          work_items_client.paginated_call(:group_work_items,
            labels: search_labels.concat(BASE_LABELS_FOR_SEARCH).uniq, state: 'opened', created_after: created_after)
        end

        def label_client
          @label_client ||= GitlabClient::LabelsClient.new(token: token, project: project)
        end

        def process_epic(epic)
          epic = fetch_work_item(epic[:iid], work_items_client, [:hierarchy, :labels])

          AnalyzedItems::AnalyzedEpic.new(epic: epic, token: token, project: project, group: group, dry_run: dry_run)
                                     .tap(&:analyze).result
        end

        def utc_time_minus_mins(mins)
          (Time.now - (mins * 60)).utc.iso8601 if mins
        end

        def report_epics(epics)
          epics.each do |epic|
            add_labels(ids_for_labels([FEATURE_READINESS_TRACKING_LABEL], label_client), epic[:epic_id], work_items_client)

            Report::FeatureReadiness::ReportOnEpic.report(epic, work_items_client)
          rescue StandardError => e
            puts "ERROR reporting epic #{epic[:epic_web_url]} due to: #{e}"
          end
        end
      end
    end
  end
end
