﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/license-manager-user-subscriptions/LicenseManagerUserSubscriptionsRequest.h>
#include <aws/license-manager-user-subscriptions/LicenseManagerUserSubscriptions_EXPORTS.h>
#include <aws/license-manager-user-subscriptions/model/IdentityProvider.h>
#include <aws/license-manager-user-subscriptions/model/Settings.h>

#include <utility>

namespace Aws {
namespace LicenseManagerUserSubscriptions {
namespace Model {

/**
 */
class RegisterIdentityProviderRequest : public LicenseManagerUserSubscriptionsRequest {
 public:
  AWS_LICENSEMANAGERUSERSUBSCRIPTIONS_API RegisterIdentityProviderRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RegisterIdentityProvider"; }

  AWS_LICENSEMANAGERUSERSUBSCRIPTIONS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>An object that specifies details for the identity provider to register.</p>
   */
  inline const IdentityProvider& GetIdentityProvider() const { return m_identityProvider; }
  inline bool IdentityProviderHasBeenSet() const { return m_identityProviderHasBeenSet; }
  template <typename IdentityProviderT = IdentityProvider>
  void SetIdentityProvider(IdentityProviderT&& value) {
    m_identityProviderHasBeenSet = true;
    m_identityProvider = std::forward<IdentityProviderT>(value);
  }
  template <typename IdentityProviderT = IdentityProvider>
  RegisterIdentityProviderRequest& WithIdentityProvider(IdentityProviderT&& value) {
    SetIdentityProvider(std::forward<IdentityProviderT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user-based subscription product.</p> <p>Valid values:
   * <code>VISUAL_STUDIO_ENTERPRISE</code> | <code>VISUAL_STUDIO_PROFESSIONAL</code>
   * | <code>OFFICE_PROFESSIONAL_PLUS</code> | <code>REMOTE_DESKTOP_SERVICES</code>
   * </p>
   */
  inline const Aws::String& GetProduct() const { return m_product; }
  inline bool ProductHasBeenSet() const { return m_productHasBeenSet; }
  template <typename ProductT = Aws::String>
  void SetProduct(ProductT&& value) {
    m_productHasBeenSet = true;
    m_product = std::forward<ProductT>(value);
  }
  template <typename ProductT = Aws::String>
  RegisterIdentityProviderRequest& WithProduct(ProductT&& value) {
    SetProduct(std::forward<ProductT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The registered identity provider’s product related configuration settings
   * such as the subnets to provision VPC endpoints.</p>
   */
  inline const Settings& GetSettings() const { return m_settings; }
  inline bool SettingsHasBeenSet() const { return m_settingsHasBeenSet; }
  template <typename SettingsT = Settings>
  void SetSettings(SettingsT&& value) {
    m_settingsHasBeenSet = true;
    m_settings = std::forward<SettingsT>(value);
  }
  template <typename SettingsT = Settings>
  RegisterIdentityProviderRequest& WithSettings(SettingsT&& value) {
    SetSettings(std::forward<SettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags that apply to the identity provider's registration.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  RegisterIdentityProviderRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  RegisterIdentityProviderRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  IdentityProvider m_identityProvider;

  Aws::String m_product;

  Settings m_settings;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_identityProviderHasBeenSet = false;
  bool m_productHasBeenSet = false;
  bool m_settingsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace LicenseManagerUserSubscriptions
}  // namespace Aws
