/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.configuration2.builder;

import org.apache.commons.configuration2.ImmutableConfiguration;
import org.apache.commons.configuration2.event.EventType;

/**
 * <p>
 * A specialized event class which is generated by a {@link ConfigurationBuilder} when a result configuration has been
 * created.
 * </p>
 * <p>
 * Events of this type are fired in the {@code getConfiguration()} method of a configuration builder each time a new
 * result object is created. At the time the event is fired, no lock is held. The newly created
 * {@code ImmutableConfiguration} object is available as a property of this event.
 * </p>
 * <p>
 * A use case for this event is to perform special initializations on newly created configuration objects. It is also an
 * indication that a builder is now fully initialized; i.e. the managed configuration is available.
 * </p>
 *
 * @since 2.0
 */
public class ConfigurationBuilderResultCreatedEvent extends ConfigurationBuilderEvent {

    /**
     * The specialized event type for a newly created result configuration. Events of this type are generated by a
     * configuration builder when a result configuration has been created.
     */
    public static final EventType<ConfigurationBuilderResultCreatedEvent> RESULT_CREATED = new EventType<>(ANY, "RESULT_CREATED");

    // SpotBugs
    // [ERROR] Medium: Class org.apache.commons.configuration2.builder.ConfigurationBuilderResultCreatedEvent defines
    // non-transient non-serializable instance field configuration
    // [org.apache.commons.configuration2.builder.ConfigurationBuilderResultCreatedEvent] In
    // ConfigurationBuilderResultCreatedEvent.java SE_BAD_FIELD
    //
    // private static final long serialVersionUID = 1L;

    /** The newly created configuration object. */
    private final ImmutableConfiguration configuration;

    /**
     * Creates a new instance of {@code ConfigurationBuilderResultCreatedEvent} and initializes its properties.
     *
     * @param source the {@code ConfigurationBuilder} object which triggered this event (must not be <strong>null</strong>)
     * @param evType the type of this event (must not be <strong>null</strong>)
     * @param createdConfiguration the newly created {@code ImmutableConfiguration} object (must not be <strong>null</strong>)
     * @throws IllegalArgumentException if a required parameter is null
     */
    public ConfigurationBuilderResultCreatedEvent(final ConfigurationBuilder<?> source,
        final EventType<? extends ConfigurationBuilderResultCreatedEvent> evType, final ImmutableConfiguration createdConfiguration) {
        super(source, evType);
        if (createdConfiguration == null) {
            throw new IllegalArgumentException("Configuration must not be null!");
        }
        configuration = createdConfiguration;
    }

    /**
     * Gets the newly created {@code ImmutableConfiguration} object.
     *
     * @return the newly created {@code ImmutableConfiguration}
     */
    public ImmutableConfiguration getConfiguration() {
        return configuration;
    }
}
