package Lire::Config::Plugin;

use strict;

use base qw/Lire::Config::Dictionary/;

use Lire::Config;
use Lire::Config::RecordSpec;
use Lire::Config::Dictionary;

use Carp;

=pod

=head1 NAME

Lire::Config::Plugin - Value object for plugin configuration.

=head1 SYNOPSIS

  use Lire::Config::Plugin;

=head1 DESCRIPTION

This configuration object hold the name of a selected plugin as
well as its options.

=cut

sub new {
    my $self = shift->SUPER::new( @_ );

    my %args = @_;

    $self->set_plugin( $args{'value'} )
      unless ( exists $self->{'_plugin'} && ! defined $args{'value'} );

    return $self;
}

=pod

=head2 get_plugin()

Returns the currently selected plugin.

=cut

sub get_plugin {
    return $_[0]{'_plugin'};
}

=pod

=head2 set_plugin( $plugin )

Changes the selected plugin. If the selected plugin is changed, a
new property set will be created from the plugin's defined properties.

=cut

sub set_plugin {
    my ( $self, $plugin ) = @_;

    my $old_plugin = $self->{'_plugin'};
    $self->{'_plugin'} = $plugin;
    my $spec = $self->get_properties_spec();
    $self->{'_properties'} = $spec->instance()
      unless ( defined $plugin && defined $old_plugin
               && $plugin eq $old_plugin );

    return;
}

=pod

=head2 get_properties_spec()

Returns the TypeSpec that is used to specify the plugin's properties.
This method will return an empty RecordSpec when the plugin didn't
define any properties. One can also use the has_properties() method to
check if the Plugin defined configuration properties.

=cut

sub get_properties_spec {
    my $self = $_[0];

    return $self->spec()->get_properties_spec( $self->{'_plugin'} );
}

=pod

=head2 has_properties()

Returns a boolean value indicating whether the current plugin is configurable.

=cut

sub has_properties {
    return  $_[0]->get_properties_spec()->components() > 0;
}

=pod

=head2 get_properties()

Returns the Lire::Config::Dictionary object which hold the
plugin configuration.

=cut

sub get_properties {
    return $_[0]{'_properties'};
}

=pod

=head2 as_value()

Returns an hash reference with two keys : 'plugin' which contains
the selected plugin and 'properties' which is an hash reference containing
the plugin's properties.

=cut

sub as_value {
    return { 'plugin' => $_[0]->{'_plugin'},
             'properties' => $_[0]->{'_properties'}->as_value() };
}

=pod

=head2 get( $name )

Delegates to the Dictionary which contains the plugin's properties.

=cut

sub get {
    return shift->{'_properties'}->get( @_ );
}

=pod

=head2 set( $name, $value )

Delegates to the Dictionary which contains the plugin's properties.

=cut

sub set {
    return shift->{'_properties'}->set( @_ );
}

=pod

=head2 is_set( $name )

Delegates to the Dictionary which contains the plugin's properties.

=cut

sub is_set {
    return shift->{'_properties'}->is_set( @_ );
}

sub is_equals {
    my ( $self, $other ) = @_;

    return 0 unless $self->Lire::Config::Value::is_equals( $other );

    no warnings 'uninitialized';
    return 0 unless $self->{'_plugin'} eq $other->{'_plugin'};

    return $self->{'_properties'}->is_equals( $other->{'_properties'} );
}

sub is_valid {
    my $self = $_[0];

    return ( $self->spec()->is_valid( $self->{'_plugin'} )
             && $self->get_properties()->is_valid() );
}

sub save_xml {
    my ( $self, $fh, $indent, $xmlns ) = @_;

    $indent ||= 0;
    $xmlns ||= '';
    return unless ( defined $self->{'_plugin'} );
    return if $self->is_default() || $self->spec()->obsolete();

    print $fh '  ' x $indent, "<${xmlns}param name=\"", $self->name(), '" value="',
      $self->{'_plugin'}, '">', "\n";
    my $spec = $self->get_properties_spec();
    foreach my $comp ( $spec->components() ) {
        $self->get( $comp->name() )->save_xml( $fh, $indent + 1, $xmlns );
    }
    print $fh '  ' x $indent, "</${xmlns}param>\n";
}

1;
