#!/usr/bin/env python3
# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-

# This is a fake SMTP server just enough to test SMTP AUTH LOGIN with
# minimal replies. It rejects DATA, VRFY, MAIL and RCPT commands.
# It is not currently shipped or hooked to the tests.

# Copyright (C) 2019 Bryce W. Harrington
# Copyright (C) 2022-2025 Matthias Andree
#
# Released under GNU GPLv2 or later, read the file 'LICENSE.GPLv2+' for
# more information.
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
# Author:  Bryce Harrington <bryce@canonical.com>

from base64 import b64decode
import os
import sys
import socket
from pathlib import Path

DEBUGGING = True
DEFAULT_HOST = ''
DEFAULT_PORT = 11025
NEWLINE = b"\r\n"

def dbg(msg):
    """Prints information to stdout if debugging is enabled"""
    if DEBUGGING:
        sys.stdout.write("{}\n".format(msg))

def command_ehlo(data, _):
    return b'250-smtp.example.org\r\n250 AUTH LOGIN'

def command_rset(data, _):
    return b'250 OK'

def command_helo(data, _):
    return b'250 smtp.example.org'

def command_quit(data, _):
    return b'221 bye'

def command_auth(data, mailserver):
    l = data.split(None, 4)
    if len(l) < 2:
        return b'501 Syntax error, AUTH mech'
    if l[1].upper() != b'LOGIN':
        return b'535 Unsupported AUTH mechanism'
    if len(l) > 2:
        return b'535 Garbage in command'
    mailserver.send(b'334 ')
    x = mailserver.receive().rstrip(b'\r\n')
    if x == b'*':
        return b'501 AUTH aborted'
    try:
        un = b64decode(x)
        dbg(f"  - Login {un}")
    except:
        return b"501 Uh - can't decode login"
    mailserver.send(b'334 ')
    x = mailserver.receive().rstrip(b'\r\n')
    try:
        pw = b64decode(x)
        dbg(f"  - Password {pw}")
    except:
        return b"501 Uh - can't decode secret"
    finally:
        del pw
    return b'235 Right, go ahead'

COMMANDS = {
    b'HELO' : command_helo,
    b'EHLO' : command_ehlo,
    b'AUTH' : command_auth,
    b'RSET' : command_rset,
    b'QUIT' : command_quit,
}

class Mailserver:
    def __init__(self, conn, mbox):
        self._conn = conn
        self._mbox = mbox
        self._chunk_size = 4096

    def close(self):
        dbg("  - Server exiting")
        self._conn.close()

    def send(self, data):
        dbg("  - Server sending")
        self._conn.sendall(data + NEWLINE)

    def receive(self):
        dbg("  - Server receiving")
        data = []
        while True:
            chunk = self._conn.recv(self._chunk_size)
            if NEWLINE in chunk:
                data.append(chunk[:chunk.index(NEWLINE)])
                break
            data.append(chunk)
        return b"".join(data)

    def process(self):
        data = self.receive()
        dbg("  - Received: '{}'".format(data))
        command = data.split(None, 1)[0].upper()
        if command in COMMANDS.keys():
            response = COMMANDS[command](data, self)
        else:
            response = b'500 What?'
        try:
                dbg("  - Response: {}".format(response))
                self.send(response)
                if command == b'QUIT':
                        return False
        except BrokenPipeError:
                dbg("  - Client terminated connection")
                return False
        return True

def serve(messages_filename, host=DEFAULT_HOST, port=DEFAULT_PORT):
    dbg("Serving for {} on {}".format(host, port))

    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    sock.bind((host, port))

    try:
        sock.listen(1)
        dbg("* mock-smtp-server ready on {}:{}".format(host, port))
        while True:
            sock.listen(1)
            startmarker = os.environ.get("STARTMARKER")
            if startmarker:
                try:
                    Path(startmarker).touch()
                except FileNotFoundError:
                    # work directory has gone away, so we have not
                    # started in time.
                    os.exit(1)
            conn, address = sock.accept()
            dbg("* Connection from {}".format(address))
            server = Mailserver(conn, None)
            server.send(b"220 mock-smtp-server smtp server ready")
            try:
                dbg("* Processing commands for connection...")
                while server.process():
                    dbg("* Finished command")
                    pass
            finally:
                dbg("* Closing remote connection")
                server.close()
    except KeyboardInterrupt:
        dbg("* mock-smtp-server interrupted")
        return 1
    except SystemExit:
        dbg("* mock-smtp-server exiting")
        return 1
    finally:
        sock.shutdown(socket.SHUT_RDWR)
        sock.close()
    return 0


if __name__ == "__main__":
    try:
        exit(serve(None, host=DEFAULT_HOST, port=DEFAULT_PORT))
    except KeyError:
        sys.stderr.write("Exiting\n")
        sys.exit(1)
